/* util.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001-2008 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "util.h"
#include "lowlevelfunc.h"
#include <cctype>
#include <algorithm>
#include <stdarg.h>
#include <functional>

int createLines(const char *text,char ***ret_lines)
{
  int lines;
  int i, p;
  short ignorenext;

  lines=1;
  ignorenext = 0;
  for(i=0;i<(int)strlen(text);i++) {
    if ( ignorenext == 0 ) {
      if(text[i]=='|') lines++;
      else if ( text[i] == '\\' ) ignorenext = 1;
    } else {
      ignorenext = 0;
    }
  }
  char **liness=(char**)_allocsafe(sizeof(char*)*lines);
  if(lines==1) {
    liness[0]=dupstring(text);
  } else {
    char *tstr1,*tstr2;;
    tstr1=dupstring(text);
    for(i=0;i<lines;i++) {
      liness[i]=dupstring(tstr1);
      int j;
      ignorenext = 0;
      for(j=0;j<(int)strlen(liness[i]);j++) {
	if ( ignorenext == 0 ) {
	  if(liness[i][j]=='|') {
	    liness[i][j]=0;
	    break;
	  } else if ( liness[i][j] == '\\' ) ignorenext = 1;
	} else {
	  ignorenext = 0;
	}
      }
      for ( p = j = 0; liness[i][j] != '\0'; j++ ) {
        if ( ( liness[i][j] == '\\' ) && ( liness[i][j+1] == '|' ) ) {
	  liness[i][p++] = liness[i][++j];
        } else if ( ( liness[i][j] == '\\' ) && ( liness[i][j+1] == '\\' ) ) {
	  liness[i][p++] = liness[i][++j];
	} else {
	  liness[i][p++] = liness[i][j];
	}
      }
      liness[i][p] = '\0';
      if((i+1)>=lines) break; //last line
      tstr2=dupstring(tstr1+j+1);
      _freesafe(tstr1);
      tstr1=tstr2;
    }
    _freesafe(tstr1);
  }
  *ret_lines=liness;
  return lines;
}

namespace AGUIXUtils
{
    std::string bytes_to_human_readable( loff_t number )
    {
        std::string return_str;
        
        if ( number >= 1 * 1024 * 1024 * 1024 ) {
            MakeLong2NiceStr( number / ( 1024 * 1024 * 1024 ), return_str, false );
            return_str += " GB";
        } else if ( number >= 1 * 1024 * 1024 ) {
            MakeLong2NiceStr( number / ( 1024 * 1024 ), return_str, false );
            return_str += " MB";
        } else if ( number >= 1 * 1024 ) {
            MakeLong2NiceStr( number / 1024, return_str, false );
            return_str += " KB";
        } else {
            MakeLong2NiceStr( number, return_str, false );
            return_str += " B";
        }
        
        return return_str;
    }
    
    template <class T>
    struct my_tolower_func : public std::unary_function< T, T > 
    {
        T operator() ( const T& c ) const 
        {
            return std::tolower( c );
        }
    };

    std::string tolower( const std::string &str )
    {
        std::string res( str );
        
        std::transform( res.begin(),
                        res.end(),
                        res.begin(),
                        my_tolower_func<char>() );
        return res;
    }

    bool stringIsShorter( const char *str, int len )
    {
        if ( str == NULL ) return true;
        
        int my_len;
        
        for ( my_len = 0; str[my_len] != '\0' && my_len < len; my_len++ );
        
        if ( my_len < len ) return true;
        return false;
    }

    std::string formatStringToString( const char *format, ... )
    {
        va_list va;
        int n, size = 1024;
        char *tstr;
        std::string string1;
        
        if ( format != NULL ) {
            // first create str
            while ( 1 ) {
                tstr = (char*)_allocsafe( size );
                va_start( va, format );
                n = vsnprintf( tstr, size, format, va );
                va_end( va );
                if ( n > -1 && n < size ) break;
                _freesafe( tstr );
                size *= 2;
            }
            string1 = tstr;
            _freesafe( tstr );
            return string1;
        }
        return "";
    }

}
