/* 
   wipe2fs - securely wipe unused space in ext2/3 filesystems

   Copyright (C) 2003-5 Chuan-kai Lin

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

*/

#include <argp.h>
#include <errno.h>
#include <locale.h>
#include <stdio.h>
#include <string.h>
#include <ext2fs/ext2fs.h>
#include <sys/types.h>
#include "config.h"

#define min(a,b) ((a)<(b)?(a):(b))
#define EXIT_FAILURE 1

#if ENABLE_NLS
# include <libintl.h>
# define _(Text) gettext (Text)
#else
# undef bindtextdomain
# define bindtextdomain(Domain, Directory)
# undef textdomain
# define textdomain(Domain)
# undef locale
# define locale(Domain)
# define _(Text) Text
#endif
#define N_(Text) Text

char *xmalloc ();
char *xrealloc ();
char *xstrdup ();

static error_t parse_opt (int key, char *arg, struct argp_state *state);
static void show_version (FILE *stream, struct argp_state *state);

/* argp option keys */
enum {
  DUMMY_KEY=129
};

/* Option flags and variables.  These are initialized in parse_opt.  */
static struct argp_option options[] =
{
  { "pass", 'p', "PASS", 0,
    N_("set the number of residual wipe passes"), 0},
  { "yes", 'y', NULL, 0, N_("disable confirmation query"), 0},
  { NULL, 0, NULL, 0, NULL, 0 }
};

/* The argp functions examine these global variables.  */
const char *argp_program_bug_address = "<cklin@cs.pdx.edu>";
void (*argp_program_version_hook) (FILE *, struct argp_state *)
     = show_version;

static struct argp argp =
{
  options, parse_opt, N_("DEVICE ..."),
  N_("securely wipe unused space in ext2/3 filesystems"),
  NULL, NULL, NULL
};

#define RAND_DEVICE "/dev/random"

static int arg_start = 0;
static int query = 1;
static int passes = 0;

static ext2_filsys fsys = NULL;
static unsigned int blk_size;
static unsigned int addr_pblk, addr_pblk_2;
static char *media;
static unsigned int af_sbox[256];
static unsigned int af_i, af_j;

/* Seek user confirmation and abort program if not acknowledged.  To be
   on the safe side, anything different from "yes" is taken as "no".  */
void
confirm_or_abort(int argc, char **argv)
{
  int index;
  char answer[6];

  printf(_("You wish to wipe unused space in "));
  printf("%s", argv[arg_start]);
  for (index=arg_start+1; index<argc; index++)
    printf(", %s", argv[index]);
  printf(_(".\nIs that correct [yes/no]? "));

  fgets(answer, 6, stdin);
  if (strcmp(answer, "yes\n")) {
    puts(_("Thank you, aborting operation at user request"));
    exit(2);
  }
  puts(_("User confirmation acknowledged, proceed to wipe operation"));
}

/* Set up the S-Box of the Arcfour PRNG with RAND_DEVICE.  Return 0
   on success and return 1 otherwise.  Successful completion of the
   procedure does not depend on the success of closing the random
   source, however error messages would still be issued.  */
int
arcfour_init(void)
{
  unsigned int af_temp;
  unsigned char af_key[256];
  FILE *rand_source;

  rand_source = fopen(RAND_DEVICE, "r");
  if (!rand_source) {
    fprintf(stderr, _("Error opening random source %s: %s\n"),
	    RAND_DEVICE, strerror(errno));
    return 1;
  }

  if (fread(af_key, 1, 256, rand_source) < 256) {
    fprintf(stderr, _("Error reading from random source %s: %s\n"),
	    RAND_DEVICE, strerror(errno));
    if (fclose(rand_source))
      fprintf(stderr, _("Error closing random source %s: %s\n"),
	      RAND_DEVICE, strerror(errno));
    return 1;
  }

  for (af_i=0; af_i<256; af_i++)
    af_sbox[af_i] = af_i;

  for (af_i=af_j=0; af_i<256; af_i++) {
    af_j = (af_j+af_key[af_i]+af_sbox[af_i]) % 256;
    af_temp = af_sbox[af_i];
    af_sbox[af_i] = af_sbox[af_j];
    af_sbox[af_j] = af_temp;
  }

  memset(af_key, 0, 256);
  if (fclose(rand_source))
    fprintf(stderr, _("Error closing random source %s: %s\n"),
	    RAND_DEVICE, strerror(errno));

  return 0;
}

/* Fill the buffer with pseudo-random strings.  */
void
arcfour_fill(unsigned char *buffer, unsigned int size)
{
  unsigned int af_d, af_temp;

  while (size--) {
    af_i = (af_i+1) % 256;
    af_j = (af_j+af_sbox[af_i]) % 256;
    af_temp = af_sbox[af_i];
    af_sbox[af_i] = af_sbox[af_j];
    af_sbox[af_j] = af_temp;
    af_d = (af_sbox[af_i]+af_sbox[af_j]) % 256;
    buffer[size] = (unsigned char) af_sbox[af_d];
  }
}

/* Fill the buffer with zeros.  */
void
zero_fill(unsigned char *buffer, unsigned int size)
{
  memset(buffer, 0, size);
}

/* Close an open ext2 filesystem and return 0 on success.  */
int
ext2_close(void)
{
  errcode_t ecode;

  ecode = ext2fs_close(fsys);
  fsys = NULL;

  if (ecode) {
    fprintf(stderr, _("Error closing filesystem %s: %s\n"),
	    media, error_message(ecode));
    return 1;
  }
  return 0;
}

/* Open an existing ext2 filesystem.  Return 0 on success and store
   handle in FSYS, otherwise print error message and return 1.  */
int
ext2_open(void)
{
  errcode_t ecode;

  ecode = ext2fs_open
    (media, EXT2_FLAG_RW, 0, 0, unix_io_manager, &fsys);
  if (ecode) {
    fprintf(stderr, _("Error opening filesystem %s: %s\n"),
	    media, error_message(ecode));
    return 1;
  }

  ecode = ext2fs_read_bitmaps(fsys);
  if (ecode) {
    fprintf(stderr, _("Error reading bitmaps %s: %s\n"),
	    media, error_message(ecode));
    ext2_close();
    return 1;
  }

  blk_size = fsys->blocksize;
  addr_pblk = blk_size / sizeof (blk_t);
  addr_pblk_2 = addr_pblk*addr_pblk;

  return 0;
}

/* Translate 32-bit little endian integers to native byte order.  */
blk_t
ltohl(const unsigned char littlei[])
{
  return (((blk_t) littlei[0])<<0) + (((blk_t) littlei[1])<<8)
    + (((blk_t) littlei[2])<<16) + (((blk_t) littlei[3])<<24);
}

/* Lookup physical block address for the Nth data block of a file.
   Return 0 on error, or the block number otherwise.  */
blk_t
data_block_lookup(struct ext2_inode *inode, unsigned int index)
{
  errcode_t ecode;
  unsigned int bound;
  unsigned int blk_count;
  blk_t ind_addr, dind_addr, blk_addr;
  unsigned char *table;

  /* Sanity check: in-file block index must be in range.  */
  blk_count = (inode->i_size+blk_size-1)/blk_size;
  if (index >= blk_count)  return 0;

  /* Direct blocks: the first EXT2_NDIR_BLOCKS only.  */
  if (index < EXT2_NDIR_BLOCKS)
    return inode->i_block[index];

  table = (unsigned char *) xmalloc(blk_size);
  index -= EXT2_NDIR_BLOCKS;
  dind_addr = inode->i_block[EXT2_DIND_BLOCK];
  ind_addr = inode->i_block[EXT2_IND_BLOCK];
  blk_addr = 0;

  /* Triple-indirect blocks: reduce to double-indirect case.
     index >= addr_pblk+addr_pblk_2  */
  bound = addr_pblk+addr_pblk_2;
  if (index >= bound) {
    index -= bound;
    ecode = io_channel_read_blk
      (fsys->io, inode->i_block[EXT2_TIND_BLOCK], 1, table);
    if (ecode) {
      fprintf(stderr, _("Error reading triple-indirect block %d: %s\n"),
	      inode->i_block[EXT2_TIND_BLOCK], error_message(ecode));
      goto lookup_terminate;
    }
    dind_addr = ltohl(table+4*(index/addr_pblk_2));
    index = (index%addr_pblk_2)+addr_pblk;
  }

  /* Double-indirect blocks: reduce to single-indirect case.
     index >= addr_pblk && index < addr_pblk+addr_pblk_2  */
  bound = addr_pblk;
  if (index >= bound) {
    index -= bound;
    ecode = io_channel_read_blk(fsys->io, dind_addr, 1, table);
    if (ecode) {
      fprintf(stderr, _("Error reading double-indirect block %d: %s\n"),
	      dind_addr, error_message(ecode));
      goto lookup_terminate;
    }
    ind_addr = ltohl(table+4*(index/addr_pblk));
    index = index%addr_pblk;
  }

  /* Single-indirect blocks: lookup data block address.
     index < addr_pblk  */
  ecode = io_channel_read_blk(fsys->io, ind_addr, 1, table);
  if (ecode) {
    fprintf(stderr, _("Error reading indirect block %d: %s\n"),
	    ind_addr, error_message(ecode));
    goto lookup_terminate;
  }
  blk_addr = ltohl(table+4*index);

 lookup_terminate:
  free(table);
  return blk_addr;
}

/* Wipe out information in file slack space on the filesystem with the
   given stream generator.  Return 0 unless there are major errors.  */
int
wipe_fileslack(void)
{
  errcode_t ecode;
  ext2_inode_scan scan;
  struct ext2_inode inode;
  ext2_ino_t ino;
  blk_t blk_addr;
  unsigned int blk_index, tail_size;
  void *buffer;

  buffer = xmalloc(blk_size);
  ecode = ext2fs_open_inode_scan(fsys, 0, &scan);
  if (ecode) {
    fprintf(stderr, _("Error initiating inode scan: %s\n"),
	    error_message(ecode));
    free(buffer);
    return 1;
  }

  for ( ; ; ) {
    do ecode = ext2fs_get_next_inode(scan, &ino, &inode);
    while (ecode == EXT2_ET_BAD_BLOCK_IN_INODE_TABLE);

    /* Break when all inodes have been iterated through and skip unused
       inodes or entries with no slack space.  */
    if (!ino)  break;
    if (!ext2fs_test_inode_bitmap(fsys->inode_map, ino))  continue;
    if (inode.i_size%blk_size == 0)  continue;
    if (inode.i_blocks == 0)  continue;

    /* Obtain the address of the last (slack) data block.  */
    blk_index = inode.i_size/blk_size;
    blk_addr = data_block_lookup(&inode, blk_index);
    if (blk_addr==0) {
      fprintf(stderr, _("Error computing inode %u slack address\n"),
	      ino);
      return 1;
    }

    /* Read contents of the last (slack) data block.  */
    ecode = io_channel_read_blk(fsys->io, blk_addr, 1, buffer);
    if (ecode) {
      fprintf(stderr, _("Error reading inode %u slack block %u: %s\n"),
	      ino, blk_addr, error_message(ecode));
      continue;
    }

    /* Overwrite slack in the last data block.  */
    tail_size = inode.i_size%blk_size;
    zero_fill(buffer+tail_size, blk_size-tail_size);
    ecode = io_channel_write_blk(fsys->io, blk_addr, 1, buffer);
    if (ecode) {
      fprintf(stderr, _("Error writing inode %u slack block %u: %s\n"),
	      ino, blk_addr, error_message(ecode));
      return 1;
    }
  }

  ext2fs_close_inode_scan(scan);
  free(buffer);
  return 0;
}

/* Wipe out information in unused blocks on the filesystem with the
   given stream generator.  Return 0 unless there are major errors.  */
int
wipe_freeblock(void)
{
  errcode_t ecode;
  ext2fs_block_bitmap bitmap;
  blk_t start, end, blk_addr;
  void *buffer;

  buffer = xmalloc(blk_size);
  bitmap = fsys->block_map;
  start = bitmap->start;
  end = bitmap->end;

  zero_fill(buffer, blk_size);
  for (blk_addr=start; blk_addr<=end; blk_addr++) {
    if (ext2fs_badblocks_list_test(fsys->badblocks, blk_addr))
      continue;
    if (ext2fs_test_block_bitmap(bitmap, blk_addr))
      continue;
    ecode = io_channel_write_blk(fsys->io, blk_addr, 1, buffer);
    if (ecode) {
      fprintf(stderr, _("Error writing free block %u: %s\n"),
	      blk_addr, error_message(ecode));
      return 1;
    }
  }
  free(buffer);
  return 0;
}

/* Try to process CLEAN_WINDOW_SIZE blocks at a time in clean_residual.
   This takes a bit more memory, but should be much faster and ensure
   random overwrites actually reaches the disk platters instead of
   sitting in the disk write cache.  */
#define CLEAN_WINDOW_SIZE 40

/* Erase (alleged) residual data: first preserve data in blocks, then
   overwrite with pseudo-random strings, and finally write the original
   data back.  This should work even with write cache enabled.  */
int
clean_residual(int passes)
{
  errcode_t ecode;
  blk_t start, end, blk_addr, w_start, w_end;
  void *buffer, *original;
  char *valid;
  int pass_index, blk_index, retval;

  buffer = xmalloc(blk_size);
  original = xmalloc(CLEAN_WINDOW_SIZE * blk_size);
  valid = xmalloc(CLEAN_WINDOW_SIZE);

  start = fsys->block_map->start;
  end = fsys->block_map->end+1;
  retval = 0;

  for (w_start=start; w_start<end; w_start=w_end) {
    w_end = min(w_start+CLEAN_WINDOW_SIZE, end);
    memset(valid, 1, w_end-w_start);

    /* Preserve the original data in the window of blocks.  The valid
       array records which blocks give access errors and therefore
       should not be further processed.  */
    for (blk_addr=w_start; blk_addr<w_end; blk_addr++) {
      blk_index = blk_addr-w_start;
      if (ext2fs_badblocks_list_test(fsys->badblocks, blk_addr)) {
        valid[blk_index] = 0;
        continue;
      }
      ecode = io_channel_read_blk
        (fsys->io, blk_addr, 1, original+blk_index*blk_size);
      if (ecode) {
        fprintf(stderr, _("Error reading block %u: %s\n"),
                blk_addr, error_message(ecode));
        valid[blk_index] = 0;
        continue;
      }
    }

    /* Overwrite with random data in passes with the attempt of
       overflowing the disk cache.  */
    for (pass_index=0; pass_index<passes; pass_index++) {
      for (blk_addr=w_start; blk_addr<w_end; blk_addr++) {
        blk_index = blk_addr-w_start;
        if (!valid[blk_index])  continue;
        arcfour_fill(buffer, blk_size);
        ecode = io_channel_write_blk(fsys->io, blk_addr, 1, buffer);
        if (ecode) {
          fprintf(stderr, _("Error random writing block %u: %s\n"),
                  blk_addr, error_message(ecode));
          valid[blk_index] = 0;
          break;
        }
      }
    }

    /* Restore original data.  */
    for (blk_addr=w_start; blk_addr<w_end; blk_addr++) {
      blk_index = blk_addr-w_start;
      if (!valid[blk_index])  continue;
      ecode = io_channel_write_blk
        (fsys->io, blk_addr, 1, original+blk_index*blk_size);
      if (ecode) {
        fprintf(stderr, _("Error restoring block %u: %s\n"),
                blk_addr, error_message(ecode));
        retval = 1;
      }
    }
  }

  free(buffer);
  free(original);
  free(valid);
  return retval;
}

int
main (int argc, char **argv)
{
  int argp_flags = ARGP_IN_ORDER | ARGP_PARSE_ARGV0;
  int index;
  int ecode, flags;

  setlocale(LC_ALL, "");
  bindtextdomain(PACKAGE, LOCALEDIR);
  textdomain(PACKAGE);
  argp_parse(&argp, argc, argv, argp_flags, NULL, NULL);

  if (query)  confirm_or_abort(argc, argv);
  else  puts(_("User confirmation waived through command-line option"));

  ecode = arcfour_init();
  if (ecode)  return 2;

  for (index=arg_start; index<argc; index++) {
    media = argv[index];

    /* Do not trust the comment in ismounted.c!  The return value of
       ext2fs_check_mount_point (called by ext2fs_check_if_mounted) does
       not tell you if the device is mounted.  Check flags instead.  */
    ecode = ext2fs_check_if_mounted(media, &flags);
    if (ecode) {
      fprintf(stderr, _("Error performing mount checks on %s: %s\n"),
	      media, error_message(ecode));
      continue;
    }
    if (flags & EXT2_MF_MOUNTED) {
      fprintf(stderr, _("Filesystem %s is mounted, skipping...\n"), media);
      continue;
    }

    ecode = ext2_open();
    if (ecode)  continue;

    if ((fsys->super->s_state & EXT2_VALID_FS)==0) {
      fprintf(stderr, _("Filesystem %s is not clean, skipping...\n"),
              media);
      continue;
    }

    printf(_("Filesystem %s initial zeroing pass...\n"), media);
    if (wipe_fileslack())  goto wipe_abort;
    if (wipe_freeblock())  goto wipe_abort;
    if (passes) {
      printf(_("Filesystem %s residual passes...\n"), media);
      if (clean_residual(passes))  goto wipe_abort;
    }
    printf(_("Filesystem %s all done\n"), media);

  wipe_abort:
    ext2_close();
  }

  return 0;
}

/* Parse a single option.  */
static error_t
parse_opt (int key, char *arg, struct argp_state *state)
{
  char *endptr;

  switch (key)  {
  case ARGP_KEY_INIT:
    arg_start = state->argc;
    break;

  case 'p':
    if (arg[0]=='\0')
      argp_error(state, _("number of passes is a null string"));
    passes = strtol(arg, &endptr, 10);
    if (endptr[0])
      argp_error(state, _("number of passes is not an integer"));
    if (passes<0)
      argp_failure(state, 1, 0, _("number of passes cannot be negative"));
    break;

  case 'y':
    query = 0;
    break;

  case ARGP_KEY_ARG:
    return ARGP_ERR_UNKNOWN;

  case ARGP_KEY_ARGS:
    arg_start = state->next;
    break;

  case ARGP_KEY_NO_ARGS:
    argp_failure(state, 1, 0, _("no filesystems to operate on"));
    break;

  default:
    return ARGP_ERR_UNKNOWN;
  }
  return 0;
}

/* Show the version number and copyright information.  */
static void
show_version (FILE *stream, struct argp_state *state)
{
  (void) state;
  /* Print in small parts whose localizations can hopefully be copied
     from other programs.  */
  fputs(PACKAGE" "VERSION"\n", stream);
  fprintf(stream, _("Written by %s.\n\n"), _("Chuan-kai Lin"));
  fprintf(stream, _("Copyright (C) %s %s\n"), "2003-5",
          _("Chuan-kai Lin"));
  fputs(_("\
This program is free software; you may \
redistribute it under the terms of\n\
the GNU General Public License.  This program \
has absolutely no warranty.\n"),
	stream);
}
