/*
 * NodeViewpoint.cpp
 *
 * Copyright (C) 1999 Stephen F. White
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program (see the file "COPYING" for details); if 
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave, 
 * Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include "stdafx.h"

#include "DuneApp.h"
#include "NodeViewpoint.h"
#include "Scene.h"
#include "Proto.h"
#include "FieldValue.h"
#include "SFBool.h"
#include "SFFloat.h"
#include "SFRotation.h"
#include "SFString.h"
#include "SFTime.h"
#include "SFVec3f.h"
#include "Field.h"
#include "ExposedField.h"

ProtoViewpoint::ProtoViewpoint(Scene *scene)
  : Proto(scene, "Viewpoint")
{
    addEventIn(SFBOOL, "set_bind");
    fieldOfView.set(
          addExposedField(SFFLOAT, "fieldOfView", new SFFloat(0.785398f), 
    		          new SFFloat(0.0f), new SFFloat(PI)));
    jump.set(
         addExposedField(SFBOOL, "jump", new SFBool(true)));

    ExposedField* orient = new ExposedField(SFROTATION, "orientation", 
                                            new SFRotation(0, 0, 1, 0));
    orient->setFlags(EIF_RECOMMENDED);
    orientation.set(addExposedField(orient));

    ExposedField* pos = new ExposedField(SFVEC3F, "position", 
                                         new SFVec3f(0, 0, 10));
    pos->setFlags(EIF_RECOMMENDED);
    position.set(addExposedField(pos));

    description.set(
         addField(SFSTRING, "description", new SFString("")));
    addEventOut(SFTIME, "bindTime");
    addEventOut(SFBOOL, "isBound");
}

Node *
ProtoViewpoint::create(Scene *scene)
{ 
    return new NodeViewpoint(scene, this); 
}

NodeViewpoint::NodeViewpoint(Scene *scene, Proto *def)
  : Node(scene, def)
{
}

void NodeViewpoint::preDraw()
{
    glPushMatrix();
    apply();
    glGetFloatv(GL_MODELVIEW_MATRIX, _matrix);
    glPopMatrix();
    _scene->addViewpoint(this);
}

void NodeViewpoint::apply()
{
//    float ffieldOfView = fieldOfView()->getValue();
//    int ijump = jump()->getValue();
    const float *rot = orientation()->getValue();
    const float *pos = position()->getValue();
//    const char *strdescription = description()->getValue();

    float eyeposition=0; 
    float eyeangle=0; 
//    float screendist=0; 
    if (TheApp->useStereo())
       {
       // inexact "toe in" stereo method 
//       screendist=TheApp->getEyeScreenDist();
       if (TheApp->getEyeMode()==EM_RIGHT)
          {
          eyeposition= - TheApp->getEyeHalfDist();
          eyeangle= - TheApp->getEyeAngle();
          }
       else if (TheApp->getEyeMode()==EM_LEFT)
          {
          eyeposition= + TheApp->getEyeHalfDist();
          eyeangle= + TheApp->getEyeAngle();
          }
       }
//    glTranslatef(eyeposition, 0, screendist);
    glTranslatef(eyeposition, 0, 0);
    glRotatef(eyeangle, 0,1,0);
    glRotatef(-RAD2DEG(rot[3]), rot[0], rot[1], rot[2]);
    glTranslatef(-pos[0], -pos[1], -pos[2]);
}

Vec3f
NodeViewpoint::getPosition() const
{
    return Vec3f(((NodeViewpoint *)this)->position()->getValue());
}

void NodeViewpoint::setPosition(const Vec3f &pos)
{
    position(new SFVec3f(pos));
}

const Quaternion &NodeViewpoint::getOrientation() const
{
    return ((NodeViewpoint *)this)->orientation()->getQuat();
}


void NodeViewpoint::setOrientation(const Quaternion &quat)
{
    orientation(new SFRotation(quat));
}

void NodeViewpoint::flip(int index)
{
    position()->flip(index);
    orientation()->flip(index);
}


