/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#ifndef PROJECTDIALOGS_H
#define PROJECTDIALOGS_H

#include "commanddialog.h"

class QLineEdit;
class QComboBox;

/**
* Base class for dialogs for executing commands related to projects.
*/
class ProjectsDialog : public CommandDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param parent The parent widget.
    */
    ProjectsDialog( QWidget* parent );

    /**
    * Destructor.
    */
    ~ProjectsDialog();

protected:
    /**
    * Verify that the project name is valid.
    *
    * A warning is displayed if the name is empty or a project
    * with this name already exists.
    *
    * @param name The name to check.
    *
    * @return @c true if the name is valid, @c false otherwise.
    */
    bool checkProjectName( const QString& name );

    /**
    * Verify that the folder name is valid.
    *
    * A warning is displayed if the name is empty or a folder
    * with this name already exists.
    *
    * @param name The name to check.
    *
    * @param projectId Identifier of the project containing the folder.
    *
    * @return @c true if the name is valid, @c false otherwise.
    */
    bool checkFolderName( int projectId, const QString& name );
};

/**
* Dialog for executing the <tt>ADD PROJECT</tt> command.
*/
class AddProjectDialog : public ProjectsDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param parent The parent widget.
    */
    AddProjectDialog( QWidget* parent );

    /**
    * Destructor.
    */
    ~AddProjectDialog();

public: // overrides
    void accept();

private:
    QLineEdit* m_nameEdit;
};

/**
* Dialog for executing the <tt>RENAME PROJECT</tt> command.
*/
class RenameProjectDialog : public ProjectsDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param projectId Identifier of the project to rename.
    *
    * @param parent The parent widget.
    */
    RenameProjectDialog( int projectId, QWidget* parent );

    /**
    * Destructor.
    */
    ~RenameProjectDialog();

public: // overrides
    void accept();

private:
    int m_projectId;
    QString m_oldName;

    QLineEdit* m_nameEdit;
};

/**
* Dialog for executing the <tt>DELETE PROJECT</tt> command.
*/
class DeleteProjectDialog : public ProjectsDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param projectId Identifier of the project to delete.
    *
    * @param parent The parent widget.
    */
    DeleteProjectDialog( int projectId, QWidget* parent );

    /**
    * Destructor.
    */
    ~DeleteProjectDialog();

public: // overrides
    void accept();

private:
    int m_projectId;
};

/**
* Dialog for executing the <tt>ADD FOLDER</tt> command.
*/
class AddFolderDialog : public ProjectsDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param projectId Identifier of the project containing the folder.
    *
    * @param parent The parent widget.
    */
    AddFolderDialog( int projectId, QWidget* parent );

    /**
    * Destructor.
    */
    ~AddFolderDialog();

public: // overrides
    void accept();

private:
    int m_projectId;

    QLineEdit* m_nameEdit;
    QComboBox* m_typeCombo;
};

/**
* Dialog for executing the <tt>RENAME FOLDER</tt> command.
*/
class RenameFolderDialog : public ProjectsDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param folderId Identifier of the folder to rename.
    *
    * @param parent The parent widget.
    */
    RenameFolderDialog( int folderId, QWidget* parent );

    /**
    * Destructor.
    */
    ~RenameFolderDialog();

public: // overrides
    void accept();

private:
    int m_folderId;
    QString m_oldName;

    QLineEdit* m_nameEdit;
};

/**
* Dialog for executing the <tt>DELETE FOLDER</tt> command.
*/
class DeleteFolderDialog : public ProjectsDialog
{
    Q_OBJECT
public:
    /**
    * Constructor.
    *
    * @param folderId Identifier of the folder to delete.
    *
    * @param parent The parent widget.
    */
    DeleteFolderDialog( int folderId, QWidget* parent );

    /**
    * Destructor.
    */
    ~DeleteFolderDialog();

public: // overrides
    void accept();

private:
    int m_folderId;
};

#endif
