// -*- C++ -*-

//<copyright>
//
// Copyright (c) 1995,96
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        httpreader.h
//
// Purpose:     get an URL via HTTP protocol
//
// Created:     30 Nov 95   Michael Pichler
//
// Changed:     30 Jul 96   Michael Pichler
//
// $Id: httpreader.h,v 1.5 1997/02/25 17:03:58 mpichler Exp $
//
//</file>



#ifndef harmony_scene_httpreader_h
#define harmony_scene_httpreader_h


#include <hyperg/utils/str.h>
class HTTPHeader;
class HTTPIOHandler;
class HTTPResponse;
class ostream;


// HTTPReader
// get an URL via HTTP protocol
// opens a socket connection (linked into the dispatcher instance),
// initiates an GET request,
// after construction error should be checked,
// and destroyed if the connection could not be obtained


class HTTPReader
{
  protected:
    HTTPReader (                        // constructor
      const char* url,                  //   URL to be fetched
      const char* parenturl = "",       //   parent URL (for relative requests)
      const char* appname = 0,          //   application name (User-Agent)
      ostream* os = 0                   //   output stream for data (may also be set afterwards)
    );                                  // derived class must check error code
    virtual ~HTTPReader ();

  public:
    // proxy host and port
    static void setProxy (const RString& hostport);
    static void setProxyHost (const RString& phost)
    { proxyhost_ = phost; }
    static void setProxyPort (int pport)
    { proxyport_ = pport; }

    const RString& proxyHost ()  { return proxyhost_; }
    int proxyPort ()  { return proxyport_; }

    enum {
      err_none,                         // no error
      // on construction:
      err_nothttp,                      // other protocol than HTTP
      // on connect:
      err_noconnect,                    // no connection
      // after reading header:
      err_badheader,                    // bad or incomplete header (failure)
      err_servererror                   // error indication in header (404 etc.)
    };

    int error () const                  // has an error occured?
    { return error_; }                  //   returns 0 or an error code

    // to stop transfer, delete this class

    const RString& host ()              // host name
    { return host_; }
    const RString& fullURL ()           // http://host[:port]/path
    { return fullurl_; }

    HTTPResponse* response ()           // get response (useful in headerComplete)
    { return response_; }

    const HTTPHeader& header ();        // get response header (in headerComplete)

    ostream* getOS ()  { return os_; }  // data stream

    unsigned long bytesTotal () const   // no. of document bytes to be expected
    { return bytestotal_; }             //   (known after header received)
    unsigned long bytesRead () const    // no. of document bytes received
    { return bytesread_; }

  protected:
    void setOS (ostream* os)  // allows derived class to set output stream
    { os_ = os; }             // e.g. on headerComplete

    // callback functions
    virtual void success ()
    { delete this;  // derived class should read and process data written to os here
    }

    virtual void failure ()
    { delete this;  // derived class informed about error
    }

    virtual void headerComplete ()  // header completa; data will arrive
    { }

    virtual void documentData ()  // progress callback: some document data arrived
    { }

    // connect is to be called right after construction
    // (derived function may give progress feedback)
    // also called to handle redirect requests (see below)
    virtual void connect ();  // does the connect (blocking)

    // derived class may wish to know about redirection request
    virtual void redirectRequest ()
    {
      if (!error_)
        connect ();
      if (error_)
        failure ();
    }

  private:
    void init (const char* url);  // prepares for connecting

    static RString proxyhost_;
    static int proxyport_;

    RString parenturl_;
    RString appname_;
    RString host_;
    int port_;
    RString path_;
    RString fullurl_;
    ostream* os_;

    HTTPIOHandler* ioh_;
    HTTPResponse* response_;
    int error_;

    unsigned long bytestotal_;
    unsigned long bytesread_;

  friend class HTTPIOHandler;

}; // HTTPReader


#endif
