
// permstep.cpp
// Copyright (c) 1998-2010 by The VoxBo Development Team

// This file is part of VoxBo
// 
// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
// 
// original version written by Kosh Banerjee

#include <iostream>
#include "perm.h"

using namespace std;

void usage(const unsigned short exitValue, char *progName);

gsl_rng *theRNG = NULL;

int main(int argc, char *argv[]) {
  SEGV_HANDLER
  GSL_ERROR_HANDLER_OFF

  if (argc == 1) {
    usage(1, argv[0]);
    exit(-1);
  }

  string matrixStemName;
  string permDir;
  int exhaustive = 0;
  VB_Vector contrasts, pseudoT;
  string scale = "";
  short method = -1;
  int permIndex = 0;
/*********************************************************************
* Now processing the command line options.                           *
*                                                                    *
* -h ==> Display usage information.                                  *
* -m ==> Specifies the matrix stem name.                             *
* -d ==> Permutation directory.                                      *
* -t ==> Permutation type.                                           *
* -v ==> Print out the gobal VoxBo version number.                   *
*                                                                    *
* VARIABLES:                                                         *
* printHelp - a flag, used to determine if the "-h" command line     *
*             option was used or not.                                *
* printVersion - a flag, used to determine if the "-v" command line  *
*                option was used or not.                             *
*********************************************************************/
  GLMInfo glmi;
  arghandler a;
  a.setArgs("-h", "--help", 0);
  a.setArgs("-m", "--matrixstemname", 1);
  a.setArgs("-d", "--permdir", 1);
  a.setArgs("-t", "--method", 1);
  a.setArgs("-v", "--version", 0);
  a.setArgs("-n", "--permindex", 1);
  a.setArgs("-s", "--scale", 1);
  a.setArgs("-c", "--contrast", 1);
  a.setArgs("-p", "--pseudot", 1);
  a.setArgs("-e", "--exhaustive", 0);
  a.parseArgs(argc, argv);
  string errstring = a.badArg();
  if (errstring.size()) {
     errstring = "[E] unknown flag: " + errstring;
     printErrorMsg(VB_ERROR, errstring.c_str());
     exit(-1);
  }
  if (a.flagPresent("-h"))
     usage(0, argv[0]);
  if (a.flagPresent("-e"))
     exhaustive = 1;
  matrixStemName = a.getFlaggedArgs("-m")[0];
  permDir = a.getFlaggedArgs("-d")[0];
  method = atoi(a.getFlaggedArgs("-t")[0].c_str());
  if (a.flagPresent("-v"))
    printf("\nVoxBo v%s\n",vbversion.c_str());
  permIndex = atoi(a.getFlaggedArgs("-n")[0].c_str());
  scale = a.getFlaggedArgs("-s")[0]; 
  string ct = a.getFlaggedArgs("-c")[0];  
  tokenlist c;
  c.SetSeparator(" ,");
  c.ParseLine(ct);
  contrasts.resize(c.size());
  for (int num = 0; num < c.size(); num++)
      contrasts[num] = atoi(c[num].c_str());
  string pt = a.getFlaggedArgs("-p")[0];
  tokenlist p;
  p.SetSeparator(" ,");
  p.ParseLine(pt);
  pseudoT.resize(p.size());
  for (int num = 0; num < p.size(); num++)
      pseudoT[num] = atoi(p[num].c_str());
  if (matrixStemName.size() == 0) {
    ostringstream errorMsg;
    errorMsg << "Must specify the matrix stem name, using the \"-m\" option.";
    printErrorMsgAndExit(VB_ERROR, errorMsg.str(), 1);
  } 
  if (permDir.size() == 0) {
    ostringstream errorMsg;
    errorMsg << "Must specify the permutation directory name, using the \"-d\" option.";
    printErrorMsgAndExit(VB_ERROR, errorMsg.str(), 1);
  } 
  if ((method < 0) || (method > 2)) {
    ostringstream errorMsg;
    errorMsg << "The \"-t\" argument must be 0, 1, or 2, not [" << method << "].";
    printErrorMsgAndExit(VB_ERROR, errorMsg.str(), 1);
  } 
  if ((scale != "t") && (scale != "i") && (scale != "rb") && (scale != "beta") && (scale != "f") && (scale != "tp") &&
      (scale != "fp") && (scale != "tz") && (scale != "fz") && (scale != "t/1") &&
      (scale != "t/2") && (scale != "tp/1") && (scale != "tp/2") && (scale != "tz/1") && (scale != "tz/2")) {
    ostringstream errorMsg;
    errorMsg << "The \"-s\" was passed an invalid scale: " << scale << ".";
    printErrorMsgAndExit(VB_ERROR, errorMsg.str(), 1);
  }
  if (permIndex < 0) {
    ostringstream errorMsg;
    errorMsg << "You must enter a positive permIndex.";
    printErrorMsgAndExit(VB_ERROR, errorMsg.str(), 1);
  }

/*
  cout << "matrixStemName " << matrixStemName << endl;
  cout << "permDir " << permDir << endl;
  cout << "contrasts ";
  contrasts.print();
  cout << "scale " << scale << endl;
  cout << "pseudoT ";
  pseudoT.print();
  cout << "permIndex: " << permIndex << endl;
*/
  
  glmi.setup(matrixStemName);
  int err = 0;
  err = permStep(matrixStemName, permDir, permIndex, method, contrasts, scale, pseudoT, exhaustive);
  if (err)
     switch(err) {
         case 100:
          printErrorMsg(VB_ERROR, "permstep: no stem name specified.\n");
          break;
         case 101:
          printErrorMsg(VB_ERROR, "permstep: no perm directory specified.\n");
          return -1;
         case 102:
          printErrorMsg(VB_ERROR, "permstep: method had value other than 0, 1, or 2.\n");
          return -1;
         case 103:
          printErrorMsg(VB_ERROR, "permstep: prm file was not found.\n");
          return -1;
         case 104:
          printErrorMsg(VB_ERROR, "permstep: there were no mask values.\n");
          break;
         case 105:
          printErrorMsg(VB_ERROR, "permstep: there was not a tes list.\n");
          return -1;
         case 106:
          printErrorMsg(VB_ERROR, "permstep: G matrix file not readable.\n");
          return -1;
         case 107:
          printErrorMsg(VB_ERROR, "permstep: G matrix file not valid.\n");
          return -1;
         case 108:
          printErrorMsg(VB_ERROR, "permstep: trace state not found.\n");
          break;
         case 109:
          printErrorMsg(VB_ERROR, "permstep: V matrix not readable.\n");
          return -1;
         case 110:
          printErrorMsg(VB_ERROR, "permstep: V matrix not valid.\n");
          return -1;
         case 111:
          printErrorMsg(VB_ERROR, "permstep: V order does not equal V rank.\n");
          return -1;
         case 112:
          printErrorMsg(VB_ERROR, "permstep: Fac1 matrix not readable.\n");
          break;
         case 113:
          printErrorMsg(VB_ERROR, "permstep: Fac1 matrix not valid.\n");
          return -1;
         case 114:
          printErrorMsg(VB_ERROR, "permstep: V order does not equal Fac1 rank.\n");
          return -1;
         case 115:
          printErrorMsg(VB_ERROR, "permstep: Fac1 order does not equal number of contrasts.\n");
          return -1;
         case 116:
          printErrorMsg(VB_ERROR, "permstep: Fac3 matrix not readable.\n");
          return -1;
         case 117:
          printErrorMsg(VB_ERROR, "permstep: Fac3 matrix not valid.\n");
          break;
         case 118:
          printErrorMsg(VB_ERROR, "permstep: Fac1 order does not equal Fac3 rank.\n");
          return -1;
         case 119:
          printErrorMsg(VB_ERROR, "permstep: Fac1 rank does not equal Fac3 order.\n");
          return -1;
         case 120:
          printErrorMsg(VB_ERROR, "permstep: permutation matrix not readable.\n");
          return -1;
         case 121:
          printErrorMsg(VB_ERROR, "permstep: permutation matrix not valid.\n");
          break;
         case 122:
          printErrorMsg(VB_ERROR, "permstep: error loading data for permutation matrix.\n");
          return -1;
         case 123:
          printErrorMsg(VB_ERROR, "permstep: permutation index exceeds number of permutations.\n");
          return -1;
         case 124:
          printErrorMsg(VB_ERROR, "permstep: no betas of interest specified.\n");
          return -1;
         case 125:
          printErrorMsg(VB_ERROR, "permstep: error loading data for G matrix.\n");
          break;
         case 126:
          printErrorMsg(VB_ERROR, "permstep: failure allocating space for R matrix.\n");
          return -1;
         case 127:
          printErrorMsg(VB_ERROR, "permstep: failure allocating space for Fac1 matrix.\n");
          return -1;
         case 128:
          printErrorMsg(VB_ERROR, "permstep: failure writing statistic cube. \n");
          return -1;
         case 129:
          printErrorMsg(VB_ERROR, "permstep: failure allocating space for Gtg matrix.\n");
          return -1;
         case 130:
          printErrorMsg(VB_ERROR, "permstep: failure allocating space for invGtg matrix.\n");
          break;
         case 131:
          printErrorMsg(VB_ERROR, "permstep: failure allocating space for LUPerm structure.\n");
          return -1;
         case 132:
          printErrorMsg(VB_ERROR, "permstep: failure inverting Gtg matrix.\n");
          return -1;
         case 133:
          printErrorMsg(VB_ERROR, "permstep: failure allocating space for Ggtg matrix.\n");
          return -1;
         case 134:
          printErrorMsg(VB_ERROR, "permstep: failure allocating space for R matrix.\n");
          break;
         case 135:
          printErrorMsg(VB_ERROR, "permstep: failure allocating space for Gtvt matrix.\n");
          return -1;
         case 136:
          printErrorMsg(VB_ERROR, "permstep: failure allocating space for F3 matrix.\n");
          return -1;
         case 137:
          printErrorMsg(VB_ERROR, "permstep: failure allocating space for F3##F1 intermediate matrix.\n");
          return -1;
         default:
          printErrorMsg(VB_ERROR, "permstep: unknown error.\n");
     }
  return 0;
} 

void usage(const unsigned short, char *) {
  printf("\nVoxBo permstep (v%s)\n",vbversion.c_str());
  printf("summary: ");
  printf(" Permutation step.\n");
  printf("usage:\n");
  printf(" permstep -h -m[matrix stem name] -d[permutation directory] -s[scale]\n");
  printf("          -c[contrasts] -p[pseudot values] -t[permutation type]\n");
  printf("          -n[index of permutation selected] -v\n");
  printf("flags:\n");
  printf(" -h                        Print usage information. Optional.\n");
  printf(" -m <matrix stem name>                       Specify the matrix stem name. Required.\n");
  printf(" -d                        Permutation directory name. Required.\n");
  printf(" -t                        Permutation type. 1 means regular permutation.\n");
  printf("                           2 means sign permutation. Required.\n");
  printf(" -s                        scale value of stat cubes. Required.\n");
  printf(" -c                        contrast values, in quotes. Required.\n");
  printf(" -p                        pseudoT values, in quotes. Required.\n");
  printf(" -v                        Global VoxBo version number. Optional.\n");
  printf(" -e                        Produce cubes and their inverses. Optional.\n");
  printf(" -n                        index of permutation to be generated. Optional.\n");
  printf("scales:                                                         \n");
  printf("                           t            - t value map\n");
  printf("                           i            - intercept term percent change map\n");
  printf("                           beta or rb   - raw beta values map\n");
  printf("                           f            - F value map\n");
  printf("                           tp tp/1 tp/2 - p map of t values\n");
  printf("                           fp           - p map based on F values\n");
  printf("                           tz tz/1 tz/2 - Z map based on t values\n");
  printf("                           fz           - Z map based on F values\n");
  printf("notes:                                                         \n");
  printf("                           /1 and /2 force one tailed and two tailed, and\n");
  printf("                           note that VoxBo defaults to one tailed testing\n");
  printf(" -v                        Global VoxBo version number. Optional.\n\n");
  exit(-1);
} 
