/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
#include <iostream>
#include <string>
#include <map>
#include <list>
#include <vector>
#include <sstream>
#include <stack>
#include <algorithm>
/* -- */
#include "interface/Interface.hpp"
/* -- */
#include "common/Message.hpp"
#include "common/Session.hpp"
/* -- */
#include "render/Palette.hpp"
/* -- */
#include "trace/values/Values.hpp"
#include "trace/tree/Interval.hpp"
#include "trace/tree/Node.hpp"
#include "trace/tree/BinaryTree.hpp"
#include "trace/EntityValue.hpp"
#include "trace/EntityTypes.hpp"
#include "trace/Entitys.hpp"
#include "trace/Trace.hpp"
/* -- */
#include "common/common.hpp"
#include "common/Info.hpp" /* Dirty, should be remove */
#include "common/Message.hpp"
/* -- */
#include "trace/tree/Interval.hpp"
/* -- */
using namespace std;


Trace::Trace(): _max_date(0.0) {
    _selected_container = NULL;
    _filter = 0;
    _interval_constrained = NULL;
    _depth = 0;

#ifndef WITHOUT_QT
    if(Session::get_use_palette())
        _palette = Session::get_palette(Session::get_current_palette());
    else
#endif
        _palette = NULL;
}

template <class T>
void MyDelete(T *ptr){
  delete ptr;
};


Trace::~Trace() {
    //delete _palette;

    if(_interval_constrained != NULL) {
        delete _interval_constrained;
        _interval_constrained = NULL;
    }

    // Delete containers
    for_each(_root_containers.begin(), 
	     _root_containers.end(), 
	     MyDelete<Container>);
    _root_containers.clear();
    _root_containers.resize(0);
    // while (!_root_containers.empty()){
    //     delete _root_containers.front();
    //     _root_containers.pop_front();
    // }
    
    // Delete container types
    while (!_root_container_types.empty()){
        delete _root_container_types.front();
        _root_container_types.pop_front();
    }

    // Delete state types
    while (!_state_types.empty()){
        delete _state_types.front();
        _state_types.pop_front();
    }

    // Delete event types
    while (!_event_types.empty()){
        delete _event_types.front();
        _event_types.pop_front();
    }

    // Delete link types
    while (!_link_types.empty()){
        delete _link_types.front();
        _link_types.pop_front();
    }

    // Delete variable types
    while (!_variable_types.empty()){
        delete _variable_types.front();
        _variable_types.pop_front();
    }
}

static void delete_opt(map<string, Value *> &opt) {
    for (map<string, Value *>::iterator i = opt.begin();
        i != opt.end();
        i++) {
        delete (*i).second;
    }
}

void Trace::define_container_type(Name &name, ContainerType *parent, map<string, Value *> &opt) {
    ContainerType *type = new ContainerType(name, parent);
    if (parent)
        parent->add_child(type);
    else
        _root_container_types.push_back(type);
 
    // Delete unused extra fields
    delete_opt(opt);
}

void Trace::create_container(Date &time, Name &name, ContainerType *type, Container *parent, map<string, Value *> &opt) {
    if (!type) {
	vite_error("Trace::get_container : Type undefined");
        return;
    }
    
    Container *cont = new Container(name, time, type, parent, opt);
    if (parent){
        parent->add_child(cont);
    } else {
        _root_containers.push_back(cont);
    }

    /* Update the trace depth */
    if (cont->get_depth() > _depth){
            _depth=cont->get_depth();
            Info::Trace::depth=_depth; /* Dirty method. Should be remove */
    }

    if (time > _max_date)
        _max_date = time;
        
    // Delete unused extra fields
    //delete_opt(opt);
}

void Trace::destroy_container(Date &time, Container *cont, ContainerType *type, map<string, Value *> &opt) {
    if (cont && type)
        cont->destroy(time);
    
    if (time > _max_date)
        _max_date = time;
    
    // Delete unused extra fields
    delete_opt(opt);
}

void Trace::define_event_type(Name &name, ContainerType *container_type, map<string, Value *> &opt) {
    if (container_type)
        _event_types.push_back(new EventType(name, container_type, opt));
}

void Trace::define_state_type(Name &name, ContainerType *container_type, map<string, Value *> &opt) {
    if (container_type)
        _state_types.push_back(new StateType(name, container_type, opt));
}

void Trace::define_variable_type(Name &name, ContainerType *container_type, map<string, Value *> &opt) {
    if (container_type)
        _variable_types.push_back(new VariableType(name, container_type, opt));
}

void Trace::define_link_type(Name &name, ContainerType *ancestor, ContainerType *source, ContainerType *destination, map<string, Value *> &opt) {
    if (ancestor && source && destination)
        _link_types.push_back(new LinkType(name, ancestor, source, destination, opt));
}

void Trace::define_entity_value(Name &name, EntityType *entity_type, map<string, Value *> &opt) {
    if (entity_type) {
        if(_palette) {
            Color *c = _palette->get_color(name.to_string());
            if(c != NULL) { // We set the palette color first
                if(opt.find("Color") != opt.end()) {
                    delete opt["Color"];
                }
                opt["Color"] = new Color(*c);
            }
        }

        entity_type->add_value(new EntityValue(name, entity_type, opt));
    }
}

void Trace::set_state(Date &time, StateType *type, Container *container, EntityValue *value, map<string, Value *> &opt) {
    if (container && type)
        container->set_state(time, type, value, opt);
    
    if (time > _max_date)
        _max_date = time;
}

void Trace::push_state(Date &time, StateType *type, Container *container, EntityValue *value, map<string, Value *> &opt) {
    if (container && type)
        container->push_state(time, type, value, opt);
    
    if (time > _max_date)
        _max_date = time;
}

void Trace::pop_state(Date &time, StateType *type, Container *container, map<string, Value *> &opt) {
    if (container && type)
        container->pop_state(time);
    
    if (time > _max_date)
        _max_date = time;
    
    // Delete unused extra fields
    delete_opt(opt);
}

void Trace::new_event(Date &time, EventType *type, Container *container, EntityValue *value, map<string, Value *> &opt) {
    if (container && type)
        container->new_event(time, type, value, opt);
    
    if (time > _max_date)
        _max_date = time;
}

void Trace::set_variable(Date &time, VariableType *type, Container *container, Double value, map<string, Value *> &opt) {
    if (container && type)
        container->set_variable(time, type, value);
    
    if (time > _max_date)
        _max_date = time;
    
    // Delete unused extra fields
    delete_opt(opt);
}

void Trace::add_variable(Date &time, VariableType *type, Container *container, Double value, map<string, Value *> &opt) {
    if (container && type)
        container->add_variable(time, type, value);
    
    if (time > _max_date)
        _max_date = time;
    
    // Delete unused extra fields
    delete_opt(opt);
}

void Trace::sub_variable(Date &time, VariableType *type, Container *container, Double value, map<string, Value *> &opt) {
    if (container && type)
        container->sub_variable(time, type, value);
    
    if (time > _max_date)
        _max_date = time;
    
    // Delete unused extra fields
    delete_opt(opt);
}

void Trace::start_link(Date &time, LinkType *type, Container *ancestor, Container *source, EntityValue *value, String key, map<string, Value *> &opt) {
    if (ancestor && type && source)
        ancestor->start_link(time, type, source, value, key, opt);
    
    if (time > _max_date)
        _max_date = time;
}

void Trace::end_link(Date &time, LinkType *type, Container *ancestor, Container *destination, EntityValue */*value*/, String key, map<string, Value *> &opt) {
    if (ancestor && type && destination)
        ancestor->end_link(time, destination, key, opt);
    
    if (time > _max_date)
        _max_date = time;
}

void Trace::finish() {
    stack<Container *> containers;
    Container::VectorIt        i   = _root_containers.begin();
    Container::VectorIt const &end = _root_containers.end();

    for (; i != end; i++)
        containers.push(*i);
    
    while (!containers.empty()) {
        Container * c = containers.top();
        containers.pop();
        c->finish(_max_date);

	{
	  Container::VectorIt        i   = c->get_children()->begin();
	  Container::VectorIt const &end = c->get_children()->end();

	  for (; i != end; i++)
            containers.push(*i);
	}
    }
}

const Container::Vector *Trace::get_root_containers() const {
    return &_root_containers;
}

void Trace::get_all_containers(Container::Vector &list_to_fill) const {
    Container::VectorIt        it  = _root_containers.begin();
    Container::VectorIt const &end = _root_containers.end();

    for (; it != end; it ++) {
        list_to_fill.push_back(*it);
        // Recursivity to add the children names
        add_containers(list_to_fill, *it);
    }

}

void Trace::add_containers(Container::Vector &containers, const Container *parent) const {
    const Container::Vector           *children  = parent->get_children();
    Container::VectorIt        it  = children->begin();
    Container::VectorIt const &end = children->end();

    for (; it != end; it ++) {
        // We create the node and we do the recursivity
        containers.push_back(*it);
        add_containers(containers , *it);
    }
}

const list <StateType *> *Trace::get_state_types() const {
    return &_state_types;
}

const list <EventType *> *Trace::get_event_types() const {
    return &_event_types;
}

template<class T>
static T *search_tree(string name, T *el) { // Before : String name
    if (el->get_name() == name)
        return el;
    else {
        T *r;
        typename list<T *>::const_iterator it_end = el->get_children()->end();
        for (typename list<T *>::const_iterator it = el->get_children()->begin();
             it != it_end;
             ++ it) {
            r = search_tree<T>(name, *it);
            if (r)
                return r;
        }
    }
    return 0;
}

template<class T>
static T *search_treeV(string name, T *el) { // Before : String name
    if (el->get_name() == name)
        return el;
    else {
        T *r;
        typename vector<T *>::const_iterator it_end = el->get_children()->end();
        for (typename vector<T *>::const_iterator it = el->get_children()->begin();
             it != it_end;
             ++ it) {
            r = search_treeV<T>(name, *it);
            if (r)
                return r;
        }
    }
    return 0;
}

ContainerType *Trace::search_container_type(String name) const {
    ContainerType *r;
    
    const string &std_name = name.to_string();
    if (std_name == "0")
        return 0;
    
    const list<ContainerType *>::const_iterator &it_end = _root_container_types.end();
    for (list<ContainerType *>::const_iterator it = _root_container_types.begin();
         it != it_end;
         ++ it) {
        r = search_tree<ContainerType>(std_name, *it);
        if (r)
            return r;
    }
    
    return 0;
}

Container *Trace::search_container(String name) const {
    Container *r;
    
    const string &std_name = name.to_string();
    if (std_name == "0")
        return 0;
    
    Container::VectorIt        it  = _root_containers.begin();
    Container::VectorIt const &end = _root_containers.end();
    for( ; it != end; ++ it) {
        r = search_tree<Container>(std_name, *it);
        if (r)
             return r;
    }
     
    return 0;
}

EventType *Trace::search_event_type(String name) const {
    const list<EventType *>::const_iterator &it_end = _event_types.end();
    for (list<EventType *>::const_iterator it = _event_types.begin();
         it != it_end;
         ++ it) {
        if ((*it)->get_name() == name)
            return *it;
    }
    
    return 0;
}

StateType *Trace::search_state_type(String name) const {
    const list<StateType *>::const_iterator &it_end = _state_types.end();
    for (list<StateType *>::const_iterator it = _state_types.begin();
         it != it_end;
         ++ it) {
        if ((*it)->get_name() == name)
            return *it;
    }
    
    // *Message::get_instance() << "Unknown state type: " << name.to_string() << Message::ende;
    return 0;
}

VariableType *Trace::search_variable_type(String name) const {
    const list<VariableType *>::const_iterator &it_end = _variable_types.end();
    for (list<VariableType *>::const_iterator it = _variable_types.begin();
         it != it_end;
         ++ it) {
        if ((*it)->get_name() == name)
            return *it;
    }
    
    return 0;
}

LinkType *Trace::search_link_type(String name) const {
    const list<LinkType *>::const_iterator &it_end = _link_types.end();
    for (list<LinkType *>::const_iterator it = _link_types.begin();
         it != it_end;
         ++ it) {
        if ((*it)->get_name() == name)
            return *it;
    }
    
    return 0;
}

EntityType *Trace::search_entity_type(String name) const {
    // Search in the StateType 
    const list<StateType *>::const_iterator &stateType_end = _state_types.end();
    for (list<StateType *>::const_iterator it = _state_types.begin();
        it != stateType_end;
        ++ it) {
        if ((*it)->get_name() == name)
            return *it;
    }

    // Search in the LinkType 
    const list<LinkType *>::const_iterator &linkType_end = _link_types.end();
    for (list<LinkType *>::const_iterator it = _link_types.begin();
         it != linkType_end;
        ++ it) {
        if ((*it)->get_name() == name)
            return *it;
    }

    // Search in the EventType
    const list<EventType *>::const_iterator &eventType_end = _event_types.end();
    for (list<EventType *>::const_iterator it = _event_types.begin();
         it != eventType_end;
        ++ it) {
        if ((*it)->get_name() == name)
            return *it;
    }
    
    return 0;
}

EntityValue *Trace::search_entity_value(String name, EntityType *entity_type) const {
    if(!entity_type){
        return NULL;
    }

    const list<EntityValue *>::const_iterator &it_end = entity_type->get_values()->end();
    for (list<EntityValue *>::const_iterator it = entity_type->get_values()->begin();
         it != it_end;
         ++ it) {
        if ((*it)->get_name() == name)
            return *it;
    }
    Name alias_name;
    alias_name.set_name(name.to_string());
    EntityValue *value = new EntityValue(alias_name, entity_type, map<string, Value*>());
    entity_type->add_value(value);
    return value;
}

Date Trace::get_max_date(){
  return _max_date;
}

int Trace::get_depth(){
    return _depth;         
}


void Trace::set_selected_container(vector<const Container *> * c) {
    _selected_container = c;
}

void Trace::set_interval_constrained(Interval * i) {
    if(_interval_constrained != NULL) {
        delete _interval_constrained;
    }
    _interval_constrained = i;
}

vector<const Container *> *Trace::get_selected_container() {
    return _selected_container;
}

Interval *Trace::get_interval_constrained() {
    return _interval_constrained;
}

void Trace::set_filter(double f) {
    _filter = f;
}

double Trace::get_filter() {
    return _filter;
}

void Trace::get_all_variables(map<string, Variable *> &map_to_fill) {
    Container::Vector all_containers;
    get_all_containers(all_containers);

    Container::VectorIt const &end = all_containers.end();
    Container::VectorIt        it  = all_containers.begin();
    for ( ; it != end ; it ++) {
        const map<VariableType *, Variable *> *variables = (*it)->get_variables();
        // We store each variable
        for(map<VariableType *, Variable *>::const_iterator var_it = variables->begin() ;
            var_it != variables->end() ;
            var_it ++) {
            string name = (*it)->get_name().to_string() + " " + (*var_it).first->get_name().to_string();
            map_to_fill[name] = (*var_it).second;
        }
    }
}

void Trace::get_states_colors(map<string, Color * > &nc_map, Container *parent) {
    Container::Vector containers;
    // We look for each container
    if(parent)
        add_containers(containers, parent);
    else // We add all
        get_all_containers(containers);

    for(Container::VectorIt it = containers.begin();
        it != containers.end(); ++ it) {
        BinaryTree<StateChange> *states;
        states = (*it)->get_states();
        if(states && states->get_root()) {
            get_state_names_colors_list(nc_map, states->get_root()); 
        }
    }
}
/* Fill name_map with the names of the state contained in sub-tree node */
void get_state_names_colors_list(map<string, Color *> &name_map, Node<StateChange> *node) {
    if (!node)
        return;
    
    /* handle the node */
    const State *left_st = node->get_element()->get_left_state();
    if (left_st) {
        const EntityValue *value = left_st->get_value();
        const string name = value->get_name().to_string();
        Color *color = NULL;
        if(value->get_extra_fields()->find(string("Color")) != value->get_extra_fields()->end()) {
            color = (Color *)value->get_extra_fields()->find(string("Color"))->second;
        }

        name_map[name] = color;
    }
    
    const State *right_st = node->get_element()->get_right_state();
    if (right_st) {
        const EntityValue *value = right_st->get_value();
        const string name = value->get_name().to_string();

        Color *color = NULL;
        if(value->get_extra_fields()->find(string("Color")) != value->get_extra_fields()->end()) {
            color = (Color *)value->get_extra_fields()->find(string("Color"))->second;
        }

        name_map[name] = color;
    }
    
    /* handle its left child */
    get_state_names_colors_list(name_map, node->get_left_child());

    /* handle its right child */
    get_state_names_colors_list(name_map, node->get_right_child());
}

