/*
** This file is part of the ViTE project.
**
** This software is governed by the CeCILL-A license under French law
** and abiding by the rules of distribution of free software. You can
** use, modify and/or redistribute the software under the terms of the
** CeCILL-A license as circulated by CEA, CNRS and INRIA at the following
** URL: "http://www.cecill.info".
** 
** As a counterpart to the access to the source code and rights to copy,
** modify and redistribute granted by the license, users are provided
** only with a limited warranty and the software's author, the holder of
** the economic rights, and the successive licensors have only limited
** liability.
** 
** In this respect, the user's attention is drawn to the risks associated
** with loading, using, modifying and/or developing or reproducing the
** software by the user in light of its specific status of free software,
** that may mean that it is complicated to manipulate, and that also
** therefore means that it is reserved for developers and experienced
** professionals having in-depth computer knowledge. Users are therefore
** encouraged to load and test the software's suitability as regards
** their requirements in conditions enabling the security of their
** systems and/or data to be ensured and, more generally, to use and
** operate it in the same conditions as regards security.
** 
** The fact that you are presently reading this means that you have had
** knowledge of the CeCILL-A license and that you accept its terms.
**
**
** ViTE developers are (for version 0.* to 1.0):
**
**        - COULOMB Kevin
**        - FAVERGE Mathieu
**        - JAZEIX Johnny
**        - LAGRASSE Olivier
**        - MARCOUEILLE Jule
**        - NOISETTE Pascal
**        - REDONDY Arthur
**        - VUCHENER Clément 
**
*/
#include <iostream>
#include <string>
#include <map>
#include <list>
#include <vector>
#include <stack>
#include <algorithm>
/* -- */
#include "common/common.hpp"
#include "common/Info.hpp"
/* -- */
#include "trace/values/Values.hpp"
#include "trace/EntityTypes.hpp"
#include "trace/Entitys.hpp"
#include "trace/tree/Interval.hpp"
#include "trace/tree/Node.hpp"
#include "trace/tree/BinaryTree.hpp"
/* -- */
#include "statistics/Statistic.hpp"
/* -- */
#include "trace/Container.hpp"
/* -- */
using namespace std;



Container::Container(Name name, Date creation_time, ContainerType *type, Container *parent, map<string, Value *> &opt):
	_name(name), _creation_time(creation_time), _destruction_time(0.0), _type(type), _parent(parent),
        _n_states(0), _state_tree(0), _n_events(0), _event_tree(0), _n_variables(0), _extra_fields(opt) {

    if (parent)    
        _depth = parent->_depth+1;
    else
        _depth=0;
}

template <class T>
void MyDelete(T *ptr){
  delete ptr;
};

Container::~Container() {
    
    std::for_each(_children.begin(), 
		  _children.end(), 
		  MyDelete<Container>);
    _children.clear();
    _children.resize(0);
    
    // Delete states
    delete _state_tree;
    _state_tree = NULL;
    // Delete events
    delete _event_tree;
    _event_tree = NULL;
    
    // Delete links
    while (!_links.empty()){
        delete _links.front();
        _links.pop_front();
    }
    // Delete variables
    for(map<VariableType *, Variable *>::iterator it = _variables.begin();
        it != _variables.end();
        it++){
        delete (*it).second;
    }
    _variables.clear();
}

void Container::add_child(Container *child) {
  if(child != NULL){
    _children.push_back(child);
  }
}

void Container::add_current_state(Date end) {
    State *new_state = new State(
        _current_states.top().start,
        end,
        _current_states.top().type,
        this,
        _current_states.top().value,
        _current_states.top().opt);
        
    // Set the change to the new state
    if (!_states.empty())
        _states.back()->set_right_state(new_state);
    else {
        _n_states++;
        _states.push_back(new StateChange(new_state->get_start_time(), NULL, new_state));
    }
  
    // Set the change from the new state
    _states.push_back(new StateChange(end, new_state));
    _n_states++;
}

void Container::set_state(Date time, StateType *type, EntityValue *value, map<string, Value *> &opt) {
    if (!_current_states.empty()) {
        add_current_state(time);
        _current_states.pop();
    }
        
    current_state_t t(time, type, value, opt);
    _current_states.push(t);
}

void Container::push_state(Date time, StateType *type, EntityValue *value, map<string, Value *> &opt) {
    if (!_current_states.empty())
        add_current_state(time);
        
    current_state_t t(time, type, value, opt);		
    _current_states.push(t);
}

void Container::pop_state(Date time) {
    if (!_current_states.empty()) {
        add_current_state(time);
        _current_states.pop();
    }
    
    if (!_current_states.empty()) {
        _current_states.top().start = time;
    }
}

void Container::new_event(Date time, EventType *type, EntityValue *value, map<string, Value *> &opt) {
    _events.push_back(new Event(time, type, this, value, opt));
    _n_events++;
}


void Container::start_link(Date time, LinkType *type, Container *source, 
			   EntityValue *value, String key, map<std::string, Value *> &opt) 
{
    map<String, current_link_t, String::less_than>::iterator i = _current_links.find(key);
    if (i == _current_links.end()) 
    {
	current_link_t t(time, type, source, value, opt);
	_current_links[key] = t;
    }
    else 
    {
	for (map<string, Value *>::const_iterator j = opt.begin();
	     j != opt.end(); j++)
	{
	    (*i).second.opt[(*j).first] = (*j).second;
        }

	_links.push_back(new Link(
			     time,
			     (*i).second.start,
			     type,
			     this,
			     source,
			     (*i).second.source,
			     value,
			     (*i).second.opt));
	_current_links.erase(i);
    }
}

void Container::end_link(Date time, Container *destination, String key, map<string, Value *> &opt) {
    map<String, current_link_t, String::less_than>::iterator i = _current_links.find(key);
    if (i == _current_links.end()) 
    {
	current_link_t t(time, NULL, destination, NULL, opt);
	_current_links[key] = t;
    }
    else
    {
	for (map<string, Value *>::const_iterator j = opt.begin();
	     j != opt.end(); j++)
	{
	    (*i).second.opt[(*j).first] = (*j).second;
        }

	_links.push_back(new Link(
			     (*i).second.start,
			     time,
			     (*i).second.type,
			     this,
			     (*i).second.source,
			     destination,
			     (*i).second.value,
			     (*i).second.opt));
	_current_links.erase(i);
    }
}

void Container::set_variable(Date time, VariableType *type, Double value) {
    map<VariableType *, Variable *>::iterator i = _variables.find(type);
    if (i == _variables.end()) {
        _variables[type] = new Variable(this, type);
        _variables[type]->add_value(time, value);
        _n_variables++;
    }
    else {
        (*i).second->add_value(time, value);
    }
}

void Container::add_variable(Date time, VariableType *type, Double value) {
    map<VariableType *, Variable *>::iterator i = _variables.find(type);
    if (i == _variables.end()) {
        _variables[type] = new Variable(this, type);
        _variables[type]->add_value(time, value);
        _n_variables++;
    }
    else {
        (*i).second->add_value(time, (*i).second->get_last_value() + value);
    }

}

void Container::sub_variable(Date time, VariableType *type, Double value) {
    map<VariableType *, Variable *>::iterator i = _variables.find(type);
    if (i == _variables.end()) {
        _variables[type] = new Variable(this, type);
        _variables[type]->add_value(time, -value);
        _n_variables++;
    }
    else {
        (*i).second->add_value(time, (*i).second->get_last_value() - value);
    }
}

Name Container::get_name() const {
    return _name;
}

const Container *Container::get_parent() const {
    return _parent;
}

const ContainerType *Container::get_type() const {
    return _type;
}

const Container::Vector * Container::get_children() const {
    return &_children;
}

Date Container::get_creation_time() const {
    return _creation_time;
}

Date Container::get_destruction_time() const {
    return _destruction_time;
}

StateChange::Tree *Container::get_states() const {
    return _state_tree;
}

Event::Tree *Container::get_events() const {
    return _event_tree;
}

const Link::Vector *Container::get_links() const {
    return &_links;
}

const map<VariableType *, Variable *> *Container::get_variables() const {
    return &_variables;
}

const map<string, Value *> *Container::get_extra_fields() const {
    return &_extra_fields;
}

unsigned int Container::get_variable_number() const {
    return _n_variables;
}

unsigned int Container::get_state_number() const {
    return _n_states;
}

unsigned int Container::get_event_number() const {
    return _n_events;
}

void Container::destroy(const Date &time) {
    if (!_current_states.empty()) {
        add_current_state(time);
        _current_states.pop();
    }
    _destruction_time = time;
}

void Container::finish(const Date &time) {
    if (_destruction_time.get_value() == 0.0)
        destroy(time);
    _event_tree = new Event::Tree(_events, _n_events);
    _state_tree = new StateChange::Tree(_states,_n_states);
}



void Container::fill_stat( Statistic * stat, Interval I){

    // If the container is a proc -> no child container
    if(this->get_children()->empty()) {
        browse_stat_link(this,stat,I);
    }
    Node<StateChange> * sauv = NULL;
    Node<StateChange> ** prev = &sauv;
    Node<StateChange> * node = this->get_states()->get_root();
    double tmp;
    browse_stat_state(node,stat,I,prev);
    // To add the first partial state of the interval
    if(sauv && 
       sauv->get_element() && 
       sauv->get_element()->get_right_state()){
        if(sauv->get_element()->get_right_state()->get_end_time() > 
           I._left){
            if( sauv->get_element()->get_right_state()->get_end_time() > I._right)
                tmp = I._right - I._left;
            else
                tmp = sauv->get_element()->get_right_state()->get_end_time() - I._left;

            stat->add_state(sauv->get_element()->get_right_state()->get_value(),tmp);
        }
    }

    stat->set_nb_event(this->get_event_number());
}


void browse_stat_state(Node<StateChange> * node, Statistic * stats, Interval I, Node<StateChange>** prev){
    
    if( ! node ||
        ! node->get_element())
        return;
    
    // If the node is in the interval
    if(node->get_element()->get_time() <= I._right &&
           node->get_element()->get_time() >= I._left){

        if(node->get_left_child())
            browse_stat_state(node->get_left_child(),stats,I,prev);
        if(node->get_right_child())
            browse_stat_state(node->get_right_child(),stats,I,prev);
        

        //Add the right state of the state change
        if(node->get_element()->get_right_state())
	  {
	    if(node->get_element()->get_right_state()->get_end_time() < I._right){
	      stats->add_state(node->get_element()->get_right_state()->get_value(),node->get_element()->get_right_state()->get_duration());
            }
            else{
	      stats->add_state(node->get_element()->get_right_state()->get_value(),I._right - node->get_element()->get_right_state()->get_start_time());
            }
	  }
    }

    // Else if after the interval
    else if(node->get_element()->get_time() > I._right ){
        if(node->get_left_child()){
            browse_stat_state(node->get_left_child(),stats,I,prev);
        }
    }

    else{        // Else node is before the interval
        if( !(*prev) ){
            *prev = node;
        }

        if(node->get_element()->get_right_state()){
            if(node->get_element()->get_right_state()->get_start_time() >= 
               (*prev)->get_element()->get_right_state()->get_start_time()){
                *prev = node;
            }
        }

        if(node->get_right_child()){
            browse_stat_state(node->get_right_child(),stats,I,prev);
        }
    }
}
    

void browse_stat_link(const Container * cont, Statistic * S, Interval I){
    if(!cont)
        return;

    Link::VectorIt const  &it_end = cont->get_links()->end();
    for(list<Link *>::const_iterator it = cont->get_links()->begin() ;
        it != it_end;
        it++){
        if( (*it)->get_source() == cont  &&
            (*it)->get_destination() != cont){
            S->add_link(cont);
        }
        else if((*it)->get_source() != cont  &&
                (*it)->get_destination() == cont){
            S->add_link(cont);
        }
    }
    browse_stat_link(cont->get_parent(), S, I);
}

int Container::get_depth(){
    return _depth;         
}

