/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "interactive.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include "view.h"
#include "objectList.h"
#include <visu_object.h>
#include <visu_tools.h>
#include <visu_extension.h>
#include <renderingBackend/visu_actionInterface.h>
#include <visu_configFile.h>
#include <coreTools/toolConfigFile.h>

#include <math.h>

#define FLAG_PARAMETER_OBSERVE_METHOD "opengl_observe_method"
#define DESC_PARAMETER_OBSERVE_METHOD "Choose the observe method ; integer (0: constrained mode, 1: walker mode)"

/**
 * SECTION:interactive
 * @short_description: Gives tools to interact with the rendered
 * area.
 * 
 * <para>When one wants some interactions on the rendering area
 * (either from the mouse or from the keyboard), one should initialise
 * it using openGLInteractiveInit_session(). Then, during the
 * interactive session, several modes have been implemented:
 * <itemizedlist>
 * <listitem><para>
 * openGLInteractiveBegin_mark(), is a mode where the mouse can select
 * a node and this nodes become highlighted.
 * </para></listitem>
 * <listitem><para>
 * openGLInteractiveBegin_move(), the mouse is then used to move
 * picked nodes. Moves are possible in the plane of the screen or
 * constrained along axis of the bounding box.
 * </para></listitem>
 * <listitem><para>
 * openGLInteractiveBegin_observe(), in this mode, the mouse is used
 * to change the position of camera, its orientation and its zoom
 * characteristics.
 * </para></listitem>
 * <listitem><para>
 * openGLInteractiveBegin_pick(), this mode is the most complex
 * picking mode. It can select atoms up to two references and measure
 * distances and angles.
 * </para></listitem>
 * <listitem><para>
 * openGLInteractiveBegin_pickAndObserve(), this mode is a simplified
 * mix of obervation and picking.
 * </para></listitem>
 * </itemizedlist>
 * The observe mode has two different moving algorithms that can be
 * changed using openGLInteractiveSet_preferedObserveMethod(). The
 * first, called 'constrained' (cf. #OPENGL_OBSERVE_CONSTRAINED)
 * corresponds to movements along parallels and meridians. When the
 * mouse is moved along x axis, the camera raotates along a
 * parallel. When the camera is moved along y axis, the camera rotate
 * along a meridian. The top is always pointing to the north pole (in
 * fact, omega is always forced to 0 in this mode). This mode has a
 * 'strange' behavior when the observer is near a pole: moving mouse
 * along x axis make the box rotates on itself. It is normal, because
 * movements on x axis is equivalent to movements on parallel and near
 * the poles, parallel are small circle around the z axis. This can be
 * unnatural in some occasion and the other mode, called 'walker' (see
 * #OPENGL_OBSERVE_WALKER) can be used instead of the 'constrained'
 * mode. In the former, the moving is done has if the observer was a
 * walking ant on a sphere : moving the mouse along y axis makes the
 * ant go on or forward ; and x axis movements makes the ant goes on
 * its left or on it right. This is a more natural way to move the box
 * but it has the inconvient that it is hard to return in a given
 * position (omega has never the right value).
 * </para>
 */

struct _VisuInteractive
{
  GObject parent;

  /* Internal object gestion. */
  gboolean dispose_has_run;

  /* The kind of interactive (pick, observe...) */
  VisuInteractiveId id;

  /* Data needed in case of pick. */
  /* The last selected node. */
  gint idSelected;
  /* The current first reference. */
  gint idRef1;
  /* The current second reference. */
  gint idRef2;
  /* */
  GList *idRegion;
  /* The position of the pick. */
  int xOrig, yOrig;
  int xPrev, yPrev;

  /* Data for the move action. */
  gboolean movingPicked;
  GList *movingNodes;
  float movingAxe[3];
};

enum
  {
    INTERACTIVE_OBSERVE_SIGNAL,
    INTERACTIVE_PICK_ERROR_SIGNAL,
    INTERACTIVE_PICK_NODE_SIGNAL,
    INTERACTIVE_PICK_REGION_SIGNAL,
    INTERACTIVE_START_MOVE_SIGNAL,
    INTERACTIVE_MOVE_SIGNAL,
    INTERACTIVE_STOP_SIGNAL,
/*     INTERACTIVE_EVENT_SIGNAL, */
    N_INTERACTIVE_SIGNALS
  };

struct _VisuInteractiveClass
{
  GObjectClass parent;

  VisuInteractiveMethod preferedObserveMethod;

  /* The save camera positions are implemented as
     a circular list we a pointer on the current head. */
  GList *savedCameras, *lastCamera;

  VisuExtension *moveAtomExtension;
  int moveAtomExtension_list;
};
static VisuInteractiveClass *local_class = NULL;

/* Internal variables. */
static guint interactive_signals[N_INTERACTIVE_SIGNALS] = { 0 };

/* Object gestion methods. */
static void visuInteractive_dispose (GObject* obj);
static void visuInteractive_finalize(GObject* obj);

/* When an atom is moved (see move()), it is associated
   to a specific extension list to avoid to recreate
   the complete allElement list. */
#define  moveAtomName        "MovedANode"
#define  moveAtomNameI18n    _("Moved a node")
#define  moveAtomDescription _("Draw the node that is displaced.")

/* Local methods. */
static void exportParameters(GString *data, VisuData *dataObj);
static gboolean readOpenGLObserveMethod(gchar **lines, int nbLines, int position,
					VisuData *dataObj, GError **error);
static void onVisuDataReady(GObject *obj, VisuData *dataObj, gpointer data);
static void onPopulationChange(VisuData *dataObj, int *nodes, gpointer data);
static gboolean observe(VisuInteractive *inter, VisuData *dataObj,
			SimplifiedEvents *ev);
static gboolean pick(VisuInteractive *inter, VisuData *dataObj, SimplifiedEvents *ev);
static gboolean move(VisuInteractive *inter, VisuData *dataObj, SimplifiedEvents *ev);
static gboolean mark(VisuInteractive *inter, VisuData *dataObj, SimplifiedEvents *ev);
static gboolean pickAndObserve(VisuInteractive *inter, VisuData *dataObj,
			       SimplifiedEvents *ev);
static int getSelectElement(VisuData *data, int x, int y);
static GList* getSelectElementsRegion(VisuData *data, int x1, int y1,
				      int x2, int y2);

G_DEFINE_TYPE(VisuInteractive, visuInteractive, G_TYPE_OBJECT)

static void visuInteractive_class_init(VisuInteractiveClass *klass)
{
  GType paramPointer[1] = {G_TYPE_POINTER};
  GType paramBool[1] = {G_TYPE_BOOLEAN};
  GType paramUInt[1] = {G_TYPE_UINT};
  GType paramUInt_pointer[2] = {G_TYPE_UINT, G_TYPE_POINTER};
  VisuConfigFileEntry *resourceEntry;

  local_class = klass;

  DBG_fprintf(stderr, "visu Interactive: creating the class of the object.\n");

  DBG_fprintf(stderr, "                - adding new signals ;\n");
  /**
   * VisuInteractive::observe:
   * @obj: the object emitting the signal.
   * @bool: a boolean.
   *
   * This signal is emitted each time an observe session is start
   * (@bool is TRUE) or finished (@bool is FALSE).
   *
   * Since: 3.6
   */
  interactive_signals[INTERACTIVE_OBSERVE_SIGNAL] =
    g_signal_newv("observe", G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__BOOLEAN,
		  G_TYPE_NONE, 1, paramBool);

  /**
   * VisuInteractive::selection-error:
   * @obj: the object emitting the signal.
   * @err: an error value.
   *
   * This signal is emitted each time a selection fails, providing the
   * error in @err (see #VisuInteractivePickError).
   *
   * Since: 3.6
   */
  interactive_signals[INTERACTIVE_PICK_ERROR_SIGNAL] =
    g_signal_newv("selection-error", G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__UINT,
		  G_TYPE_NONE, 1, paramUInt);

  /**
   * VisuInteractive::node-selection:
   * @obj: the object emitting the signal.
   * @kind: a flag.
   * @nodes: an array of three node ids.
   *
   * This signal is emitted each time a single node selection succeed, providing the
   * kind in @kind (see #VisuInteractivePick). The corresponding nodes
   * are stored in @nodes.
   *
   * Since: 3.6
   */
  interactive_signals[INTERACTIVE_PICK_NODE_SIGNAL] =
    g_signal_newv("node-selection", G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__UINT_POINTER,
		  G_TYPE_NONE, 2, paramUInt_pointer);

  /**
   * VisuInteractive::region-selection:
   * @obj: the object emitting the signal.
   * @nodes: an array of node ids.
   *
   * This signal is emitted each time a region selection succeed. The corresponding nodes
   * are stored in @nodes.
   *
   * Since: 3.6
   */
  interactive_signals[INTERACTIVE_PICK_REGION_SIGNAL] =
    g_signal_newv("region-selection", G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuInteractive::start-move:
   * @obj: the object emitting the signal.
   * @nodes: an array of node ids.
   *
   * This signal is emitted each time a set of nodes are clicked to be
   * moved. The corresponding nodes are stored in @nodes.
   *
   * Since: 3.6
   */
  interactive_signals[INTERACTIVE_START_MOVE_SIGNAL] =
    g_signal_newv("start-move", G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuInteractive::move:
   * @obj: the object emitting the signal.
   * @nodes: an array of node ids.
   *
   * This signal is emitted each time a set of nodes are moved. The
   * corresponding nodes are stored in @nodes.
   *
   * Since: 3.6
   */
  interactive_signals[INTERACTIVE_MOVE_SIGNAL] =
    g_signal_newv("move", G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__POINTER,
		  G_TYPE_NONE, 1, paramPointer);

  /**
   * VisuInteractive::stop:
   * @obj: the object emitting the signal.
   *
   * This signal is emitted each time a set of nodes are stopped to be
   * moved.
   *
   * Since: 3.6
   */
  interactive_signals[INTERACTIVE_STOP_SIGNAL] =
    g_signal_newv("stop", G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL, NULL, NULL, g_cclosure_marshal_VOID__VOID,
		  G_TYPE_NONE, 0, NULL);

  /* Connect freeing methods. */
  G_OBJECT_CLASS(klass)->dispose  = visuInteractive_dispose;
  G_OBJECT_CLASS(klass)->finalize = visuInteractive_finalize;

  DBG_fprintf(stderr, "                - add the resources.\n");
  resourceEntry = visu_configFile_addEntry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_OBSERVE_METHOD,
					  DESC_PARAMETER_OBSERVE_METHOD,
					  1, readOpenGLObserveMethod);
  visu_configFile_addExportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParameters);

  /* Set local variables to default. */
  DBG_fprintf(stderr, "                - setup the local variables.\n");
  klass->preferedObserveMethod = interactive_constrained;
  klass->moveAtomExtension = (VisuExtension*)0;
}

static void visuInteractive_init(VisuInteractive *obj)
{
  DBG_fprintf(stderr, "Visu Interactive: creating a new interactive session (%p).\n",
	      (gpointer)obj);

  obj->dispose_has_run = FALSE;
  obj->idRef1          = -99;
  obj->idRef2          = -99;
  obj->idSelected      = -99;
  obj->idRegion        = (GList*)0;
  obj->movingAxe[0]    = 1.f;
  obj->movingAxe[1]    = 0.f;
  obj->movingAxe[2]    = 0.f;
  obj->movingPicked    = FALSE;
  obj->movingNodes     = (GList*)0;

  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(onVisuDataReady), (gpointer)obj);
}

static void onVisuDataReady(GObject *obj _U_, VisuData *dataObj, gpointer data)
{
  VisuInteractive *inter;

  inter = VISU_INTERACTIVE(data);
  g_return_if_fail(inter);

  if (inter->idRef1 >= 0 &&
      (!dataObj || !visu_data_getNodeFromNumber(dataObj, inter->idRef1)))
    inter->idRef1 = -99;
  if (inter->idRef2 >= 0 &&
      (!dataObj || !visu_data_getNodeFromNumber(dataObj, inter->idRef2)))
    inter->idRef2 = -99;
  if (inter->idSelected >= 0 &&
      (!dataObj || !visu_data_getNodeFromNumber(dataObj, inter->idSelected)))
    inter->idSelected = -99;
  DBG_fprintf(stderr, "Visu Interactive: specific nodes are now:\n");
  DBG_fprintf(stderr, " - selected    %d.\n", inter->idSelected);
  DBG_fprintf(stderr, " - ref. 1      %d.\n", inter->idRef1);
  DBG_fprintf(stderr, " - ref. 2      %d.\n", inter->idRef2);

  if (inter->idRegion)
    g_list_free(inter->idRegion);
  inter->idRegion = (GList*)0;

  if (dataObj)
    g_signal_connect(G_OBJECT(dataObj), "NodePopulationDecrease",
		     G_CALLBACK(onPopulationChange), (gpointer)inter);
}
static void onPopulationChange(VisuData *dataObj _U_, int *nodes, gpointer data)
{
  int i;
  VisuInteractive *inter;

  inter = VISU_INTERACTIVE(data);
  g_return_if_fail(inter);

  DBG_fprintf(stderr, "Visu Interactive:  remove the specific nodes.\n");
  for (i = 0; nodes[i] >= 0; i++)
    {
      if (inter->idRef1 == nodes[i])
	inter->idRef1 = -1;
      if (inter->idRef2 == nodes[i])
	inter->idRef2 = -1;
      if (inter->idSelected == nodes[i])
	inter->idSelected = -1;
    }
  if (inter->idRegion)
    g_list_free(inter->idRegion);
  inter->idRegion = (GList*)0;
}

/* This method can be called several times.
   It should unref all of its reference to
   GObjects. */
static void visuInteractive_dispose(GObject* obj)
{
  DBG_fprintf(stderr, "Visu Interactive: dispose object %p.\n", (gpointer)obj);

  if (VISU_INTERACTIVE(obj)->dispose_has_run)
    return;

  VISU_INTERACTIVE(obj)->dispose_has_run = TRUE;
  /* Chain up to the parent class */
  G_OBJECT_CLASS(visuInteractive_parent_class)->dispose(obj);
}
/* This method is called once only. */
static void visuInteractive_finalize(GObject* obj)
{
  g_return_if_fail(obj);

  DBG_fprintf(stderr, "Visu Interactive: finalize object %p.\n", (gpointer)obj);

  if (VISU_INTERACTIVE(obj)->movingNodes)
    g_list_free(VISU_INTERACTIVE(obj)->movingNodes);
  g_free(VISU_INTERACTIVE(obj));
  if (VISU_INTERACTIVE(obj)->idRegion)
    g_list_free(VISU_INTERACTIVE(obj)->idRegion);

  /* Chain up to the parent class */
  G_OBJECT_CLASS(visuInteractive_parent_class)->finalize(obj);
}

/**
 * visuInteractiveNew:
 * @type: a #VisuInteractiveId flag.
 *
 * Creates a new interactive session of the given @type.
 *
 * Returns: a newly created object.
 */
VisuInteractive* visuInteractiveNew(VisuInteractiveId type)
{
  VisuInteractive *inter;
  VisuInteractiveClass *klass;

  inter = VISU_INTERACTIVE(g_object_new(VISU_INTERACTIVE_TYPE, NULL));
  g_return_val_if_fail(inter, (VisuInteractive*)0);

  DBG_fprintf(stderr, "Visu Interactive: start new interactive session %p (%d).\n",
	      (gpointer)inter, type);

  inter->id = type;
  
  klass = VISU_INTERACTIVE_GET_CLASS(inter);
  if (type == interactive_move && !klass->moveAtomExtension)
    {
      klass->moveAtomExtension_list = visu_openGL_objectList_new(1);
      klass->moveAtomExtension = visu_extension_new(moveAtomName, moveAtomNameI18n,
						     moveAtomDescription,
						     klass->moveAtomExtension_list,
						     NULL);
      visu_extension_setPriority(klass->moveAtomExtension,
				  VISU_EXTENSION_PRIORITY_NODE_DECORATIONS);
      visuExtensions_add(klass->moveAtomExtension);
    }

  return inter;
}
/**
 * visuInteractiveGet_type:
 * @inter: a #VisuInteractive object.
 *
 * It returns the kind of interactive session.
 *
 * Returns: a #VisuInteractiveId value.
 */
VisuInteractiveId visuInteractiveGet_type(VisuInteractive *inter)
{
  g_return_val_if_fail(IS_VISU_INTERACTIVE(inter), interactive_none);

  return inter->id;
}

/**
 * visuInteractivePop_savedCamera:
 * @inter: a #VisuInteractive object.
 *
 * @inter object stores camera settings as a ring. This routine goes
 * to the next camera in the ring and returns the current one. The
 * popped camera is not actually removed from the ring.
 *
 * Since: 3.6
 *
 * Returns: a pointer to the previously current #VisuOpenGLCamera. It
 * is owned by V_Sim and should not be touched.
 */
VisuOpenGLCamera* visuInteractivePop_savedCamera(VisuInteractive *inter)
{
  VisuOpenGLCamera *cur;
  VisuInteractiveClass *klass;

  klass = VISU_INTERACTIVE_GET_CLASS(inter);
  g_return_val_if_fail(klass, (VisuOpenGLCamera*)0);

  if (!klass->lastCamera)
    return (VisuOpenGLCamera*)0;

  cur = (VisuOpenGLCamera*)klass->lastCamera->data;

  klass->lastCamera = g_list_next(klass->lastCamera);
  if (!klass->lastCamera)
    klass->lastCamera = klass->savedCameras;

  DBG_fprintf(stderr, "Interactive: pop, pointing now at camera %d.\n",
	      g_list_position(klass->savedCameras, klass->lastCamera));

  return cur;
}
/**
 * visuInteractiveGet_savedCameras:
 * @inter: a #VisuInteractive object.
 * @cameras: a location to store a list of cameras.
 * @head: a location to store a list of cameras.
 *
 * @inter object stores camera settings as a ring. One can access the
 * set of saved cameras thanks to @cameras or to the current position
 * in the ring thanks to @head. @cameras or @head are not copied and
 * are owned by V_Sim. They should be considered read-only.
 *
 * Since: 3.6
 */
void visuInteractiveGet_savedCameras(VisuInteractive *inter,
				     GList **cameras, GList **head)
{
  VisuInteractiveClass *klass;

  klass = VISU_INTERACTIVE_GET_CLASS(inter);
  g_return_if_fail(klass);
  
  *cameras = klass->savedCameras;
  *head    = klass->lastCamera;
}
static gboolean cmpCameras(VisuOpenGLCamera *c1, VisuOpenGLCamera *c2)
{
  return (c1 == c2) ||
    (c1->theta == c2->theta &&
     c1->phi == c2->phi &&
     c1->omega == c2->omega &&
     c1->xs == c2->xs &&
     c1->ys == c2->ys/*  && */
     /* c1->gross == c2->gross && */
     /* c1->d_red == c2->d_red */);
}
/**
 * visuInteractivePush_savedCamera:
 * @inter: a #VisuInteractive object.
 * @camera: a #VisuOpenGLCamera object.
 *
 * @inter object stores camera settings as a ring. The given @camera
 * is copied in the ring if its values not already exist. The current camera
 * is set to this new one.
 *
 * Since: 3.6
 */
void visuInteractivePush_savedCamera(VisuInteractive *inter, VisuOpenGLCamera *camera)
{
  VisuOpenGLCamera *tmp;
  VisuInteractiveClass *klass;

  klass = VISU_INTERACTIVE_GET_CLASS(inter);
  g_return_if_fail(klass && camera);

  for (klass->lastCamera = klass->savedCameras ;
       klass->lastCamera &&
	 !cmpCameras((VisuOpenGLCamera*)klass->lastCamera->data, camera);
       klass->lastCamera = g_list_next(klass->lastCamera));

  /* Case we don't find it, we add. */
  if (!klass->lastCamera || (VisuOpenGLCamera*)klass->lastCamera->data != camera)
    {
      tmp = g_malloc(sizeof(VisuOpenGLCamera));
      tmp->theta = camera->theta;
      tmp->phi   = camera->phi;
      tmp->omega = camera->omega;
      tmp->xs    = camera->xs;
      tmp->ys    = camera->ys;
      tmp->gross = camera->gross;
      tmp->d_red = camera->d_red;
      klass->savedCameras = g_list_prepend(klass->savedCameras, (gpointer)tmp);
    }
  klass->lastCamera = klass->savedCameras;
  DBG_fprintf(stderr, "Interactive: push, storing now %d cameras.\n",
	      g_list_length(klass->savedCameras));
}
gboolean visuInteractiveRemove_savedCamera(VisuInteractive *inter, VisuOpenGLCamera *camera)
{
  VisuInteractiveClass *klass;
  GList *lst;

  klass = VISU_INTERACTIVE_GET_CLASS(inter);
  g_return_val_if_fail(klass, FALSE);

  for (lst = klass->savedCameras ;
       lst && !cmpCameras((VisuOpenGLCamera*)lst->data, camera);
       lst = g_list_next(lst));

  /* Case we don't find it, we return. */
  if (!lst)
    return FALSE;

  g_free(lst->data);
  klass->savedCameras = g_list_delete_link(klass->savedCameras, lst);
  if (klass->lastCamera == lst)
    klass->lastCamera = lst->next;
  if (!klass->lastCamera)
    klass->lastCamera = klass->savedCameras;
  DBG_fprintf(stderr, "Interactive: remove, storing now %d cameras.\n",
	      g_list_length(klass->savedCameras));

  return TRUE;
}


/**
 * visuInteractiveHandle_event:
 * @inter: a #VisuInteractive object ;
 * @dataObj: the #VisuData object to interact with ;
 * @ev: a simplified event.
 *
 * This routine should be called by the rendering window when some
 * event is raised on the rendering surface.
 */
void visuInteractiveHandle_event(VisuInteractive *inter, VisuData *dataObj,
				 SimplifiedEvents *ev)
{
  gboolean stop;

  g_return_if_fail(IS_VISU_INTERACTIVE(inter));

  switch (inter->id)
    {
    case interactive_observe:
      stop = observe(inter, dataObj, ev);
      break;
    case interactive_measureAndObserve:
      stop = pickAndObserve(inter, dataObj, ev);
      break;
    case interactive_measure:
    case interactive_pick:
      stop = pick(inter, dataObj, ev);
      break;
    case interactive_move:
      stop = move(inter, dataObj, ev);
      break;
    case interactive_mark:
      stop = mark(inter, dataObj, ev);
      break;
    default:
      stop = FALSE;
      break;
    }

  if (stop)
    g_signal_emit(G_OBJECT(inter),
		  interactive_signals[INTERACTIVE_STOP_SIGNAL], 0, NULL);
}
/**
 * visuInteractiveSet_references:
 * @inter: a #VisuInteractive object.
 * @from: another #VisuInteractive object.
 *
 * Copies all node ids used as reference from @from to @inter.
 */
void visuInteractiveSet_references(VisuInteractive *inter, VisuInteractive *from)
{
  g_return_if_fail(IS_VISU_INTERACTIVE(inter) && IS_VISU_INTERACTIVE(from));

  inter->idSelected = from->idSelected;
  inter->idRef1     = from->idRef1;
  inter->idRef2     = from->idRef2;
}

/******************************************************************************/

static gboolean pickAndObserve(VisuInteractive *inter, VisuData *dataObj,
			       SimplifiedEvents *ev)
{
  /* If button 1 or 2, use observe mode */
  if (ev->button != 3)
    return observe(inter, dataObj, ev);
  /* If button 3, use pick mode with button 3 as button 1 */
  else
    {
      if (ev->shiftMod && !ev->controlMod)
	{
	  ev->button = 2;
	}
      else if (!ev->shiftMod && ev->controlMod)
	{
	  ev->button = 2;
	}
      else if (ev->shiftMod && ev->controlMod)
	{
	  ev->shiftMod = 0;
	  ev->controlMod = 1;
	  ev->button = 1;
	}
      else
	ev->button = 1;
      ev->motion = 0;
      
      return pick(inter, dataObj, ev);
    }
}

static gboolean observe(VisuInteractive *inter, VisuData *dataObj,
			SimplifiedEvents *ev)
{
  int reDrawNeeded;
  int sign_theta;
  int dx, dy;
  VisuOpenGLView *view;
  VisuOpenGLCamera *camera;
  float angles[3], ratio;
  gboolean zoom;

  g_return_val_if_fail(ev && inter, TRUE);
  if (ev->button == 3)
    return (ev->buttonType == BUTTON_TYPE_PRESS);
  /* If the realese event is triggered, we exit only if it's
     not a scroll event (button 4 and 5). */
  if (ev->button > 0 && ev->buttonType == BUTTON_TYPE_RELEASE)
    if (ev->button != 4 && ev->button != 5 )
      {
	if (ev->button == 1)
	  g_signal_emit(G_OBJECT(inter),
			interactive_signals[INTERACTIVE_OBSERVE_SIGNAL],
			0 , FALSE, NULL);
	return FALSE;
      }

  view = visu_data_getOpenGLView(dataObj);
  g_return_val_if_fail(view, FALSE);

/*   fprintf(stderr, "%d %d, %d\n", ev->x, ev->y, ev->button); */
/*   fprintf(stderr, "%d %d, %d\n", ev->shiftMod, ev->controlMod, ev->motion); */
  reDrawNeeded = 0;
  /* Support de la roulette en zoom et perspective. */
  if (ev->specialKey == Key_Page_Up || ev->specialKey == Key_Page_Down ||
      ev->button == 4 || ev->button == 5)
    {
      if ((ev->button == 4 || ev->specialKey == Key_Page_Up) && !ev->shiftMod)
	reDrawNeeded = visu_data_setZoomOfView(dataObj, view->camera->gross * 1.1);
      else if ((ev->button == 4 || ev->specialKey == Key_Page_Up) && ev->shiftMod)
	reDrawNeeded = visu_data_setPerspectiveOfView(dataObj, view->camera->d_red / 1.1);
      else if ((ev->button == 5 || ev->specialKey == Key_Page_Down) && !ev->shiftMod)
	reDrawNeeded = visu_data_setZoomOfView(dataObj, view->camera->gross / 1.1);
      else if ((ev->button == 5 || ev->specialKey == Key_Page_Down) && ev->shiftMod)
	reDrawNeeded = visu_data_setPerspectiveOfView(dataObj, view->camera->d_red * 1.1);
    }
  else if (ev->button && !ev->motion)
    {
      inter->xOrig = ev->x;
      inter->yOrig = ev->y;
      if (ev->button == 1)
	g_signal_emit(G_OBJECT(inter),
		      interactive_signals[INTERACTIVE_OBSERVE_SIGNAL],
		      0 , TRUE, NULL);
    }
  else if (ev->motion ||
	   ev->specialKey == Key_Arrow_Down  || ev->specialKey == Key_Arrow_Up ||
	   ev->specialKey == Key_Arrow_Right || ev->specialKey == Key_Arrow_Left)
    {
      if (ev->motion)
	{
	  dx =   ev->x - inter->xOrig;
	  dy = -(ev->y - inter->yOrig);
	}
      else
	{
	  dx = 0;
	  dy = 0;
	  if (ev->specialKey == Key_Arrow_Left)
	    dx = -10;
	  else if (ev->specialKey == Key_Arrow_Right)
	    dx = +10;
	  else if (ev->specialKey == Key_Arrow_Down)
	    dy = -10;
	  else if (ev->specialKey == Key_Arrow_Up)
	    dy = +10;
	}
      zoom = (ev->button == 2);
      if(!zoom && !ev->shiftMod && !ev->controlMod)
	{
	  if (local_class->preferedObserveMethod == interactive_constrained)
	    {
	      if(view->camera->theta > 0.0)
		sign_theta = 1;
	      else
		sign_theta = -1;
	      openGLViewRotate_box(view, dy * 180.0f / view->window->height,
				   -dx * 180.0f / view->window->width * sign_theta, angles);
	      reDrawNeeded = visu_data_setAngleOfView(dataObj, angles[0], angles[1],
						     0., VISU_CAMERA_THETA | VISU_CAMERA_PHI);
	    }
	  else if (local_class->preferedObserveMethod == interactive_walker)
	    {
	      openGLViewRotate_camera(view, dy * 180.0f / view->window->height,
				      -dx * 180.0f / view->window->width, angles);
	      reDrawNeeded = visu_data_setAngleOfView(dataObj, angles[0], angles[1], angles[2],
						     VISU_CAMERA_THETA | VISU_CAMERA_PHI | VISU_CAMERA_OMEGA);
	    }
	}
      else if(!zoom && ev->shiftMod && !ev->controlMod)
	{
	  ratio = 1. / MIN(view->window->width, view->window->height) /
	    view->camera->gross * (view->camera->d_red - 1.f) / view->camera->d_red;
	  reDrawNeeded =
	    visu_data_setPositionOfView(dataObj,
				       view->camera->xs + (float)dx * ratio,
				       view->camera->ys + (float)dy * ratio,
				       MASK_XS | MASK_YS);
	}
      else if(!zoom && ev->controlMod && !ev->shiftMod)
	{
	  if (abs(dx) > abs(dy))
	    reDrawNeeded = visu_data_setAngleOfView(dataObj, 0., 0.,
						   view->camera->omega + dx *
						   180.0f / view->window->width,
						   VISU_CAMERA_OMEGA);
	  else
	    reDrawNeeded = visu_data_setAngleOfView(dataObj, 0., 0.,
						   view->camera->omega + dy *
						   180.0f / view->window->height,
						   VISU_CAMERA_OMEGA);
	}
      else if(zoom && !ev->shiftMod)
	reDrawNeeded = visu_data_setZoomOfView(dataObj, view->camera->gross *
					      (1. + (float)dy *
					       3.0f / view->window->height));
      else if(zoom && ev->shiftMod)
	reDrawNeeded = visu_data_setPerspectiveOfView(dataObj, view->camera->d_red *
						     (1. - (float)dy *
						      5.0f / view->window->height));
      if (ev->motion)
	{
	  inter->xOrig = ev->x;
	  inter->yOrig = ev->y;
	}
    }
  else if (ev->letter == 'r')
    {
      camera = visuInteractivePop_savedCamera(inter);
      if (camera)
	{
	  reDrawNeeded = visu_data_setAngleOfView(dataObj,
						 camera->theta,
						 camera->phi,
						 camera->omega,
						 VISU_CAMERA_THETA |
						 VISU_CAMERA_PHI | VISU_CAMERA_OMEGA) ||
	    reDrawNeeded;
	  reDrawNeeded = visu_data_setPositionOfView(dataObj,
						    camera->xs,
						    camera->ys,
						    MASK_XS | MASK_YS) ||
	    reDrawNeeded;
	  reDrawNeeded = visu_data_setZoomOfView(dataObj, camera->gross) ||
	    reDrawNeeded;
	  reDrawNeeded = visu_data_setPerspectiveOfView(dataObj, camera->d_red) ||
	    reDrawNeeded;
	}
    }
  else if (ev->letter == 's' && !ev->shiftMod && !ev->controlMod)
    visuInteractivePush_savedCamera(inter, view->camera);
  else if (ev->letter == 's' && ev->shiftMod && !ev->controlMod)
    visuInteractiveRemove_savedCamera(inter, view->camera);
  if (reDrawNeeded)
    {
      DBG_fprintf(stderr, "Visu Interactive: redraw on observe.\n");
      g_idle_add_full(G_PRIORITY_HIGH_IDLE, visu_object_redrawForce,
		      (gpointer)__func__, (GDestroyNotify)0);
    }
  return FALSE;
}
static void glDrawSelection(VisuInteractive *inter, int x, int y)
{
  int viewport[4];

  glPushAttrib(GL_ENABLE_BIT);
  glDisable(GL_FOG);
  glEnable(GL_BLEND);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_LIGHTING);
  glBlendFunc(GL_ONE_MINUS_DST_COLOR, GL_ONE_MINUS_SRC_COLOR);
  glGetIntegerv(GL_VIEWPORT, viewport);
  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D(0.0, (float)viewport[2], 0., (float)viewport[3]);   
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glLoadIdentity();
  glColor4f(1.f, 1.f, 1.f, 1.f);
  glLineWidth(2);
  glDrawBuffer(GL_FRONT);
  /* We erase the previous selecting rectangle. */
  glBegin(GL_LINE_LOOP);
  glVertex3f(inter->xOrig, (float)viewport[3] - inter->yOrig, 0.f);
  glVertex3f(inter->xPrev, (float)viewport[3] - inter->yOrig, 0.f);
  glVertex3f(inter->xPrev, (float)viewport[3] - inter->yPrev, 0.f);
  glVertex3f(inter->xOrig, (float)viewport[3] - inter->yPrev, 0.f);
  glEnd();
  glFlush();

  /* We draw the new selecting rectangle. */
  if (x > 0 && y > 0)
    {
      glBegin(GL_LINE_LOOP);
      glVertex3f(inter->xOrig, (float)viewport[3] - inter->yOrig, 0.f);
      glVertex3f(x, (float)viewport[3] - inter->yOrig, 0.f);
      glVertex3f(x, (float)viewport[3] - y, 0.f);
      glVertex3f(inter->xOrig, (float)viewport[3] - y, 0.f);
      glEnd();
      glFlush();
    }
  glDrawBuffer(GL_BACK);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glPopMatrix();
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);
  glPopAttrib();
}

static gboolean pick(VisuInteractive *inter, VisuData *dataObj, SimplifiedEvents *ev)
{
  int nodeId, nodeRefId, nodeRef2Id;
  GList *region;
  gboolean pickOnly;
  VisuInteractivePick pick;
  VisuInteractivePickError error;
  VisuNode *nodes[3];

  g_return_val_if_fail(ev && inter, TRUE);
  /* We store the pickInfo into the VisuData on press,
     but we apply it only at release if there was no drag action. */
  if (ev->button == 3)
    {
      if (ev->buttonType == BUTTON_TYPE_PRESS)
	return TRUE;
      else
	return FALSE;
    }

  pickOnly = (inter->id == interactive_pick);

  if (ev->button == 1 && ev->motion && !pickOnly)
    {
      /* We drag a selecting area. */
      /* A nodeInfo should have been stored, we retrieve it. */
      DBG_fprintf(stderr, "Interactive: pick, drag to %dx%d.\n", ev->x, ev->y);
      glDrawSelection(inter, ev->x, ev->y);
      inter->xPrev = ev->x;
      inter->yPrev = ev->y;
    }
  else if (ev->buttonType == BUTTON_TYPE_PRESS)
    {
      /* We create and store a nodeInfo data. */
      DBG_fprintf(stderr, "Interactive: pick, press at %dx%d.\n", ev->x, ev->y);
      inter->xOrig = ev->x;
      inter->yOrig = ev->y;
      inter->xPrev = ev->x;
      inter->yPrev = ev->y;
    }
  else if (ev->buttonType == BUTTON_TYPE_RELEASE)
    {
      /* Reset current selection. */
      if (inter->idRegion)
	g_list_free(inter->idRegion);
      inter->idRegion = (GList*)0;
      inter->idSelected = -1;

      /* Set new selection from click. */
      pick = PICK_NONE;
      /* If no drag action, we select the node and compute the pick mesure. */
      if ((inter->xOrig == inter->xPrev &&
	   inter->yOrig == inter->yPrev) || pickOnly)
	{
	  nodeId = getSelectElement(dataObj, ev->x, ev->y);
	  nodeRefId = nodeRef2Id = -1;

	  error = PICK_ERROR_NO_SELECTION;

	  DBG_fprintf(stderr, "Visu Interactive: set selection (single %d %d).\n",
		      ev->shiftMod, ev->controlMod);
	  if (pickOnly && (ev->button == 1 || ev->button == 2))
	    {
	      DBG_fprintf(stderr, "Visu Interactive: set pick node %d.\n", nodeId);
	      inter->idSelected = nodeId;
	      pick = (nodeId >= 0)?PICK_SELECTED:PICK_NONE;
	    }
	  else if (!pickOnly && ev->button == 1 && !ev->controlMod)
	    {
	      if (nodeId >= 0)
		{
		  pick = PICK_SELECTED;
		  if (nodeId == inter->idRef1 || nodeId == inter->idRef2)
		    {
		      error = PICK_ERROR_SAME_REF;
		      pick = PICK_NONE;
		    }

		  if (pick != PICK_NONE)
		    {
		      inter->idSelected = nodeId;
		      if (inter->idRef1 >= 0)
			pick = (inter->idRef2 < 0)?PICK_DISTANCE:PICK_ANGLE;
		    }
		}
	      else
		pick = PICK_NONE;
	    }
	  else if (!pickOnly && ev->button == 1 && ev->controlMod)
	    pick = (nodeId < 0)?PICK_NONE:PICK_HIGHLIGHT;
	  else if (!pickOnly && ev->button == 2 && ev->shiftMod && !ev->controlMod)
	    {
	      DBG_fprintf(stderr, "Visu Interactive: set ref1 info for node %d.\n",
			  nodeId);
	      pick = PICK_REFERENCE_1;
	      /* The error case. */
	      if (nodeId >= 0 && inter->idRef2 == nodeId)
		{
		  error = PICK_ERROR_SAME_REF;
		  pick = PICK_NONE;
		}
	      else if (nodeId < 0 && inter->idRef2 >= 0)
		{
		  error = PICK_ERROR_REF2;
		  pick = PICK_NONE;
		}

	      if (pick != PICK_NONE)
		{
		  inter->idRef1 = nodeId;
		  pick = (nodeId < 0)?PICK_UNREFERENCE_1:PICK_REFERENCE_1;
		}
	    }
	  else if (!pickOnly && ev->button == 2 && !ev->shiftMod && ev->controlMod)
	    {
	      DBG_fprintf(stderr, "Visu Interactive: set ref2 info for node %d.\n",
			  nodeId);
	      pick = PICK_REFERENCE_2;
	      /* The error case. */
	      if (nodeId >= 0 && inter->idRef1 < 0)
		{
		  error = PICK_ERROR_REF1;
		  pick = PICK_NONE;
		}
	      else if (nodeId >= 0 && inter->idRef1 == nodeId)
		{
		  error = PICK_ERROR_SAME_REF;
		  pick = PICK_NONE;
		}

	      if (pick != PICK_NONE)
		{
		  inter->idRef2 = nodeId;
		  pick = (nodeId < 0)?PICK_UNREFERENCE_2:PICK_REFERENCE_2;
		}
	    }
	  else if (!pickOnly && ev->button == 2 && !ev->shiftMod && !ev->controlMod)
	    pick = (nodeId < 0)?PICK_NONE:PICK_INFORMATION;
	  else
	    return FALSE;

	  DBG_fprintf(stderr, " | OK.\n");
	  if (pick != PICK_NONE)
	    {
	      nodes[0] = visu_data_getNodeFromNumber(dataObj, nodeId);
	      nodes[1] = visu_data_getNodeFromNumber
		(dataObj, (pick != PICK_REFERENCE_1)?inter->idRef1:nodeId);
	      nodes[2] = visu_data_getNodeFromNumber
		(dataObj, (pick != PICK_REFERENCE_2)?inter->idRef2:nodeId);
	      g_signal_emit(G_OBJECT(inter),
			    interactive_signals[INTERACTIVE_PICK_NODE_SIGNAL],
			    0 , pick, (gpointer)nodes, NULL);
	    }
	  else if (pick == PICK_NONE && error != PICK_ERROR_NONE)
	    g_signal_emit(G_OBJECT(inter),
			  interactive_signals[INTERACTIVE_PICK_ERROR_SIGNAL],
			  0 , error, NULL);
	}
      /* If drag action, we compute all the nodes in the rectangle
	 and we erase it. */
      else
	{
	  /* We get the list of selected nodes. */
	  glDrawSelection(inter, -1, -1);
	  /* Get all nodes in the region. */
	  region = getSelectElementsRegion(dataObj,
					  inter->xOrig, inter->yOrig,
					  ev->x, ev->y);
	  DBG_fprintf(stderr, "Interactive: set selection (drag).\n");
	  /* Copy the region list. */
	  if (region)
	    {
	      inter->idRegion = region;
	      g_signal_emit(G_OBJECT(inter),
			    interactive_signals[INTERACTIVE_PICK_REGION_SIGNAL],
			    0 , region, NULL);
	    }
	  else
	    g_signal_emit(G_OBJECT(inter),
			  interactive_signals[INTERACTIVE_PICK_ERROR_SIGNAL],
			  0 , PICK_ERROR_NO_SELECTION, NULL);
	}
    }
  return FALSE;
}

static gboolean move(VisuInteractive *inter, VisuData *dataObj, SimplifiedEvents *ev)
{
  int dx, dy, nodeId;
  float ratio, z, xyz[3];
  VisuOpenGLView *view;
  VisuNode *node;
  GList *tmpLst;
  float delta[3], centre[3];

  g_return_val_if_fail(ev && inter, TRUE);
  if (ev->button == 3)
    {
      if (ev->buttonType == BUTTON_TYPE_PRESS)
	return TRUE;
      else
	return FALSE;
    }
  if (ev->button != 1 && ev->button != 2 && ev->button != 4 && ev->button != 5 &&
      ev->specialKey != Key_Arrow_Left && ev->specialKey != Key_Arrow_Right &&
      ev->specialKey != Key_Arrow_Up && ev->specialKey != Key_Arrow_Down)
    return FALSE;
  DBG_fprintf(stderr, "Visu Interactive: event (%d).\n", ev->button);

  if ((ev->motion == 1 || ev->button == 4 || ev->button == 5) &&
      inter->movingNodes)
    {
      DBG_fprintf(stderr, "Visu Interactive: drag action (%dx%d).\n", ev->x, ev->y);
      dx =   ev->x - inter->xPrev;
      dy = -(ev->y - inter->yPrev);
      DBG_fprintf(stderr, " | dx x dy : %d x %d\n", dx, dy);

      /* Get the camera orientation. */
      view = visu_data_getOpenGLView(dataObj);      
      if (!ev->shiftMod && !ev->controlMod)
	{
	  if (ev->button == 1)
	    {
	      visu_data_getBoxCentre(dataObj, centre);
	      glMatrixMode(GL_MODELVIEW); 
	      glPushMatrix();
	      glTranslated(-centre[0], -centre[1], -centre[2]);
	      if (inter->movingPicked)
		{
		  node = visu_data_getNodeFromNumber
		    (dataObj, GPOINTER_TO_INT(inter->movingNodes->data));
		  visu_data_getNodePosition(dataObj, node, xyz);
		  z = openGLViewGet_zCoordinate(view, xyz);
		}
	      else
		{
		  z = 0.f;
		  openGLViewGet_realCoordinates(view, xyz,
						(float)inter->xPrev,
						(float)inter->yPrev, z);
		}
	      /* Now, xyz contains the prev real space coord. */

	      openGLViewGet_realCoordinates(view, delta, (float)ev->x,
					    (float)ev->y, z);
	      glPopMatrix();
	      /* Now, delta contains the new real space coord. */

	      delta[0] -= xyz[0];
	      delta[1] -= xyz[1];
	      delta[2] -= xyz[2];
	    }
	  else if (ev->button > 1)
	    {
	      if (ev->button == 4)
		dy = -5;
	      else if (ev->button == 5)
		dy = +5;
	      ratio = VisuOpenGLViewGet_fileUnitPerPixel(view) * dy;
	      delta[0] = inter->movingAxe[0] * ratio;
	      delta[1] = inter->movingAxe[1] * ratio;
	      delta[2] = inter->movingAxe[2] * ratio;
	    }
	}
      else if (ev->button == 1)
	{
	  delta[0] = 0.f;
	  delta[1] = 0.f;
	  delta[2] = 0.f;
	  ratio = VisuOpenGLViewGet_fileUnitPerPixel(view);
	  if (ABS(dy) > ABS(dx))
	    ratio *= ((dy > 0)?1.f:-1.f);
	  else
	    ratio *= ((dx > 0)?1.f:-1.f);
	  if (ev->shiftMod && !ev->controlMod)
	    delta[0] = ratio * sqrt(dx * dx + dy * dy);
	  else if (ev->controlMod && !ev->shiftMod)
	    delta[1] = ratio * sqrt(dx * dx + dy * dy);
	  else if (ev->controlMod && ev->shiftMod)
	    delta[2] = ratio * sqrt(dx * dx + dy * dy);
	}
      else
	return FALSE;

      DBG_fprintf(stderr, "Visu Interactive: drag a list of %d nodes of %gx%gx%g.\n",
		  g_list_length(inter->movingNodes),
		  delta[0], delta[1], delta[2]);

      glNewList(local_class->moveAtomExtension_list, GL_COMPILE);
      for (tmpLst = inter->movingNodes; tmpLst; tmpLst = g_list_next(tmpLst))
	{
	  node = visu_data_getNodeFromNumber(dataObj,
					    GPOINTER_TO_INT(tmpLst->data));
	  if (node)
	    {
	      node->xyz[0] += delta[0];
	      node->xyz[1] += delta[1];
	      node->xyz[2] += delta[2];
	      if (ev->button != 4 && ev->button != 5)
		visu_data_createNode(dataObj, node);
	    }
	}
      glEndList();
      if (ev->button == 4 || ev->button == 5)
	visu_data_createAllNodes(dataObj);
      visu_data_emitNodePositionChanged(dataObj);

      /* Update stored position for drag info. */
      inter->xPrev = ev->x;
      inter->yPrev = ev->y;

      DBG_fprintf(stderr, "Visu Interactive: emit the 'move' signal.\n");
      g_signal_emit(G_OBJECT(inter),
		    interactive_signals[INTERACTIVE_MOVE_SIGNAL],
		    0 , delta, NULL);
    }
/*   else if (ev->specialKey != Key_None) */
/*     { */
/*       DBG_fprintf(stderr, "Interactive: keyboard drag action.\n"); */
      
/*       dx = dy = 0; */
/*       switch (ev->specialKey) */
/* 	{ */
/* 	case Key_Arrow_Left: */
/* 	  dx = -1; dy =  0; break; */
/* 	case Key_Arrow_Right: */
/* 	  dx = +1; dy =  0; break; */
/* 	case Key_Arrow_Up: */
/* 	  dx =  0; dy = +1; break; */
/* 	case Key_Arrow_Down: */
/* 	  dx =  0; dy = -1; break; */
/* 	default: */
/* 	  g_error("Key not handled for drag."); */
/* 	} */

/*       view = visu_data_getOpenGLView(dataObj); */
/*       for (i = 0; i < 3; i++) */
/* 	{ */
/* 	  xAxis[i] = 0.; */
/* 	  yAxis[i] = 0.; */
/* 	  zAxis[i] = 0.; */
/* 	} */
/*       if (!ev->shiftMod && !ev->controlMod) */
/* 	VisuOpenGLViewGet_screenAxes(view, xAxis, yAxis); */
/*       else */
/* 	{ */
/* 	  if (ev->shiftMod) */
/* 	    { */
/* 	      xAxis[0] = 1.; */
/* 	      yAxis[1] = 1.; */
/* 	    } */
/* 	  else if (ev->controlMod) */
/* 	    zAxis[2] = 1.; */
/* 	} */
/*       ratio = VisuOpenGLViewGet_fileUnitPerPixel(view); */
/*       pickMesureSet_dragMove(inter->pick, */
/* 			     ratio * (dx * xAxis[0] + dy * yAxis[0] + dy * zAxis[0]), */
/* 			     ratio * (dx * xAxis[1] + dy * yAxis[1] + dy * zAxis[1]), */
/* 			     ratio * (dx * xAxis[2] + dy * yAxis[2] + dy * zAxis[2])); */
/*     } */
  else if ((ev->button == 1 || ev->button == 2) &&
	   ev->buttonType == BUTTON_TYPE_PRESS)
    {
      nodeId = getSelectElement(dataObj, ev->x, ev->y);
      /* Store the position to find the drag values. */
      inter->xOrig = inter->xPrev = ev->x;
      inter->yOrig = inter->yPrev = ev->y;

      /* Case no list set yet. */
      inter->movingPicked = FALSE;
      if (inter->movingNodes == (GList*)0)
	{
	  if (nodeId < 0)
	    return FALSE;

	  inter->movingNodes = g_list_append(inter->movingNodes,
						   GINT_TO_POINTER(nodeId));
	  inter->movingPicked = TRUE;
	}

      /* Setup the OpenGL trick to move nodes. */
      /* Hide all nodes. */
      node = (VisuNode*)0;
      for (tmpLst = inter->movingNodes; tmpLst; tmpLst = g_list_next(tmpLst))
	{
	  node = visu_data_getNodeFromNumber(dataObj,
					    GPOINTER_TO_INT(tmpLst->data));
	  visu_node_setVisibility(node, FALSE);
	}
      g_return_val_if_fail(node, FALSE);
      g_return_val_if_fail(dataObj, FALSE);

      /* If we have only one element, we recreate the list of only this
	 element, otherwise, we recreate all the elements. */
      if (!inter->movingNodes->next)
	visu_data_createNodes
	  (dataObj,
	   dataObj->fromIntToVisuElement[node->posElement]);
      else
	visu_data_createAllNodes(dataObj);

      /* Show again the selected nodes and add it in
	 the specific moveAtom list. */
      glNewList(local_class->moveAtomExtension_list, GL_COMPILE);
      for (tmpLst = inter->movingNodes; tmpLst; tmpLst = g_list_next(tmpLst))
	{
	  node = visu_data_getNodeFromNumber(dataObj,
					    GPOINTER_TO_INT(tmpLst->data));
	  visu_node_setVisibility(node, TRUE);
	  visu_data_createNode(dataObj, node);
	}
      glEndList();
      local_class->moveAtomExtension->used = 1;

      DBG_fprintf(stderr, "Visu Interactive: emit the 'start-move' signal.\n");
      g_signal_emit(G_OBJECT(inter),
		    interactive_signals[INTERACTIVE_START_MOVE_SIGNAL],
		    0, inter->movingNodes, NULL);
    }
  else if ((ev->button == 1 || ev->button == 2) &&
	   ev->buttonType == BUTTON_TYPE_RELEASE &&
	   inter->movingNodes)
    {
      DBG_fprintf(stderr, "Visu Interactive: stop dragging a list of %d nodes.\n",
		  g_list_length(inter->movingNodes));
      /* Show again the selected element in the allElement list. */
      if (!inter->movingNodes->next)
	{
	  node = visu_data_getNodeFromNumber(dataObj,
					    GPOINTER_TO_INT(inter->movingNodes->data));
	  g_return_val_if_fail(node, FALSE);

	  visu_data_createNodes(dataObj,
			       dataObj->fromIntToVisuElement[node->posElement]);
	}
      else
	visu_data_createAllNodes(dataObj);

      /* Stop the move extension. */
      local_class->moveAtomExtension->used = 0;

      /* Remove the movingList if only one node. */
      if (inter->movingPicked)
	{
	  g_list_free(inter->movingNodes);
	  inter->movingNodes = (GList*)0;
	  inter->movingPicked = FALSE;
	}
    }
  else
    return FALSE;

  /* Force redraw */
  g_idle_add_full(G_PRIORITY_HIGH_IDLE, visu_object_redrawForce,
		  (gpointer)__func__, (GDestroyNotify)0);

  return FALSE;
}
/**
 * visuInteractiveSet_movingNodes:
 * @inter: a #VisuInteractive object.
 * @nodeIds: a list of node ids.
 *
 * Defines the nodes that should be moved if @inter is a move
 * action session. The list is actually copied.
 */
void visuInteractiveSet_movingNodes(VisuInteractive *inter, GList *nodeIds)
{
  DBG_fprintf(stderr, "Visu Interactive: set the list of %d nodes to move.\n",
	      g_list_length(nodeIds));

  g_return_if_fail(IS_VISU_INTERACTIVE(inter) && inter->id == interactive_move);

  /* Empty the node list. */
  if (inter->movingNodes)
    g_list_free(inter->movingNodes);
  inter->movingNodes = g_list_copy(nodeIds);
  inter->movingPicked = FALSE;
}
/**
 * visuInteractiveSet_movingAxe:
 * @inter: a #VisuInteractive object.
 * @axe: a direction.
 *
 * Defines the axe that can be used to move along if @inter is a move
 * action session.
 */
void visuInteractiveSet_movingAxe(VisuInteractive *inter, float axe[3])
{
  float norm;

  norm = 1.f / sqrt(axe[0] * axe[0] + axe[1] * axe[1] + axe[2] * axe[2]);
  inter->movingAxe[0] = axe[0] * norm;
  inter->movingAxe[1] = axe[1] * norm;
  inter->movingAxe[2] = axe[2] * norm;
}

static gboolean mark(VisuInteractive *inter, VisuData *dataObj, SimplifiedEvents *ev)
{
  int nodeId;
  VisuNode *nodes[3];

  g_return_val_if_fail(ev && inter, TRUE);
  if (ev->button == 3 && ev->buttonType == BUTTON_TYPE_PRESS)
    return TRUE;
  if (ev->buttonType == BUTTON_TYPE_RELEASE)
    return FALSE;

  nodeId = getSelectElement(dataObj, ev->x, ev->y);

  if (nodeId >= 0)
    {
      nodes[0] = visu_data_getNodeFromNumber(dataObj, nodeId);
      nodes[1] = (VisuNode*)0;
      nodes[2] = (VisuNode*)0;
      g_signal_emit(G_OBJECT(inter),
		    interactive_signals[INTERACTIVE_PICK_NODE_SIGNAL],
		    0 , PICK_HIGHLIGHT, (gpointer)nodes, NULL);
    }
  else
    g_signal_emit(G_OBJECT(inter),
		  interactive_signals[INTERACTIVE_PICK_ERROR_SIGNAL],
		  0 , PICK_ERROR_NO_SELECTION, NULL);
  return FALSE;
}

static GList* getSelectElementsRegion(VisuData *dataObj, int x1, int y1,
				      int x2, int y2)
{
   GLsizei bufsize;
   GLuint *select_buf;
   GLint viewport[4] = {0, 0, 0, 0};
   int hits, names, ptr, i;
   VisuOpenGLView *view;
   GList *lst;
   float centre[3];
 
   g_return_val_if_fail(IS_VISU_DATA_TYPE(dataObj), (GList*)0);
   
   DBG_fprintf(stderr, "Interactive: get elements in region %dx%d - %dx%d.\n",
	       x1, x2, y1, y2);

   if ((x1 == x2) || (y1 == y2))
     return (GList*)0;

   view = visu_data_getOpenGLView(dataObj);
   visu_data_getBoxCentre(dataObj, centre);

   bufsize = visu_data_getNodeArray(dataObj)->nbOfAllStoredNodes * 4;
   select_buf = g_malloc(sizeof(GLuint) * bufsize);
   glSelectBuffer(bufsize, select_buf);
   hits = glRenderMode(GL_SELECT);
   glInitNames();
   glPushName(-1);
   
   lst = (GList*)0;
   viewport[2] = view->window->width;
   viewport[3] = view->window->height;

   glNewList(10, GL_COMPILE);
   gluPickMatrix(0.5f * (x1 + x2) , (float)view->window->height - 0.5f * (y1 + y2),
		 (float)ABS(x2 - x1), (float)ABS(y2 - y1), viewport);
   glEndList();

   glMatrixMode(GL_PROJECTION);
   glPushMatrix();
   glLoadIdentity();
   glCallList(10);
   glFrustum(view->window->left, view->window->right, view->window->bottom,
	     view->window->top, view->window->near, view->window->far);
   glMatrixMode(GL_MODELVIEW); 
   glPushMatrix();
   glTranslated(-centre[0], -centre[1], -centre[2]);

   glCallList(visu_data_getObjectList(dataObj));
   glFlush();

   hits = glRenderMode(GL_RENDER);
   DBG_fprintf(stderr, "%d elements are on the z buffer %dx%d - %dx%d.\n", hits,
	       x1, y1, x2, y2);
   ptr = 0;

   /* return the buffer to normal */
   glPopMatrix();
   glMatrixMode(GL_PROJECTION);
   glPopMatrix();
   glMatrixMode(GL_MODELVIEW); 

   for(i=0; i<hits; i++)
     {
       names = select_buf[ptr];
       if (names != 1)
	 {
	   g_warning("OpenGL picking is not working???\n");
	   return (GList*)0;
	 }
       ptr = ptr + 3;
       lst = g_list_prepend(lst, GINT_TO_POINTER((int)select_buf[ptr]));
       ptr = ptr + 1;
     }
   g_free(select_buf);

   return lst;
}

static int getSelectElement(VisuData *dataObj, int x, int y)
{
#define bufsize 512
   GLuint select_buf[bufsize];
   GLint viewport[4] = {0, 0, 0, 0};
#define wpck 2.0
#define hpck 2.0
   int hits, names, ptr, i;
   unsigned int z1;
   unsigned int z1_sauve = UINT_MAX;
   int number;
   int found;
   VisuOpenGLView *view;
   float centre[3];
 
   g_return_val_if_fail(IS_VISU_DATA_TYPE(dataObj), 0);

   view = visu_data_getOpenGLView(dataObj);
   visu_data_getBoxCentre(dataObj, centre);

   glSelectBuffer(bufsize, select_buf);
   hits = glRenderMode(GL_SELECT);
   glInitNames();
   glPushName(-1);
      
   viewport[2] = view->window->width;
   viewport[3] = view->window->height;
   glNewList(10, GL_COMPILE);
      gluPickMatrix(1.0*x, 1.0*(view->window->height-y), wpck, hpck, viewport);
   glEndList();

   glMatrixMode(GL_PROJECTION);
   glPushMatrix();
   glLoadIdentity();
   glCallList(10);
   DBG_fprintf(stderr, "Interactive: frustum %f %f.\n",
	       view->window->near, view->window->far);
   glFrustum(view->window->left, view->window->right, view->window->bottom,
	     view->window->top, view->window->near, view->window->far);
   glMatrixMode(GL_MODELVIEW); 
   glPushMatrix();
   glTranslated(-centre[0], -centre[1], -centre[2]);
   DBG_fprintf(stderr, "Interactive: translated %f %f %f.\n",
	       -centre[0], -centre[1], -centre[2]);

   glCallList(visu_data_getObjectList(dataObj));
   glFlush();

   hits = glRenderMode(GL_RENDER);
   DBG_fprintf(stderr, "Interactive: %d elements are on the z buffer.\n", hits);
   ptr = 0;

   /* return the buffer to normal */
   glPopMatrix();
   glMatrixMode(GL_PROJECTION);
   glPopMatrix();
   glMatrixMode(GL_MODELVIEW); 

   found = 0;
   number = -1;
   for(i=0; i<hits; i++) {
      names = select_buf[ptr];
      if (names != 1)
	{
	  g_warning("OpenGL picking is not working???\n");
	  return -1;
	}
      ptr = ptr + 1;
      z1 = select_buf[ptr];
      DBG_fprintf(stderr, " | z position %f for %d\n", (float)z1/0x7fffffff,
		  (int)select_buf[ptr + 2]);
      ptr = ptr + 2;
      if (z1 < z1_sauve) {
         z1_sauve = z1;
	 number = (int)select_buf[ptr];
	 found = 1;
      }
      ptr = ptr + 1;
   }
   if (found && number >= 0)
     return number;
   else
     return -1;
}

static gboolean readOpenGLObserveMethod(gchar **lines, int nbLines, int position,
					VisuData *dataObj _U_, GError **error)
{
  int val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!tool_configFile_readInteger(lines[0], position, &val, 1, error))
    return FALSE;
  if (val != interactive_constrained && val != interactive_walker)
    {
      *error = g_error_new(TOOL_CONFIGFILE_ERROR, TOOL_CONFIGFILE_ERROR_VALUE,
			   _("Parse error at line %d: width must be in %d-%d.\n"),
			   position, 0, 500);
      return FALSE;
    }
  visuInteractiveClassSet_preferedObserveMethod(val);

  return TRUE;
}
static void exportParameters(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_OBSERVE_METHOD);
  g_string_append_printf(data, "%s[gtk]: %d\n\n", FLAG_PARAMETER_OBSERVE_METHOD,
			 visuInteractiveClassGet_preferedObserveMethod());
}
void visuInteractiveClassSet_preferedObserveMethod(VisuInteractiveMethod method)
{
  g_return_if_fail(method == interactive_constrained ||
		   method == interactive_walker);

  if (!local_class)
    visuInteractive_get_type();

  local_class->preferedObserveMethod = method;
}
VisuInteractiveMethod visuInteractiveClassGet_preferedObserveMethod()
{
  if (!local_class)
    visuInteractive_get_type();

  return local_class->preferedObserveMethod;
}

/* From an array of nodes ids, we specify their 2D coordinates */
void getNodes2DCoordinates(VisuData *dataObj, unsigned int *nodeIds,
    	unsigned int nNodes, GLfloat *coordinates2D, unsigned int *size)
{
  int i,j;
  unsigned int k;
  VisuNode *node;
  float xyz[3], centre[3];
  GLfloat *coord;
  GLint nValues;
  VisuOpenGLView *view;

  view = visu_data_getOpenGLView(dataObj);
  visu_data_getBoxCentre(dataObj, centre);
  coord = g_malloc(sizeof(GLfloat) * nNodes * (1 + 2));
  glFeedbackBuffer(nNodes * (2 + 1), GL_2D, coord);
  glRenderMode(GL_FEEDBACK);
  glPushMatrix();
  glTranslated(-centre[0], -centre[1], -centre[2]);

  /* Get nodes from ids and store their 2D coordinates */
  glBegin(GL_POINTS);
  for(k = 0; k < nNodes; k++){
    node = visu_data_getNodeFromNumber(dataObj, nodeIds[k]);
    if(node != NULL) 
    {
      visu_data_getNodePosition(dataObj, node, xyz);
      glVertex3fv(xyz);
    }
  }
  glEnd();
  glPopMatrix();
  nValues = glRenderMode(GL_RENDER);
  i = 0;
  j = 0;
  /* Keep only the coordinates */
  while (i < nValues)
  {
    if (coord[i] == GL_POINT_TOKEN)
    {
      coordinates2D[j] = coord[i + 1];
      coordinates2D[j+1] = coord[i + 2];
      i += 3;
      j += 2;
    }
    else
      i++;
  }
  *size = j;
}
