# Copyright (C) 2007  Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


"""Publish various files from source tree or build area."""


import os
import shutil
import subprocess
import urlparse

import unperishlib


class SftpError(Exception): #pragma: no cover

    def __init__(self, stderr):
        self._msg = "Error sending file via sftp:\n%s" % stderr
    
    def __str__(self):
        return self._msg


class Publish(unperishlib.Operation):

    """Publish various files from source tree and build area."""

    name = "publish"
    
    required_options = ["publish_to"]
    
    def add_options(self, parser):
        parser.add_public_option("--publish-to", metavar="DIR",
                                 help="Copy files to DIR to publish them.")

        parser.add_public_option("--publish-file", action="append", 
                                 metavar="FILE", help="""\
Add FILE to list of files to publish. Default list: %default.""")
        parser.set_defaults(publish_file=["NEWS", "README"])

        parser.add_public_option("--publish-nothing", action="store_const",
                                 const=None, dest="publish_file",
                                 help="Clear list of files to publish.")

    def local_copy(self, filename, location):
        shutil.copy(filename, location)

    def parse_url(self, location):
        x = urlparse.urlparse(location)
        if x.scheme == "sftp" and x.path.startswith("/~/"):
            return urlparse.ParseResult(x.scheme, x.netloc, x.path[3:],
                                        x.params, x.query, x.fragment)
        else:
            return x

    def network_copy(self, filename, location): #pragma: no cover
        x = self.parse_url(location)
        p = subprocess.Popen(["sftp", "%s:%s" % (x.netloc, x.path)],
                             stdin=subprocess.PIPE, stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE)
        stdout, stderr = p.communicate("put %s" % filename)
        if p.returncode:
            raise SftpError(stderr)

    def copy_to_location(self, filename, location):
        if os.path.exists(filename):
            if location.startswith("sftp:"):
                self.network_copy(filename, location)
            else:
                self.local_copy(filename, location)

    def do_it(self, options):
        for filename in options.publish_file:
            srcpath = os.path.join(options.directory, filename)
            self.copy_to_location(srcpath, options.publish_to)

        if hasattr(options, "upstream_tarball") and options.upstream_tarball:
            self.copy_to_location(options.upstream_tarball, 
                                  options.publish_to)

        if hasattr(options, "changelog") and options.changelog:
            self.copy_to_location(options.changelog, options.publish_to)
