# Copyright (C) 2007  Lars Wirzenius <liw@iki.fi>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along
# with this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.


"""Unit tests for the Unperish plugin for bzr operations."""


import os
import shutil
import tempfile
import unittest

import unperishlib


class IsBzrBranchTests(unittest.TestCase):

    def setUp(self):
        self.dirname = tempfile.mkdtemp()
        pm = unperishlib.PluginManager()
        pm.set_plugin_directories(["plugins"])
        pm.load_plugins()
        self.oper = pm.get_operation("is-bzr-branch")
        self.options = unperishlib.FakeOptions(directory=self.dirname,
                                               is_bzr_branch=False)

    def tearDown(self):
        shutil.rmtree(self.dirname)

    def testNoticesNonBzrDirectory(self):
        self.failUnlessEqual(self.oper.do_it(self.options), None)
        self.failUnlessEqual(self.options.is_bzr_branch, False)

    def testAcceptsBzrDirectory(self):
        unperishlib.run(["bzr", "init", "--quiet"], cwd=self.dirname)
        self.failUnlessEqual(self.oper.do_it(self.options), None)
        self.failUnlessEqual(self.options.is_bzr_branch, True)


class BuildTarballFromBzrTests(unittest.TestCase):

    def setUp(self):
        self.app = unperishlib.Application(plugins=["plugins"])

    def testAddsUpstreamTarballOption(self):
        options, _ = self.app.parse_args(["--upstream-tarball=pink.tar.gz"])
        self.failUnlessEqual(options.upstream_tarball, "pink.tar.gz")

    def testCreatesTarball(self):
        dirname = tempfile.mkdtemp()
        build_area = tempfile.mkdtemp()

        f = file(os.path.join(dirname, "pink"), "w")
        f.write("pretty")
        f.close()

        unperishlib.run(["bzr", "init", "--quiet"], cwd=dirname)
        unperishlib.run(["bzr", "add", "--quiet"], cwd=dirname)
        unperishlib.run(["bzr", "commit", "--quiet", "-m."], cwd=dirname)
        
        options, opers = self.app.parse_args(["--project-name=pink",
                                              "--project-version=4.2", 
                                              "--directory=" + dirname,
                                              "--build-area=" + build_area])
        self.failUnlessEqual(options.upstream_tarball, None)
        
        pm = self.app.get_plugin_manager()
        oper = pm.get_operation("build-tarball-from-bzr")
        oper.do_it(options)
        
        tarball_base = "pink-4.2.tar.gz"
        tarball_path = os.path.join(build_area, tarball_base)

        self.failUnless(os.path.exists(tarball_path))
        self.failUnlessEqual(options.upstream_tarball, tarball_path)

        shutil.rmtree(dirname)
        shutil.rmtree(build_area)


class BzrChangelogTests(unittest.TestCase):

    def testGeneratesChangelog(self):
        dirname = tempfile.mkdtemp()
        build_area = tempfile.mkdtemp()

        f = file(os.path.join(dirname, "pink"), "w")
        f.write("pretty")
        f.close()

        unperishlib.run(["bzr", "init", "--quiet"], cwd=dirname)
        unperishlib.run(["bzr", "add", "--quiet"], cwd=dirname)
        unperishlib.run(["bzr", "commit", "--quiet", "-m."], cwd=dirname)
        
        app = unperishlib.Application(plugins=["plugins"])
        app.run(["bzr-changelog", 
                 "--project-name=pink",
                 "--project-version=4.2", 
                 "--directory=" + dirname,
                 "--build-area=" + build_area])
        
        changelog_path = os.path.join(build_area, "pink.changelog")

        self.failUnless(os.path.exists(changelog_path))

        shutil.rmtree(dirname)
        shutil.rmtree(build_area)
