/*

  Copyright (c) 2003-2008 uim Project http://code.google.com/p/uim/

  All rights reserved.

  Redistribution and use in source and binary forms, with or without
  modification, are permitted provided that the following conditions
  are met:

  1. Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
  2. Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
  3. Neither the name of authors nor the names of its contributors
     may be used to endorse or promote products derived from this software
     without specific prior written permission.

  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS'' AND
  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
  ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE LIABLE
  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
  SUCH DAMAGE.

*/

#include <config.h>

#include <stdlib.h>
#include <string.h>
#include <iconv.h>
#include <assert.h>

#ifdef HAVE_ALLOCA_H
# include <alloca.h>
#endif

#include "uim.h"
#include "uim-internal.h"
#include "uim-util.h"


#define MBCHAR_LEN_MAX 6  /* assumes CESU-8 */

static void *uim_iconv_open(const char *tocode, const char *fromcode);
static int uim_iconv_is_convertible(const char *tocode, const char *fromcode);
static void *uim_iconv_create(const char *tocode, const char *fromcode);
static char *uim_iconv_code_conv(void *obj, const char *str);
static void uim_iconv_release(void *obj);

static int check_encoding_equivalence(const char *tocode,
                                      const char *fromcode);
static const char **uim_get_encoding_alias(const char *encoding);


static struct uim_code_converter uim_iconv_tbl = {
  uim_iconv_is_convertible,
  uim_iconv_create,
  uim_iconv_code_conv,
  uim_iconv_release
};
struct uim_code_converter *uim_iconv = &uim_iconv_tbl;

#include "encoding-table.c"


static int
check_encoding_equivalence(const char *tocode, const char *fromcode)
{
  const char **alias_tocode;
  const char **alias_fromcode;
  int i, j;
  int alias_tocode_alloced = 0;
  int alias_fromcode_alloced = 0;
  int found = 0;

  assert(tocode);
  assert(fromcode);

  alias_tocode = uim_get_encoding_alias(tocode);
  alias_fromcode = uim_get_encoding_alias(fromcode);

  if (!alias_tocode) {
    alias_tocode = uim_malloc(sizeof(char *) * 2);
    alias_tocode[0] = tocode;
    alias_tocode[1] = NULL;
    alias_tocode_alloced = 1;
  }
  if (!alias_fromcode) {
    alias_fromcode = uim_malloc(sizeof(char *) * 2);
    alias_fromcode[0] = fromcode;
    alias_fromcode[1] = NULL;
    alias_fromcode_alloced = 1;
  }

  for (i = 0; alias_tocode[i]; i++) {
    for (j = 0; alias_fromcode[j]; j++) {
      if (!strcmp(alias_tocode[i], alias_fromcode[j])) {
        found = 1;
	break;
      }
    }
    if (found)
      break;
  }

  if (alias_tocode_alloced)
    free(alias_tocode);
  if (alias_fromcode_alloced)
    free(alias_fromcode);
  return found;
}

static int
uim_iconv_is_convertible(const char *tocode, const char *fromcode)
{
  iconv_t ic;
  uim_bool result;

  if (UIM_CATCH_ERROR_BEGIN())
    return UIM_FALSE;

  assert(tocode);
  assert(fromcode);

  do {
    if (check_encoding_equivalence(tocode, fromcode)) {
      result = UIM_TRUE;
      break;
    }

    /* TODO cache the result */
    ic = (iconv_t)uim_iconv_open(tocode, fromcode);
    if (ic == (iconv_t)-1) {
      result = UIM_FALSE;
      break;
    }
    iconv_close(ic);
    result = UIM_TRUE;
  } while (/* CONSTCOND */ 0);

  UIM_CATCH_ERROR_END();

  return result;
}

static const char **
uim_get_encoding_alias(const char *encoding)
{
  int i, j;
  const char **alias;

  assert(encoding);

  for (i = 0; (alias = uim_encoding_list[i]); i++) {
    for (j = 0; alias[j]; j++) {
      if (!strcmp(alias[j], encoding))
        return alias;
    }
  }
  return NULL;
}

static void *
uim_iconv_open(const char *tocode, const char *fromcode)
{
  iconv_t cd = (iconv_t)-1;
  int i, j;
  const char **alias_tocode, **alias_fromcode;
  int alias_tocode_alloced = 0;
  int alias_fromcode_alloced = 0;
  int opened = 0;

  assert(tocode);
  assert(fromcode);

  alias_tocode = uim_get_encoding_alias(tocode);
  alias_fromcode = uim_get_encoding_alias(fromcode);

  if (!alias_tocode) {
    alias_tocode = uim_malloc(sizeof(char *) * 2);
    alias_tocode[0] = tocode;
    alias_tocode[1] = NULL;
    alias_tocode_alloced = 1;
  }
  if (!alias_fromcode) {
    alias_fromcode = uim_malloc(sizeof(char *) * 2);
    alias_fromcode[0] = fromcode;
    alias_fromcode[1] = NULL;
    alias_fromcode_alloced = 1;
  }

  for (i = 0; alias_tocode[i]; i++) {
    for (j = 0; alias_fromcode[j]; j++) {
      cd = iconv_open(alias_tocode[i], alias_fromcode[j]);
      if (cd != (iconv_t)-1) {
	opened = 1;
	break;
      }
    }
    if (opened)
      break;
  }

  if (alias_tocode_alloced)
    free(alias_tocode);
  if (alias_fromcode_alloced)
    free(alias_fromcode);
  return (void *)cd;
}

static void *
uim_iconv_create(const char *tocode, const char *fromcode)
{
  iconv_t ic;

  if (UIM_CATCH_ERROR_BEGIN())
    return NULL;

  assert(tocode);
  assert(fromcode);

  do {
    if (check_encoding_equivalence(tocode, fromcode)) {
      ic = (iconv_t)0;
      break;
    }

    ic = (iconv_t)uim_iconv_open(tocode, fromcode);
    if (ic == (iconv_t)-1) {
      /* since iconv_t is not explicit pointer, use 0 instead of NULL */
      ic = (iconv_t)0;
    }
  } while (/* CONSTCOND */ 0);

  UIM_CATCH_ERROR_END();

  return (void *)ic;
}

static char *
uim_iconv_code_conv(void *obj, const char *str)
{
  iconv_t ic;
  size_t len, bufsize;
  char *outbuf, *realbuf, *copied;
  const char *inbuf, *src;

  if (UIM_CATCH_ERROR_BEGIN())
    return NULL;

  do {
    if (!str) {
      copied = NULL;
      break;
    }

    ic = (iconv_t)obj;
    if (ic) {
      len = strlen(str);
      bufsize = (len + sizeof("")) * MBCHAR_LEN_MAX;
      realbuf = alloca(bufsize);
      bufsize--;

      inbuf = str;
      outbuf = realbuf;
      iconv(ic, (ICONV_CONST char **)&inbuf, &len, &outbuf, &bufsize);
      iconv(ic, NULL, NULL, &outbuf, &bufsize);
      *outbuf = '\0';
      src = realbuf;
    } else {
      src = str;
    }

    copied = uim_strdup(src);
  } while (/* CONSTCOND */ 0);

  UIM_CATCH_ERROR_END();

  return copied;
}

static void
uim_iconv_release(void *obj)
{
  int err;

  if (UIM_CATCH_ERROR_BEGIN())
    return;

  if (obj)
    err = iconv_close((iconv_t)obj);

  UIM_CATCH_ERROR_END();
}
