/*
 * Copyright (C) 2006 Kay Sievers <kay@vrfy.org>
 *
 *	This program is free software; you can redistribute it and/or modify it
 *	under the terms of the GNU General Public License as published by the
 *	Free Software Foundation version 2 of the License.
 * 
 *	This program is distributed in the hope that it will be useful, but
 *	WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	General Public License for more details.
 * 
 *	You should have received a copy of the GNU General Public License along
 *	with this program; if not, write to the Free Software Foundation, Inc.,
 *	51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

#include <stdlib.h>
#include <stddef.h>
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <dirent.h>
#include <fcntl.h>
#include <syslog.h>
#include <getopt.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "udev.h"
#include "udevd.h"

#define DEFAULT_TIMEOUT			180
#define LOOP_PER_SECOND			20

static void print_queue(const char *dir)
{
	LIST_HEAD(files);
	struct name_entry *item;

	if (add_matching_files(&files, dir, NULL) < 0)
		return;

	printf("\n\nAfter the udevadm settle timeout, the events queue contains:\n\n");

	list_for_each_entry(item, &files, node) {
		char target[NAME_SIZE];
		size_t len;
		const char *filename = strrchr(item->name, '/');

		if (filename == NULL)
			continue;
		filename++;
		if (*filename == '\0')
			continue;

		len = readlink(item->name, target, sizeof(target));
		if (len < 0)
			continue;
		target[len] = '\0';

		printf("%s: %s\n", filename, target);
	}

	printf("\n\n");
}

int udevsettle(int argc, char *argv[], char *envp[])
{
	char queuename[PATH_SIZE];
	char filename[PATH_SIZE];
	unsigned long long seq_kernel;
	unsigned long long seq_udev;
	char seqnum[32];
	int fd;
	ssize_t len;
	int timeout = DEFAULT_TIMEOUT;
	int loop;
	static const struct option options[] = {
		{ "timeout", 1, NULL, 't' },
		{ "help", 0, NULL, 'h' },
		{}
	};
	int option;
	int rc = 1;
	int seconds;

	logging_init("udevsettle");
	udev_config_init();
	dbg("version %s\n", UDEV_VERSION);
	sysfs_init();

	while (1) {
		option = getopt_long(argc, argv, "t:h", options, NULL);
		if (option == -1)
			break;

		switch (option) {
		case 't':
			seconds = atoi(optarg);
			if (seconds > 0)
				timeout = seconds;
			else
				fprintf(stderr, "invalid timeout value\n");
			dbg("timeout=%i\n", timeout);
			break;
		case 'h':
			printf("Usage: udevadm settle [--help] [--timeout=<seconds>]\n\n");
			goto exit;
		}
	}

	strlcpy(queuename, udev_root, sizeof(queuename));
	strlcat(queuename, "/" EVENT_QUEUE_DIR, sizeof(queuename));

	loop = timeout * LOOP_PER_SECOND;
	while (loop--) {
		/* wait for events in queue to finish */
		while (loop--) {
			struct stat statbuf;

			if (stat(queuename, &statbuf) < 0) {
				info("queue is empty\n");
				break;
			}
			usleep(1000 * 1000 / LOOP_PER_SECOND);
		}
		if (loop <= 0) {
			info("timeout waiting for queue\n");
			print_queue(queuename);
			goto exit;
		}

		/* read current udev seqnum */
		strlcpy(filename, udev_root, sizeof(filename));
		strlcat(filename, "/" EVENT_SEQNUM, sizeof(filename));
		fd = open(filename, O_RDONLY);
		if (fd < 0)
			goto exit;
		len = read(fd, seqnum, sizeof(seqnum)-1);
		close(fd);
		if (len <= 0)
			goto exit;
		seqnum[len] = '\0';
		seq_udev = strtoull(seqnum, NULL, 10);
		info("udev seqnum = %llu\n", seq_udev);

		/* read current kernel seqnum */
		strlcpy(filename, sysfs_path, sizeof(filename));
		strlcat(filename, "/kernel/uevent_seqnum", sizeof(filename));
		fd = open(filename, O_RDONLY);
		if (fd < 0)
			goto exit;
		len = read(fd, seqnum, sizeof(seqnum)-1);
		close(fd);
		if (len <= 0)
			goto exit;
		seqnum[len] = '\0';
		seq_kernel = strtoull(seqnum, NULL, 10);
		info("kernel seqnum = %llu\n", seq_kernel);

		/* make sure all kernel events have arrived in the queue */
		if (seq_udev >= seq_kernel) {
			info("queue is empty and no pending events left\n");
			rc = 0;
			goto exit;
		}
		usleep(1000 * 1000 / LOOP_PER_SECOND);
		info("queue is empty, but events still pending\n");
	}

exit:
	sysfs_cleanup();
	logging_close();
	return rc;
}
