# -*- coding: iso-8859-1 -*-
#
# Copyright (C) 2006 Panoramic Feedback <abentley@panoramicfeedback.com>
# All rights reserved.
#
# This software may be used and distributed according to the terms
# of the GNU General Public License, incorporated herein by reference.


import urllib

from bzrlib import bzrdir, osutils
from bzrlib.tests import treeshape
from trac import versioncontrol

from tracbzr import tests

class TestRepository(tests.MultiBranchTracTestCase):

    def test_get_changeset(self):
        self.assertRaises(versioncontrol.NoSuchChangeset, 
                          self.trac_repo().get_changeset, 'tree,foo')
        self.assertRaises(versioncontrol.NoSuchChangeset, 
                          self.trac_repo().get_changeset, 'treo,foo')
        self.assertRaises(versioncontrol.NoSuchChangeset, 
                          self.trac_repo().get_changeset, 'foo')
        self.assertRaises(versioncontrol.NoSuchChangeset, 
                          self.trac_repo().get_changeset, ',foo')
        revision_id = self.commit_foo_bar()
        assert revision_id is not None
        changeset = self.trac_repo().get_changeset('tree,%s' %
                                                   urllib.quote(revision_id))
        self.assertRaises(versioncontrol.NoSuchChangeset, 
                          self.trac_repo().get_changeset, 'tree/../,%s' %
                                                   urllib.quote(revision_id))
        self.assertRaises(versioncontrol.NoSuchChangeset, 
                          self.trac_repo().get_changeset, 'tree,23')

    def commit_foo_bar(self):
        """Create a simple revision"""
        treeshape.build_tree_contents([('repo/tree/a/',), ('repo/tree/a/b', 
                                        'contents of b')])
        self.tree.add(['a', 'a/b'])
        return self.tree.commit('tree contents', rev_id='foo%bar')

    def commit_foo_bar_baz(self):
        """Create a simple revision"""
        self.commit_foo_bar()
        other = self.tree.bzrdir.sprout('repo/other').open_workingtree()
        treeshape.build_tree_contents([
            ('repo/other/a/b', 'new contents of b'),
            ('repo/other/a/c', 'contents of c')])
        other.add(['a/c'])
        other.commit('added c, changed b', rev_id='baz')
        self.tree.merge_from_branch(other.branch)
        return self.tree.commit('merged from other', rev_id='qux')

    def test_has_node(self):
        self.commit_foo_bar()
        self.assertTrue(self.trac_repo().has_node('tree/a', 'tree,foo%25bar'))
        self.trac_repo().has_node('b', 'tree,foo%25bar')
        self.trac_repo().has_node('b', 'tree,foo%25bar')
        self.assertFalse(self.trac_repo().has_node('b', 'tree,foo%25bar'))
        self.assertFalse(self.trac_repo().has_node('tree/b', 'tree,foo%25bar'))
        self.assertTrue(self.trac_repo().has_node('tree/a/b', 
                                                  'tree,foo%25bar'))
        self.assertRaises(versioncontrol.NoSuchChangeset, 
                          self.trac_repo().has_node, 'tree/a/b', 
                          'tree,foo%25baa')
        self.assertRaises(versioncontrol.NoSuchChangeset, 
                          self.trac_repo().has_node, 'a/b', 'treo,foo%25bar')
        self.assertRaises(versioncontrol.NoSuchChangeset, 
                          self.trac_repo().has_node, 'a/b', 'tree,23')
        self.assertRaises(versioncontrol.NoSuchChangeset, 
                          self.trac_repo().has_node, 'a/b', 'treo,23')

    def test_get_node(self):
        self.commit_foo_bar()
        a_node = self.trac_repo().get_node('tree/a', 'tree,foo%25bar')
        self.assertTrue(a_node.isdir)
        self.assertFalse(a_node.isfile)
        self.assertTrue(a_node.get_content() is None)
        self.assertEqual(list('tree/a/b'), 
                         list(a_node.get_entries().next().path))
        self.assertRaises(versioncontrol.NoSuchNode, 
                          self.trac_repo().get_node, 'tree/b', 
                          'tree,foo%25bar')
        b_node = self.trac_repo().get_node('tree/a/b', 'tree,foo%25bar')
        self.assertFalse(b_node.isdir)
        self.assertTrue(b_node.isfile)
        self.assertEqual('contents of b', b_node.get_content().read())
        self.assertRaises(versioncontrol.NoSuchChangeset, 
                          self.trac_repo().get_node, 'tree/a/b', 
                          'tree,foo%25baa')
        self.assertRaises(versioncontrol.NoSuchChangeset, 
                          self.trac_repo().get_node, 'a/b', 'treo,foo%25bar')
        self.assertRaises(versioncontrol.NoSuchChangeset, 
                          self.trac_repo().get_node, 'a/b', 'tree,23')
        self.assertRaises(versioncontrol.NoSuchChangeset, 
                          self.trac_repo().get_node, 'a/b', 'treo,23')

    def test_rev(self):
        self.commit_foo_bar()
        self.assertEqual('tree,1', self.trac_repo().get_youngest_rev())
        self.assertEqual('tree,1', self.trac_repo().get_oldest_rev())
        oldest = self.tree.commit('again')
        repo = self.trac_repo()
        self.assertEqual('tree,2', self.trac_repo().get_youngest_rev())
        self.assertEqual('tree,1', self.trac_repo().get_oldest_rev())

    def test_next_prev(self):
        self.commit_foo_bar_baz()
        repo = self.trac_repo()
        self.assertEqual('other,2', repo.previous_rev('tree,2'))
        self.assertEqual('tree,1', repo.previous_rev('other,2'))
        self.assertEqual('null:', repo.previous_rev('tree,1'))
        self.assertEqual(None, repo.previous_rev('null:'))
        self.assertEqual('tree,1.1.1', repo.next_rev('tree,1', 'tree'))
        self.assertEqual(None, repo.next_rev('tree,2'))
        self.assertRaises(versioncontrol.NoSuchChangeset, repo.next_rev, 
                          'tree,255')
        self.assertRaises(versioncontrol.NoSuchChangeset, repo.next_rev, 
                          'tree,3')

    def test_get_path_history(self):
        self.commit_foo_bar()
        repo = self.trac_repo()
        self.assertRaises(versioncontrol.NoSuchNode, 
                          repo.get_path_history, 'b')
        self.assertEqual([(u'tree/a/b', 'tree,1', 'add')], 
                         list(repo.get_path_history('tree/a/b')))

    def test_normalize_path(self):
        repo = self.trac_repo()
        self.assertEqual('', repo.normalize_path(None))
        self.assertEqual('foo', repo.normalize_path('foo'))
        self.assertEqual('foo', repo.normalize_path('/foo/'))

    def test_normalize_rev(self):
        self.commit_foo_bar()
        rev1 = self.tree.commit('again')
        repo = self.trac_repo()
        self.assertEqual('current:', repo.normalize_rev(None))
        self.assertEqual('tree,1', repo.normalize_rev('tree,foo%25bar'))
        self.assertRaises(versioncontrol.NoSuchChangeset, repo.normalize_rev, 
                          'garbage,foo%25bar')

    def test_short_rev(self):
        self.commit_foo_bar_baz()
        repo = self.trac_repo()
        self.assertEqual('1', repo.short_rev('tree,1'))
        self.assertEqual('2', repo.short_rev('tree,qux'))
        self.assertEqual('~1', repo.short_rev('tree,baz'))

    def test_string_rev(self):
        self.tree.commit('Commit with comma in revid', rev_id='hi,there')
        self.assertEqual('tree,hi%2Cthere', 
                         self.trac_repo()._string_rev_revid('tree', 
                         'hi,there'))

    def test_branch_path(self):
        branch = self.tree.branch
        self.assertEqual('tree', self.trac_repo().branch_path(branch))

    def test_parse_rev(self):
        repository = self.trac_repo()
        branch, revid = repository._parse_rev('tree,hi%2Cthere')
        self.assertEqual('tree', repository.branch_path(branch))
        self.assertEqual('hi,there', revid)

    def test_sync(self):
        self.trac_repo().sync()
