/*  spwg -  the Simplest Previewer in the World with GTK          */
/*  Copyright (C) 1999  Hirotsugu Kakugawa. All rights reserved.  */
/*  See "COPYING" for distribution of this software.              */
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRING_H
#  include <string.h>
#endif
#if HAVE_STRINGS_H
#  include <strings.h>
#endif
#include <gtk/gtk.h>

#include "libdvi29.h"
#include "defs.h"



int              param_dpi           = DPI;
char            *param_kpathsea_mode = KPATHSEA_MODE;
char            *param_vflibcap      = VFLIBCAP;
double           param_shrink        = SHRINK;
char            *param_paper         = "A4"; 
int              param_orient        = 0;
char            *param_foregr        = COLOR_FG;
char            *param_backgr        = COLOR_BG;
char            *dvi_file; 
DVI              dvi   = NULL;
DVI_DEVICE       dev   = NULL;
int              page, ui_cmd, page_state, status;
int              paper_w, paper_h, ofs_x, ofs_y;
GtkWidget       *window          = NULL;
GtkWidget       *win_scrolled    = NULL;
GtkWidget       *win_drawing     = NULL;
GdkPixmap       *back_pixmap     = NULL;
GdkGC          **color_gcs       = NULL;
GdkColor         color_fg, color_bg;

char    **paper_list = NULL;
char    **mode_list  = NULL;

#define EXIT_MSG1(a1)   {fprintf(stderr,a1); fprintf(stderr,"\n"); exit(1);}
#define EXIT_MSG2(a1,a2){fprintf(stderr,a1,a2); fprintf(stderr,"\n"); exit(1);}

#define PAGE_EMPTY        0
#define PAGE_DRAWING      1
#define PAGE_DRAW_FINISH  2
#define STAT_RUNNING      0
#define STAT_MAINLOOP     1
#define CMD_NONE          0
#define CMD_DRAW_PAGE     1

void  usage(void)
{
  printf("SPWG %s - the Simplest Previewer in the World with GTK\n", 
	 SPWG_VERSION);
  printf("SPWG is a previewer for TeX DVI files on X Window System.\n");
  printf("Usage: spwg [PaperSize] [Options] DVI-FILE\n");
  printf("PaperSize:\n");
  printf("  -a1,..,-a7,-b1,..,-b7,-letter,-us,-legal\n");
  printf("  -l      Landscape mode\n");
  printf("Options:  (Each value enclosed by [ ] is the default.)\n");
  printf("  -m M         Magnification (%f<=M<=%f) [1.0]\n", MAG_MIN, MAG_MAX);
  printf("  -v VFLIBCAP  A path name of vflibcap [%s]\n", VFLIBCAP);
  printf("  -cx, -sparcptr, -ljfour, .... -MODE\n");
  printf("  -mode MODE   Device mode name for kpathsea [%s]\n", KPATHSEA_MODE);
  printf("  -dpi DPI     Device resolution for font selection [%d]\n", DPI);
  printf("  -fg COLOR    Foreground color [%s]\n", COLOR_FG);
  printf("  -bg COLOR    Background color [%s]\n", COLOR_BG);
  printf("  -paper-list  Show list of paper sizes\n");
  printf("  -mode-list   Show device mode names\n");
  printf("Key and Mouse operation:\n");
  printf("  Mouse Right Button    Next page\n");
  printf("  Mouse Left Button     Previous page\n");
  printf("  q                     Quit\n");
  exit(0);
}

void
show_paper_list(void)
{
  char  **p;
  int   i;
  
  if ((p = paper_list) == NULL){
    printf("Paper size list is not found.\n");
  } else {
    i = 0;
    printf("Paper Size: \n");
    while (*p != NULL){
      printf("%s%s", *p, (*(p+1) == NULL) ? "\n" : ", ");
      i++;
      p++;
    }
  }
}

void
show_device_list(void)
{
  char **p, *desc;
  int  i;
  
  if ((p = mode_list) == NULL){
    printf("Device mode list is not found.\n");
  } else {
    i = 0;
    printf("Device Mode Names: \n");
    printf("   Mode Name     Description\n");
    while (*p != NULL){
      DVI_parse_device_mode(*p, NULL, NULL, &desc);
      printf("%12s     %s\n", *p, desc);
      i++;
      p++;
    }
  }
}

int  window_size(char *paper, int ori)
{
  double  pw, ph;

  if (DVI_parse_paper_size(paper, &pw, &ph) < 0)
    return -1;

  paper_w = ((ori == 0)?pw:ph) * param_dpi/param_shrink;
  paper_h = ((ori == 0)?ph:pw) * param_dpi/param_shrink;
  return 0;
}

char *parse_args(int argc, char **argv)
{
  double  d, mag = 1.0;

  for (--argc, argv++; argc > 0;  argc--, argv++){
    if ((strcmp(*argv, "-m") == 0) && (argc > 1)){
      argc--; argv++;
      mag = atof(*argv);
    } else if (strcmp(*argv, "-l") == 0){
      param_orient = 1;
    } else if ((strcmp(*argv, "-v") == 0)  && (argc > 1)){
      argc--;  argv++;
      param_vflibcap = *argv;
    } else if (DVI_parse_device_mode(&argv[0][1], NULL, NULL, NULL) >= 0){
      param_kpathsea_mode = &argv[0][1];
    } else if ((strcmp(*argv, "-mode") == 0) && (argc > 1)){
      argc--;  argv++;
      param_kpathsea_mode = *argv;
    } else if ((strcmp(*argv, "-dpi") == 0) && (argc > 1)){
      argc--;  argv++;
      param_dpi = atoi(*argv);
    } else if (DVI_parse_paper_size(&argv[0][1], NULL, NULL) >= 0){
      param_paper = &argv[0][1];
    } else if ((strcmp(*argv, "-fg") == 0) && (argc > 1)){
      argc--;  argv++;
      param_foregr = *argv;
    } else if ((strcmp(*argv, "-bg") == 0) && (argc > 1)){
      argc--;  argv++;
      param_backgr = *argv;
    } else if (**argv != '-'){
      dvi_file = *argv;
    } else if (window_size(&argv[0][1], param_orient) >= 0){
      param_paper = &argv[0][1];
    } else if ((strcmp(*argv, "-h") == 0) || (strcmp(*argv, "--help") == 0)){
      usage();
    } else if (strcmp(*argv, "-paper-list") == 0){
      show_paper_list();
      exit(0);
    } else if (strcmp(*argv, "-mode-list") == 0){
      show_device_list();
      exit(0);
    } else 
      EXIT_MSG2("Unknow option: %s\n", *argv);
  }
  if (mag < MAG_MIN)
    EXIT_MSG1("Maginification is too small.");
  if (MAG_MAX < mag)
    EXIT_MSG1("Maginification is too large.");
  if (param_kpathsea_mode == NULL){
    param_kpathsea_mode = KPATHSEA_MODE;
    if (param_dpi <= 0)
      param_dpi = DPI;
  } else {
    if (DVI_parse_device_mode(param_kpathsea_mode, &d, NULL, NULL) >= 0)
      param_dpi = d;
    else if (param_dpi <= 0)
      param_dpi = DPI;
  }
  param_shrink = (SHRINK * param_dpi) / (mag * DPI);
  if (window_size(param_paper, param_orient) < 0)
    EXIT_MSG2("Unknown paper size: %s", param_paper);
  ofs_x = OFFSET_X * param_dpi / param_shrink;
  ofs_y = OFFSET_Y * param_dpi / param_shrink;
  return dvi_file;
}

void  DEV_put_bitmap(DVI_DEVICE dev, DVI dvi, DVI_BITMAP bm, int font_id, 
		     long key2, long code_point, long pos_x, long pos_y)
{
  int   x, y;
  static unsigned char bits[] = {
    0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01 };
  
  for (y = 0; y < bm->height; y++){
    for (x = 0; x < bm->width; x++){
      if ((bm->bitmap[bm->raster * y + (x/8)] & bits[x%8]) != 0)
	gdk_draw_point(back_pixmap, color_gcs[1],
		       x+pos_x+ofs_x, y+pos_y+ofs_y);
    }
  }
  gdk_draw_pixmap(win_drawing->window, color_gcs[1], back_pixmap, 
		  pos_x+ofs_x, pos_y+ofs_y, 
		  pos_x+ofs_x, pos_y+ofs_y, bm->width, bm->height);
}

void  DEV_put_rectangle(DVI_DEVICE dev, DVI dvi, 
			long pos_x, long pos_y, long w, long h)
{
  gdk_draw_rectangle(back_pixmap, color_gcs[1], TRUE, 
		     pos_x+ofs_x, pos_y+ofs_y, w, h);
  gdk_draw_pixmap(win_drawing->window, color_gcs[1], back_pixmap, 
		  pos_x+ofs_x, pos_y+ofs_y, pos_x+ofs_x, pos_y+ofs_y, w, h);
}

int  DEV_poll(DVI_DEVICE dev, DVI dvi, int poll_type)
{
  static int  t = 0;

  if (poll_type != DVI_POLL_PAGE)
    return 0;
  if ((--t) > 0)
    return 0;
  t = POLLING_INTERVAL;
  while (gtk_events_pending()){
    gtk_main_iteration();
  }
  if (ui_cmd != CMD_NONE)
    return 1;
  return 0;
}

DVI_DEVICE  create_dev(void)
{
  if ((dev = DVI_DEVICE_ALLOC()) == NULL)
    return NULL;
  dev->h_dpi = dev->v_dpi   = param_dpi;
  dev->device_polling       = DEV_poll;
  dev->put_bitmap           = DEV_put_bitmap;
  dev->put_rectangle        = DEV_put_rectangle;
  dev->message_advice       = NULL;
  return dev;
}

void clear_page()
{
  gdk_draw_rectangle(back_pixmap, color_gcs[0], TRUE, 0, 0, paper_w, paper_h);
  gdk_draw_pixmap(win_drawing->window, 
		  win_drawing->style->fg_gc[GTK_WIDGET_STATE(win_drawing)], 
		  back_pixmap, 0, 0, 0, 0, paper_w, paper_h);
  page_state = PAGE_EMPTY;
}

void  draw_page(void)
{
  if (status != STAT_MAINLOOP)
    return;
  status = STAT_RUNNING;
  while (ui_cmd == CMD_DRAW_PAGE){
    clear_page();
    ui_cmd = CMD_NONE;
    page_state = PAGE_DRAWING;
    if (DVI_DRAW_PAGE(dvi, dev, page, param_shrink) != DVI_DRAW_INTERRUPTED){
      page_state = PAGE_DRAW_FINISH;
      ui_cmd = CMD_NONE;
    }
  }
  status = STAT_MAINLOOP;
}

gint
event_key_press(GtkWidget *w, GdkEventKey *ev)
{
  int     opage;

  switch (ev->keyval){
  default:
    break;
  case 'q':
    gtk_main_quit();
    break;
  case ' ': 
    page = (page < dvi->pages) ? (page + 1) : page; 
    break;
  case 'b':
    page = (1 < page) ? (page - 1) : page; 
    break;
  }
  if (page != opage){
    ui_cmd = CMD_DRAW_PAGE;
    draw_page();
  }
  return TRUE;
}

gint  event_configure(GtkWidget *w, GdkEventConfigure *ev)
{
  GdkColormap     *colormap;

  if (color_gcs == NULL){
    colormap = gdk_window_get_colormap(win_drawing->window);
    color_gcs = g_new(GdkGC*, 2);
    color_gcs[0] = gdk_gc_new(win_drawing->window);
    color_gcs[1] = gdk_gc_new(win_drawing->window);
    if (gdk_color_parse(param_backgr, &color_bg) == FALSE)
      color_bg.red = color_bg.green = color_bg.blue = (guint16)65535;
    gdk_color_alloc(colormap, &color_bg);
    gdk_gc_set_foreground(color_gcs[0], &color_bg);
    if (gdk_color_parse(param_foregr, &color_fg) == FALSE)
      color_fg.red = color_fg.green = color_fg.blue = (guint16)0;
    gdk_color_alloc(colormap, &color_fg);
    gdk_gc_set_foreground(color_gcs[1], &color_fg);

  }
  if (back_pixmap != NULL)
    gdk_pixmap_unref(back_pixmap);
  back_pixmap = gdk_pixmap_new(win_drawing->window, paper_w, paper_h, -1);
  gdk_draw_rectangle(back_pixmap, color_gcs[0], TRUE, 0, 0, paper_w, paper_h);
  page_state = PAGE_EMPTY;
  ui_cmd = CMD_DRAW_PAGE;
  return TRUE;
}

gint  event_expose(GtkWidget *w, GdkEventExpose *ev)
{
  if (back_pixmap == NULL)
    return FALSE;
  if (page_state == PAGE_DRAW_FINISH){
    gdk_draw_pixmap(win_drawing->window, 
		    win_drawing->style->fg_gc[GTK_WIDGET_STATE(win_drawing)], 
		    back_pixmap, ev->area.x, ev->area.y, 
		    ev->area.x, ev->area.y, ev->area.width, ev->area.height);
  } else {
    draw_page();
  }
  return FALSE;
}

void  gui_init(void)
{
  window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), "SPWG");
  gtk_widget_set_events(window, GDK_EXPOSURE_MASK|GDK_KEY_PRESS_MASK);
  gtk_signal_connect(GTK_OBJECT(window), "key_press_event", 
		     GTK_SIGNAL_FUNC(event_key_press), NULL);
  gtk_signal_connect(GTK_OBJECT(window), "destroy", 
		     GTK_SIGNAL_FUNC(gtk_main_quit), NULL);
  gtk_container_set_border_width(GTK_CONTAINER(window), 2);
  gtk_widget_set_usize(GTK_WIDGET(window), WIN_WIDTH, WIN_HEIGHT);
  win_scrolled = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(win_scrolled),
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  win_drawing = gtk_drawing_area_new();
  gtk_drawing_area_size(GTK_DRAWING_AREA(win_drawing), paper_w, paper_h);
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(win_scrolled),
					win_drawing);
  gtk_signal_connect(GTK_OBJECT(win_drawing), "configure_event",
		     (GtkSignalFunc)event_configure, NULL);
  gtk_signal_connect(GTK_OBJECT(win_drawing), "expose_event",
		     GTK_SIGNAL_FUNC(event_expose), NULL);
  gtk_container_add(GTK_CONTAINER(window), win_scrolled);
  gtk_widget_show(win_drawing);
  gtk_widget_show(win_scrolled);
  gtk_widget_show(window);
}

int  main(int argc, char **argv)
{
  char  params[3*256];
  DVI_PROPERTY  dvi_props;

  DVI_SETUP();
  paper_list = DVI_get_paper_size_list();
  mode_list  = DVI_get_device_mode_list();
  gtk_init(&argc, &argv);
  if (parse_args(argc, argv) == NULL)
    EXIT_MSG1("No DVI file.");
  sprintf(params, "TeX_DPI=%d, TeX_KPATHSEA_MODE=%s, TeX_KPATHSEA_PROGRAM=%s",
	  param_dpi, param_kpathsea_mode, "spwg");
  if ((DVI_INIT(param_vflibcap, params) < 0) || (create_dev() == NULL))
    EXIT_MSG1("Failed to initialize DVI reader.\n");
  gui_init();
  dvi_props = DVI_PROPERTY_ALLOC_DEFAULT();
  DVI_PROPERTY_SET(dvi_props, DVI_PROP_ASYNC_GS_INVOCATION);
  DVI_PROPERTY_SET(dvi_props, DVI_PROP_INCREMENTAL_EPS_DISPLAY);
  if ((dvi = DVI_CREATE(dev, dvi_file, dvi_props)) == NULL)
    EXIT_MSG2("Can't open a DVI file: %s", dvi_file);
  if (DVI_OPEN_FONT(dvi, dev) < 0)
    EXIT_MSG1("Failed to open fonts.");
  page       = 1; 
  ui_cmd     = CMD_DRAW_PAGE;
  page_state = PAGE_EMPTY;
  status     = STAT_MAINLOOP;
  gtk_main ();
  return 0;
}
