/*
 * spc_grf.c - embedded PostScript code analizer 
 *             for graphic{s,x}.sty package.
 * by Hirotsugu Kakugawa
 *
 */
/*
 * Copyright (C) 1996-1999  Hirotsugu Kakugawa. 
 * All rights reserved.
 *
 * This file is part of the DVIlib Library.  This library is free
 * software; you can redistribute it and/or modify it under the terms of
 * the GNU Library General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  This library is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "../config.h"
#include <stdio.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#ifdef HAVE_MALLOC_H
#  include <malloc.h>
#endif
#if defined(HAVE_STRING_H)
#  include  <string.h>
#endif
#if defined(HAVE_STRINGS_H)
#  include  <strings.h>
#endif
#include <ctype.h>
#include <sys/errno.h>
#include <sys/param.h>
#include <signal.h>

#include "libdvi29.h"
#include "defs.h"
#include "cache.h"
#include "private.h"
#include "spc_ps.h"
#include "spc_grf.h"


#if 0
#  define  DEBUG_PS_TYPE
#endif





/* 
 * PostScript code analizer 
 */

struct s_ps_code_type {
  char   *code_pattern[32];
  int    code_type;
  double params[PS_CODE_MAX_PARAMS];
};



Private struct s_ps_code_type  ps_code_type[] = {

  /* before scaling */
  { {"currentpoint", "currentpoint", "translate", 
     PS_CODE_PARAM, PS_CODE_PARAM, "scale", 
     "neg", "exch", "neg", "exch", "translate", NULL},
    PS_CODE_TYPE_SCALE_BEGIN },

  /* after scaling */
  { {"currentpoint", "currentpoint", "translate", 
     PS_CODE_PARAM, PS_CODE_PARAM, "div", 
     PS_CODE_PARAM, PS_CODE_PARAM, "div", "scale", 
     "neg", "exch", "neg", "exch", "translate", NULL},
    PS_CODE_TYPE_SCALE_END }, 

  /* before rotation */
  { {"gsave", "currentpoint", "currentpoint", "translate", 
     PS_CODE_PARAM, "neg", "rotate", "neg", "exch", "neg", 
     "exch", "translate", NULL},
    PS_CODE_TYPE_ROTATE_BEGIN }, 

  /* after rotation */
  { {"currentpoint", "grestore", "moveto", NULL},
    PS_CODE_TYPE_ROTATE_END }, 

  { {NULL}, 
    -1}
};


Glocal int 
dvi_ps_code_type(DVI dvi, char *code)
{
  int     i, c, token, param, trans;
  char   *p, *pat;
  double  sh, sv;
  int     an;

#ifdef DEBUG_PS_TYPE
  printf("PS: %s\n", code);
#endif

  for (i = 0; ps_code_type[i].code_pattern[0] != NULL; i++){
    for (p = code; (*p != '\0') && isspace((int)*p); p++)
      ;
    if (*p == '\0')
      return  PS_CODE_TYPE_UNKNOWN;

#ifdef DEBUG_PS_TYPE
    printf("Check pattern %d\n", i);
#endif
    param = 0;
    for (token = 0; ps_code_type[i].code_pattern[token] != NULL; token++){
      if (*p == '\0')
	break;
      pat = ps_code_type[i].code_pattern[token];
#ifdef DEBUG_PS_TYPE
      printf("  %s", pat);
#endif
      if (strcmp(pat, PS_CODE_PARAM) == 0){
	if (param < PS_CODE_MAX_PARAMS)
	  ps_code_type[i].params[param++] = atof(p);
	while ((*p != '\0') && !isspace((int)*p))
	  p++;
      } else {
	for (c = 0; (pat[c] != '\0') && (pat[c] == *p); p++, c++)
	  ;
	if ((pat[c] != '\0') || ((*p != '\0') && !isspace((int)*p)))
	  goto TryNext;
      }
      while ((*p != '\0') && isspace((int)*p))
	p++;
    }
    if ((ps_code_type[i].code_pattern[token] == NULL) && (*p == '\0'))
      break;
TryNext:
#ifdef DEBUG_PS_TYPE
    printf("  \n");
#endif
    ;
  }
#ifdef DEBUG_PS_TYPE
  printf("  \n");
#endif

#ifdef DEBUG_PS_TYPE
  printf("PS type = %d   param1=%.3f param2=%.3f\n",
	 ps_code_type[i].code_type, 
	 ps_code_type[i].params[0], ps_code_type[i].params[1]);
#endif

  switch (ps_code_type[i].code_type){
  case PS_CODE_TYPE_SCALE_BEGIN:
    dvi_ps_gstate_push(dvi, PS_CODE_TYPE_SCALE_BEGIN,
		       ps_code_type[i].params[0], ps_code_type[i].params[1], 
		       0);
    break;
  case PS_CODE_TYPE_SCALE_END:
    dvi_ps_gstate_pop(dvi, PS_CODE_TYPE_SCALE_BEGIN, NULL, NULL, NULL);
    break;
  case PS_CODE_TYPE_ROTATE_BEGIN:
    dvi_ps_gstate_push(dvi, PS_CODE_TYPE_ROTATE_BEGIN,
		       1, 1, ps_code_type[i].params[0]);
    break;
  case PS_CODE_TYPE_ROTATE_END:
    dvi_ps_gstate_pop(dvi, PS_CODE_TYPE_ROTATE_BEGIN, NULL, NULL, NULL);
    break;
  default:
    return  PS_CODE_TYPE_UNKNOWN;
  }

  trans = dvi_ps_gstate_get_current(dvi, &sh, &sv, &an);
  PD(dvi,gstate_transformed) = trans;
  PD(dvi,gstate_scale_h) = sh;
  PD(dvi,gstate_scale_v) = sv;
  PD(dvi,gstate_angle)   = an;
#if 0
  printf("** ps: type=%d, %.3f, %.3f, %.3f\n", 
	 ps_code_type[i].code_type, sh, sv, an);
#endif
  
  return  ps_code_type[i].code_type;
}



/*
 * Grahics State Stack
 */
   
Glocal void
dvi_ps_gstate_push(DVI dvi, int type, 
		   double scale_h, double scale_v, int angle)
{
  int     sp;

#if 0
  printf("gstate push  sp=%d: %d, %.3f, %.3f, %.3f\n", 
	 PD(dvi,gstate_stack_ptr), type, scale_h, scale_v, angle);
#endif

  PD(dvi,gstate_stack_ptr) = PD(dvi,gstate_stack_ptr) + 1;
  sp = PD(dvi,gstate_stack_ptr);
  if (sp < DVI_GSTATE_STACK_SIZE){
    PD(dvi,gstate_stack[sp-1]).type = type;
    if (sp == 1){
      PD(dvi,gstate_stack[sp-1]).scale_h = scale_h;
      PD(dvi,gstate_stack[sp-1]).scale_v = scale_v;
      PD(dvi,gstate_stack[sp-1]).angle   = angle;
    } else {
      PD(dvi,gstate_stack[sp-1]).scale_h 
	= PD(dvi,gstate_stack[sp-2]).scale_h * scale_h;
      PD(dvi,gstate_stack[sp-1]).scale_v 
	= PD(dvi,gstate_stack[sp-2]).scale_v * scale_v;
      PD(dvi,gstate_stack[sp-1]).angle 
	= PD(dvi,gstate_stack[sp-2]).angle   + angle;
    }
  }
}

Glocal void
dvi_ps_gstate_pop(DVI dvi, int expected_type,
		  double *p_scale_h, double *p_scale_v, int *p_angle)
{
#if 0
  printf("gstate pop  sp=%d\n", PD(dvi,gstate_stack_ptr));
#endif

  PD(dvi,gstate_stack_ptr) = PD(dvi,gstate_stack_ptr) - 1;
  if (PD(dvi,gstate_stack_ptr) < 0)
    PD(dvi,gstate_stack_ptr) = 0;
  (void) dvi_ps_gstate_get_current(dvi, p_scale_h, p_scale_v, p_angle);
}

Glocal int
dvi_ps_gstate_get_current(DVI dvi,
			  double *p_scale_h, double *p_scale_v, 
			  int *p_angle)
{
  int     sp, an;
  double  sh, sv;

  sh = sv = 1;
  an = 0;
  if (PD(dvi,gstate_stack_ptr) > 0){
    sp = PD(dvi,gstate_stack_ptr);
    if (sp >= DVI_GSTATE_STACK_SIZE)
      sp = DVI_GSTATE_STACK_SIZE - 1;
    sh = PD(dvi,gstate_stack[sp-1]).scale_h;
    sv = PD(dvi,gstate_stack[sp-1]).scale_v;
    an = PD(dvi,gstate_stack[sp-1]).angle;
  }
  if (p_scale_h != NULL)
    *p_scale_h = sh;
  if (p_scale_v != NULL)
    *p_scale_v = sv;
  if (p_angle   != NULL)
    *p_angle   = an;

  if (PD(dvi,gstate_stack_ptr) > 0)
    return 1;  /* transformed */
  return 0;    /* not transformed */
}


/*EOF*/
