package mccombe.terrain;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import mccombe.mapping.*;
import java.io.*;
import java.util.zip.*;
import javax.swing.JComponent;

/**
 *
 * @author Mike
 */
public class SRTMReader extends PropertyChangeSupport {
    
    public SRTMReader(JComponent item) throws MissingDataFileException {
        super(item);
        PropertyChangeListener[] listeners = item.getPropertyChangeListeners();
        for (PropertyChangeListener ear : listeners) {
            addPropertyChangeListener(ear);
        }
        addPropertyChangeListener(listener);
        item.addPropertyChangeListener(listener);
        File dir = new File(DIRECTORY);
        if(!dir.isDirectory()){
            boolean madeDirectory = dir.mkdir();
            if(!madeDirectory) {
                throw new MissingDataFileException(String.format("Failed to create data directory %s%n",DIRECTORY));
            }
        }
        try {
            File readMeTxt = new File(DIRECTORY+"ReadMe.txt");
            if(!readMeTxt.isFile()){
                PrintWriter readMe = new PrintWriter(new FileWriter(readMeTxt));
                readMe.printf(
                        
                        "This directory contains copies of raw compressed data files downloaded from NASA. TerrainTool will re-use the files it needs%n" +
                        "instead of downloading them again, saving a lot of time. However, you can save disk space by manually deleting any files you%n" +
                        "no longer need.%n%n" +
                        "Data is read in zipped form. DO NOT DECOMPRESS THESE FILES.");
                
                readMe.close();
            }
            
        } catch (IOException ex) {
            //Do nothing if there's an error writing the ReadMe file
        }
        
        
    }
     public double getHeight(LatLong place) throws MissingDataFileException {
        double lat = Math.floor(place.lat());
        double lon = Math.floor(place.lon());
        String ew = "E";
        String ns = "N";
        if (lat < 0) {
            ns = "S";
        }
        if (lon < 0) {
            ew = "W";
        }
        lat = Math.abs(lat);
        lon = Math.abs(lon);
        double x0 = tile(place.lon());
        double y0 = (double) (RECORD_LENGTH - 1) - tile(place.lat());
        int xtile = (int) x0;
        int ytile = (int) y0;
        String pagename = String.format("%1s%02d%1s%03d", ns, (int) lat, ew, (int) lon);
        double[] h = new double[3];
        double[] x = new double[3];
        double[] w = new double[3];
        double[] y = new double[3];
        int k = Math.max(ytile - 1, 0);
        int m = 0;
        while (m < 3 && k < RECORD_LENGTH && k<ytile+3) {
              CacheEntry page = getRow(pagename, k);
            if (page == null) {
                return MISSING;
            }
            int i = 0;
            int j = xtile;
            while (i < 3 && j < RECORD_LENGTH && j<xtile+4) {
                double v = page.getValue(j);
                if (v != MISSING) {
                    x[i] = (double) j;
                    h[i] = v;
                    i++;
                } else {
                    missing++;
                }
                j++;
            }
            j = xtile - 1;
            while (i < 3 && j >=0 && j>xtile-3) {
                double v = page.getValue(j);
                if (v != MISSING) {
                    x[i] = (double) j;
                    h[i] = v;
                    i++;
                } else {
                    missing++;
                }
                j--;
            }
            if(i==3){
                java.awt.geom.Point2D.Double p0 = new java.awt.geom.Point2D.Double(x[0], h[0]);
                java.awt.geom.Point2D.Double p1 = new java.awt.geom.Point2D.Double(x[1], h[1]);
                java.awt.geom.Point2D.Double p2 = new java.awt.geom.Point2D.Double(x[2], h[2]);
                y[m] = lagrangian(x0, p0, p1, p2);
                w[m] = (double) k;
                m++;
            }
            k++;
        }
        if(m==3){
            java.awt.geom.Point2D.Double q0 = new java.awt.geom.Point2D.Double(w[0], y[0]);
            java.awt.geom.Point2D.Double q1 = new java.awt.geom.Point2D.Double(w[1], y[1]);
            java.awt.geom.Point2D.Double q2 = new java.awt.geom.Point2D.Double(w[2], y[2]);
            double height = lagrangian(y0, q0, q1, q2);
            resultcount++;
            return height;
        }
        return MISSING ;
    }
    
    CacheEntry getRow(String name, int ytile) throws MissingDataFileException {
        tries++;
        cycle++;
        String shortname = String.format("%s#%04d", name, ytile);
        if (cacheEnable) {
            CacheEntry page = cache.get(shortname);
            if (page != null) {
                hits++;
                page.setLastUsed();
                return page;
            }
        }
// Record not in cache or cache not enabled
        try {
            
            String filename = DIRECTORY + name + ".hgt.zip";
            File infile = new File(filename);
            if (!infile.isFile()) {
                String tempname = name + ".hgt.zip";
                try {
                    downloadFile(tempname);
                } catch (IOException e) {
                    throw new MissingDataFileException(String.format("Unable to dowload missing file %s%n%s%n", tempname, e.toString()));
                }
            }
            in = new java.util.zip.ZipInputStream(new FileInputStream(infile));
            ZipEntry entry = in.getNextEntry();
            int recordno = 0;
            try {
                int[] heights;
                while (true) {
                    heights = readRecord();
                    int n = heights.length;
                    if (recordno == ytile) {
                        break;
                    }
                    recordno++;
                }
                if (cacheEnable) {
                    if (cache.size() >= MAX_CACHE_SIZE) {
//Find the oldest entry and remove it
                        CacheEntry oldest = null;
                        long age = cycle;
                        for (CacheEntry test : cache.values()) {
                            if (test.lastUsed() < age) {
                                age = test.lastUsed();
                                oldest = test;
                            }
                        }
                        String key = oldest.getName();
                        cache.remove(key);
                    }
                    CacheEntry page = new CacheEntry(shortname, heights);
                    cache.put(shortname, page);
                    return page;
                }
                CacheEntry page = new CacheEntry(shortname, heights);
                return page;
            } catch (EOFException e) {
               throw new MissingDataFileException("Hit end of file");
            }
        } 
        catch (IOException e) {
               throw new MissingDataFileException("Unable to read file - " + e.toString());
       }
    }
    
    public static int[] readRecord() throws EOFException, IOException {
        byte[] buffer = new byte[RECORD_LENGTH * 2];
        int[] outbuffer = new int[RECORD_LENGTH];
        int sofar = 0;
        //Keep reading until we have the whole record
        while (sofar < RECORD_LENGTH * 2) {
            int res = in.read(buffer, sofar, RECORD_LENGTH * 2 - sofar);
            if (res == -1) {
                throw new EOFException();
            }
            sofar += res;
        }
        for (int i = 0; i < RECORD_LENGTH; i++) {
            short temp = (short) (buffer[2 * i] << 8 | (0xff & buffer[2 * i + 1]));
            outbuffer[i] = temp;
        }
        return outbuffer;
    }
    
    public double tile(double x) {
        double q = Math.floor(x);
        double r = (RECORD_LENGTH - 1) * (x - q);
        return r;
    }
    
    private void downloadFile(String filename) throws IOException, MissingDataFileException {
        if (!download) {
            
            String msg = String.format("Needs data file %s - Auto Download is disabled", filename);
            throw new MissingDataFileException(msg);
        }
        String urlString = getProperty(TerrainProperties.FTP) + getProperty(TerrainProperties.REGION) + "/" + filename;
        setMessage(String.format("Downloading data: %s", filename));
        java.net.URL url = new java.net.URL(urlString);
        InputStream ins = url.openStream();
        java.io.DataInputStream instream = new java.io.DataInputStream(ins);
        String outfilename = DIRECTORY + filename;
        File outfile = new java.io.File(outfilename);
        DataOutputStream outstream = new DataOutputStream(new FileOutputStream(outfile));
        
        int bytecount = 0;
        long sofar = 0;
        long guess = 900000;
        while (bytecount != -1) {
            byte[] buffer = new byte[BUFFERLENGTH];
            bytecount = instream.read(buffer);
            if (bytecount > 0) {
                outstream.write(buffer, 0, bytecount);
                sofar += bytecount;
                int progress = (int) (100 * sofar / guess);
                setProgress(progress);
                setMessage(String.format("Downloading data: %s", filename));
            }
        }
        outstream.close();
        instream.close();
        setMessage("");
        return;
    }
    
    private class CacheEntry {
        
        public CacheEntry(String name, int[] buffer) {
            page_name = name;
            last_used = cycle;
            data = buffer;
        }
        
        public void setLastUsed() {
            last_used = cycle;
        }
        
        public String getName() {
            return page_name;
        }
        
        public int getValue(int i) {
            return data[i];
        }
        
        public long lastUsed() {
            return last_used;
        }
        private long last_used;
        private String page_name;
        private int[] data;
    }
    
    public static double lagrangian(double x, java.awt.geom.Point2D.Double... points) {
        int n = points.length;
        double tot = 0.0;
        for (int i = 0; i < n; i++) {
            double prod = 1.0;
            for (int j = 0; j < n; j++) {
                if (j != i) {
                    prod *= (x - points[j].getX()) / (points[i].getX() - points[j].getX());
                }
            }
            tot += prod * points[i].getY();
        }
        return tot;
    }
    
    public long hits() {
        return hits;
    }
    
    public long resultcount() {
        return resultcount;
    }
    
    public long tries() {
        return tries;
    }
    
    public long missing() {
        return missing;
    }
    
    public void resetCounts() {
        hits = 0;
        tries = 0;
        missing = 0;
        resultcount = 0;
    }
    
    private void setMessage(String msg) {
        if (!msg.equals(lastMessage)) {
            firePropertyChange("message", lastMessage, msg);
        }
        lastMessage = msg;
    }
    
    private void setProgress(int val) {
        int v = Math.min(100, val);
        v = Math.max(0, v);
        if (lastValue != v) {
            firePropertyChange("progress", lastValue, v);
        }
        lastValue = v;
    }
    
    private String getProperty(TerrainProperties propertyName) {
        return TerrainFrame.properties.get(propertyName);
    }
    
    public void setDownload(boolean flag) {
        download = flag;
    }
    public PropertyChangeListener[] getPropertyChangeListeners(){
        PropertyChangeListener[] res = new PropertyChangeListener[1];
        res[0] = listener ;
        return res ;
    }
    
    
    private PropertyChangeListener listener = new java.beans.PropertyChangeListener() {
        
        public void propertyChange(java.beans.PropertyChangeEvent evt) {
            String propertyName = evt.getPropertyName();
            String propertyValue = evt.getNewValue().toString();
            if(propertyName.equalsIgnoreCase("download")){
                download = propertyValue.equalsIgnoreCase("true");
            }
            if(propertyName.equalsIgnoreCase("message")){
                lastMessage = propertyValue ;
            }
        }
    };
    
    private static java.util.zip.ZipInputStream in;
    private static final String DIRECTORY = "data/";
    private static final int RECORD_LENGTH = 1201;
    private long cycle = 0;
    private java.util.HashMap<String, CacheEntry> cache = new java.util.HashMap<String, CacheEntry>();
    private boolean cacheEnable = true;
    private static final int MAX_CACHE_SIZE = 16;
    private long hits = 0;
    private long tries = 0;
    private long missing = 0;
    private long resultcount = 0;
    public static final double MISSING = -32768.0;
    private static final int BUFFERLENGTH = 1024;
    private String lastMessage = "";
    private boolean download = true;
    private int lastValue = 0;
    private static final java.util.Locale LOCALE = java.util.Locale.UK ; //Force use of UK locale for number formatting
}
