/*
 * Telepathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 *
 * ti-page-properties.c:
 * A GtkNotebook page exposing
 * org.freedesktop.Telepathy.Properties functionality.
 *
 * Copyright (C) 2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2008 Nokia Corporation
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Originally by Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "page-properties.h"

#include <string.h>

#include <glade/glade.h>
#include <telepathy-glib/enums.h>

#include "page-priv.h"
#include "util.h"

enum {
    TI_COLUMN_PROP_ID = 0,
    TI_COLUMN_PROP_NAME,
    TI_COLUMN_PROP_SIGNATURE,
    TI_COLUMN_PROP_FLAGS,
    TI_COLUMN_PROP_VALUE,
    TI_PROP_N_COLUMNS
};

struct _TIPagePropertiesClass {
    TIPageClass parent;
};

G_DEFINE_TYPE (TIPageProperties, ti_page_properties, TI_TYPE_PAGE);

/* Function prototypes */
static void _ti_page_properties_setup_page (TIPage *page, GladeXML *glade_xml);
static void _ti_page_properties_restart_page (TIPage *page);
static void _ti_page_properties_build_tree_view_props (TIPageProperties *self, GladeXML *glade_xml);
static void _ti_page_properties_list (TIPageProperties *self);
static void _ti_page_properties_get (TIPageProperties *self);
static void _ti_page_properties_set (TIPageProperties *self, guint id, const gchar *signature, const gchar *new_value);
static void _ti_page_properties_add_prop (TIPageProperties *self, guint id, const gchar *name,
                                          const gchar *signature, guint flags);
static void _ti_page_properties_update_prop_value (TIPageProperties *self, guint id, const GValue *value);
static void _ti_page_properties_update_prop_flags (TIPageProperties *self, guint id, guint flags);
static void _ti_page_properties_get_prop_row (TIPageProperties *self, GtkTreeIter *iter, guint id);
static void _ti_page_properties_property_value_edited (TIPageProperties *self, gchar *row_index, gchar *new_value);
static gchar *_ti_prop_flags_to_string (guint flags);

/*
 * Instance private data.
 */
struct _TIPagePropertiesPrivate {
    TpProxy *proxy;

    GtkTreeView *tree_view_props;
    GtkListStore *list_store_props;
    GtkTreeSelection *selection_props;
};
typedef struct _TIPagePropertiesPrivate TIPagePropertiesPrivate;

#define TI_PAGE_PROPERTIES_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_PAGE_PROPERTIES, TIPagePropertiesPrivate))

static void
ti_page_properties_dispose (GObject *object)
{
    TIPageProperties *self = TI_PAGE_PROPERTIES (object);
    TIPagePropertiesPrivate *priv = TI_PAGE_PROPERTIES_GET_PRIVATE (self);

    if (priv->proxy != NULL)
      {
        g_object_unref (priv->proxy);
        priv->proxy = NULL;
      }

    G_OBJECT_CLASS (ti_page_properties_parent_class)->dispose (object);
}

static void
ti_page_properties_class_init (TIPagePropertiesClass *page_properties_class)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (page_properties_class);
    TIPageClass *page_class = TI_PAGE_CLASS (page_properties_class);

    gobject_class->dispose = ti_page_properties_dispose;

    page_class->setup_page = _ti_page_properties_setup_page;
    page_class->restart_page = _ti_page_properties_restart_page;

    /* Add private */
    g_type_class_add_private (page_properties_class, sizeof (TIPagePropertiesPrivate));
}

/*
 * Instance initialization.
 */
static void
ti_page_properties_init (TIPageProperties *self)
{
}

static void _ti_page_properties_on_props_changed (TpProxy *,
    const GPtrArray *, gpointer, GObject *);
static void _ti_page_properties_on_prop_flags_changed (TpProxy *,
    const GPtrArray *, gpointer, GObject *);

/*
 * Returns a new instance.
 */
TIPageProperties*
ti_page_properties_new (GtkNotebook *parent_notebook,
                        TpProxy *proxy)
{
    TIPage *page_properties;
    TIPagePropertiesPrivate *priv;

    page_properties = g_object_new (TI_TYPE_PAGE_PROPERTIES, NULL);
    priv = TI_PAGE_PROPERTIES_GET_PRIVATE (page_properties);

    priv->proxy = g_object_ref (proxy);

    tp_cli_properties_interface_connect_to_properties_changed (proxy,
        _ti_page_properties_on_props_changed, NULL, NULL,
        (GObject *) page_properties, NULL);

    tp_cli_properties_interface_connect_to_property_flags_changed (proxy,
        _ti_page_properties_on_prop_flags_changed, NULL, NULL,
        (GObject *) page_properties, NULL);

    _ti_page_new (&page_properties, parent_notebook, "page-properties.xml");

    return (TIPageProperties *) page_properties;
}

/*
 * Setup Page
 */
static void
_ti_page_properties_setup_page (TIPage *page, GladeXML *glade_xml)
{
    TIPageProperties *self = TI_PAGE_PROPERTIES (page);
    GtkWidget *widget;

    _ti_page_properties_build_tree_view_props (self, glade_xml);

    widget = glade_xml_get_widget (glade_xml, "button_refresh");
    g_assert (widget != NULL && GTK_IS_BUTTON (widget));
    g_signal_connect_swapped (widget, "clicked", G_CALLBACK (_ti_page_properties_list), self);

    widget = glade_xml_get_widget (glade_xml, "button_get");
    g_assert (widget != NULL && GTK_IS_BUTTON (widget));
    g_signal_connect_swapped (widget, "clicked", G_CALLBACK (_ti_page_properties_get), self);
}

/*
 * Restart Page
 */
static void
_ti_page_properties_restart_page (TIPage *page)
{
    TIPageProperties *self = TI_PAGE_PROPERTIES (page);

    _ti_page_properties_list (self);
}

/*
 * Build Treeview Props
 */
static void
_ti_page_properties_build_tree_view_props (TIPageProperties *self,
                                           GladeXML *glade_xml)
{
  TIPagePropertiesPrivate *priv = TI_PAGE_PROPERTIES_GET_PRIVATE (self);
  GtkCellRenderer *renderer;
  GtkCellRenderer *value_renderer;
  GtkTreeViewColumn *column;

  priv->tree_view_props = GTK_TREE_VIEW (glade_xml_get_widget (glade_xml,
        "treeview_props"));
  g_assert (priv->tree_view_props != NULL &&
      GTK_IS_TREE_VIEW (priv->tree_view_props));

  priv->list_store_props = gtk_list_store_new (TI_PROP_N_COLUMNS,
           G_TYPE_UINT,
           G_TYPE_STRING,
           G_TYPE_STRING,
           G_TYPE_STRING,
           G_TYPE_STRING,
           G_TYPE_STRING);

  gtk_tree_view_set_model (priv->tree_view_props,
      GTK_TREE_MODEL (priv->list_store_props));

  renderer = gtk_cell_renderer_text_new ();

  value_renderer = gtk_cell_renderer_text_new ();

  g_object_set (value_renderer,
      "editable", TRUE,
      NULL);

  g_signal_connect_swapped (value_renderer, "edited",
      G_CALLBACK (_ti_page_properties_property_value_edited), self);

  column = gtk_tree_view_column_new_with_attributes ("ID", renderer,
      "text", TI_COLUMN_PROP_ID,
      NULL);
  gtk_tree_view_append_column (priv->tree_view_props, column);

  column = gtk_tree_view_column_new_with_attributes ("Name", renderer,
      "text", TI_COLUMN_PROP_NAME,
      NULL);
  gtk_tree_view_append_column (priv->tree_view_props, column);

  column = gtk_tree_view_column_new_with_attributes ("Sig", renderer,
      "text", TI_COLUMN_PROP_SIGNATURE,
      NULL);
  gtk_tree_view_append_column (priv->tree_view_props, column);

  column = gtk_tree_view_column_new_with_attributes ("Flags", renderer,
      "text", TI_COLUMN_PROP_FLAGS,
      NULL);
  gtk_tree_view_append_column (priv->tree_view_props, column);

  column = gtk_tree_view_column_new_with_attributes ("Value", value_renderer,
      "text", TI_COLUMN_PROP_VALUE,
      NULL);
  gtk_tree_view_append_column (priv->tree_view_props, column);

  priv->selection_props = gtk_tree_view_get_selection (priv->tree_view_props);
  gtk_tree_selection_set_mode (priv->selection_props, GTK_SELECTION_MULTIPLE);
}

static void
_ti_page_properties_list (TIPageProperties *self)
{
  TIPagePropertiesPrivate *priv = TI_PAGE_PROPERTIES_GET_PRIVATE (self);
  GError *error = NULL;
  GPtrArray *props = NULL;
  GValueArray *property = NULL;
  guint i;
  guint id;
  const gchar *name;
  const gchar *signature;
  guint flags;

  gtk_list_store_clear (priv->list_store_props);

  if (!tp_cli_properties_interface_run_list_properties (priv->proxy,
        -1, &props, &error, NULL))
    {
      g_printerr ("ListProperties(): %s\n", error->message);
      g_error_free (error);
      return;
    }

  for (i = 0; i < props->len; i++)
    {
      property = (GValueArray *) g_ptr_array_index (props, i);

      id = g_value_get_uint (g_value_array_get_nth (property, 0));
      name = g_value_get_string (g_value_array_get_nth (property, 1));
      signature = g_value_get_string (g_value_array_get_nth (property, 2));
      flags = g_value_get_uint (g_value_array_get_nth (property, 3));

      _ti_page_properties_add_prop (self, id, name, signature, flags);

      g_value_array_free (property);
    }

  g_ptr_array_free (props, TRUE);
}

/*
 * Get Properties
 * Called when the "Get" button is clicked
 */
static void
_ti_page_properties_get (TIPageProperties *self)
{
  TIPagePropertiesPrivate *priv = TI_PAGE_PROPERTIES_GET_PRIVATE (self);
  guint i;
  guint id;
  GValue *value;
  GArray *prop_ids = NULL;
  GError *error = NULL;
  GPtrArray *prop_values = NULL;
  GValueArray *prop_value = NULL;

  prop_ids = ti_get_selected_elements (priv->selection_props,
      TI_COLUMN_PROP_ID, G_TYPE_UINT);

  if (prop_ids == NULL)
      goto CLEAN_UP;

  if (!tp_cli_properties_interface_run_get_properties (priv->proxy,
        -1, prop_ids, &prop_values, &error, NULL))
    {
      g_printerr ("GetProperties(): %s\n", error->message);
      g_error_free (error);
      goto CLEAN_UP;
    }

  for (i = 0; i < prop_values->len; i++)
    {
      prop_value = (GValueArray *) g_ptr_array_index (prop_values, i);

      id = g_value_get_uint (g_value_array_get_nth (prop_value, 0));
      value = (GValue *) g_value_get_boxed (g_value_array_get_nth (prop_value,
            1));

      g_assert (G_IS_VALUE (value));

      _ti_page_properties_update_prop_value (self, id, value);

      g_value_array_free (prop_value);
    }

CLEAN_UP:
  if (prop_ids != NULL)
    g_array_free (prop_ids, TRUE);

  if (prop_values != NULL)
    g_ptr_array_free (prop_values, TRUE);
}

static void
_ti_page_properties_set (TIPageProperties *self,
                         guint id,
                         const gchar *signature,
                         const gchar *new_value)
{
  TIPagePropertiesPrivate *priv = TI_PAGE_PROPERTIES_GET_PRIVATE (self);
  GError *error = NULL;
  GPtrArray *prop_values = NULL;
  GValueArray *prop_value_struct = NULL;
  GValue value;

  memset (&value, 0, sizeof (GValue));
  ti_string_to_value (&value, signature, new_value);

  prop_value_struct = g_value_array_new (2);

  g_value_array_append (prop_value_struct, NULL);
  g_value_init (g_value_array_get_nth (prop_value_struct, 0), G_TYPE_UINT);
  g_value_set_uint (g_value_array_get_nth (prop_value_struct, 0), id);

  g_value_array_append (prop_value_struct, NULL);
  g_value_init (g_value_array_get_nth (prop_value_struct, 1), G_TYPE_VALUE);
  g_value_set_boxed (g_value_array_get_nth (prop_value_struct, 1), &value);

  prop_values = g_ptr_array_new ();
  g_ptr_array_add (prop_values, prop_value_struct);

  if (!tp_cli_properties_interface_run_set_properties (priv->proxy,
        -1, prop_values, &error, NULL))
    {
      g_printerr ("SetProperties(): %s\n", error->message);
      g_error_free (error);
      return;
    }

  if (prop_values != NULL)
    g_ptr_array_free (prop_values, TRUE);

  if (prop_value_struct != NULL)
    g_value_array_free (prop_value_struct);

  g_value_unset (&value);
}

/*
 * Add Property
 * Adds a property (without its value) to the properties list.
 */
static void
_ti_page_properties_add_prop (TIPageProperties *self,
                              guint id,
                              const gchar *name,
                              const gchar *signature,
                              guint flags)
{
  TIPagePropertiesPrivate *priv = TI_PAGE_PROPERTIES_GET_PRIVATE (self);
  GtkTreeIter iter;
  gboolean ok;
  gboolean found = FALSE;
  guint curr_id;
  gchar *flags_str = NULL;

  ok = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (priv->list_store_props),
      &iter);
  while (ok && !found)
    {
      gtk_tree_model_get (GTK_TREE_MODEL (priv->list_store_props), &iter,
          TI_COLUMN_PROP_ID, &curr_id,
          -1);

      if (curr_id == id)
        {
          found = TRUE;
        }
      else
        {
          ok = gtk_tree_model_iter_next (
              GTK_TREE_MODEL (priv->list_store_props), &iter);
        }
    }

  if (!found)
    {
      gtk_list_store_append (priv->list_store_props, &iter);
    }

  flags_str = _ti_prop_flags_to_string (flags);

  gtk_list_store_set (priv->list_store_props, &iter,
      TI_COLUMN_PROP_ID, id,
      TI_COLUMN_PROP_NAME, name,
      TI_COLUMN_PROP_SIGNATURE, signature,
      TI_COLUMN_PROP_FLAGS, flags_str,
      -1);

  g_free (flags_str);
}

static void
_ti_page_properties_on_props_changed (TpProxy *proxy,
                                      const GPtrArray *prop_values,
                                      gpointer unused,
                                      GObject *object)
{
  TIPageProperties *self = TI_PAGE_PROPERTIES (object);
  GValueArray *prop_value;
  guint i;
  guint id;
  GValue *value;

  for (i = 0; i < prop_values->len; i++)
    {
      prop_value = (GValueArray *) g_ptr_array_index (prop_values, i);

      id = g_value_get_uint (g_value_array_get_nth (prop_value, 0));
      value = (GValue *) g_value_get_boxed (
          g_value_array_get_nth (prop_value, 1));

      g_assert (G_IS_VALUE (value));

      _ti_page_properties_update_prop_value (self, id, value);
    }
}

static void
_ti_page_properties_on_prop_flags_changed (TpProxy *proxy,
                                           const GPtrArray *flags,
                                           gpointer user_data,
                                           GObject *object)
{
  TIPageProperties *self = TI_PAGE_PROPERTIES (object);
  GValueArray *prop_flags;
  guint i;
  guint id;
  guint flags_uint;

  for (i = 0; i < flags->len; i++)
    {
      prop_flags = (GValueArray*) g_ptr_array_index (flags, i);

      id = g_value_get_uint (g_value_array_get_nth (prop_flags, 0));
      flags_uint = g_value_get_uint (g_value_array_get_nth (prop_flags, 1));

      _ti_page_properties_update_prop_flags (self, id, flags_uint);
    }
}

/*
 * Update Property Value
 * Updates the value of the property that has the given ID
 *
 * This is done on the local list that gets displayed to the user, not on the
 * connection manager.
 */
static void
_ti_page_properties_update_prop_value (TIPageProperties *self,
                                       guint id,
                                       const GValue *value)
{
  TIPagePropertiesPrivate *priv = TI_PAGE_PROPERTIES_GET_PRIVATE (self);
  GtkTreeIter iter;
  gchar *value_str = NULL;

  _ti_page_properties_get_prop_row (self, &iter, id);

  value_str = ti_value_to_string (value);

  gtk_list_store_set (priv->list_store_props, &iter,
      TI_COLUMN_PROP_VALUE, value_str,
      -1);

  g_free (value_str);
}

/*
 * Update Property Flags
  * Updates the flags of the property that has the given ID
 *
 * This is done on the local list that gets displayed to the user, not on the
 * connection manager.
 */
static void
_ti_page_properties_update_prop_flags (TIPageProperties *self,
                                       guint id,
                                       guint flags)
{
  TIPagePropertiesPrivate *priv = TI_PAGE_PROPERTIES_GET_PRIVATE (self);
  GtkTreeIter iter;
  gchar *flags_str = NULL;

  _ti_page_properties_get_prop_row (self, &iter, id);

  flags_str = _ti_prop_flags_to_string (flags);

  gtk_list_store_set (priv->list_store_props, &iter,
      TI_COLUMN_PROP_FLAGS, flags_str,
      -1);

  g_free (flags_str);
}

static void
_ti_page_properties_get_prop_row (TIPageProperties *self,
                                  GtkTreeIter *iter,
                                  guint id)
{
  TIPagePropertiesPrivate *priv = TI_PAGE_PROPERTIES_GET_PRIVATE (self);
  gboolean ok;
  gboolean found = FALSE;
  guint curr_id;

  ok = gtk_tree_model_get_iter_first (
      GTK_TREE_MODEL (priv->list_store_props), iter);
  while (ok && !found)
    {
      gtk_tree_model_get (GTK_TREE_MODEL (priv->list_store_props), iter,
          TI_COLUMN_PROP_ID, &curr_id,
          -1);

      if (curr_id == id)
        {
          found = TRUE;
        }
      else
        {
          ok = gtk_tree_model_iter_next (
              GTK_TREE_MODEL (priv->list_store_props), iter);
        }
    }

  if (!found)
    {
      g_error ("_ti_page_properties_get_prop_row: Haven't found a row for "
          "the given property id.");
    }
}

/*
 * Property Value Edited
 * Called after the user edits a property value (the cell at the "Value"
 * column of a given property row).
 */
static void
_ti_page_properties_property_value_edited (TIPageProperties *self,
                                           gchar *row_index,
                                           gchar *new_value)
{
  TIPagePropertiesPrivate *priv = TI_PAGE_PROPERTIES_GET_PRIVATE (self);
  GtkTreeIter iter;
  gchar *current_value = NULL;
  gchar *signature = NULL;
  guint id;
  gboolean ok;

  ok = gtk_tree_model_get_iter_from_string (
      GTK_TREE_MODEL (priv->list_store_props), &iter, row_index);
  g_assert (ok);

  gtk_tree_model_get (GTK_TREE_MODEL (priv->list_store_props), &iter,
      TI_COLUMN_PROP_ID, &id,
      TI_COLUMN_PROP_SIGNATURE, &signature,
      TI_COLUMN_PROP_VALUE, &current_value,
      -1);

  if (current_value == NULL)
    {
      current_value = g_strdup ("");
    }

  if (!g_str_equal (current_value, new_value))
    {
      _ti_page_properties_set (self, id, signature, new_value);
    }

  g_free (current_value);
  g_free (signature);
}

static gchar *_ti_prop_flags_to_string (guint flags)
{
  GString *flags_str = g_string_new (NULL);

  if ((flags & TP_PROPERTY_FLAG_READ) != 0)
    g_string_append (flags_str, "r");

  if ((flags & TP_PROPERTY_FLAG_WRITE) != 0)
    g_string_append (flags_str, "w");

  return g_string_free (flags_str, FALSE);
}
