/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2010 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "touchpadadaptor.h"
#include <QtDBus/QDBusConnection>
#include <QtDBus/QDBusMessage>

/*
 * Implementation of adaptor class TouchpadAdaptor
 */

using namespace synaptiks;


TouchpadAdaptor::TouchpadAdaptor(synaptiks::Touchpad *parent):
    QDBusAbstractAdaptor(parent) {
}

TouchpadAdaptor::~TouchpadAdaptor() {}

QString TouchpadAdaptor::name() const {
    return this->propertyOrDBusError("name").toString();
}

bool TouchpadAdaptor::isOn() const {
    return this->propertyOrDBusError("on").toBool();
}

void TouchpadAdaptor::setOn(bool on) {
    this->setPropertyOrDBusError("on", on);
}

double TouchpadAdaptor::minimumSpeed() const {
    return this->propertyOrDBusError("minimumSpeed").toDouble();
}

void TouchpadAdaptor::setMinimumSpeed(double speed) {
    this->setPropertyOrDBusError("minimumSpeed", speed);
}

double TouchpadAdaptor::maximumSpeed() const {
    return this->propertyOrDBusError("maximumSpeed").toDouble();
}

void TouchpadAdaptor::setMaximumSpeed(double speed) {
    this->setPropertyOrDBusError("maximumSpeed", speed);
}

double TouchpadAdaptor::accelerationFactor() const {
    return this->propertyOrDBusError("accelerationFactor").toDouble();
}

void TouchpadAdaptor::setAccelerationFactor(double accel) {
    this->setPropertyOrDBusError("accelerationFactor", accel);
}

bool TouchpadAdaptor::edgeMotionAlways() const {
    return this->propertyOrDBusError("edgeMotionAlways").toBool();
}

void TouchpadAdaptor::setEdgeMotionAlways(bool enabled) {
    this->setPropertyOrDBusError("edgeMotionAlways", enabled);
}

bool TouchpadAdaptor::tapAndDragGesture() const {
    return this->propertyOrDBusError("tapAndDragGesture").toBool();
}

void TouchpadAdaptor::setTapAndDragGesture(bool enabled) {
    this->setPropertyOrDBusError("tapAndDragGesture", enabled);
}

bool TouchpadAdaptor::lockedDrags() const {
    return this->propertyOrDBusError("lockedDrags").toBool();
}

void TouchpadAdaptor::setLockedDrags(bool enabled) {
    this->setPropertyOrDBusError("lockedDrags", enabled);
}

int TouchpadAdaptor::lockedDragsTimeout() const {
    return this->propertyOrDBusError("lockedDragsTimeout").toInt();
}

void TouchpadAdaptor::setLockedDragsTimeout(int timeout) {
    this->setPropertyOrDBusError("lockedDragsTimeout", timeout);
}

bool TouchpadAdaptor::circularScrolling() const {
    return this->propertyOrDBusError("circularScrolling").toBool();
}

void TouchpadAdaptor::setCircularScrolling(bool enabled) {
    this->setPropertyOrDBusError("circularScrolling", enabled);
}

uchar TouchpadAdaptor::circularScrollingTrigger() const {
    return this->propertyOrDBusError(
        "circularScrollingTrigger").value<uchar>();
}

void TouchpadAdaptor::setCircularScrollingTrigger(uchar trigger) {
    if (trigger > Touchpad::TopLeftCornerTrigger) {
        QString message = i18nc("touchpad dbus error",
                                "The argument must be in [%1; %2]",
                                Touchpad::AnyEdgeTrigger,
                                Touchpad::TopLeftCornerTrigger);
        this->parent()->sendErrorReply("org.kde.TouchpadError", message);
    } else {
        this->setPropertyOrDBusError("circularScrollingTrigger", trigger);
    }
}

double TouchpadAdaptor::circularScrollingDistance() const {
    return this->propertyOrDBusError(
        "circularScrollingDistance").toDouble();
}

void TouchpadAdaptor::setCircularScrollingDistance(double angle) {
    this->setPropertyOrDBusError("circularScrollingDistance", angle);
}

bool TouchpadAdaptor::horizontalTwoFingerScrolling() const {
    return this->propertyOrDBusError(
        "horizontalTwoFingerScrolling").toBool();
}

void TouchpadAdaptor::setHorizontalTwoFingerScrolling(bool enabled) {
    this->setPropertyOrDBusError("horizontalTwoFingerScrolling", enabled);
}

bool TouchpadAdaptor::verticalTwoFingerScrolling() const {
    return this->propertyOrDBusError(
        "verticalTwoFingerScrolling").toBool();
}

void TouchpadAdaptor::setVerticalTwoFingerScrolling(bool enabled) {
    this->setPropertyOrDBusError("verticalTwoFingerScrolling", enabled);
}

bool TouchpadAdaptor::horizontalEdgeScrolling() const {
    return this->propertyOrDBusError("horizontalEdgeScrolling").toBool();
}

void TouchpadAdaptor::setHorizontalEdgeScrolling(bool enabled) {
    this->setPropertyOrDBusError("horizontalEdgeScrolling", enabled);
}

bool TouchpadAdaptor::verticalEdgeScrolling() const {
    return this->propertyOrDBusError("verticalEdgeScrolling").toBool();
}

void TouchpadAdaptor::setVerticalEdgeScrolling(bool enabled) {
    this->setPropertyOrDBusError("verticalEdgeScrolling", enabled);
}

bool TouchpadAdaptor::cornerCoasting() const {
    return this->propertyOrDBusError("cornerCoasting").toBool();
}

void TouchpadAdaptor::setCornerCoasting(bool enabled) {
    this->setPropertyOrDBusError("cornerCoasting", enabled);
}

double TouchpadAdaptor::coastingSpeed() const {
    return this->propertyOrDBusError("coastingSpeed").toDouble();
}

void TouchpadAdaptor::setCoastingSpeed(double speed) {
    this->setPropertyOrDBusError("coastingSpeed", speed);
}

int TouchpadAdaptor::horizontalScrollingDistance() const {
    return this->propertyOrDBusError("horizontalScrollingDistance").toInt();
}

void TouchpadAdaptor::setHorizontalScrollingDistance(int distance) {
    this->setPropertyOrDBusError("horizontalScrollingDistance", distance);
}

int TouchpadAdaptor::verticalScrollingDistance() const {
    return this->propertyOrDBusError("verticalScrollingDistance").toInt();
}

void TouchpadAdaptor::setVerticalScrollingDistance(int distance) {
    this->setPropertyOrDBusError("verticalScrollingDistance", distance);
}

QByteArray TouchpadAdaptor::cornerButtons() const {
    return this->propertyOrDBusError("cornerButtons").toByteArray();
}

void TouchpadAdaptor::setCornerButtons(const QByteArray &buttons) {
    if (buttons.size() < 4) {
        QString message = i18ncp(
            "touchpad dbus error",
            "The argument must have at least one element",
            "The argument must have at least %1 elements", 4);
        this->parent()->sendErrorReply("org.kde.TouchpadError", message);
    } else {
        this->setPropertyOrDBusError("cornerButtons", buttons);
    }
}

QByteArray TouchpadAdaptor::fingerButtons() const {
    return this->propertyOrDBusError("fingerButtons").toByteArray();
}

void TouchpadAdaptor::setFingerButtons(const QByteArray &buttons) {
    if (buttons.size() < 3) {
        QString message = i18np(
            "The argument must have at least one element",
            "The argument must have at least %1 elements", 3);
        this->parent()->sendErrorReply("org.kde.TouchpadError", message);
    } else {
        this->setPropertyOrDBusError("fingerButtons", buttons);
    }
}

bool TouchpadAdaptor::fastTaps() const {
    return this->propertyOrDBusError("fastTaps").toBool();
}

void TouchpadAdaptor::setFastTaps(bool enabled) {
    this->setPropertyOrDBusError("fastTaps", enabled);
}

bool TouchpadAdaptor::circularTouchpad() const {
    return this->propertyOrDBusError("circularTouchpad").toBool();
}

void TouchpadAdaptor::setCircularTouchpad(bool circular) {
    this->setPropertyOrDBusError("circularTouchpad", circular);
}

bool TouchpadAdaptor::hasLeftButton() const {
    return this->propertyOrDBusError("hasLeftButton").toBool();
}

bool TouchpadAdaptor::hasMiddleButton() const {
    return this->propertyOrDBusError("hasMiddleButton").toBool();
}

bool TouchpadAdaptor::hasRightButton() const {
    return this->propertyOrDBusError("hasRightButton").toBool();
}

int TouchpadAdaptor::fingerDetection() const {
    return this->propertyOrDBusError("fingerDetection").toInt();
}

QVariant TouchpadAdaptor::propertyOrDBusError(const char *name) const {
    try {
        return this->parent()->property(name);
    } catch (const QXDeviceError &error) {
        this->parent()->sendErrorReply(
            "org.kde.TouchpadError", error.toString());
        return QVariant();
    }
}

void TouchpadAdaptor::setPropertyOrDBusError(const char *name,
                                             const QVariant &value) {
    try {
        this->parent()->setProperty(name, value);
    } catch (const QXDeviceError &error) {
        this->parent()->sendErrorReply(
            "org.kde.TouchpadError", error.toString());
    }
}
