/*
 * synaptiks -- a touchpad control tool
 *
 *
 * Copyright (C) 2010 Sebastian Wiesner <basti.wiesner@gmx.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */


#ifndef SYNAPTIKSDAEMON_H
#define SYNAPTIKSDAEMON_H

#include <KDEDModule>
#include <QtCore/QList>
#include <QtCore/QVariant>


namespace synaptiks {
    class SynaptiksDaemonPrivate;

    /**
     * @brief The @b synaptiks daemon module.
     *
     * This module manages the touchpad and exports a convenient DBus API
     * for touchpad configuration and management.
     *
     * It registers the service @c "org.kde.synaptiks" and exports the
     * following objects on this service:
     *
     * @li @c /modules/synaptiks – this object
     * @li @c /MouseDevicesMonitor – an instance of
     *     synaptiks::MouseDevicesMonitor
     * @li @c /TouchpadManager – an instance of
     *     synaptiks::TouchpadManager (use this object to switch the
     *     touchpad on or off!)
     * @li @c /Touchpad – the touchpad object itself (an
     *     instance of synaptiks::Touchpad)
     *
     * @note The two latter objects are only exported, if a configurable
     * touchpad was found, so use isTouchpadAvailable() before accessing
     * these objects to make sure, that a touchpad is available.
     */
    class KDE_EXPORT SynaptiksDaemon: public KDEDModule {
        Q_OBJECT
        Q_CLASSINFO("D-Bus Interface", "org.kde.Synaptiks")

    public:
        /**
         * @brief Create a new daemon module.
         *
         * @param parent the parent object
         * @param args ignored, required by KPlugin signature
         */
        SynaptiksDaemon(QObject *parent, const QList<QVariant>& args);

        /**
         * @brief Destroy this module.
         */
        virtual ~SynaptiksDaemon();

    public Q_SLOTS:
        /**
         * @brief Reparse the daemon and the touchpad configuration.
         *
         * If called, this method will re-configure this module, and update
         * the touchpad properties according to the configuration file
         * contents.
         *
         * This function is exported on DBus.
         */
        Q_SCRIPTABLE void reparseConfiguration();

        /**
         * @brief Check whether a configurable touchpad is available.
         *
         * Exported over D-Bus.
         *
         * Call touchpadNotAvailableMessage() to get a human-readable error
         * message, if this returns @c false.
         *
         * @return @c true, if a touchpad is available, @c false otherwise.
         * @sa touchpadNotAvailableMessage()
         */
        Q_SCRIPTABLE bool isTouchpadAvailable() const;

        /**
         * @brief Return a human-readable error message if no touchpad is
         * available.
         *
         * Exported over D-Bus.
         *
         * @return a human-readable error message, if no touchpad is
         *         available, empty QString() otherwise.
         * @sa isTouchpadAvailable()
         */
        Q_SCRIPTABLE QString touchpadNotAvailableMessage() const;

    private Q_SLOTS:
        /**
         * @brief Show the current state of the touchpad.
         */
        void showTouchpadState() const;

        /**
         * @brief Called to show the touchpad state as given by the
         * arguments.
         *
         * @param on is the touchpad on?
         * @param reason the reason for the touchpad switch
         * @param closure additional data
         */
        void showTouchpadState(bool on, const QString &reason,
                               const QVariant &closure=QVariant()) const;

        /**
         * @brief Notify about an error.
         *
         * @param message a human readable error message
         */
        void notifyError(const QString &message) const;

        /**
         * @brief Called to handle the global shortcut.
         */
        void touchpadOnTriggered(bool) const;

    private:
        Q_DECLARE_PRIVATE(SynaptiksDaemon)

        /**
         * @brief Dump the touchpad state to the configuration object.
         *
         * This function reads all configurable properties from the
         * touchpad, and writes them to the configuration.
         *
         * This is used to initialize the configuration with defaults from
         * the touchpad driver.
         */
        void dumpTouchpadToConfiguration();

        /**
         * @brief Set the given touchpad @p property to the given @p value.
         *
         * This function does not propagate any exceptions from
         * synaptiks::Touchpad, but catches them all and calls
         * notifyError(const QString&) instead.
         *
         * Do @e always use this function, if you access touchpad properties
         * from slots or DBus calls!
         */
        void setTouchpadProperty(const char *property,
                                 const QVariant &value);

        /**
         * @brief Get the given touchpad @p property.
         *
         * This function does not propagate any exceptions from
         * synaptiks::Touchpad, but catches and silently ignores them.  In
         * this case, an invalid QVariant is returned.
         *
         * Do @e always use this function, if you access touchpad properties
         * from slots or DBus calls!
         *
         * @return the property value, or an invalid QVariant in case of
         *         exception
         */
        QVariant touchpadProperty(const char *property) const;

        SynaptiksDaemonPrivate *const d_ptr;
    };
}

#endif /* SYNAPTIKSDAEMON_H */
