// Copyright (C) 2009-2009 Martin Sandve Alnæs
// Licensed under the GNU LGPL Version 3.0.
//
// First added:  2009-01-01
// Last changed: 2009-04-01
//
// This demo program computes a projection of a function
// between two mixed finite element spaces.

#include <dolfin.h>
#include "generated_code/ProjectionMixed.h"

using namespace dolfin;
using namespace ProjectionMixed;

class FScalar: public Function
{
public:
  FScalar(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    values[0] = 3.14159*x[0]*x[1];
  }
};

class FVector: public Function
{
public:
  FVector(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    double dx = x[0] - 0.5;
    double dy = x[1] - 0.5;
    values[0] = 2.0 * x[0];
    values[1] = 3.0 * x[1];
  }
};

class FTensor: public Function
{
public:
  FTensor(const FunctionSpace & V):
    Function(V) {}

  void eval(double* values, const double* x) const
  {
    double dx = x[0] - 0.5;
    double dy = x[1] - 0.5;
    values[0] = 0.1 * x[0] * x[1];
    values[1] = 0.2 * x[0] * x[1];
    values[2] = 0.3 * x[0] * x[1];
    values[3] = 0.4 * x[0] * x[1];
  }
};

int main(int argc, char ** argv)
{
  UnitSquare mesh(10, 10);

  BilinearForm::TrialSpace V(mesh);

  CoefficientSpace_ft FT(mesh);
  CoefficientSpace_fv FV(mesh);
  CoefficientSpace_fs FS(mesh);

  FTensor ft(FT);
  FVector fv(FV);
  FScalar fs(FS);

  Function u(V);

  BilinearForm a(V, V);
  LinearForm L(V);

  L.ft = ft;
  L.fv = fv;
  L.fs = fs;

  Matrix A;
  Vector b;
  Assembler::assemble(A, a);
  Assembler::assemble(b, L);

  LUSolver solver(symmetric);
  solver.solve(A, u.vector(), b);

  // Get subfunctions
  Function ut = u[0];
  Function uv = u[1];
  Function us = u[2];
  
  // Write function and subfunctions to file
  File u_file("u.pvd");
  u_file << u;
  File ut_file("ut.pvd");
  ut_file << ut;
  File uv_file("uv.pvd");
  uv_file << uv;
  File us_file("us.pvd");
  us_file << us;

  return 0;
}

