/*==================================================================
 * menutbar.c - Menu and tool bar interface and callback routines
 *
 * Swami
 * Copyright (C) 1999-2003 Josh Green <jgreen@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA or point your web browser to http://www.gnu.org.
 *
 * To contact the author of this program:
 * Email: Josh Green <jgreen@users.sourceforge.net>
 * Swami homepage: http://swami.sourceforge.net
 *==================================================================*/
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <gtk/gtk.h>

#include <libswami/SwamiObject.h>
#include "GObjSup.h"

#include "menutbar.h"

#include "SwamiUIMidiCtrl.h"
#include "SwamiUIModEdit.h"
#include "SwamiUIObject.h"
#include "SwamiUISpanWin.h"
#include "i18n.h"
#include "pixmap.h"
#include "util.h"

static GtkWidget *tbar_drivers_togbtn;
static GtkWidget *tbar_gen_view_togbtn;
static GtkWidget *tbar_gen_ctrl_togbtn;

#ifdef CANVAS_SUPPORT
static GtkWidget *tbar_gen_graph_togbtn;
#endif

static GtkWidget *tbar_sam_view_togbtn;
static GtkWidget *tbar_mod_edit_togbtn;
static GtkWidget *tbar_piano_togbtn;
static GtkWidget *tbar_velocity_togbtn;

static GtkWidget *tbar_drivers_on_pm;
static GtkWidget *tbar_drivers_off_pm;
static gboolean tbar_drivers_on = FALSE;

static gboolean block_menu_lowpane_select = FALSE;
static gboolean block_menu_spanwin_mode = FALSE;
static gboolean block_tbar_lowpane_btns = FALSE;
static gboolean block_tbar_piano_mode_btns = FALSE ;

/* function prototypes */

static void swamiui_menu_lowpane_select (int mode);
static void swamiui_menu_spanwin_mode (GtkWidget *widg, int mode);

static gboolean tbar_cb_drivers_toggle (GtkWidget *btn);
static gboolean tbar_cb_lower_view_button_toggled (GtkWidget *rad,
						   int wintype);
static void tbar_cb_piano_mode_button_toggled (GtkWidget *rad, int mode);

/* menu functions */

/** main menu callback routine for "Quit" */
void
swamiui_menu_cb_quit (void)
{
  swamiui_quit ();
}

/** main menu callback to create a new sound font */
void
swamiui_menu_cb_new_sfont (void)
{
  swamiui_new_item (NULL, IPITEM_SFONT);
}

/** main menu callback to save files */
void
swamiui_menu_cb_save (GtkWidget *mitem, gpointer data)
{
  gboolean saveas = GPOINTER_TO_INT (data);
  SwamiUITree *tree;
  GList *items;

  tree = SWAMIUI_TREE (swamiui_lookup_object ("SwamiUITree"));
  if (!tree) return;

  items = swamiui_tree_get_selection (tree);
  swamiui_save_files (items, saveas);
  swamiui_tree_free_selection (items);
}

/* menu callback to edit global modulators */
void
swamiui_menu_cb_edit_global_mods (GtkMenuItem *menuitem)
{
  SwamiUIModEdit *modedit;

  swamiui_lowpane_set_view (SWAMIUI_LOWPANE_MODEDIT);
  swamiui_menu_set_lowpane_radio_item (SWAMIUI_LOWPANE_MODEDIT);
  swamiui_tbar_set_lowpane_togbtn (SWAMIUI_LOWPANE_MODEDIT);

  modedit = SWAMIUI_MODEDIT (swamiui_lookup_object ("SwamiUIModEdit"));
  swamiui_modedit_global_mods (modedit);
}

/* menu callback for enable global modulators check menu item */
void
swamiui_menu_cb_enable_global_mods_toggle (GtkMenuItem *menuitem)
{
  /* GTK 1.2 doesn't have any accessors for the active field */
  swamiui_object->enable_global_mods = GTK_CHECK_MENU_ITEM (menuitem)->active;
  swamiui_update_global_mods ();
}

/* menu radio item callback to select generator view */
void
swamiui_menu_cb_lowpane_genview (GtkMenuItem *menuitem)
{
  swamiui_menu_lowpane_select (SWAMIUI_LOWPANE_VIEW);
}

/* menu radio item callback to select generator control */
void
swamiui_menu_cb_lowpane_genctrl (GtkMenuItem *menuitem)
{
  swamiui_menu_lowpane_select (SWAMIUI_LOWPANE_CTRL);
}

/* menu radio item callback to select generator graph */
void
swamiui_menu_cb_lowpane_gengraph (GtkMenuItem *menuitem)
{
  swamiui_menu_lowpane_select (SWAMIUI_LOWPANE_GRAPH);
}

/* menu radio item callback to select sample viewer */
void
swamiui_menu_cb_lowpane_samview (GtkMenuItem *menuitem)
{
  swamiui_menu_lowpane_select (SWAMIUI_LOWPANE_SAMVIEW);
}

/* menu radio item callback to select modulator editor */
void
swamiui_menu_cb_lowpane_modedit (GtkMenuItem *menuitem)
{
  swamiui_menu_lowpane_select (SWAMIUI_LOWPANE_MODEDIT);
}

/* menu radio item callback to select spanwin piano mode */
void
swamiui_menu_cb_spanwin_piano (GtkMenuItem *menuitem)
{
  swamiui_menu_spanwin_mode (GTK_WIDGET (menuitem), SWAMIUI_SPANWIN_PIANO);
}

/* menu radio item callback to select spanwin piano mode */
void
swamiui_menu_cb_spanwin_velocity (GtkMenuItem *menuitem)
{
  swamiui_menu_spanwin_mode (GTK_WIDGET (menuitem), SWAMIUI_SPANWIN_VELOCITY);
}

/** HACK: Grrr. Glade overwrites callback data when menus are saved so we
    have to have a callback for each radio item, this routine sets the
    low pane view */
static void
swamiui_menu_lowpane_select (int mode)
{
  if (block_menu_lowpane_select) return;

  swamiui_lowpane_set_view (mode);
  swamiui_tbar_set_lowpane_togbtn (mode);
}

/** HACK: Grrr. Glade overwrites callback data when menus are saved so we
    have to have a callback for each radio item, this routine sets the
    spanwin mode */
static void
swamiui_menu_spanwin_mode (GtkWidget *widg, int mode)
{
  SwamiUISpanWin *spanwin;

  if (block_menu_spanwin_mode) return;

  if (GTK_CHECK_MENU_ITEM (widg)->active)
    {
      spanwin = SWAMIUI_SPANWIN (swamiui_lookup_object ("SwamiUISpanWin"));
      swamiui_spanwin_set_mode (spanwin, mode);
      swamiui_tbar_set_piano_mode_togbtn (mode);
    }
}

/** main menu callback to restart drivers */
void
swamiui_menu_cb_restart_drivers (GtkWidget *widg)
{
  SwamiWavetbl *wavetbl;
  SwamiMidi *midi;
  SwamiUIMidiCtrl *midictrl;
  gboolean active;

  wavetbl = SWAMI_WAVETBL (swami_get_object_by_type (G_OBJECT (swami_object),
						     "SwamiWavetbl"));
  midi = SWAMI_MIDI (swami_get_object_by_type (G_OBJECT (swami_object),
					       "SwamiMidi"));
  midictrl = SWAMIUI_MIDICTRL (swamiui_lookup_object ("SwamiUIMidiCtrl"));

  if (!wavetbl || !midi) return;

  g_object_get (G_OBJECT (wavetbl), "active", &active, NULL);

  if (active)
    {
      swami_wavetbl_close_driver (wavetbl);
      swami_midi_close_driver (midi);
    }

  swami_wavetbl_init_driver (wavetbl);
  swami_midi_init_driver (midi);
  swamiui_midictrl_midi_update_all (midictrl);

  swamiui_tbar_update_drivers_togbtn ();
}

/** routine to update main menu radio button to active lowpane mode */
void
swamiui_menu_set_lowpane_radio_item (int mode)
{
  GtkWidget *mnuitem;

  switch (mode)
    {
    case SWAMIUI_LOWPANE_VIEW:
      mnuitem = gtk_object_get_data (GTK_OBJECT (swamiui_object->main_window),
				     "MNUgenview");
      break;
    case SWAMIUI_LOWPANE_CTRL:
      mnuitem = gtk_object_get_data (GTK_OBJECT (swamiui_object->main_window),
				     "MNUgenctrl");
      break;
#ifdef CANVAS_SUPPORT
    case SWAMIUI_LOWPANE_GRAPH:
      mnuitem = gtk_object_get_data (GTK_OBJECT (swamiui_object->main_window),
				     "MNUgengraph");
      break;
#endif
    case SWAMIUI_LOWPANE_SAMVIEW:
      mnuitem = gtk_object_get_data (GTK_OBJECT (swamiui_object->main_window),
				     "MNUsamview");
      break;
    case SWAMIUI_LOWPANE_MODEDIT:
      mnuitem = gtk_object_get_data (GTK_OBJECT (swamiui_object->main_window),
				     "MNUmodedit");
      break;
    default:
      return;
    }
  block_menu_lowpane_select = TRUE;
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mnuitem), TRUE);
  block_menu_lowpane_select = FALSE;
}

/** routine to update main menu radio button for active piano/velocity mode */
void
swamiui_menu_set_piano_mode_radio_item (int mode)
{
  GtkWidget *mnuitem;

  switch (mode)
    {
    case SWAMIUI_SPANWIN_PIANO:
      mnuitem = gtk_object_get_data (GTK_OBJECT (swamiui_object->main_window),
				     "MNUpiano");
      break;
    case SWAMIUI_SPANWIN_VELOCITY:
      mnuitem = gtk_object_get_data (GTK_OBJECT (swamiui_object->main_window),
				     "MNUvelocity");
      break;
    }
  block_menu_spanwin_mode = TRUE;
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (mnuitem), TRUE);
  block_menu_spanwin_mode = FALSE;
}


/* Toolbar functions */

/** toolbar create function */
GtkWidget *
swamiui_tbar_new (void)
{
  GtkWidget *toolbar;
  GtkWidget *icon;

  toolbar = gtk_toolbar_new (GTK_ORIENTATION_HORIZONTAL, GTK_TOOLBAR_ICONS);
  gtk_toolbar_set_space_size (GTK_TOOLBAR (toolbar), 10);
  gtk_container_set_border_width (GTK_CONTAINER (toolbar), 5);
  gtk_toolbar_set_tooltips (GTK_TOOLBAR (toolbar), TRUE);


  /* drivers on/off toggle button (pixmap toggles too) */

  tbar_drivers_on_pm = swamiui_util_create_pixmap (drivers_on_xpm);
  gtk_widget_show (tbar_drivers_on_pm);
  gtk_widget_ref (tbar_drivers_on_pm);	/* reference inactive pixmap widget */

  tbar_drivers_off_pm = swamiui_util_create_pixmap (drivers_off_xpm);
  gtk_widget_show (tbar_drivers_off_pm);

  tbar_drivers_togbtn = gtk_toggle_button_new ();
  gtk_container_add (GTK_CONTAINER (tbar_drivers_togbtn), tbar_drivers_off_pm);
  gtk_widget_show (tbar_drivers_togbtn);

  gtk_signal_connect (GTK_OBJECT (tbar_drivers_togbtn), "toggled",
    (GtkSignalFunc)tbar_cb_drivers_toggle, NULL);

  gtk_toolbar_append_widget (GTK_TOOLBAR (toolbar), tbar_drivers_togbtn,
    _("Toggle drivers on/off"), NULL);


  gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));


  /* lower pane toggle buttons */

  icon = swamiui_util_create_pixmap (gen_view_xpm);
  tbar_gen_view_togbtn = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
    GTK_TOOLBAR_CHILD_RADIOBUTTON, NULL, _("Gen View"),
    _("View generator values"), NULL,
    icon, NULL, NULL);
  gtk_widget_show (tbar_gen_view_togbtn);

  icon = swamiui_util_create_pixmap (gen_ctrl_xpm);
  tbar_gen_ctrl_togbtn = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
    GTK_TOOLBAR_CHILD_RADIOBUTTON, tbar_gen_view_togbtn, _("Gen Control"),
    _("Set generator values"), NULL,
    icon, NULL, NULL);
  gtk_widget_show (tbar_gen_ctrl_togbtn);

#ifdef CANVAS_SUPPORT
  icon = swamiui_util_create_pixmap (gen_graph_xpm);
  tbar_gen_graph_togbtn = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
    GTK_TOOLBAR_CHILD_RADIOBUTTON, tbar_gen_ctrl_togbtn, _("Gen Graph"),
    _("Graph generator values"), NULL,
    icon, NULL, NULL);
  gtk_widget_show (tbar_gen_graph_togbtn);

  gtk_signal_connect (GTK_OBJECT (tbar_gen_graph_togbtn), "toggled",
		      (GtkSignalFunc)tbar_cb_lower_view_button_toggled,
		      GINT_TO_POINTER (SWAMIUI_LOWPANE_GRAPH));
#endif

  icon = swamiui_util_create_pixmap (samview_xpm);
  tbar_sam_view_togbtn = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
    GTK_TOOLBAR_CHILD_RADIOBUTTON, tbar_gen_ctrl_togbtn, _("Sample View"),
    _("Set sample loop parameters"), NULL,
    icon, NULL, NULL);
  gtk_widget_show (tbar_sam_view_togbtn);

  icon = swamiui_util_create_pixmap (modedit_xpm);
  tbar_mod_edit_togbtn = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
    GTK_TOOLBAR_CHILD_RADIOBUTTON, tbar_sam_view_togbtn,
    _("Modulator Editor"), _("Edit modulators"), "MOD", icon, NULL, NULL);
  gtk_widget_show (tbar_mod_edit_togbtn);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (tbar_gen_view_togbtn),
				TRUE);

  gtk_signal_connect (GTK_OBJECT (tbar_gen_view_togbtn), "toggled",
		      (GtkSignalFunc)tbar_cb_lower_view_button_toggled,
		      GINT_TO_POINTER (SWAMIUI_LOWPANE_VIEW));
  gtk_signal_connect (GTK_OBJECT (tbar_gen_ctrl_togbtn), "toggled",
		      (GtkSignalFunc)tbar_cb_lower_view_button_toggled,
		      GINT_TO_POINTER (SWAMIUI_LOWPANE_CTRL));
  gtk_signal_connect (GTK_OBJECT (tbar_sam_view_togbtn), "toggled",
		      (GtkSignalFunc)tbar_cb_lower_view_button_toggled,
		      GINT_TO_POINTER (SWAMIUI_LOWPANE_SAMVIEW));
  gtk_signal_connect (GTK_OBJECT (tbar_mod_edit_togbtn), "toggled",
		      (GtkSignalFunc)tbar_cb_lower_view_button_toggled,
		      GINT_TO_POINTER (SWAMIUI_LOWPANE_MODEDIT));

  gtk_toolbar_append_space (GTK_TOOLBAR (toolbar));


  /* piano/velocity toggle buttons */

  icon = swamiui_util_create_pixmap (piano_xpm);
  tbar_piano_togbtn = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
    GTK_TOOLBAR_CHILD_RADIOBUTTON, NULL, _("Piano"), _("Piano mode"),
    NULL, icon, NULL, NULL);
  gtk_widget_show (tbar_piano_togbtn);

  icon = swamiui_util_create_pixmap (velocity_xpm);
  tbar_velocity_togbtn = gtk_toolbar_append_element (GTK_TOOLBAR (toolbar),
    GTK_TOOLBAR_CHILD_RADIOBUTTON, tbar_piano_togbtn, _("Velocity"),
    _("Velocity mode"), NULL, icon, NULL, NULL);
  gtk_widget_show (tbar_velocity_togbtn);

  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (tbar_piano_togbtn), TRUE);

  gtk_signal_connect (GTK_OBJECT (tbar_piano_togbtn), "toggled",
    (GtkSignalFunc)tbar_cb_piano_mode_button_toggled,
    GINT_TO_POINTER (SWAMIUI_SPANWIN_PIANO));
  gtk_signal_connect (GTK_OBJECT (tbar_velocity_togbtn), "toggled",
    (GtkSignalFunc)tbar_cb_piano_mode_button_toggled,
    GINT_TO_POINTER (SWAMIUI_SPANWIN_VELOCITY));

  swamiui_tbar_update_drivers_togbtn ();

  return (toolbar);
}

/** updates the drivers toggle button with current state of drivers */
void
swamiui_tbar_update_drivers_togbtn (void)
{
  SwamiWavetbl *wavetbl;
  gboolean active = FALSE;

  wavetbl = SWAMI_WAVETBL (swami_get_object_by_type (G_OBJECT (swami_object),
						     "SwamiWavetbl"));
  if (wavetbl)
    g_object_get (G_OBJECT (wavetbl), "active", &active, NULL);

  swamiui_tbar_set_drivers_togbtn (active);
}

/** sets the state of the drivers toggle button */
void
swamiui_tbar_set_drivers_togbtn (gboolean active)
{
  GtkWidget *old_pm, *new_pm;

  if (tbar_drivers_on != active)	/* change pixmap? */
    {
      if (tbar_drivers_on) /* change from "on" state to "off" */
	{
	  old_pm = tbar_drivers_on_pm;
	  new_pm = tbar_drivers_off_pm;
	}
      else				/* change from "off" state to "on" */
	{
	  old_pm = tbar_drivers_off_pm;
	  new_pm = tbar_drivers_on_pm;
	}
      gtk_widget_ref (old_pm);		/* ref so doesn't get destroyed */
      gtk_container_remove (GTK_CONTAINER (tbar_drivers_togbtn), old_pm);
      gtk_container_add (GTK_CONTAINER (tbar_drivers_togbtn), new_pm);
      gtk_widget_unref (new_pm);	/* unref bogus reference */

      tbar_drivers_on = active;
    }

  gtk_signal_handler_block_by_func (GTK_OBJECT (tbar_drivers_togbtn),
    GTK_SIGNAL_FUNC (tbar_cb_drivers_toggle), NULL);
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (tbar_drivers_togbtn),
				active);
  gtk_signal_handler_unblock_by_func (GTK_OBJECT (tbar_drivers_togbtn),
    GTK_SIGNAL_FUNC (tbar_cb_drivers_toggle), NULL);
}

/** callback routine for toolbar driver toggle button */
static gboolean
tbar_cb_drivers_toggle (GtkWidget *btn)
{
  SwamiWavetbl *wavetbl;
  SwamiMidi *midi;
  SwamiUIMidiCtrl *midictrl;
  gboolean active;

  wavetbl = SWAMI_WAVETBL (swami_get_object_by_type (G_OBJECT (swami_object),
						     "SwamiWavetbl"));
  midi = SWAMI_MIDI (swami_get_object_by_type (G_OBJECT (swami_object),
					       "SwamiMidi"));
  if (!wavetbl || !midi) return (FALSE);

  midictrl = SWAMIUI_MIDICTRL (swamiui_lookup_object ("SwamiUIMidiCtrl"));

  g_object_get (G_OBJECT (wavetbl), "active", &active, NULL);

  if (active)
    {
      swami_wavetbl_close_driver (wavetbl);
      swami_midi_close_driver (midi);
    }
  else
    {
      swami_wavetbl_init_driver (wavetbl);
      swami_midi_init_driver (midi);
      swamiui_midictrl_midi_update_all (midictrl);
    }

  swamiui_tbar_update_drivers_togbtn ();

  return (FALSE);
}

/* callback for lower view toggle buttons */
static gboolean
tbar_cb_lower_view_button_toggled (GtkWidget *rad, int wintype)
{
  if (!GTK_TOGGLE_BUTTON (rad)->active || block_tbar_lowpane_btns)
    return (TRUE);

  swamiui_lowpane_set_view (wintype);

  /* update the view menu radio buttons */
  swamiui_menu_set_lowpane_radio_item (wintype);

  return (FALSE);
}

/* callback for piano mode toggle buttons */
static void
tbar_cb_piano_mode_button_toggled (GtkWidget *rad, int mode)
{
  SwamiUISpanWin *spanwin;

  if (!GTK_TOGGLE_BUTTON (rad)->active || block_tbar_piano_mode_btns)
    return;

  spanwin = SWAMIUI_SPANWIN (swamiui_lookup_object ("SwamiUISpanWin"));
  swamiui_spanwin_set_mode (SWAMIUI_SPANWIN (spanwin), mode);

  /* update the view menu radio buttons */
  swamiui_menu_set_piano_mode_radio_item (mode);
}

/* sets which lower pane toggle button is active on toolbar */
void
swamiui_tbar_set_lowpane_togbtn (int mode)
{
  GtkWidget *btn;

  switch (mode)
    {
    case SWAMIUI_LOWPANE_VIEW:
      btn = tbar_gen_view_togbtn;
      break;
    case SWAMIUI_LOWPANE_CTRL:
      btn = tbar_gen_ctrl_togbtn;
      break;
#ifdef CANVAS_SUPPORT
    case SWAMIUI_LOWPANE_GRAPH:
      btn = tbar_gen_graph_togbtn;
      break;
#endif
    case SWAMIUI_LOWPANE_SAMVIEW:
      btn = tbar_sam_view_togbtn;
      break;
    case SWAMIUI_LOWPANE_MODEDIT:
      btn = tbar_mod_edit_togbtn;
      break;
    default:
      return;
    }

  block_tbar_lowpane_btns = TRUE;
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (btn), TRUE);
  block_tbar_lowpane_btns = FALSE;
}

/* sets piano or velocity button to active on toolbar */
void
swamiui_tbar_set_piano_mode_togbtn (gint mode)
{
  GtkWidget *btn;

  switch (mode)
    {
    case SWAMIUI_SPANWIN_PIANO:
      btn = tbar_piano_togbtn;
      break;
    case SWAMIUI_SPANWIN_VELOCITY:
      btn = tbar_velocity_togbtn;
      break;
    }

  block_tbar_piano_mode_btns = TRUE;
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (btn), TRUE);
  block_tbar_piano_mode_btns = FALSE;
}
