/* Copyright (C) 2000-2004  Thomas Bopp, Thorsten Hampel, Ludger Merkens
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 * 
 * $Id: searching.pike,v 1.1.1.1 2006/03/27 12:40:12 exodusd Exp $
 */
inherit "/kernel/db_searching";

#include <database.h>
#include <macros.h>

string get_identifier() { return "searching"; }

#define DEBUG_QUERY 1

#ifdef DEBUG_QUERY
#define LOG_QUERY(a, b...) werror(a,b);
#else
#define LOG_QUERY(a, b...)
#endif

static int searches = 0;
static object service;

class SearchToken {
  string storage;
  string query;
  string andor;
  string key;
  mixed value;
  
  void create(string store, string k, mixed v, string ao) {
    storage = store;
    key = k;
    if ( objectp(v) ) { 
      value = "%" + v->get_object_id();
    }
    value = v;
    andor = ao;
  }
  mapping get() {
    return ([ 
      "storage": storage,
      "key": key,
      "value": value,
    ]);
  }
}

string compose_value_expression(SearchToken st)
{
  string query;
  if ( st->storage == STORE_ATTRIB ) 
    query = " (ob_ident = 'attrib' and ob_attr='"+st->key+"' and ob_data "+
      st->value[0]+" '"+st->value[1]+"') ";
  else if ( st->storage == "doc_data" ) 
    query = " (match(doc_data) against (\""+st->value[1]+"\") ";
  else
    query = " (ob_data "+ st->value[0]+" '" + st->value[1]+"') ";
  return query;
}

string compose_value_expressions(array tokens)
{
  string query = "";

  SearchToken last = tokens[0];
  string exp = compose_value_expression(last);

  for ( int i = 1; i < sizeof(tokens); i++ ) {
    SearchToken token = tokens[i];
    query += exp + " or ";
    exp = compose_value_expression(token);
  }
  query += exp;
  
  return query;
}

string get_table_name() { return "ob_data"; }

class Result {
  
  function f;
  mixed args;
  void create(function func, mixed a) { f = func; args = a; }

  void asyncResult(mixed id, mixed result) {
    f(args, result);
  }
}

static mapping results = ([ ]);

class Search {

  array eq(string|int value) {
    return ({ "=",
	      intp(value) ? (string)value : fDb()->quote(serialize(value))
    });
  }
  array gt(string|int value) {
    return ({ ">", intp(value) ? (string)value : fDb()->quote(value)});
  }
  array lt(string|int value) {
    return ({ "<", intp(value) ? (string)value : fDb()->quote(value)});
  }
  array like(string value) {
    return ({ "like", intp(value) ? (string)value : fDb()->quote(value)});
  }
  array lte(string|int value) {
    return ({ "<=", intp(value) ? (string)value : fDb()->quote(value)});
  }
  array gte(string|int value) {
    return ({ ">=", intp(value) ? (string)value : fDb()->quote(value)});
  }
  array btw(string|int low, string|int up) {
    return and(gte(low), lte(up));
  }
  array or(array a, array b) {
    return ({ "or", ({a, b}) });
  }
  array and(array a, array b) {
    return ({ "and", ({a, b}) });
  }

  array extends = ({ });  
  array limits = ({ });
  array fulltext = ({ });
  array(string) classes;
  int search_id;
  
  void create(int sid, array cl) {
    classes = cl;
    search_id = sid;
    service = get_module("ServiceManager");
  }

  void search_attribute(string key, string value) {
    extends += ({ search(STORE_ATTRIB, key, like(value), "or") });
  }

  void first_query(string store, string key, mixed value, mixed filter) {
    extends += ({ search(store, key, value, "or") });
  }

  void limit(string store, string key, mixed value) {
    limits += ({ search(store, key, value, "and") });
  }
  void extend(string store, string key, mixed value) {
    extends += ({ search(store, key, value, "or") });
  }
  
  void extend_ft(string pattern) {
    fulltext += ({ SearchToken("doc_ft", "doc_data", pattern, "or") });
  }
 
  SearchToken search(string store, string key, mixed value, string andor) {
    return SearchToken(store, key, value, andor);
  }
  mapping serialize_token(SearchToken s) {
    return s->get();
  }
  void execute() { run(); }
  void run() {
    if ( !service->is_service("search") ) {
      FATAL("Unable to locate Search Service - running locally !");
      string _query = "select distinct ob_id from ( ob_data ";
      
      if ( arrayp(classes) && sizeof(classes) > 0 ) {
	_query += "INNER JOIN ob_class on ob_class.ob_id=ob_data.ob_id and ("+
	  ( "ob_class = "+classes*" or ob_class=")+")";
      }
      _query += ") where";
      
      
      _query += compose_value_expressions(extends);
      LOG_QUERY("Query is: %s", _query);
      array res = query(_query);
      array sresult = ({ });
      foreach(res, mixed r) 
	if ( mappingp(r) )
	  sresult += ({ r->ob_id });
      
      handle_service(search_id, sresult);
    }
    else {
	object result = service->call_service_async("search", 
				    ({ this(), search_id, 
					   classes,
					   map(extends, serialize_token),
					   map(limits, serialize_token),
					   map(fulltext, serialize_token) }) );
	// process result;
	result->vars = ([ "id": search_id, ]);
	result->processFunc = handle_result;
	result->resultFunc = results[search_id]->asyncResult;
    }
  }
  object run_async() {
      if ( !service->is_service("search") )
	  steam_error("Unable to locate search service !");
      return
	  service->call_service_async("search", ({ this(), search_id, 
					     classes,
					     map(extends, serialize_token),
					     map(limits, serialize_token),
					     map(fulltext, serialize_token) }) );
  }
}

array handle_result(array res)
{
  int size = sizeof(res);
  array result = ({ });

  for (int i =0; i<size; i++) {
    object o = find_object((int)res[i]);
    if ( objectp(o) && o->status() >= 0 )
      result += ({ o });
  }
  return result;
}

void handle_service(int id, array result) 
{
  result = handle_result(result);
  Result r = results[id];
  werror("Calling Callback function for search result %O!\n", result);
  r->f(r->args, result);
}

Search searchQuery(function result_cb, mixed result_args, mixed ... params)
{
  results[++searches] = Result(result_cb, result_args);
  return Search(searches, @params);
}

object 
searchAsyncAttribute(string key, mixed val, mixed ... params) 
{
  Async.Return r = Async.Return();
  werror("searchAsyncAttribute(%s, %O, %O)\n", key, val, params);
  results[++searches] = Result(r->asyncResult, 0);
  Search s = Search(searches, @params);
  s->search_attribute(key, val);
  s->run();
  return r;
}
  
object 
searchAsync(array extends, array limits, array fulltext, void|int classBits)
{

  object aResult = get_module("ServiceManager")->call_service_async(
					 "search", 
					 ({ this(), searches, ({ }), 
					      extends, limits, fulltext }));
  aResult->processFunc = handle_result;
  return aResult;
}
