/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

#include "vdb-copy.vers.h"
#include "vdb-copy-includes.h"
#include "definitions.h"
#include "context.h"
#include "helper.h"
#include "coldefs.h"
#include "get_platform.h"
#include "progressbar.h"
#include "copy_meta.h"
#include "type_matcher.h"
#include "redactval.h"

#include <kapp/main.h>

/* this is here to detect the md5-mode of the src-table */
#include <kdb/kdb-priv.h>
#include <kdb/table.h>

#include <sysalloc.h>

/*
#if _DEBUGGING
#define _CRTDBG_MAP_ALLOC 1
#include <crtdbg.h>
#endif
*/

#include <stdlib.h>
#include <string.h>

static const char * table_usage[] = { "table-name", NULL };
static const char * rows_usage[] = { "set of rows to be copied(default = all)", NULL };
static const char * columns_usage[] = { "set of columns to be copied(default = all)", NULL };
static const char * schema_usage[] = { "schema-name", NULL };
static const char * without_accession_usage[] = { "without accession-test", NULL };
static const char * ignore_reject_usage[] = { "ignore SRA_FILTER_REJECT values", NULL };
static const char * ignore_redact_usage[] = { "ignore SRA_FILTER_REDACTED values", NULL };
static const char * kfg_path_usage[] = { "use this path to find the file vdb-copy.kfg", NULL };
static const char * show_matching_usage[] = { "show type-matching results", NULL };
static const char * show_progress_usage[] = { "show progress in percent while copying", NULL };
static const char * ignore_incomp_usage[] = { "ignore incompatible columns", NULL };
static const char * reindex_usage[] = { "reindex columns after copy", NULL };
static const char * show_redact_usage[] = { "show redaction-process", NULL };
static const char * excluded_columns_usage[] = { "exclude these columns from copy", NULL };
static const char * show_meta_usage[] = { "show metadata-copy-process", NULL };
static const char * md5mode_usage[] = { "MD5-mode def.: auto, '1'...forced ON, '0'...forced OFF)", NULL };
static const char * force_usage[] = { "forces an existing target to be overwritten", NULL };
static const char * unlock_usage[] = { "forces a locked target to be unlocked", NULL };

OptDef MyOptions[] =
{
    { OPTION_TABLE, ALIAS_TABLE, NULL, table_usage, 1, true, false },
    { OPTION_ROWS, ALIAS_ROWS, NULL, rows_usage, 1, true, false },
    { OPTION_COLUMNS, ALIAS_COLUMNS, NULL, columns_usage, 1, true, false },
    { OPTION_SCHEMA, ALIAS_SCHEMA, NULL, schema_usage, 5, true, false },
    { OPTION_WITHOUT_ACCESSION, ALIAS_WITHOUT_ACCESSION, NULL, without_accession_usage, 1, false, false },
    { OPTION_IGNORE_REJECT, ALIAS_IGNORE_REJECT, NULL, ignore_reject_usage, 1, false, false },
    { OPTION_IGNORE_REDACT, ALIAS_IGNORE_REDACT, NULL, ignore_redact_usage, 1, false, false },
#if ALLOW_EXTERNAL_CONFIG
    { OPTION_KFG_PATH, ALIAS_KFG_PATH, NULL, kfg_path_usage, 1, true, false },
#endif
    { OPTION_SHOW_MATCHING, ALIAS_SHOW_MATCHING, NULL, show_matching_usage, 1, false, false },
    { OPTION_SHOW_PROGRESS, ALIAS_SHOW_PROGRESS, NULL, show_progress_usage, 1, false, false },
    { OPTION_IGNORE_INCOMP, ALIAS_IGNORE_INCOMP, NULL, ignore_incomp_usage, 1, false, false },
    { OPTION_REINDEX, ALIAS_REINDEX, NULL, reindex_usage, 1, false, false },
    { OPTION_SHOW_REDACT, ALIAS_SHOW_REDACT, NULL, show_redact_usage, 1, false, false },
    { OPTION_EXCLUDED_COLUMNS, ALIAS_EXCLUDED_COLUMNS, NULL, excluded_columns_usage, 1, true, false },
    { OPTION_SHOW_META, ALIAS_SHOW_META, NULL, show_meta_usage, 1, false, false },
    { OPTION_MD5_MODE, ALIAS_MD5_MODE, NULL, md5mode_usage, 1, true, false },
    { OPTION_FORCE, ALIAS_FORCE, NULL, force_usage, 1, false, false },
    { OPTION_UNLOCK, ALIAS_UNLOCK, NULL, unlock_usage, 1, false, false }
};

static const char def_name[] = "vdb-copy";


static void summary ( const char * progname )
{
    OUTMSG ( ("\n"
        "Usage:\n"
        "  %s <src_path> <dst_path> [options]\n"
        "\n", progname) );
}

MINIUSAGE( def_name )

rc_t CC Usage ( const Args * args )
{
    const char * progname;
    const char * fullpath;
    rc_t rc;

    if ( args == NULL )
        rc = RC (rcApp, rcArgv, rcAccessing, rcSelf, rcNull);
    else
        rc = ArgsProgram (args, &fullpath, &progname);
    if ( rc )
        progname = fullpath = def_name;

    summary ( progname );

    OUTMSG (( "Options:\n" ));
    HelpOptionLine ( ALIAS_TABLE, OPTION_TABLE, "table", table_usage );
    HelpOptionLine ( ALIAS_ROWS, OPTION_ROWS, "rows", rows_usage );
    HelpOptionLine ( ALIAS_COLUMNS, OPTION_COLUMNS, "columns", columns_usage );
    HelpOptionLine ( ALIAS_SCHEMA, OPTION_SCHEMA, "schema", schema_usage );
    HelpOptionLine ( ALIAS_WITHOUT_ACCESSION, OPTION_WITHOUT_ACCESSION, NULL, without_accession_usage );
    HelpOptionLine ( ALIAS_IGNORE_REJECT, OPTION_IGNORE_REJECT, NULL, ignore_reject_usage );
    HelpOptionLine ( ALIAS_IGNORE_REDACT, OPTION_IGNORE_REDACT, NULL, ignore_redact_usage );
#if ALLOW_EXTERNAL_CONFIG
    HelpOptionLine ( ALIAS_KFG_PATH, OPTION_KFG_PATH, NULL, kfg_path_usage );
#endif
    HelpOptionLine ( ALIAS_SHOW_MATCHING, OPTION_SHOW_MATCHING, NULL, show_matching_usage );
    HelpOptionLine ( ALIAS_SHOW_PROGRESS, OPTION_SHOW_PROGRESS, NULL, show_progress_usage );
    HelpOptionLine ( ALIAS_IGNORE_INCOMP, OPTION_IGNORE_INCOMP, NULL, ignore_incomp_usage );
    HelpOptionLine ( ALIAS_REINDEX, OPTION_REINDEX, NULL, reindex_usage );
    HelpOptionLine ( ALIAS_SHOW_REDACT, OPTION_SHOW_REDACT, NULL, show_redact_usage );
    HelpOptionLine ( ALIAS_EXCLUDED_COLUMNS, OPTION_EXCLUDED_COLUMNS, NULL, excluded_columns_usage );
    HelpOptionLine ( ALIAS_SHOW_META, OPTION_SHOW_META, NULL, show_meta_usage );
    HelpOptionLine ( ALIAS_FORCE, OPTION_FORCE, NULL, force_usage );
    HelpOptionLine ( ALIAS_UNLOCK, OPTION_UNLOCK, NULL, unlock_usage );

    HelpOptionsStandard ();

    HelpVersion ( fullpath, KAppVersion() );

    return rc;
}


/* Version  EXTERN
 *  return 4-part version code: 0xMMmmrrrr, where
 *      MM = major release
 *      mm = minor release
 *    rrrr = bug-fix release
 */
ver_t CC KAppVersion ( void )
{
    return VDB_COPY_VERS;
}

/* ----------------------------------------------------------------------------------- */
typedef struct copy_ctx
{
    /* common to source and destination */
    KDirectory *directory;
    KConfig *config_mgr;
    VDBManager *vdb_manager;
    progressbar *progress;
    matcher *type_matcher;
    char * filter_col_name;
    char * redactable_columns;
    char * redactable_types;
    char * do_not_redact_columns;
    char * meta_ignore_nodes;
    redact_vals *rvals;
    void * redact_buffer;
    uint32_t redact_buffer_len;
    uint64_t row_id;

    /* for the source table*/
    VSchema *src_schema;
    const VDatabase *src_database;
    const VTable *src_table;
    const VCursor* src_cursor;
    col_defs *columns;
    char * src_schema_tabname;
    bool src_is_legacy;
    char * src_platform;

    /* for the destination table*/
    VSchema *dst_schema;
    VDatabase *dst_database;
    VTable *dst_table;
    VCursor* dst_cursor;
    char * dst_schema_tabname;

    /* legacy related parameters */
    char * legacy_schema_file;
    char * legacy_dont_copy;
    
    bool dont_remove_target;
} copy_ctx;
typedef copy_ctx* p_copy_ctx;


static rc_t vdb_copy_init_copy_ctx( p_copy_ctx cctx )
{
    rc_t rc;

    memset( cctx, 0, sizeof cctx[0] );
    cctx->dont_remove_target = false;
    rc = make_progressbar( &cctx->progress );
    DISP_RC( rc, "vdb_copy_init_copy_ctx:make_progressbar() failed" );
    rc = matcher_init( &cctx->type_matcher );
    DISP_RC( rc, "vdb_copy_init_copy_ctx:matcher_init() failed" );
    rc = redact_vals_init( &cctx->rvals );
    DISP_RC( rc, "vdb_copy_init_copy_ctx:redact_vals_init() failed" );
    return rc;
}


static void vdb_copy_destroy_cctx_str( char **s )
{
    if ( *s != NULL )
    {
        free( *s );
        *s = NULL;
    }
}


static void vdb_copy_release_dst( p_copy_ctx cctx )
{
    if ( cctx->dst_cursor != NULL )
    {
        rc_t rc = VCursorRelease( cctx->dst_cursor );
        DISP_RC( rc, "vdb_copy_destroy_copy_ctx:VCursorRelease(dst_cursor) failed" );
        cctx->dst_cursor = NULL;
    }
    if ( cctx->dst_table != NULL )
    {
        rc_t rc = VTableRelease( cctx->dst_table );
        DISP_RC( rc, "vdb_copy_destroy_copy_ctx:VTableRelease(dst) failed" );
        cctx->dst_table = NULL;
    }
    if ( cctx->dst_database != NULL )
    {
        rc_t rc = VDatabaseRelease( cctx->dst_database );
        DISP_RC( rc, "vdb_copy_destroy_copy_ctx:VDatabaseRelease(dst) failed" );
        cctx->dst_database = NULL;
    }
    if ( cctx->dst_schema != NULL )
    {
        rc_t rc = VSchemaRelease( cctx->dst_schema );
        DISP_RC( rc, "vdb_copy_destroy_copy_ctx:VSchemaRelease(src) failed" );
        cctx->dst_schema = NULL;
    }
}

static void vdb_copy_destroy_copy_ctx( p_copy_ctx cctx )
{
    /* ********** destination stuff in reverse order... */
    vdb_copy_destroy_cctx_str( &cctx->legacy_dont_copy );
    vdb_copy_destroy_cctx_str( &cctx->legacy_schema_file );
    vdb_copy_destroy_cctx_str( &cctx->dst_schema_tabname );

    vdb_copy_release_dst( cctx );

    /* ********** source stuff in reverse order... */
    vdb_copy_destroy_cctx_str( &cctx->src_platform );
    vdb_copy_destroy_cctx_str( &cctx->src_schema_tabname );
    if ( cctx->columns != NULL )
    {
        col_defs_destroy( cctx->columns );
        cctx->columns = NULL;
    }
    if ( cctx->src_cursor != NULL )
    {
        rc_t rc = VCursorRelease( cctx->src_cursor );
        DISP_RC( rc, "vdb_copy_destroy_copy_ctx:VCursorRelease(src) failed" );
        cctx->src_cursor = NULL;
    }
    if ( cctx->src_table != NULL )
    {
        rc_t rc = VTableRelease( cctx->src_table );
        DISP_RC( rc, "vdb_copy_destroy_copy_ctx:VTableRelease(src) failed" );
        cctx->src_table = NULL;
    }
    if ( cctx->src_database != NULL )
    {
        rc_t rc = VDatabaseRelease( cctx->src_database );
        DISP_RC( rc, "vdb_copy_destroy_copy_ctx:VDatabaseRelease(src) failed" );
        cctx->src_database = NULL;
    }
    if ( cctx->src_schema != NULL )
    {
        rc_t rc = VSchemaRelease( cctx->src_schema );
        DISP_RC( rc, "vdb_copy_destroy_copy_ctx:VSchemaRelease(src) failed" );
        cctx->src_schema = NULL;
    }

    /* ********** global stuff in reverse order at last... */
    vdb_copy_destroy_cctx_str( &cctx->filter_col_name );
    vdb_copy_destroy_cctx_str( &cctx->redactable_columns );
    vdb_copy_destroy_cctx_str( &cctx->redactable_types );
    vdb_copy_destroy_cctx_str( &cctx->do_not_redact_columns );
    vdb_copy_destroy_cctx_str( &cctx->meta_ignore_nodes );
    
    if ( cctx->progress != NULL )
    {
        rc_t rc = destroy_progressbar( cctx->progress );
        DISP_RC( rc, "vdb_copy_destroy_copy_ctx:destroy_progressbar() failed" );
        cctx->progress = NULL;
    }
   if ( cctx->vdb_manager != NULL )
    {
        rc_t rc = VDBManagerRelease( cctx->vdb_manager );
        DISP_RC( rc, "vdb_copy_destroy_copy_ctx:VDBManagerRelease() failed" );
        cctx->vdb_manager = NULL;
    }
    if ( cctx->config_mgr != NULL )
    {
        rc_t rc = KConfigRelease( cctx->config_mgr );
        DISP_RC( rc, "vdb_copy_destroy_copy_ctx:KConfigRelease() failed" );
        cctx->config_mgr = NULL;
    }
    if ( cctx->directory != NULL )
    {
        rc_t rc = KDirectoryRelease( cctx->directory );
        DISP_RC( rc, "vdb_copy_destroy_copy_ctx:KDirectoryRelease() failed" );
        cctx->directory = NULL;
    }
    if ( cctx->type_matcher != NULL )
    {
        rc_t rc = matcher_destroy( cctx->type_matcher );
        DISP_RC( rc, "vdb_copy_destroy_copy_ctx:matcher_destroy() failed" );
        cctx->type_matcher = NULL;
    }
    if ( cctx->rvals != NULL )
    {
        rc_t rc = redact_vals_destroy( cctx->rvals );
        DISP_RC( rc, "vdb_copy_destroy_copy_ctx:redact_vals_destroy() failed" );
        cctx->rvals = NULL;
    }
    if ( cctx-> redact_buffer != NULL )
    {
        free( cctx->redact_buffer );
        cctx->redact_buffer = NULL;
        cctx->redact_buffer_len = 0;
    }
}


static rc_t vdb_copy_redact_cell( p_copy_ctx cctx, const p_col_def col,
                                  const bool show_redact )
{
    rc_t rc;
    const void * src_buffer;
    uint32_t offset_in_bits;
    uint32_t n_elements;
    uint32_t elem_bits;

    /* we read the original cell-data to detect how big the data is before redacting */
    rc = VCursorCellData( cctx->src_cursor, col->src_idx, &elem_bits,
                          &src_buffer, &offset_in_bits, &n_elements );
    if ( rc != 0 )
    {
        PLOGERR( klogInt,
                 (klogInt,
                 rc,
                 "VCursorCellData( col:$(col_name) at row #$(row_nr) ) failed",
                 "col_name=%s,row_nr=%lu",
                  col->name, cctx->row_id ));
    }

    DISP_RC( rc, "vdb_copy_redact_cell:VCursorCellData(src) failed" );
    if ( rc == 0 )
    {
        size_t bufsize = ( ( elem_bits * n_elements ) + 8 ) >> 3;
        if ( cctx->redact_buffer_len < bufsize || cctx->redact_buffer == NULL )
        {
            /* allocate or re-allocate the buffer */
            if ( cctx->redact_buffer_len == 0 )
                cctx->redact_buffer = malloc( bufsize );
            else
                cctx->redact_buffer = realloc( cctx->redact_buffer, bufsize );

            /* exit */
            if ( cctx->redact_buffer == NULL )
                return RC( rcVDB, rcNoTarg, rcConstructing, rcMemory, rcExhausted );
            cctx->redact_buffer_len = bufsize;
        }

        if ( col->r_val != NULL )
        {
            if ( show_redact )
            {
                char * c = ( char * )col->r_val->value;
                OUTMSG (( "redacting #%lu %s -> 0x%.02x\n", cctx->row_id, col->dst_cast, *c ));
            }
            redact_val_fill_buffer( col->r_val, cctx->redact_buffer, bufsize );
        }
        else
        {
            if ( show_redact )
                OUTMSG (( "redacting #%lu %s -> 0\n", cctx->row_id, col->dst_cast ));
            memset( cctx->redact_buffer, 0, bufsize );
        }

        rc = VCursorWrite( cctx->dst_cursor, col->dst_idx, elem_bits,
                           cctx->redact_buffer, 0, n_elements );
        if ( rc != 0 )
        {
            PLOGERR( klogInt,
                     (klogInt,
                     rc,
                     "VCursorWrite( col:$(col_name) at row #$(row_nr) ) failed",
                     "col_name=%s,row_nr=%lu",
                      col->name, cctx->row_id ));
        }
    }
    return rc;
}


static rc_t vdb_copy_cell( p_copy_ctx cctx, const p_col_def col ) 
{
    rc_t rc;

    const void *buffer;
    uint32_t offset_in_bits;
    uint32_t number_of_elements;
    uint32_t elem_bits;

    /* introduce a artificial error at row #100 to test
       that the tool removes the output on error
    if ( cctx->row_id == 100 ) return -1;
    */

    /*
    OUTMSG (( " - copy cell %s ( src_idx=%u / dst_idx=%u )\n", 
              col->name, col->src_idx, col->dst_idx ));
    */
    rc = VCursorCellData( cctx->src_cursor, col->src_idx, &elem_bits,
                          &buffer, &offset_in_bits, &number_of_elements );
    if ( rc != 0 )
    {
        PLOGERR( klogInt,
                 (klogInt,
                 rc,
                 "VCursorCellData( col:$(col_name) at row #$(row_nr) ) failed",
                 "col_name=%s,row_nr=%lu",
                  col->name, cctx->row_id ));
    }
    if ( rc != 0 ) return rc;

    /*
    OUTMSG (( "bit-offset = %u / elements = %u / element-bits = %u\n", 
       offset_in_bits, number_of_elements, elem_bits ));
    */
    rc = VCursorWrite( cctx->dst_cursor, col->dst_idx, elem_bits,
                       buffer, offset_in_bits, number_of_elements );
    if ( rc != 0 )
    {
        PLOGERR( klogInt,
                 (klogInt,
                 rc,
                 "VCursorWrite( col:$(col_name) at row #$(row_nr) ) failed",
                 "col_name=%s,row_nr=%lu",
                  col->name, cctx->row_id ));
    }
    return rc;
}


/* ----------------------------------------------------------------------------------- */
static rc_t vdb_copy_row( const p_context ctx, p_copy_ctx cctx,
                          bool redact ) 
{
    uint32_t len, idx = 0;
    rc_t rc = VCursorOpenRow( cctx->dst_cursor );
    if ( rc != 0 )
    {
        PLOGERR( klogInt,
                 (klogInt,
                 rc,
                 "VCursorOpenRow(dst) row #$(row_nr) failed",
                 "row_nr=%lu",
                  cctx->row_id ));
        return rc;
    }

    len = VectorLength( &(cctx->columns->cols) );
    /* loop through the columns and copy them if they have to be copied */
    while ( idx < len && rc == 0 )
    {
        p_col_def col = (p_col_def) VectorGet ( &(cctx->columns->cols), idx++ );
        if ( col != NULL )
        {
            if ( col->to_copy )
            {
                if ( redact && col->redactable )
                    rc = vdb_copy_redact_cell( cctx, col, ctx->show_redact );
                else
                    rc = vdb_copy_cell( cctx, col );
            }
        }
    }
    if ( rc == 0 )
    {
        rc = VCursorCommitRow( cctx->dst_cursor );
        if ( rc != 0 )
        {
            PLOGERR( klogInt,
                     (klogInt,
                     rc,
                     "VCursorCommitRow(dst) row #$(row_nr) failed",
                     "row_nr=%lu",
                      cctx->row_id ));
        }

        rc = VCursorCloseRow( cctx->dst_cursor );
        if ( rc != 0 )
        {
            PLOGERR( klogInt,
                     (klogInt,
                     rc,
                     "VCursorCloseRow(dst) row #$(row_nr) failed",
                     "row_nr=%lu",
                      cctx->row_id ));
        }
    }
    return rc;
}


static rc_t vdb_copy_read_row_flags( const p_context ctx, const VCursor *cursor,
        const uint32_t src_idx, bool *pass, bool *redact )
{
    uint64_t filter;
    rc_t rc = helper_read_vdb_int_row_open( cursor, src_idx, &filter );
    if ( rc != 0 ) return rc;

    switch( filter )
    {
    case SRA_READ_FILTER_REJECT   : 
        if ( ctx->ignore_reject == false ) *pass = false;
        break;

    case SRA_READ_FILTER_REDACTED : 
        if ( ctx->ignore_redact == false ) *redact = true;
        break;
    }
    return rc;
}


static uint8_t vdb_copy_calc_fract_digits( const num_gen_iter *iter )
{
    uint8_t res = 0;
    uint64_t count;
    if ( num_gen_iterator_count( iter, &count ) == 0 )
    {
        if ( count > 10000 )
        {
            if ( count > 100000 )
                res = 2;
            else
                res = 1;
        }
    }
    return res;
}


static rc_t vdb_copy_row_loop( const p_context ctx, p_copy_ctx cctx ) 
{
    rc_t rc;
    const num_gen_iter *iter;
    uint64_t count;
    uint32_t percent;
    uint8_t fract_digits;
    p_col_def filter_col_def = NULL;

    if ( cctx->columns->filter_idx != -1 )
        filter_col_def = col_defs_get( cctx->columns, cctx->columns->filter_idx );

    rc = num_gen_iterator_make( ctx->row_generator, &iter );
    if ( rc != 0 ) return rc;

    col_defs_find_redact_vals( cctx->columns, cctx->rvals );

    fract_digits = vdb_copy_calc_fract_digits( iter );
    count = 0;
    while ( ( num_gen_iterator_next( iter, &(cctx->row_id) ) == 0 )&&
            ( rc == 0 ) )
    {
        rc = Quitting();    /* to be able to cancel the loop by signal */
        if ( rc != 0 ) break;

        rc = VCursorSetRowId( cctx->src_cursor, cctx->row_id );
        if ( rc != 0 )
        {
            PLOGERR( klogInt,
                     (klogInt,
                     rc,
                     "VCursorSetRowId(src) row #$(row_nr) failed",
                     "row_nr=%lu",
                      cctx->row_id ));
        }
        if ( rc == 0 )
        {
            rc = VCursorOpenRow( cctx->src_cursor );
            if ( rc != 0 )
            {
                PLOGERR( klogInt,
                         (klogInt,
                         rc,
                         "VCursorOpenRow(src) row #$(row_nr) failed",
                         "row_nr=%lu",
                          cctx->row_id ));
            }
            else
            {
                bool pass = true;
                bool redact = false;
                rc_t rc1;

                if ( filter_col_def != NULL )
                    vdb_copy_read_row_flags( ctx, cctx->src_cursor,
                                filter_col_def->src_idx, &pass, &redact );
                if ( pass )
                {
                    rc = vdb_copy_row( ctx, cctx, redact );
                    if ( rc == 0 ) count++;
                }

                rc1 = VCursorCloseRow( cctx->src_cursor );
                if ( rc1 != 0 )
                {
                    PLOGERR( klogInt,
                             (klogInt,
                             rc1,
                             "VCursorCloseRow(src) row #$(row_nr) failed",
                             "row_nr=%lu",
                              cctx->row_id ));
                }
            }
        }
        if ( ctx->show_progress )
            if ( num_gen_iterator_percent( iter, fract_digits, &percent ) == 0 )
                update_progressbar( cctx->progress, fract_digits, percent );

    }
    OUTMSG (( "\n" ));
    OUTMSG (( "%lu rows copied\n", count ));
    if ( rc == 0 )
    {
        rc = VCursorCommit( cctx->dst_cursor );
        DISP_RC( rc, "vdb_copy_row_loop:VCursorCommit( dst ) failed" );
    }
    num_gen_iterator_destroy( iter );
        
    return rc;
}


static rc_t vdb_copy_make_dst_table( const p_context ctx, p_copy_ctx cctx,
                                     VTable **dst_table )
{
    rc_t rc = 0;
    KCreateMode cmode = kcmParents;

    if ( ctx->force_kcmInit )
        cmode |= kcmInit;
    else
        cmode |= kcmCreate;

    switch( ctx->md5_mode )
    {
    case MD5_MODE_ON :
    case MD5_MODE_AUTO : cmode |= kcmMD5; break;
    }

    /* different ways to make the schema for the dest-table */
    if ( cctx->src_is_legacy )
    {
        /* load it from a file */
        OUTMSG (( "we are using '%s'\n", cctx->legacy_schema_file ));
        if ( cctx->dst_schema == NULL )
        {
            rc = VDBManagerMakeSRASchema( cctx->vdb_manager, &cctx->dst_schema );
            DISP_RC( rc, "vdb_copy_make_dst_table:VDBManagerMakeSRASchema(dst) failed" );
        }
        rc = VSchemaParseFile ( cctx->dst_schema, cctx->legacy_schema_file );
        DISP_RC( rc, "vdb_copy_make_dst_table:VSchemaParseFile() failed" );
    }
    else
    {
        /* in case of a non-legacy-table, do nothing,
           keep using the src-schema-object */
        cctx->dst_schema = cctx->src_schema;
        VSchemaAddRef( cctx->src_schema );
    }
    if ( rc != 0 ) return rc;

    rc = VDBManagerCreateTable( cctx->vdb_manager, dst_table,
                         cctx->dst_schema, cctx->dst_schema_tabname,
                         cmode, ctx->dst_path );
    DISP_RC( rc, "vdb_copy_make_dst_table:VDBManagerCreateTable() failed" );
    return rc;
}


static rc_t vdb_copy_open_dest_table( const p_context ctx, p_copy_ctx cctx )
{
    rc_t rc;

    /* copy the metadata */
    rc = copy_table_meta( cctx->src_table, cctx->dst_table, 
                          cctx->meta_ignore_nodes, 
                          ctx->show_meta, cctx->src_is_legacy );
    if ( rc != 0 ) return rc;

    /* mark all columns which are to be found writable as to_copy */
    rc = col_defs_mark_writable_columns( cctx->columns, cctx->dst_table );
    DISP_RC( rc, "vdb_copy_open_dest_table:col_defs_mark_writable_columns() failed" );
    if ( rc != 0 ) return rc;

    /* make a writable cursor */
    rc = VTableCreateCursorWrite( cctx->dst_table, &(cctx->dst_cursor), kcmInsert );
    DISP_RC( rc, "vdb_copy_open_dest_table:VTableCreateCursorWrite(dst) failed" );
    if ( rc != 0 ) return rc;

    /* add all marked ( as to copy ) columns to the writable cursor */
    rc = col_defs_add_to_wr_cursor( cctx->columns, cctx->dst_cursor );
    DISP_RC( rc, "vdb_copy_open_dest_table:col_defs_add_to_wr_cursor(dst) failed" );
    if ( rc != 0 ) return rc;

    /* opens the dst cursor */
    rc = VCursorOpen( cctx->dst_cursor );
    DISP_RC( rc, "vdb_copy_open_dest_table:VCursorOpen(dst) failed" );

    return rc;
}


/* detect the row-range of the source-table
   check if the requested row-range is within this range
   otherwise correct the requested row-range
*/
static rc_t vdb_copy_check_range( p_context ctx, p_copy_ctx cctx )
{
    int64_t  first;
    uint64_t count;
    rc_t rc = VCursorIdRange( cctx->src_cursor, 0, &first, &count );
    DISP_RC( rc, "vdb_copy_check_range:VCursorIdRange() failed" );
    if ( rc != 0 ) return rc;

    rc = context_range_check( ctx, first, count );
    DISP_RC( rc, "vdb_copy_check_range:context_range_check() failed" );
    return rc;
}


static rc_t vdb_copy_prepare_legacy_tab( p_context ctx, p_copy_ctx cctx )
{
    rc_t rc;
    if ( cctx->src_database != NULL )
        rc = get_db_platform( ctx->src_path, ctx->table, &(cctx->src_platform), '_' );
    else
        rc = get_table_platform( ctx->src_path, &(cctx->src_platform), '_' );
    DISP_RC( rc, "vdb_copy_prepare_legacy_tab:get_db/table_platform() failed" );
    if ( rc != 0 ) return rc;

    OUTMSG (( "legacy-platform: %s\n", cctx->src_platform ));

    rc = helper_get_legacy_write_schema_from_config( cctx->config_mgr,
                cctx->src_platform,
                cctx->src_schema_tabname,
                &(cctx->legacy_schema_file),
                &(cctx->dst_schema_tabname),
                &(cctx->legacy_dont_copy) );
    DISP_RC( rc, "vdb_copy_prepare_legacy_tab:helper_get_legacy_write_schema_from_config() failed" );
    if ( rc != 0 ) return rc;

    if ( ctx->show_matching )
    {
        OUTMSG (( "-file: %s\n", cctx->legacy_schema_file ));
        OUTMSG (( "-tab : %s\n", cctx->dst_schema_tabname ));
        OUTMSG (( "-dont: %s\n", cctx->legacy_dont_copy ));
    }
    return rc;
}


static rc_t vdb_copy_find_out_what_columns_to_use( p_context ctx, p_copy_ctx cctx )
{
    bool cols_requested = ( ( ctx->columns != NULL ) &&
                            ( nlt_strcmp( ctx->columns, "*" ) != 0 ) );

    /* no matter if specific columns are requested, we first discover all of them
       to later mark the columns which are requested */
    rc_t rc = col_defs_extract_from_table( cctx->columns, cctx->src_table );
    DISP_RC( rc, "vdb_copy_find_out_what_columns_to_use:col_defs_extract_from_table() failed" );
    if ( rc != 0 ) return rc;

    if ( cols_requested )
    /* the user requested a specific subset of columns... */
        rc = col_defs_mark_requested_columns( cctx->columns, ctx->columns );
    else
    /* no specific subset of columns is requested --> copy what we can... */
        rc = col_defs_mark_requested_columns( cctx->columns, NULL );
    DISP_RC( rc, "vdb_copy_find_out_what_columns_to_use:col_defs_mark_requested_columns() failed" );
    if ( rc != 0 ) return rc;

    if ( ctx->excluded_columns != NULL )
    {
        rc = col_defs_exclude_these_columns( cctx->columns, ctx->excluded_columns );
        DISP_RC( rc, "vdb_copy_find_out_what_columns_to_use:col_defs_unmark_writable_columns() failed" );
    }

    return rc;
}


/* find the filter-column and redactable columns */
static void vdb_copy_find_filter_and_redact_columns( p_context ctx, p_copy_ctx cctx )
{
    /* it is ok to not find a filter-column: no error in this case */
    col_defs_detect_filter_col( cctx->columns, cctx->filter_col_name );

/*
    col_defs_detect_redactable_cols_by_name( cctx->columns,
             cctx->redactable_columns );
*/

    /* it is ok to not find redactable types: no error in this case */
    col_defs_detect_redactable_cols_by_type( cctx->columns,
             cctx->src_schema, cctx->type_matcher, cctx->redactable_types );

    /* it is ok to not find columns excluded from redacting: no error in this case */
    col_defs_unmark_do_not_redact_columns( cctx->columns,
                    cctx->do_not_redact_columns );
}


/* detect the columns of the source-table to use
   ( either from the commandline or from the source-table )
   - if the user did not specify the columns, extract them from the table
   - if the used did specify the colums, take these...
   - add the columns to the source-cursor
*/
static rc_t vdb_copy_prepare_columns( p_context ctx, p_copy_ctx cctx )
{
    /* detects the name of the schema-table used by the source-table */
    rc_t rc = helper_get_schema_tab_name( cctx->src_table, &cctx->src_schema_tabname );
    DISP_RC( rc, "vdb_copy_prepare_columns:helper_get_schema_tab_name() failed" );
    if ( rc != 0 ) return rc;

    rc = helper_is_tablename_legacy( cctx->vdb_manager, 
                     cctx->src_schema_tabname, &( cctx->src_is_legacy ) );
    if ( rc != 0 ) return rc;
    if ( cctx->src_is_legacy )
    {
        OUTMSG (( "used legacy schema: %s\n", cctx->src_schema_tabname ));
        rc = vdb_copy_prepare_legacy_tab( ctx, cctx );
        if ( rc != 0 ) return rc;
    }
    else
    {
        OUTMSG (( "used schema: %s\n", cctx->src_schema_tabname ));
        cctx->dst_schema_tabname = string_dup_measure ( cctx->src_schema_tabname, NULL );
    }
    return rc;
}


/* we have the src-columns collected (incl. there types):
   no columns have been added to a cursor (rd and wr),
   now we can find the columns which have to be copied
   including the right type-casts for every one of these columns */
static rc_t vdb_copy_match_columns( p_context ctx, p_copy_ctx cctx )
{
    rc_t rc;

    matcher_input mi;

    mi.manager = cctx->vdb_manager;
    mi.add_schemas = ctx->src_schema_list;
    mi.cfg = cctx->config_mgr;
    mi.dir = cctx->directory;
    rc = col_defs_as_string( cctx->columns, (char**)&mi.columns );
    if ( rc != 0 ) return rc;
    mi.src_path = ctx->src_path;
    mi.dst_path = ctx->dst_path;
    mi.legacy_schema = cctx->legacy_schema_file;
    mi.dst_tabname = cctx->dst_schema_tabname;
    mi.excluded_columns = cctx->legacy_dont_copy;
    mi.force_kcmInit = ctx->force_kcmInit;
    mi.force_unlock = ctx->force_unlock;

    rc = matcher_execute( cctx->type_matcher, &mi );
    if ( rc != 0 )
    {
        if ( GetRCState( rc ) == rcExists )
        {
            OUTMSG(( "vdb-copy cannot create the target-table, because it already exists!\n" ));
            OUTMSG(( "try to use the force-option (-f)\n" ));
            cctx->dont_remove_target = true;
        }
        else
            DISP_RC( rc, "vdb_copy_match_columns:matcher_execute() failed" );
    }
    if ( rc == 0 )
    {
        if ( ctx->show_matching )
            matcher_report( cctx->type_matcher, true );
        rc = col_defs_apply_casts( cctx->columns, cctx->type_matcher );
        DISP_RC( rc, "vdb_copy_match_columns:col_defs_apply_casts() failed" );
    }
    free( ( void * )mi.columns );
    return rc;
}


/* this is the common entry-point for the copy-operation in both cases
   "given path is database" OR "given path is table" */
static rc_t vdb_copy_open_source_table( p_context ctx, p_copy_ctx cctx ) 
{
    rc_t rc = VTableCreateCursorRead( cctx->src_table, &(cctx->src_cursor) );
    DISP_RC( rc, "vdb_copy_open_source_table:VTableCreateCursorRead() failed" );
    if ( rc != 0 ) return rc;

    /* everything what has to be done before adding the columns
       to the source-cursor and opening the source-cursor */
    rc = vdb_copy_prepare_columns( ctx, cctx );
    if ( rc != 0 ) return rc;

    /* type-match between src <---> dst columns, find best typecast */
    rc = vdb_copy_match_columns( ctx, cctx );
    if ( rc != 0 ) return rc;

    /* in case of legacy table make new schema - parse matched schema
       in case on non-legacy use the src-schema as dst-schema */
    rc = vdb_copy_make_dst_table( ctx, cctx, &(cctx->dst_table) );
    if ( rc != 0 ) return rc;

    /* the prepared and eventually shortened src-column-list is used */
    rc = col_defs_add_to_rd_cursor( cctx->columns, cctx->src_cursor );
    if ( rc != 0 ) return rc;

    rc = VCursorOpen( cctx->src_cursor );
    DISP_RC( rc, "vdb_copy_open_source_table:VCursorOpen() failed" );
    if ( rc != 0 ) return rc;

    /* range check requires an open cursor */
    rc = vdb_copy_check_range( ctx, cctx );
    return rc;
}


static bool vdb_copy_detect_src_md5( p_copy_ctx cctx )
{
    rc_t rc;
    const struct KTable * ktab;
    bool res = false;

    rc = VTableOpenKTableRead ( cctx->src_table, &ktab );
    if ( rc == 0 )
    {
        const struct KDirectory *dir;
        rc = KTableOpenDirectoryRead ( ktab, &dir );
        if ( rc == 0 )
        {
            /* ask for pathtype of "md5" */
            uint32_t pt = KDirectoryPathType ( dir, "md5" );
            if ( ( pt == kptFile )||( pt == ( kptFile | kptAlias ) ) )
                res = true;
            KDirectoryRelease( dir );
        }
        KTableRelease( ktab );
    }
    return res;
}

static rc_t vdb_copy_table( const p_context ctx, p_copy_ctx cctx )
{
    rc_t rc;

    if ( ctx->md5_mode == MD5_MODE_AUTO )
    {
        bool src_md5_mode = vdb_copy_detect_src_md5( cctx );
        ctx->md5_mode = ( src_md5_mode ? MD5_MODE_ON : MD5_MODE_OFF );
    }

    rc = col_defs_init( &(cctx->columns) );
    DISP_RC( rc, "vdb_copy_table:col_defs_init() failed" );
    if ( rc != 0 ) return rc;

    rc = vdb_copy_find_out_what_columns_to_use( ctx, cctx );
    if ( rc != 0 ) return rc;

    rc = vdb_copy_open_source_table( ctx, cctx );
    if ( rc != 0 ) return rc;

    rc = vdb_copy_open_dest_table( ctx, cctx );
    if ( rc != 0 ) return rc;

    /* this function does not fail, because it is ok to not find
       filter-column, redactable types and excluded columns */
    vdb_copy_find_filter_and_redact_columns( ctx, cctx );

    rc = vdb_copy_row_loop( ctx, cctx );
    if ( rc != 0 ) return rc;

    if ( ctx->reindex )
    {
        /* releasing the cursor is necessary for reindex */
        if ( cctx->dst_cursor != NULL )
        {
            VCursorRelease( cctx->dst_cursor );
            cctx->dst_cursor = NULL;
        }
        rc = VTableReindex( cctx->dst_table );
        DISP_RC( rc, "vdb_copy_table:VTableReindex() failed" );
    }
    return rc;
}


/* remove the target-table if something went wrong */
static rc_t vdb_copy_remove_table( const p_context ctx, p_copy_ctx cctx )
{
    rc_t rc;

    OUTMSG (( "removing '%s'\n", ctx->dst_path ));
    vdb_copy_release_dst( cctx );
    rc = KDirectoryRemove ( cctx->directory, true, ctx->dst_path );
    DISP_RC( rc, "vdb_copy_remove_table:KDirectoryRemove() failed" );
    return rc;
}

static rc_t vdb_copy_database_table( const p_context ctx, p_copy_ctx cctx,
                                     const char * tab_name )
{
    rc_t rc = VDatabaseOpenTableRead( cctx->src_database, &(cctx->src_table), tab_name );
    DISP_RC( rc, "vdb_copy_database_table:VDatabaseOpenTableRead(src) failed" );
    if ( rc != 0 ) return rc;

    rc = vdb_copy_table( ctx, cctx );
    if ( rc != 0 && !cctx->dont_remove_target )
        vdb_copy_remove_table( ctx, cctx );
    return rc;
}

/* this function is used if the given path is a database */
static rc_t vdb_copy_database( const p_context ctx, p_copy_ctx cctx )
{
    char *typespec = NULL;
    rc_t rc = VDBManagerOpenDBRead ( cctx->vdb_manager,
                &(cctx->src_database), cctx->src_schema, ctx->src_path );
    DISP_RC( rc, "vdb_copy_database:VDBManagerOpenDBRead(src) failed" );
    if ( rc != 0 ) return rc;

    /* TBD: detect the database-typespec of the source... */
    /* !!! */
    
    rc = VDBManagerCreateDB ( cctx->vdb_manager, 
                &(cctx->dst_database), cctx->dst_schema,
                typespec, kcmInit | kcmParents, ctx->dst_path );
    DISP_RC( rc, "vdb_copy_database:VDBManagerCreateDB( dst ) failed" );
    if ( rc != 0 ) return rc;
    
    if ( ctx->table == NULL )
    {
        KNamelist *table_names;
        /* the user did not specify a particular table: copy all of them */
        rc = VDatabaseListTbl( cctx->src_database, &table_names );
        DISP_RC( rc, "vdb_copy_database:VDatabaseListTbl() failed" );
        if ( rc == 0 )
        {
            uint32_t idx, count;
            rc = KNamelistCount( table_names, &count );
            DISP_RC( rc, "vdb_copy_database:KNamelistCount() failed" );
            if ( rc == 0 )
                for ( idx = 0; idx < count && rc == 0; ++idx )
                {
                    const char *table_name;
                    rc = KNamelistGet( table_names, idx, &table_name );
                    DISP_RC( rc, "vdb_copy_database:KNamelistGet() failed" );
                    if ( rc == 0 )
                        rc = vdb_copy_database_table( ctx, cctx, table_name );
                }
            KNamelistRelease( table_names );
        }
    }
    else
        rc = vdb_copy_database_table( ctx, cctx, ctx->table );
    return rc;
}


static void vdb_copy_read_redact_value(  p_copy_ctx cctx, const char * type_name )
{
    rc_t rc;
    String key_prefix, key_type, key_postfix;
    const String * p1;

    StringInitCString( &key_prefix, REDACTVALUE_PREFIX );
    StringInitCString( &key_type, type_name );
    StringInitCString( &key_postfix, REDACTVALUE_VALUE_POSTFIX );
    rc = StringConcat ( &p1, &key_prefix, &key_type );
    if ( rc == 0 )
    {
        const String * key;
        rc = StringConcat ( &key, p1, &key_postfix );
        if ( rc == 0 )
        {
            char * value_str;
            rc = helper_read_cfg_str( cctx->config_mgr, key->addr, &value_str );
            StringWhack ( key );
            if ( rc == 0 )
            {
                StringInitCString( &key_postfix, REDACTVALUE_LEN_POSTFIX );
                rc = StringConcat ( &key, p1, &key_postfix );
                if ( rc == 0 )
                {
                    uint32_t idx, l, len = 1;
                    char * endp, * len_str, * real_type;

                    rc = helper_read_cfg_str( cctx->config_mgr, key->addr, &len_str );
                    StringWhack ( key );
                    if ( rc == 0 )
                    {
                        len = strtol( len_str, &endp, 0 );
                        free( len_str );
                    }

                    real_type = string_dup_measure ( type_name, NULL );
                    l = string_size ( real_type );
                        
                    /* transform the typename back into '_' ---> ':' real typename */
                    for ( idx = 0; idx < l; ++idx )
                        if ( real_type[idx] == '_' )
                            real_type[idx] = ':';

                    /* finally insert the redact-value-pair into the list */
                    redact_vals_add( cctx->rvals, real_type, len, value_str );
                    free( real_type );
                }
                free( value_str );
            }
        }
        StringWhack ( p1 );
    }
}


static void vdb_copy_read_redact_values(  p_copy_ctx cctx )
{
    rc_t rc;
    char * type_list_str;
    const KNamelist *type_list;

    /* first we read the list of redactable types */
    helper_read_cfg_str( cctx->config_mgr, 
            REDACTABLE_LIST_KEY, &type_list_str );
    if ( type_list_str == NULL ) return;
    rc = nlt_make_namelist_from_string( &type_list, type_list_str );
    if ( rc == 0 && type_list != NULL )
    {
        uint32_t idx, count;
        rc = KNamelistCount( type_list, &count );
        if ( rc == 0 && count > 0 )
            for ( idx = 0; idx < count; ++idx )
            {
                const char *type_name;
                rc = KNamelistGet( type_list, idx, &type_name );
                if ( rc == 0 )
                    vdb_copy_read_redact_value(  cctx, type_name );
            }
        KNamelistRelease( type_list );
    }
    free( type_list_str );
}

static void vdb_copy_read_config_values(  p_copy_ctx cctx )
{
    /* key's and default-values in definitions.h */

    /* look for the name of the filter-column */
    helper_read_cfg_str( cctx->config_mgr, 
            READ_FILTER_COL_NAME_KEY, &(cctx->filter_col_name) );
    if ( cctx->filter_col_name == NULL )
        cctx->filter_col_name = string_dup_measure ( READ_FILTER_COL_NAME, NULL );

    /* look for the comma-separated list of meta-nodes to be ignored */
    helper_read_cfg_str( cctx->config_mgr, 
            META_IGNORE_NODES_KEY, &(cctx->meta_ignore_nodes) );
    if ( cctx->meta_ignore_nodes == NULL )
        cctx->meta_ignore_nodes = string_dup_measure ( META_IGNROE_NODES_DFLT, NULL );

    /* look for the comma-separated list of redactable columns */
    helper_read_cfg_str( cctx->config_mgr, 
            REDACTABLE_COLUMNS_KEY, &(cctx->redactable_columns) );
/*
    if ( cctx->redactable_columns == NULL )
        cctx->redactable_columns = string_dup_measure ( REDACTABLE_COLUMNS, NULL );
*/

    /* look for the comma-separated list of redactable types */
    helper_read_cfg_str( cctx->config_mgr, 
            REDACTABLE_TYPES_KEY, &(cctx->redactable_types) );
/*
    if ( cctx->redactable_types == NULL )
        cctx->redactable_types = string_dup_measure ( REDACTABLE_TYPES, NULL );
*/

    /* look for the comma-separated list of columns which are protected from redaction */
    helper_read_cfg_str( cctx->config_mgr, 
            DO_NOT_REDACT_KEY, &(cctx->do_not_redact_columns) );

    vdb_copy_read_redact_values( cctx );
}


static
bool vdb_copy_detect_path_clues( const char * path )
{
    bool res = false;
    size_t i, n = string_size ( path );
    for ( i = 0; i < n && !res; ++i )
    {
        char c = path[ i ];
        res = ( c == '.' || c == '/' || c == '\\' );
    }
    return res;
}

/***************************************************************************
    vdb_copy_main:
    * called by "KMain()"
    * make the "native directory"
    * make a vdb-manager for write
      all subsequent copy-functions will use this manager...
    * check if the given path is database-path ( by trying to open it )
      if it is one: continue wit copy_database()
    * check if the given path is table-path ( by trying to open it )
      if it is one: continue wit copy_table()
    * release manager and directory

ctx        [IN] ... contains path, tablename, columns, row-range etc.
***************************************************************************/
static rc_t vdb_copy_main( const p_context ctx )
{
    rc_t rc;
    copy_ctx cctx;

    vdb_copy_init_copy_ctx( &cctx );

    rc = KDirectoryNativeDir( &(cctx.directory) );
    DISP_RC( rc, "vdb_copy_main:KDirectoryNativeDir() failed" );
    if ( rc != 0 )
    {
        vdb_copy_destroy_copy_ctx( &cctx );
        return rc;
    }

    rc = helper_make_config_mgr( &(cctx.config_mgr), ctx->kfg_path );
    DISP_RC( rc, "vdb_copy_main:helper_make_config_mgr() failed" );
    if ( rc != 0 )
    {
        vdb_copy_destroy_copy_ctx( &cctx );
        return rc;
    }

    vdb_copy_read_config_values( &cctx );
    
    if ( !ctx->dont_check_accession )
        ctx->dont_check_accession = vdb_copy_detect_path_clues( ctx->src_path );
    if ( !ctx->dont_check_accession )
    {
        rc_t rc1 = helper_resolve_accession( cctx.directory, (char**)&( ctx->src_path ) );
        DISP_RC( rc1, "vdb_copy_main:helper_check_accession() failed" );
    }

    rc = VDBManagerMakeUpdate ( &(cctx.vdb_manager), cctx.directory );
    DISP_RC( rc, "vdb_copy_main:VDBManagerMakeRead() failed" );
    if ( rc != 0 )
    {
        vdb_copy_destroy_copy_ctx( &cctx );
        return rc;
    }

    rc = helper_parse_schema( cctx.vdb_manager,
                              &(cctx.src_schema),
                              ctx->src_schema_list );
    DISP_RC( rc, "vdb_copy_main:helper_parse_schema(src) failed" );
    if ( rc != 0 )
    {
        vdb_copy_destroy_copy_ctx( &cctx );
        return rc;
    }

    /* if the path is a database-path... (from helper.c) */
    if ( helper_is_path_database( cctx.vdb_manager, cctx.src_schema, 
                                  ctx->src_path ) )
    {
        OUTMSG (( "a copy of a database is not implemented yet...\n" ));
#if 0
        rc = vdb_copy_database( ctx, &cctx );
        DISP_RC( rc, "vdb_copy_main:vdb_copy_database() failed" );
#endif
        vdb_copy_destroy_copy_ctx( &cctx );
        return 0;
    }
    
    /* if the path is a table-path... (from helper.c) */
    if ( helper_is_path_table( cctx.vdb_manager, cctx.src_schema,
                               ctx->src_path ) )
    {
        rc = VDBManagerOpenTableRead( cctx.vdb_manager,
                       &(cctx.src_table), cctx.src_schema, ctx->src_path );
        DISP_RC( rc, "vdb_copy_main:VDBManagerOpenTableRead() failed" );
        if ( rc == 0 )
            rc = vdb_copy_table( ctx, &cctx );
        if ( rc != 0 && !cctx.dont_remove_target )
            vdb_copy_remove_table( ctx, &cctx );
        vdb_copy_destroy_copy_ctx( &cctx );
        return rc;
    }

    OUTMSG ( ( "\nthe path >%s< cannot be opened as vdb-database or vdb-table !!!\n", 
                ctx->src_path ) );
    if ( context_schema_count( ctx ) == 0 )
        OUTMSG ( ( "Maybe it is a legacy table. If so, specify a schema with the -S option\n" ) );

    vdb_copy_destroy_copy_ctx( &cctx );
    return RC( rcVDB, rcNoTarg, rcCopying, rcItem, rcNotFound );
}


/***************************************************************************
    Main:
    * create the copy-context
    * parse the commandline for arguments and options
    * react to help/usage - requests ( no dump in this case )
      these functions are in vdb-copy-context.c
    * call copy_main() to execute the copy-operation
    * destroy the copy-context
***************************************************************************/
rc_t CC KMain ( int argc, char *argv [] )
{
    Args * args;
    context *ctx;

    rc_t rc = ArgsMakeAndHandle ( &args, argc, argv, 1,
                             MyOptions, sizeof MyOptions / sizeof ( OptDef ) );
    DISP_RC( rc, "KMain:ArgsMakeAndHandle() failed" );
    if ( rc != 0 ) return rc;

    KLogHandlerSetStdErr();
    rc = context_init( &ctx );
    DISP_RC( rc, "KMain:copy_context_init() failed" );
    if ( rc != 0 )
    {
        ArgsWhack ( args );
        return rc;
    }

    rc = context_capture_arguments_and_options( args, ctx );
    DISP_RC( rc, "KMain:context_capture_arguments_and_options() failed" );
    if ( rc != 0 )
    {
        context_destroy( ctx );
        ArgsWhack ( args );
        return rc;
    }

    if ( ctx->usage_requested )
        MiniUsage( args );
    else
        rc = vdb_copy_main( ctx );

    context_destroy( ctx );
    ArgsWhack (args);
    return rc;
}
