/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

/*

Generates 1, 2, or 3 files depending on the existence of paired reads and/or
fragment reads. The files will only contain biologically significant bases and
qualities, technical reads such as the 'linker' for 454 are excluded. Reads
less than defaultMinReadLength will be excluded.

For a fragment-based run, only 1 fastq file is created.

For a pair-based run, either 1, 2, or 3 fastq files will created. If all the
reads can be paired and both reads in the pairs are > defaultMinReadLength, then 2 files
are created. If some of the reads cannot be paired, then a third, fragment file
is created. If none of the reads can be paired, then 1 file will be created
(e.g. a Solexa run where only the first application read was provided even though
the experiment involved a paired library).

A fragment read file will have the form:

accession.fastq

The two paired read files will have the form:

accession_1.fastq
accession_2.fastq
....
accession_x.fastq

Corresponding reads in the paired files will occur at the same line in each file.

*/
#include <klib/log.h>
#include <klib/container.h>
#include <kapp/main.h>

#include <sra/sradb.h>
#include <sra/fastq.h>

#include <os-native.h>
#include <sysalloc.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "core.h"
#include "fastq-dump.vers.h"
#include "debug.h"

struct FastqArgs_struct {
    uint8_t platform;

    int maxReads;
    bool skipTechnical;

    uint32_t minReadLen;
    bool applyClip;
    bool dumpOrigFmt;
    bool dumpBase;
    bool dumpCs;
    bool readIds;
    int offset;
    bool qual_filter;
    const char* b_deffmt;
    SLList* b_defline;
    const char* q_deffmt;
    SLList* q_defline;
    const char *desiredCsKey;
    bool single_file;
    bool one_line;
    bool fasta;
    const char* file_extension;
} FastqArgs;

typedef enum DefNodeType_enum {
    DefNode_Unknown = 0,
    DefNode_Text = 1,
    DefNode_Optional,
    DefNode_Accession,
    DefNode_SpotId,
    DefNode_SpotName,
    DefNode_SpotGroup,
    DefNode_SpotLen,
    DefNode_ReadId,
    DefNode_ReadName,
    DefNode_ReadLen,
    DefNode_Last
} DefNodeType;

typedef struct DefNode_struct {
    SLNode node;
    DefNodeType type;
    union {
        SLList* optional;
        char* text;
    } data;
} DefNode;

typedef struct DeflineData_struct {
    rc_t rc;
    bool optional;
    union {
        spotid_t* id;
        struct {
            const char* s;
            size_t sz;
        } str;
        uint32_t* u32;
    } values[DefNode_Last];
    char* buf;
    size_t buf_sz;
    size_t* writ;
} DeflineData;

static
bool CC Defline_Builder( SLNode *node, void *data )
{
    DefNode* n = (DefNode*)node;
    DeflineData* d = (DeflineData*)data;
    char s[256];
    size_t w;
    int x = 0;

    s[0] = '\0';
    switch(n->type) {
        case DefNode_Optional:
            d->optional = true;
            w = *d->writ;
            *d->writ = 0;
            SLListDoUntil( n->data.optional, Defline_Builder, data );
            x = (*d->writ == 0) ? 0 : 1;
            d->optional = false;
            *d->writ = w;
            if( x > 0 ) {
                SLListDoUntil( n->data.optional, Defline_Builder, data );
            }
            break;

        case DefNode_Text:
            if( d->optional ) {
                break;
            }
            d->values[n->type].str.s = n->data.text;
            d->values[n->type].str.sz = 0;
        case DefNode_Accession:
        case DefNode_SpotName:
        case DefNode_SpotGroup:
        case DefNode_ReadName:
            if( d->values[n->type].str.s != NULL ) {
                x = d->values[n->type].str.sz > 0 ? d->values[n->type].str.sz : strlen(d->values[n->type].str.s);
                if( x < sizeof(s) ) {
                    strncpy(s, d->values[n->type].str.s, x);
                    s[x] = '\0';
                    *d->writ = *d->writ + x;
                } else {
                    d->rc = RC(rcExe, rcNamelist, rcExecuting, rcBuffer, rcInsufficient);
                }
            }
            break;

        case DefNode_SpotId:
            if( d->values[n->type].id != NULL && (!d->optional || *d->values[n->type].id > 0) ) {
                x = snprintf(s, sizeof(s), "%u", *d->values[n->type].id);
                if( x < 0 || x >= sizeof(s) ) {
                    d->rc = RC(rcExe, rcNamelist, rcExecuting, rcBuffer, rcInsufficient);
                } else {
                    *d->writ = *d->writ + x;
                }
            }
            break;

        case DefNode_ReadId:
        case DefNode_SpotLen:
        case DefNode_ReadLen:
            if( d->values[n->type].u32 != NULL && (!d->optional || *d->values[n->type].u32 > 0) ) {
                x = snprintf(s, sizeof(s), "%u", *d->values[n->type].u32);
                if( x < 0 || x >= sizeof(s) ) {
                    d->rc = RC(rcExe, rcNamelist, rcExecuting, rcBuffer, rcInsufficient);
                } else {
                    *d->writ = *d->writ + x;
                }
            }
            break;

        default:
            d->rc = RC(rcExe, rcNamelist, rcExecuting, rcId, rcInvalid);
    }
    if( d->rc == 0 && !d->optional && n->type != DefNode_Optional ) {
        if( *d->writ < d->buf_sz ) {
            strcat(d->buf, s);
        } else {
            d->rc = RC(rcExe, rcNamelist, rcExecuting, rcBuffer, rcInsufficient);
        }
    }
    return d->rc != 0;
}

static
rc_t Defline_Bind(DeflineData* data, const char* accession,
                  spotid_t* spotId, const char* spot_name, size_t spotname_sz,
                  const char* spot_group, size_t sgrp_sz, uint32_t* spot_len,
                  uint32_t* readId, const char* read_name, INSDC_coord_len rlabel_sz, INSDC_coord_len* read_len)
{
    if( data == NULL ) {
        return RC(rcExe, rcNamelist, rcExecuting, rcMemory, rcInsufficient);
    }
    data->values[DefNode_Unknown].str.s = NULL;
    data->values[DefNode_Text].str.s = NULL;
    data->values[DefNode_Optional].str.s = NULL;
    data->values[DefNode_Accession].str.s = accession;
    data->values[DefNode_Accession].str.sz = 0;
    data->values[DefNode_SpotId].id = spotId;
    data->values[DefNode_SpotName].str.s = spot_name;
    data->values[DefNode_SpotName].str.sz = spotname_sz;
    data->values[DefNode_SpotGroup].str.s = spot_group;
    data->values[DefNode_SpotGroup].str.sz = sgrp_sz;
    data->values[DefNode_SpotLen].u32 = spot_len;
    data->values[DefNode_ReadId].u32 = readId;
    data->values[DefNode_ReadName].str.s = read_name;
    data->values[DefNode_ReadName].str.sz = rlabel_sz;
    data->values[DefNode_ReadLen].u32 = read_len;
    return 0;
}

static
rc_t Defline_Build(const SLList* def, DeflineData* data, char* buf, size_t buf_sz, size_t* writ)
{
    if( data == NULL ) {
        return RC(rcExe, rcNamelist, rcExecuting, rcMemory, rcInsufficient);
    }

    data->rc = 0;
    data->optional = false;
    data->buf = buf;
    data->buf_sz = buf_sz;
    data->writ = writ;

    data->buf[0] = '\0';
    *data->writ = 0;

    SLListDoUntil( def, Defline_Builder, data );
    return data->rc;
}

static
rc_t DeflineNode_Add(SLList* list, DefNode** node, DefNodeType type, const char* text, size_t text_sz)
{
    rc_t rc = 0;

    *node = calloc(1, sizeof(**node));
    if( *node == NULL ) {
        rc = RC(rcExe, rcNamelist, rcConstructing, rcMemory, rcExhausted);
    } else if(type == DefNode_Text && (text == NULL || text_sz == 0) ) {
        rc = RC(rcExe, rcNamelist, rcConstructing, rcParam, rcInvalid);
    } else {
        (*node)->type = type;
        if( type == DefNode_Text ) {
            (*node)->data.text = strndup(text, text_sz);
            if( (*node)->data.text == NULL ) {
                rc = RC(rcExe, rcNamelist, rcConstructing, rcMemory, rcExhausted);
                free(*node);
            }
        } else if( type == DefNode_Optional ) {
            (*node)->data.optional = malloc(sizeof(SLList));
            if( (*node)->data.optional == NULL ) {
                rc = RC(rcExe, rcNamelist, rcConstructing, rcMemory, rcExhausted);
                free(*node);
            } else {
                SLListInit((*node)->data.optional);
            }
        }
        if( rc == 0 ) {
            SLListPushTail(list, &(*node)->node);
        }
    }
    return rc;
}

static
void Defline_Release(SLList* list);

static
void CC DeflineNode_Whack( SLNode* node, void* data )
{
    if( node != NULL ) {
        DefNode* n = (DefNode*)node;
        if( n->type == DefNode_Text ) {
            free(n->data.text);
        } else if( n->type == DefNode_Optional ) {
            Defline_Release(n->data.optional);
        }
        free(node);
    }
}

static
void Defline_Release( SLList* list )
{
    if( list != NULL ) {
        SLListForEach( list, DeflineNode_Whack, NULL );
        free(list);
    }
}

#if _DEBUGGING
static
void CC Defline_Dump( SLNode* node, void* data )
{
    DefNode* n = (DefNode*)node;
    const char* s = NULL, *t;
    if( n->type == DefNode_Text ) {
        s = n->data.text;
    }
    switch(n->type) {
        case DefNode_Unknown:
            t = "Unknown";
            break;
        case DefNode_Text:
            t = "Text";
            break;
        case DefNode_Optional:
            t = "Optional";
            break;
        case DefNode_Accession:
            t = "Accession";
            break;
        case DefNode_SpotId:
            t = "SpotId";
            break;
        case DefNode_SpotName:
            t = "SpotName";
            break;
        case DefNode_SpotGroup:
            t = "SpotGroup";
            break;
        case DefNode_SpotLen:
            t = "SpotLen";
            break;
        case DefNode_ReadId:
            t = "ReadId";
            break;
        case DefNode_ReadName:
            t = "ReadName";
            break;
        case DefNode_ReadLen:
            t = "ReadLen";
            break;
        default:
            t = "ERROR";
    }

    SRA_DUMP_DBG(3, ("%s type %s", data, t));
    if( s ) {
        SRA_DUMP_DBG(3, (": '%s'", s));
    }
    SRA_DUMP_DBG(3, ("\n"));
    if( n->type == DefNode_Optional ) {
        SLListForEach( n->data.optional, Defline_Dump, "+-->" );
    }
}
#endif

static
rc_t Defline_Parse(SLList** def, const char* line)
{
    rc_t rc = 0;
    size_t i, sz, text = 0, opt_vars = 0;
    DefNode* node;
    SLList* list = NULL;

    if( def == NULL || line == NULL ) {
        return RC(rcExe, rcNamelist, rcConstructing, rcParam, rcInvalid);
    }
    *def = malloc(sizeof(SLList));
    if( *def  == NULL ) {
        return RC(rcExe, rcNamelist, rcConstructing, rcMemory, rcExhausted);
    }
    sz = strlen(line);
    list = *def;
    SLListInit(list);

    for(i = 0; rc == 0 && i < sz; i++ ) {
        if( line[i] == '[' ) {
            if( (i + 1) < sz && line[i + 1] == '[' ) {
                i++;
            } else {
                if( list != *def ) {
                    rc = RC(rcExe, rcNamelist, rcConstructing, rcFormat, rcIncorrect);
                } else if( i > text ) {
                    rc = DeflineNode_Add(list, &node, DefNode_Text, &line[text], i - text);
                }
                if( rc == 0 && (rc = DeflineNode_Add(list, &node, DefNode_Optional, NULL, 0)) == 0 ) {
                    opt_vars = 0;
                    list = node->data.optional;
                    text = i + 1;
                }
            }
        } else if( line[i] == ']' ) {
            if( (i + 1) < sz && line[i + 1] == ']' ) {
                i++;
            } else {
                if( list == *def ) {
                    rc = RC(rcExe, rcNamelist, rcConstructing, rcFormat, rcIncorrect);
                } else {
                    if( opt_vars < 1 ) {
                        rc = RC(rcExe, rcNamelist, rcConstructing, rcConstraint, rcEmpty);
                    } else if( i > text ) {
                        rc = DeflineNode_Add(list, &node, DefNode_Text, &line[text], i - text);
                    }
                    list = *def;
                    text = i + 1;
                }
            }
        } else if( line[i] == '$' ) {
            if( (i + 1) < sz && line[i + 1] == '$' ) {
                i++;
            } else {
                DefNodeType type = DefNode_Unknown;
                switch(line[++i]) {
                    case 'a':
                        switch(line[++i]) {
                            case 'c':
                                type = DefNode_Accession;
                                break;
                        }
                        break;
                    case 's':
                        switch(line[++i]) {
                            case 'i':
                                type = DefNode_SpotId;
                                break;
                            case 'n':
                                type = DefNode_SpotName;
                                break;
                            case 'g':
                                type = DefNode_SpotGroup;
                                break;
                            case 'l':
                                type = DefNode_SpotLen;
                                break;
                        }
                        break;
                    case 'r':
                        switch(line[++i]) {
                            case 'i':
                                type = DefNode_ReadId;
                                break;
                            case 'n':
                                type = DefNode_ReadName;
                                break;
                            case 'l':
                                type = DefNode_ReadLen;
                                break;
                        }
                        break;
                }
                if( type == DefNode_Unknown ) {
                    rc = RC(rcExe, rcNamelist, rcConstructing, rcName, rcUnrecognized);
                } else {
                    if( (i - 2) > text ) {
                        rc = DeflineNode_Add(list, &node, DefNode_Text, &line[text], i - text - 2);
                    }
                    if( rc == 0 && (rc = DeflineNode_Add(list, &node, type, NULL, 0)) == 0 ) {
                        opt_vars++;
                        text = i + 1;
                    }
                }
            }
        }
    }
    if( rc == 0 ) {
        if( list != *def ) {
            rc = RC(rcExe, rcNamelist, rcConstructing, rcFormat, rcInvalid);
        } else if( i > text ) {
            rc = DeflineNode_Add(list, &node, DefNode_Text, &line[text], i - text);
        }
    }
    if( rc != 0 ) {
        i = i < sz ? i : sz;
        PLOGERR(klogErr,(klogErr, rc, "$(l1) -->$(c)<-- $(l2)",
                            "l1=%.*s,c=%c,l2=%.*s", i - 1, line, line[i - 1], sz - i, &line[i]));
    }
#if _DEBUGGING
    SRA_DUMP_DBG(3, ("| defline\n"));
    SLListForEach( *def, Defline_Dump, "+->" );
#endif
    return rc;
}

/* ============== FASTQ read type (bio/tech) filter ============================ */

typedef struct FastqBioFilter_struct {
    const FastqReader* reader;
} FastqBioFilter;


/* filter out non-bio reads */
static
rc_t FastqBioFilter_GetKey(const SRASplitter* cself, const char** key, spotid_t spot, uint32_t* readmask)
{
    rc_t rc = 0;
    FastqBioFilter* self = (FastqBioFilter*)cself;

    if( self == NULL || key == NULL ) {
        rc = RC(rcExe, rcNode, rcExecuting, rcParam, rcInvalid);
    } else {
        uint32_t num_reads = 0;

        if( (rc = FastqReaderSeekSpot(self->reader, spot)) == 0 ) {
            if( (rc = FastqReader_SpotInfo(self->reader, NULL, NULL, NULL, NULL, NULL, &num_reads)) == 0 ) {
                uint32_t readId, new_mask = 0;
                SRAReadTypes read_type = SRA_READ_TYPE_TECHNICAL;

                SRA_DUMP_DBG(3, ("%s %u row reads: ", __func__, spot));
                for(readId = 0; rc == 0 && readId < num_reads; readId++) {
                    if( (rc = FastqReader_SpotReadInfo(self->reader, readId + 1, &read_type, NULL, NULL, NULL, NULL)) == 0 ) {
                        if( ( read_type & SRA_READ_TYPE_BIOLOGICAL ) != 0 ) {
                            new_mask |= (1 << readId);
                            SRA_DUMP_DBG(3, (" %u", readId));
                        }
                    }
                }
                *key = "";
                *readmask = new_mask;
                SRA_DUMP_DBG(3, (" key '%s'\n", *key));
            }
        } else if( GetRCState(rc) == rcNotFound ) {
            SRA_DUMP_DBG(3, ("%s skipped %u row\n", __func__, spot));
            *key = NULL;
            rc = 0;
        }
    }
    return rc;
}

typedef struct FastqBioFilterFactory_struct {
    const char* accession;
    const SRATable* table;
    const FastqReader* reader;
} FastqBioFilterFactory;

static
rc_t FastqBioFilterFactory_Init(const SRASplitterFactory* cself)
{
    rc_t rc = 0;
    FastqBioFilterFactory* self = (FastqBioFilterFactory*)cself;

    if( self == NULL ) {
        rc = RC(rcExe, rcType, rcConstructing, rcParam, rcNull);
    } else {
        rc = FastqReaderMake(&self->reader, self->table, self->accession,
                             /* preserve orig spot format to save on conversions */
                             FastqArgs.platform == SRA_PLATFORM_ABSOLID, false, false, 
                             false, !FastqArgs.applyClip, 0,
                             FastqArgs.offset, '\0', 0, 0);
    }
    return rc;
}

static
rc_t FastqBioFilterFactory_NewObj(const SRASplitterFactory* cself, const SRASplitter** splitter)
{
    rc_t rc = 0;
    FastqBioFilterFactory* self = (FastqBioFilterFactory*)cself;

    if( self == NULL ) {
        rc = RC(rcExe, rcType, rcExecuting, rcParam, rcNull);
    } else {
        if( (rc = SRASplitter_Make(splitter, sizeof(FastqBioFilter), FastqBioFilter_GetKey, NULL, NULL, NULL)) == 0 ) {
            ((FastqBioFilter*)(*splitter))->reader = self->reader;
        }
    }
    return rc;
}

static
void FastqBioFilterFactory_Release(const SRASplitterFactory* cself)
{
    if( cself != NULL ) {
        FastqBioFilterFactory* self = (FastqBioFilterFactory*)cself;
        FastqReaderWhack(self->reader);
    }
}

static
rc_t FastqBioFilterFactory_Make(const SRASplitterFactory** cself, const char* accession, const SRATable* table)
{
    rc_t rc = 0;
    FastqBioFilterFactory* obj = NULL;

    if( cself == NULL || accession == NULL || table == NULL ) {
        rc = RC(rcExe, rcType, rcConstructing, rcParam, rcNull);
    } else if( (rc = SRASplitterFactory_Make(cself, eSplitterSpot, sizeof(*obj),
                                             FastqBioFilterFactory_Init,
                                             FastqBioFilterFactory_NewObj,
                                             FastqBioFilterFactory_Release)) == 0 ) {
        obj = (FastqBioFilterFactory*)*cself;
        obj->accession = accession;
        obj->table = table;
    }
    return rc;
}

/* ============== FASTQ number of reads filter ============================ */

typedef struct FastqRNumberFilter_struct {
    const FastqReader* reader;
} FastqRNumberFilter;


static
rc_t FastqRNumberFilter_GetKey(const SRASplitter* cself, const char** key, spotid_t spot, uint32_t* readmask)
{
    rc_t rc = 0;
    FastqRNumberFilter* self = (FastqRNumberFilter*)cself;

    if( self == NULL || key == NULL ) {
        rc = RC(rcExe, rcNode, rcExecuting, rcParam, rcInvalid);
    } else {
        uint32_t num_reads = 0;

        if( (rc = FastqReaderSeekSpot(self->reader, spot)) == 0 ) {
            if( (rc = FastqReader_SpotInfo(self->reader, NULL, NULL, NULL, NULL, NULL, &num_reads)) == 0 ) {
                int readId, q;
                uint32_t new_mask = 0;

                SRA_DUMP_DBG(3, ("%s %u row reads: ", __func__, spot));
                for(readId = 0, q = 0; readId < num_reads && q < FastqArgs.maxReads; readId++) {
                    if( (*readmask) & (1 << readId) ) {
                        new_mask |= (1 << readId);
                        q++;
                        SRA_DUMP_DBG(3, (" %u", readId));
                    }
                }
                *key = "";
                *readmask = new_mask;
                SRA_DUMP_DBG(3, (" key '%s'\n", *key));
            }
        } else if( GetRCState(rc) == rcNotFound ) {
            SRA_DUMP_DBG(3, ("%s skipped %u row\n", __func__, spot));
            *key = NULL;
            rc = 0;
        }
    }
    return rc;
}

typedef struct FastqRNumberFilterFactory_struct {
    const char* accession;
    const SRATable* table;
    const FastqReader* reader;
} FastqRNumberFilterFactory;

static
rc_t FastqRNumberFilterFactory_Init(const SRASplitterFactory* cself)
{
    rc_t rc = 0;
    FastqRNumberFilterFactory* self = (FastqRNumberFilterFactory*)cself;

    if( self == NULL ) {
        rc = RC(rcExe, rcType, rcConstructing, rcParam, rcNull);
    } else {
        rc = FastqReaderMake(&self->reader, self->table, self->accession,
                             /* preserve orig spot format to save on conversions */
                             FastqArgs.platform == SRA_PLATFORM_ABSOLID, false, false, 
                             false, !FastqArgs.applyClip, 0,
                             FastqArgs.offset, '\0', 0, 0);
    }
    return rc;
}

static
rc_t FastqRNumberFilterFactory_NewObj(const SRASplitterFactory* cself, const SRASplitter** splitter)
{
    rc_t rc = 0;
    FastqRNumberFilterFactory* self = (FastqRNumberFilterFactory*)cself;

    if( self == NULL ) {
        rc = RC(rcExe, rcType, rcExecuting, rcParam, rcNull);
    } else {
        if( (rc = SRASplitter_Make(splitter, sizeof(FastqRNumberFilter), FastqRNumberFilter_GetKey, NULL, NULL, NULL)) == 0 ) {
            ((FastqRNumberFilter*)(*splitter))->reader = self->reader;
        }
    }
    return rc;
}

static
void FastqRNumberFilterFactory_Release(const SRASplitterFactory* cself)
{
    if( cself != NULL ) {
        FastqRNumberFilterFactory* self = (FastqRNumberFilterFactory*)cself;
        FastqReaderWhack(self->reader);
    }
}

static
rc_t FastqRNumberFilterFactory_Make(const SRASplitterFactory** cself, const char* accession, const SRATable* table)
{
    rc_t rc = 0;
    FastqRNumberFilterFactory* obj = NULL;

    if( cself == NULL || accession == NULL || table == NULL ) {
        rc = RC(rcExe, rcType, rcConstructing, rcParam, rcNull);
    } else if( (rc = SRASplitterFactory_Make(cself, eSplitterSpot, sizeof(*obj),
                                             FastqRNumberFilterFactory_Init,
                                             FastqRNumberFilterFactory_NewObj,
                                             FastqRNumberFilterFactory_Release)) == 0 ) {
        obj = (FastqRNumberFilterFactory*)*cself;
        obj->accession = accession;
        obj->table = table;
    }
    return rc;
}

/* ============== FASTQ quality filter ============================ */

typedef struct FastqQFilter_struct {
    const FastqReader* reader;
} FastqQFilter;


/* filter out reads by leading/trialing quality */
static
rc_t FastqQFilter_GetKey(const SRASplitter* cself, const char** key, spotid_t spot, uint32_t* readmask)
{
    rc_t rc = 0;
    FastqQFilter* self = (FastqQFilter*)cself;

    if( self == NULL || key == NULL ) {
        rc = RC(rcExe, rcNode, rcExecuting, rcParam, rcInvalid);
    } else {
        uint32_t num_reads = 0;


        if( (rc = FastqReaderSeekSpot(self->reader, spot)) == 0 ) {
            if( (rc = FastqReader_SpotInfo(self->reader, NULL, NULL, NULL, NULL, NULL, &num_reads)) == 0 ) {
                uint32_t readId, new_mask = 0;
                INSDC_coord_len read_len = 0;
                static char const* const baseStr = "NNNNNNNNNN";
                static char const* const colorStr = "..........";
                static const int xLen = 10;
                char buf[512 * 1024];

                SRA_DUMP_DBG(3, ("%s %u row reads: ", __func__, spot));
                for(readId = 0; rc == 0 && readId < num_reads; readId++) {
                    if( (rc = FastqReader_SpotReadInfo(self->reader, readId + 1, NULL, NULL, NULL, NULL, &read_len)) == 0 ) {
                        if( (rc = FastqReaderBase(self->reader, readId + 1, buf, sizeof(buf), NULL)) == 0 ) {
                            if( (FastqArgs.platform == SRA_PLATFORM_ABSOLID &&
                                 strncmp(&buf[1], colorStr, xLen) == 0 && strcmp(&buf[read_len - xLen + 1], colorStr) == 0) ||
                                (strncmp(buf, baseStr, xLen) == 0 && strcmp(&buf[read_len - xLen], baseStr) == 0) )  {
                                continue;
                            }
                        }
                        new_mask |= (1 << readId);
                        SRA_DUMP_DBG(3, (" %u", readId));
                    }
                }
                *key = "";
                *readmask = new_mask;
                SRA_DUMP_DBG(3, (" key '%s'\n", *key));
            }
        } else if( GetRCState(rc) == rcNotFound ) {
            SRA_DUMP_DBG(3, ("%s skipped %u row\n", __func__, spot));
            *key = NULL;
            rc = 0;
        }
    }
    return rc;
}

typedef struct FastqQFilterFactory_struct {
    const char* accession;
    const SRATable* table;
    const FastqReader* reader;
} FastqQFilterFactory;

static
rc_t FastqQFilterFactory_Init(const SRASplitterFactory* cself)
{
    rc_t rc = 0;
    FastqQFilterFactory* self = (FastqQFilterFactory*)cself;

    if( self == NULL ) {
        rc = RC(rcExe, rcType, rcConstructing, rcParam, rcNull);
    } else {
        rc = FastqReaderMake(&self->reader, self->table, self->accession,
                             /* preserve orig spot format to save on conversions */
                             FastqArgs.platform == SRA_PLATFORM_ABSOLID, false, false, 
                             false, !FastqArgs.applyClip, 0,
                             FastqArgs.offset, '\0', 0, 0);
    }
    return rc;
}

static
rc_t FastqQFilterFactory_NewObj(const SRASplitterFactory* cself, const SRASplitter** splitter)
{
    rc_t rc = 0;
    FastqQFilterFactory* self = (FastqQFilterFactory*)cself;

    if( self == NULL ) {
        rc = RC(rcExe, rcType, rcExecuting, rcParam, rcNull);
    } else {
        if( (rc = SRASplitter_Make(splitter, sizeof(FastqQFilter), FastqQFilter_GetKey, NULL, NULL, NULL)) == 0 ) {
            ((FastqQFilter*)(*splitter))->reader = self->reader;
        }
    }
    return rc;
}

static
void FastqQFilterFactory_Release(const SRASplitterFactory* cself)
{
    if( cself != NULL ) {
        FastqQFilterFactory* self = (FastqQFilterFactory*)cself;
        FastqReaderWhack(self->reader);
    }
}

static
rc_t FastqQFilterFactory_Make(const SRASplitterFactory** cself, const char* accession, const SRATable* table)
{
    rc_t rc = 0;
    FastqQFilterFactory* obj = NULL;

    if( cself == NULL || accession == NULL || table == NULL ) {
        rc = RC(rcExe, rcType, rcConstructing, rcParam, rcNull);
    } else if( (rc = SRASplitterFactory_Make(cself, eSplitterSpot, sizeof(*obj),
                                             FastqQFilterFactory_Init,
                                             FastqQFilterFactory_NewObj,
                                             FastqQFilterFactory_Release)) == 0 ) {
        obj = (FastqQFilterFactory*)*cself;
        obj->accession = accession;
        obj->table = table;
    }
    return rc;
}

/* ============== FASTQ min read length splitter ============================ */

char* FastqReadLenSplitter_key_buf = NULL;

typedef struct FastqReadLenSplitter_struct {
    const FastqReader* reader;
} FastqReadLenSplitter;

/* if all active reads len >= minreadlen, reads are splitted into separate files, otherwise on single key "" is returnded for all reads */
static
rc_t FastqReadLenSplitter_GetKeySet(const SRASplitter* cself, SRASplitter_Keys* key, int* keys, spotid_t spot, uint32_t readmask)
{
    rc_t rc = 0;
    FastqReadLenSplitter* self = (FastqReadLenSplitter*)cself;
    const size_t key_offset = 4, key_max_sz = sizeof(readmask) * 8;

    if( self == NULL || key == NULL || *keys < FastqArgs.maxReads ) {
        rc = RC(rcExe, rcNode, rcExecuting, rcParam, rcInvalid);
    } else {
        uint32_t num_reads = 0;
        *keys = 0;

        if( FastqReadLenSplitter_key_buf == NULL ) {
            /* initial alloc key_buf: "  1\0  2\0...\0  9\0 10\0 11\0...\0220\0221\0" */
            FastqReadLenSplitter_key_buf = malloc(key_max_sz * key_offset);
            if( FastqReadLenSplitter_key_buf == NULL ) {
                rc = RC(rcExe, rcNode, rcExecuting, rcMemory, rcExhausted);
            } else {
                /* fill buffer w/keys */
                int i;
                char* p = FastqReadLenSplitter_key_buf;
                for(i = 1; rc == 0 && i <= key_max_sz; i++) {
                    if( sprintf(p, "%3u", i) <= 0 ) {
                        rc = RC(rcExe, rcNode, rcExecuting, rcTransfer, rcIncomplete);
                    }
                    p += key_offset;
                }
            }
        }
        if( rc == 0 && (rc = FastqReaderSeekSpot(self->reader, spot)) == 0 ) {
            if( (rc = FastqReader_SpotInfo(self->reader, NULL, NULL, NULL, NULL, NULL, &num_reads)) == 0 ) {
                uint32_t readId, all = 0, good  = 0;
                INSDC_coord_len read_len = 0;
                SRAReadTypes read_type;

                SRA_DUMP_DBG(3, ("%s %u row reads: ", __func__, spot));
                for(readId = 0; rc == 0 && readId < num_reads; readId++) {
                    rc = FastqReader_SpotReadInfo(self->reader, readId + 1, &read_type, NULL, NULL, NULL, &read_len);
                    if( ( read_type & SRA_READ_TYPE_BIOLOGICAL ) == 0 && FastqArgs.skipTechnical ) {
                        continue;
                    }
                    all++;
                    if( (readmask & (1 << readId)) == 0 ) {
                        continue;
                    }
                    if( rc != 0 || read_len < FastqArgs.minReadLen ) {
                        continue;
                    }
                    key[good].key = &FastqReadLenSplitter_key_buf[(good * key_offset)];
                    while( key[good].key[0] == ' ' && key[good].key[0] != '\0' ) {
                        key[good].key++;
                    }
                    key[good].readmask = (1 << readId);
                    SRA_DUMP_DBG(3, (" key['%s']+=%u", key[good].key, readId));
                    good++;
                }
                *keys = good;
                if( all != good || FastqArgs.single_file || (all == 1 && good == 1) ) {
                    /* some are short -> reset keys to same value for all valid reads */
                    /* run has just one read -> no suffix */
                    /* or single file was requested */
                    for(readId = 0; readId < good; readId++) {
                        key[readId].key = "";
                    }
                    SRA_DUMP_DBG(3, (" all keys joined to ''"));
                }
                SRA_DUMP_DBG(3, ("\n"));
            }
        } else if( GetRCState(rc) == rcNotFound ) {
            SRA_DUMP_DBG(3, ("%s skipped %u row\n", __func__, spot));
            *keys = 0;
            rc = 0;
        }
    }
    return rc;
}

typedef struct FastqReadLenSplitterFactory_struct {
    const char* accession;
    const SRATable* table;
    const FastqReader* reader;
} FastqReadLenSplitterFactory;

static
rc_t FastqReadLenSplitterFactory_Init(const SRASplitterFactory* cself)
{
    rc_t rc = 0;
    FastqReadLenSplitterFactory* self = (FastqReadLenSplitterFactory*)cself;

    if( self == NULL ) {
        rc = RC(rcExe, rcType, rcConstructing, rcParam, rcNull);
    } else {
        rc = FastqReaderMake(&self->reader, self->table, self->accession,
                             /* preserve orig spot format to save on conversions */
                             FastqArgs.platform == SRA_PLATFORM_ABSOLID, false, false, 
                             false, !FastqArgs.applyClip, 0,
                             FastqArgs.offset, '\0', 0, 0);
    }
    return rc;
}

static
rc_t FastqReadLenSplitterFactory_NewObj(const SRASplitterFactory* cself, const SRASplitter** splitter)
{
    rc_t rc = 0;
    FastqReadLenSplitterFactory* self = (FastqReadLenSplitterFactory*)cself;

    if( self == NULL ) {
        rc = RC(rcExe, rcType, rcExecuting, rcParam, rcNull);
    } else {
        if( (rc = SRASplitter_Make(splitter, sizeof(FastqReadLenSplitter), NULL, FastqReadLenSplitter_GetKeySet, NULL, NULL)) == 0 ) {
            ((FastqReadLenSplitter*)(*splitter))->reader = self->reader;
        }
    }
    return rc;
}

static
void FastqReadLenSplitterFactory_Release(const SRASplitterFactory* cself)
{
    if( cself != NULL ) {
        FastqReadLenSplitterFactory* self = (FastqReadLenSplitterFactory*)cself;
        FastqReaderWhack(self->reader);
        free(FastqReadLenSplitter_key_buf);
    }
}

static
rc_t FastqReadLenSplitterFactory_Make(const SRASplitterFactory** cself, const char* accession, const SRATable* table)
{
    rc_t rc = 0;
    FastqReadLenSplitterFactory* obj = NULL;

    if( cself == NULL || accession == NULL || table == NULL ) {
        rc = RC(rcExe, rcType, rcConstructing, rcParam, rcNull);
    } else if( (rc = SRASplitterFactory_Make(cself, eSplitterRead, sizeof(*obj),
                                             FastqReadLenSplitterFactory_Init,
                                             FastqReadLenSplitterFactory_NewObj,
                                             FastqReadLenSplitterFactory_Release)) == 0 ) {
        obj = (FastqReadLenSplitterFactory*)*cself;
        obj->accession = accession;
        obj->table = table;
    }
    return rc;
}

/* ============== FASTQ formatter object  ============================ */

typedef struct FastqFormatterSplitter_struct {
    const char* accession;
    const FastqReader* reader;
} FastqFormatterSplitter;

static
rc_t FastqFormatterSplitter_DumpByRead(const SRASplitter* cself, spotid_t spot, uint32_t readmask)
{
    rc_t rc = 0;
    FastqFormatterSplitter* self = (FastqFormatterSplitter*)cself;

    if( self == NULL ) {
        rc = RC(rcExe, rcType, rcExecuting, rcParam, rcNull);
    } else if( readmask == 0 ) {
    } else {
        if( (rc = FastqReaderSeekSpot(self->reader, spot)) == 0 ) {
            if( (rc = SRASplitter_FileActivate(cself, FastqArgs.file_extension)) == 0 ) {
                DeflineData def_data;
                const char* spot_name = NULL, *spot_group = NULL, *read_name = NULL;
                uint32_t readIdx, spot_len = 0;
                uint32_t num_reads, readId, k;
                char l[4][16 * 1024];
                size_t lsz[4], sname_sz, sgrp_sz;
                INSDC_coord_len rlabel_sz = 0, read_len = 0;

                if( FastqArgs.b_defline || FastqArgs.q_defline ) {
                    rc = FastqReader_SpotInfo(self->reader, &spot_name, &sname_sz, &spot_group, &sgrp_sz, &spot_len, &num_reads);
                } else {
                    rc = FastqReader_SpotInfo(self->reader, NULL, NULL, NULL, NULL, NULL, &num_reads);
                }

                for(readId = 1, readIdx = 1; rc == 0 && readId <= num_reads; readId++, readIdx++) {
                    if( ( readmask & (1 << (readId - 1)) ) == 0 ) {
                        continue;
                    }
                    if( FastqArgs.b_defline || FastqArgs.q_defline ) {
                        if( (rc = FastqReader_SpotReadInfo(self->reader, readId, NULL, &read_name, &rlabel_sz, NULL, &read_len)) == 0 ) {
                            rc = Defline_Bind(&def_data, self->accession, &spot, spot_name, sname_sz, spot_group, sgrp_sz,
                                              &spot_len, &readIdx, read_name, rlabel_sz, &read_len);
                        }
                    }
                    if( rc == 0 ) {
                        if( FastqArgs.b_defline ) {
                            rc = Defline_Build(FastqArgs.b_defline, &def_data, l[0], sizeof(l[0]), &lsz[0]);
                        } else {
                            rc = FastqReaderBaseName(self->reader, readId, &FastqArgs.dumpCs, l[0], sizeof(l[0]), &lsz[0]);
                        }
                    }
                    if( rc == 0 ) {
                        rc = FastqReaderBase(self->reader, readId, l[1], sizeof(l[1]), &lsz[1]);
                    }
                    if( !FastqArgs.fasta && rc == 0 ) {
                        if( FastqArgs.q_defline ) {
                            rc = Defline_Build(FastqArgs.q_defline, &def_data, l[2], sizeof(l[2]), &lsz[2]);
                        } else {
                            rc = FastqReaderQualityName(self->reader, readId, &FastqArgs.dumpCs, l[2], sizeof(l[2]), &lsz[2]);
                        }
                        if( rc == 0 ) {
                            rc = FastqReaderQuality(self->reader, readId, l[3], sizeof(l[3]), &lsz[3]);
                        }
                    } else if( l[0][0] == '@' ) {
                        l[0][0] = '>';
                    }
                    for(k = 0; rc == 0 && k < (FastqArgs.fasta ? 2 : 4); k++) {
                        if( (rc = SRASplitter_FileWrite(cself, spot, l[k], lsz[k])) == 0 ) {
                            rc = SRASplitter_FileWrite(cself, spot, "\n", 1);
                        }
                    }
                }
            }
        } else if( GetRCState(rc) == rcNotFound ) {
            SRA_DUMP_DBG(3, ("%s skipped %u row\n", __func__, spot));
            rc = 0;
        }
    }
    return rc;
}

static
rc_t FastqFormatterSplitter_DumpBySpot(const SRASplitter* cself, spotid_t spot, uint32_t readmask)
{
    rc_t rc = 0;
    FastqFormatterSplitter* self = (FastqFormatterSplitter*)cself;

    if( self == NULL ) {
        rc = RC(rcExe, rcType, rcExecuting, rcParam, rcNull);
    } else if( readmask == 0 ) {
    } else {
        if( (rc = FastqReaderSeekSpot(self->reader, spot)) == 0 ) {
            if( (rc = SRASplitter_FileActivate(cself, FastqArgs.file_extension)) == 0 ) {
                DeflineData def_data;
                const char* spot_name = NULL, *spot_group = NULL;
                uint32_t num_reads, readId = 1, spot_len = 0;
                char l[4][16 * 1024];
                size_t lsz[4], writ, sname_sz, sgrp_sz;
                const size_t lmax = sizeof(l[0]);

                if( FastqArgs.b_defline || FastqArgs.q_defline ) {
                    if( (rc = FastqReader_SpotInfo(self->reader, &spot_name, &sname_sz, &spot_group, &sgrp_sz, &spot_len, &num_reads)) == 0 ) {
                        rc = Defline_Bind(&def_data, self->accession, &spot, spot_name, sname_sz, spot_group, sgrp_sz, 
                                          &spot_len, &readId, NULL, 0, &spot_len);
                    }
                } else {
                    rc = FastqReader_SpotInfo(self->reader, NULL, NULL, NULL, NULL, NULL, &num_reads);
                }
                if( rc == 0 ) {
                    if( FastqArgs.b_defline ) {
                        rc = Defline_Build(FastqArgs.b_defline, &def_data, l[0], sizeof(l[0]), &lsz[0]);
                    } else {
                        rc = FastqReaderBaseName(self->reader, 0, &FastqArgs.dumpCs, l[0], sizeof(l[0]), &lsz[0]);
                    }
                    lsz[1] = 0;
                    if( !FastqArgs.fasta && rc == 0 ) {
                        if( FastqArgs.q_defline ) {
                            rc = Defline_Build(FastqArgs.q_defline, &def_data, l[2], sizeof(l[2]), &lsz[2]);
                        } else {
                            rc = FastqReaderQualityName(self->reader, 0, &FastqArgs.dumpCs, l[2], sizeof(l[2]), &lsz[2]);
                        }
                        lsz[3] = 0;
                    } else if( l[0][0] == '@' ) {
                        l[0][0] = '>';
                    }
                }
                for(readId = 1; rc == 0 && readId <= num_reads; readId++) {
                    if( ( readmask & (1 << (readId - 1)) ) == 0 ) {
                        continue;
                    }
                    rc = FastqReaderBase(self->reader, readId, &l[1][lsz[1]], lmax - lsz[1], &writ);
                    lsz[1] += writ;
                    if( !FastqArgs.fasta && rc == 0 ) {
                        rc = FastqReaderQuality(self->reader, readId, &l[3][lsz[3]], lmax - lsz[3], &writ);
                        lsz[3] += writ;
                    }
                }
                for(readId = 0; rc == 0 && readId < (FastqArgs.fasta ? 2 : 4); readId++) {
                    if( (rc = SRASplitter_FileWrite(cself, spot, l[readId], lsz[readId])) == 0 ) {
                        rc = SRASplitter_FileWrite(cself, spot, "\n", 1);
                    }
                }
            }
        } else if( GetRCState(rc) == rcNotFound ) {
            SRA_DUMP_DBG(3, ("%s skipped %u row\n", __func__, spot));
            rc = 0;
        }
    }
    return rc;
}

typedef struct FastqFormatterFactory_struct {
    const char* accession;
    const SRATable* table;
    const FastqReader* reader;
} FastqFormatterFactory;

static
rc_t FastqFormatterFactory_Init(const SRASplitterFactory* cself)
{
    rc_t rc = 0;
    FastqFormatterFactory* self = (FastqFormatterFactory*)cself;

    if( self == NULL ) {
        rc = RC(rcExe, rcType, rcConstructing, rcParam, rcNull);
    } else {
        rc = FastqReaderMake(&self->reader, self->table, self->accession,
                             FastqArgs.dumpCs, FastqArgs.dumpOrigFmt, FastqArgs.dumpCs,
                             FastqArgs.readIds, !FastqArgs.applyClip, 0,
                             FastqArgs.offset, FastqArgs.desiredCsKey[0], 0, 0);
    }
    return rc;
}

static
rc_t FastqFormatterFactory_NewObj(const SRASplitterFactory* cself, const SRASplitter** splitter)
{
    rc_t rc = 0;
    FastqFormatterFactory* self = (FastqFormatterFactory*)cself;

    if( self == NULL ) {
        rc = RC(rcExe, rcType, rcExecuting, rcParam, rcNull);
    } else {
        if( (rc = SRASplitter_Make(splitter, sizeof(FastqFormatterSplitter), NULL, NULL,
            FastqArgs.one_line ? FastqFormatterSplitter_DumpBySpot : FastqFormatterSplitter_DumpByRead, NULL)) == 0 ) {
            ((FastqFormatterSplitter*)(*splitter))->accession = self->accession;
            ((FastqFormatterSplitter*)(*splitter))->reader = self->reader;
        }
    }
    return rc;
}

static
void FastqFormatterFactory_Release(const SRASplitterFactory* cself)
{
    if( cself != NULL ) {
        FastqFormatterFactory* self = (FastqFormatterFactory*)cself;
        FastqReaderWhack(self->reader);
    }
}

static
rc_t FastqFormatterFactory_Make(const SRASplitterFactory** cself, const char* accession, const SRATable* table)
{
    rc_t rc = 0;
    FastqFormatterFactory* obj = NULL;

    if( cself == NULL || accession == NULL || table == NULL ) {
        rc = RC(rcExe, rcType, rcConstructing, rcParam, rcNull);
    } else if( (rc = SRASplitterFactory_Make(cself, eSplitterFormat, sizeof(*obj),
                                             FastqFormatterFactory_Init,
                                             FastqFormatterFactory_NewObj,
                                             FastqFormatterFactory_Release)) == 0 ) {
        obj = (FastqFormatterFactory*)*cself;
        obj->accession = accession;
        obj->table = table;
    }
    return rc;
}

/* ### External entry points ##################################################### */

ver_t CC KAppVersion( void )
{
    return FASTQ_DUMP_VERS;
}

rc_t FastqDumper_Release(const SRADumperFmt* fmt)
{
    if( fmt == NULL ) {
        return RC(rcExe, rcFormatter, rcDestroying, rcParam, rcInvalid);
    } else {
        Defline_Release(FastqArgs.b_defline);
        Defline_Release(FastqArgs.q_defline);
    }
    return 0;
}

bool FastqDumper_AddArg(const SRADumperFmt* fmt, GetArg* f, int* i, int argc, char *argv[])
{
    const char* arg = NULL;

    if( fmt == NULL || f == NULL || i == NULL || argv == NULL ) {
        LOGERR(klogErr, RC(rcExe, rcArgv, rcReading, rcParam, rcInvalid), "null param");
        return false;
    } else if( f("M", "minReadLen", i, argc, argv, &arg) ) {
        FastqArgs.minReadLen = AsciiToU32(arg, NULL, NULL);
    } else if( f("W", "clip", i, argc, argv, NULL) ) {
        FastqArgs.applyClip = true;
    } else if( f("F", "origfmt", i, argc, argv, NULL) ) {
        FastqArgs.dumpOrigFmt = true;
    } else if( f("C", "dumpcs", i, argc, argv, &FastqArgs.desiredCsKey) ) {
        FastqArgs.dumpCs = true;
    } else if( f("B", "dumpbase", i, argc, argv, NULL) ) {
        FastqArgs.dumpBase = true;
    } else if( f("Q", "offset", i, argc, argv, &arg) ) {
        FastqArgs.offset = AsciiToU32(arg, NULL, NULL);
    } else if( f("I", "readids", i, argc, argv, NULL) ) {
        FastqArgs.readIds = true;
    } else if( f("E", "qual-filter", i, argc, argv, NULL) ) {
        FastqArgs.qual_filter = true;
    } else if( f("DB", "defline-seq", i, argc, argv, &arg) ) {
        FastqArgs.b_deffmt = arg;
    } else if( f("DQ", "defline-qual", i, argc, argv, &arg) ) {
        FastqArgs.q_deffmt = arg;
    } else if( f("TR", "skip-technical", i, argc, argv, NULL) ) {
        FastqArgs.skipTechnical = true;
    } else if( f("SF", "single-file", i, argc, argv, NULL) ) {
        FastqArgs.single_file = true;
    } else if( f("SL", "one-line", i, argc, argv, NULL) ) {
        FastqArgs.single_file = true;
        FastqArgs.one_line = true;
    } else if( f("HS", "helicos", i, argc, argv, NULL) ) {
        FastqArgs.b_deffmt = "@$sn";
        FastqArgs.q_deffmt =  "+";
    } else if( f("FA", "fasta", i, argc, argv, NULL) ) {
        FastqArgs.fasta = true;
        FastqArgs.file_extension = ".fasta";
    } else {
        return false;
    }
    return true;
}

rc_t FastqDumper_Factories(const SRADumperFmt* fmt, const SRASplitterFactory** factory)
{
    rc_t rc = 0;
    const SRASplitterFactory* parent = NULL, *child = NULL;

    if( fmt == NULL || factory == NULL ) {
        return RC(rcExe, rcFormatter, rcReading, rcParam, rcInvalid);
    }
    *factory = NULL;

    {
        const SRAColumn* c = NULL;
        const uint8_t *platform;
        bitsz_t o, z;

        if( (rc = SRATableOpenColumnRead(fmt->table, &c, "PLATFORM", sra_platform_id_t)) != 0 ) {
            return rc;
        }
        if( (rc = SRAColumnRead(c, 1, (const void **)&platform, &o, &z)) != 0 ) {
            return rc;
        }
        FastqArgs.platform = *platform;
        if( !FastqArgs.dumpCs && !FastqArgs.dumpBase ) {
            if( FastqArgs.platform == SRA_PLATFORM_ABSOLID ) {
                FastqArgs.dumpCs = true;
            } else {
                FastqArgs.dumpBase = true;
            }
        }
        SRAColumnRelease(c);
    }

    if( rc == 0 && FastqArgs.skipTechnical ) {
        if( (rc = FastqBioFilterFactory_Make(&child, fmt->accession, fmt->table)) == 0 ) {
            if( parent != NULL ) {
                if( (rc = SRASplitterFactory_AddNext(parent, child)) != 0 ) {
                    SRASplitterFactory_Release(child);
                } else {
                    parent = child;
                }
            } else {
                parent = child;
                *factory = parent;
            }
        }
    }
    if( rc == 0 && FastqArgs.maxReads > 0 ) {
        if( (rc = FastqRNumberFilterFactory_Make(&child, fmt->accession, fmt->table)) == 0 ) {
            if( parent != NULL ) {
                if( (rc = SRASplitterFactory_AddNext(parent, child)) != 0 ) {
                    SRASplitterFactory_Release(child);
                } else {
                    parent = child;
                }
            } else {
                parent = child;
                *factory = parent;
            }
        }
    }
    if( rc == 0 && FastqArgs.qual_filter ) {
        if( (rc = FastqQFilterFactory_Make(&child, fmt->accession, fmt->table)) == 0 ) {
            if( parent != NULL ) {
                if( (rc = SRASplitterFactory_AddNext(parent, child)) != 0 ) {
                    SRASplitterFactory_Release(child);
                } else {
                    parent = child;
                }
            } else {
                parent = child;
                *factory = parent;
            }
        }
    }

    if( rc == 0 ) {
        if( (rc = FastqReadLenSplitterFactory_Make(&child, fmt->accession, fmt->table)) == 0 ) {
            if( parent != NULL ) {
                if( (rc = SRASplitterFactory_AddNext(parent, child)) != 0 ) {
                    SRASplitterFactory_Release(child);
                } else {
                    parent = child;
                }
            } else {
                parent = child;
                *factory = parent;
            }
        }
    }
    if( rc == 0 ) {
        if( FastqArgs.b_deffmt != NULL ) {
            rc = Defline_Parse(&FastqArgs.b_defline, FastqArgs.b_deffmt);
        }
        if( FastqArgs.q_deffmt != NULL ) {
            rc = Defline_Parse(&FastqArgs.q_defline, FastqArgs.q_deffmt);
        }
        if( rc == 0 && (rc = FastqFormatterFactory_Make(&child, fmt->accession, fmt->table)) == 0 ) {
            if( parent != NULL ) {
                if( (rc = SRASplitterFactory_AddNext(parent, child)) != 0 ) {
                    SRASplitterFactory_Release(child);
                }
            } else {
                *factory = child;
            }
        }
    }
    return rc;
}

/* main entry point of the file */
rc_t SRADumper_Init(SRADumperFmt* fmt)
{
    static const SRADumperFmt_Arg arg[] =
        {
            {"M", "minReadLen <len>", "Filter by minimum read length", NULL},
            {"E", "qual-filter", "Filter by quality", NULL},
            {"TR", "skip-technical", "Filter technical reads", NULL},
            {"W", "clip", "Clip quality left and right for spot\n\n"
            "NOTE: above 4 options were changed, to restore behaviour run with: -M 25 -E -TR -W\n", NULL},

            {"F", "origfmt", "Excludes SRR accession & length on defline", NULL},
            {"C", "dumpcs [cskey]", "Dump color space sequence (default for ABI SOLID), single letter color space key is optional", NULL},
            {"B", "dumpbase", "Dump base sequence (default for other than ABI SOLID)", NULL},
            {"Q", "offset", "Offset to use for quality conversion", "33"},
            {"I", "readids", "Append read id after spot id as 'accession.spot.readid' on defline", NULL},
            {"SF", "complete", "Dump reads in a single file", NULL},
            {"SL", "one-line", "Join reads on a single line", NULL},
            {"DB", "defline-seq", "sequence defline format specification. String of characters and/or variables.\n"
                              "\n\t\tVariables are one of:\n"
                              "\t\t  $ac - accession, $si - spot id, $sn - spot name, $sg - spot group (barcode),\n"
                              "\t\t  $sl - spot length in bases, $ri - read number, $rn - read name, $rl - read length in bases.\n"
                              "\t\t'[]' could be used for an optional output: if all vars in [] yield empty values whole group is not printed.\n"
                              "\t\tEmpty value is empty string or 0 for numeric variables.\n"
                              "\t\tEx: @$sn[_$rn]/$ri - '_$rn' is omitted if name is empty\n", NULL},
            {"DQ", "defline-qual", "quailty defline format specification. If -DB is provided but -DQ ommitted, -DB value is used", NULL},
            {"HS", "helicos", "Helicos style", NULL},
            {"FA", "fasta", "Fasta only, no qualities", NULL},
            {NULL, NULL, NULL, NULL}
        };

    FastqArgs.platform = SRA_PLATFORM_UNDEFINED;

    FastqArgs.maxReads = ~0;
    FastqArgs.skipTechnical = false;
    FastqArgs.minReadLen = 0;
    FastqArgs.applyClip = false;
    FastqArgs.dumpOrigFmt = false;
    FastqArgs.dumpBase = false;
    FastqArgs.dumpCs = false;
    FastqArgs.readIds = false;
    FastqArgs.offset = 33;
    FastqArgs.desiredCsKey = "";
    FastqArgs.qual_filter = false;
    FastqArgs.b_deffmt = NULL;
    FastqArgs.b_defline = NULL;
    FastqArgs.q_deffmt = NULL;
    FastqArgs.q_defline = NULL;
    FastqArgs.single_file = false;
    FastqArgs.one_line = false;
    FastqArgs.fasta = false;
    FastqArgs.file_extension = ".fastq";

    if( fmt == NULL ) {
        return RC(rcExe, rcFileFormat, rcConstructing, rcParam, rcNull);
    }
    fmt->release = FastqDumper_Release;
    fmt->arg_desc = arg;
    fmt->add_arg = FastqDumper_AddArg;
    fmt->get_factory = FastqDumper_Factories;

    return 0;
}
