-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

separate (Sem.Walk_Expression_P.Walk_Annotation_Expression)
procedure Up_Wf_Quantifier
  (Node    : in     STree.SyntaxNode;
   E_Stack : in out Exp_Stack.Exp_Stack_Type;
   Scope   : in out Dictionary.Scopes) is
   Errors_Found                   : Boolean;
   Range_Node                     : STree.SyntaxNode;
   Predicate_Result, Range_Result : Sem.Exp_Record;
   Quantifier_Type                : Dictionary.Symbol;
begin
   -- top of E_Stack is predicate data
   -- 2nd tos E_Stack is range info iff a range node exists

   Exp_Stack.Pop (Item  => Predicate_Result,
                  Stack => E_Stack); -- result of predicate node
   Errors_Found := Predicate_Result.Errors_In_Expression;
   if not Dictionary.IsBooleanTypeMark (Predicate_Result.Type_Symbol) then -- error case
      Errors_Found := True;
      ErrorHandler.Semantic_Error
        (Err_Num   => 326,
         Reference => ErrorHandler.No_Reference,
         Position  => STree.Node_Position (Node => STree.Last_Sibling_Of (Start_Node => STree.Child_Node (Current_Node => Node))),
         Id_Str    => LexTokenManager.Null_String);
   end if;

   Range_Node :=
     STree.Next_Sibling
     (Current_Node => STree.Next_Sibling
        (Current_Node => STree.Next_Sibling (Current_Node => STree.Child_Node (Current_Node => Node))));
   -- ASSUME Range_Node = annotation_arange OR predicate
   if STree.Syntax_Node_Type (Node => Range_Node) = SP_Symbols.annotation_arange then
      -- ASSUME Range_Node = annotation_arange
      -- if the arange node exists there will be other stuff on the stack here
      Exp_Stack.Pop (Item  => Range_Result,
                     Stack => E_Stack);
      Errors_Found := Errors_Found or else Range_Result.Errors_In_Expression;

      Quantifier_Type := Dictionary.GetType (Dictionary.GetRegion (Scope));

      -- If the quantifier is over Boolean or a subtype of Boolean, then
      -- an explicit range is illegal.
      if Dictionary.TypeIsBoolean (Quantifier_Type) then
         ErrorHandler.Semantic_Error
           (Err_Num   => 412,
            Reference => ErrorHandler.No_Reference,
            Position  => STree.Node_Position (Node => Range_Node),
            Id_Str    => LexTokenManager.Null_String);

      elsif not Range_Result.Is_ARange then
         Errors_Found := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 98,
            Reference => ErrorHandler.No_Reference,
            Position  => STree.Node_Position (Node => Range_Node),
            Id_Str    => LexTokenManager.Null_String);
      elsif not Dictionary.CompatibleTypes (Scope, Quantifier_Type, Range_Result.Type_Symbol) then
         Errors_Found := True;
         ErrorHandler.Semantic_Error
           (Err_Num   => 106,
            Reference => ErrorHandler.No_Reference,
            Position  => STree.Node_Position (Node => Range_Node),
            Id_Str    => LexTokenManager.Null_String);
      end if;
   elsif STree.Syntax_Node_Type (Node => Range_Node) /= SP_Symbols.predicate then
      SystemErrors.Fatal_Error
        (Sys_Err => SystemErrors.Invalid_Syntax_Tree,
         Msg     => "Expect Range_Node = annotation_arange OR predicate in Up_Wf_Quantifier");
   end if;

   -- turn result into a Boolean type result
   Exp_Stack.Push
     (X     => Sem.Exp_Record'(Type_Symbol             => Dictionary.GetPredefinedBooleanType,
                               Other_Symbol            => Dictionary.NullSymbol,
                               Stream_Symbol           => Dictionary.NullSymbol,
                               Tagged_Parameter_Symbol => Dictionary.NullSymbol,
                               Variable_Symbol         => Dictionary.NullSymbol,
                               Param_Count             => 0,
                               Param_List              => Lists.Null_List,
                               Sort                    => Sem.Type_Result,
                               Arg_List_Found          => False,
                               Is_AVariable            => False,
                               Is_An_Entire_Variable   => False,
                               Errors_In_Expression    => Errors_Found,
                               Has_Operators           => False,
                               Is_Constant             => False,
                               Is_Static               => False,
                               Is_ARange               => False,
                               String_Value            => LexTokenManager.Null_String,
                               Value                   => Maths.NoValue,
                               Range_RHS               => Maths.NoValue),
      Stack => E_Stack);
   -- leave local scope of quantifier
   Scope := Dictionary.GetEnclosingScope (Scope);
end Up_Wf_Quantifier;
