<?php
/**
 * Forwards_Driver_qmail:: implements the Forwards_Driver API for FTP driven
 * Qmail mail servers.
 *
 * $Horde: forwards/lib/Driver/qmail.php,v 1.17.2.2 2007/01/02 13:54:06 jan Exp $
 *
 * Copyright 2001-2007 Eric Rostetter <eric.rostetter@physics.utexas.edu>
 *
 * See the enclosed file LICENSE for license information (BSDL). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Eric Rostetter <eric.rostetter@physics.utexas.edu>
 * @since   Forwards 2.2
 * @package Forwards
 */
class Forwards_Driver_qmail extends Forwards_Driver {

    /**
     * The FTP stream we open via the VFS class.
     *
     * @var VFS_ftp
     */
    var $_vfs;

    /**
     * Checks if the realm has a specific configuration.
     *
     * If not, try to fall back on the default configuration.  If
     * still not a valid configuration then exit with an error.
     *
     * @param string $realm  The realm of the user, or "default" if none.
     *                       Note: passed by reference so we can change
     *                       its value!
     */
    function checkConfiguration(&$realm)
    {
        // If no realm passed in, or no host config for the realm
        // passed in, then we fall back to the default realm.
        if (empty($realm) || empty($this->_params[$realm]['host'])) {
            $realm = 'default';
        }

        // If still no host/port, then we have a misconfigured module.
        if (empty($this->_params[$realm]['host']) ||
            empty($this->_params[$realm]['port']) ) {
            $this->_error = _("The module is not properly configured!");
            return false;
        }
        return true;
    }

    /**
     * Begins forwarding of mail for a user.
     *
     * @param string $user        The username of the user.
     * @param string $realm       The realm of the user.
     * @param string $password    The password of the user.
     * @param string $target      The email address that mail should be
     *                            forwarded to.
     * @param boolean $keeplocal  Keep a copy of forwarded mail in the local
     *                            mailbox.
     *
     * @return boolean  True on success, false otherwise.
     */
    function enableForwarding($user, $realm = 'default', $password, $target,
                              $keeplocal = false)
    {
        // Make sure the configuration file is correct.
        if (!$this->checkConfiguration($realm)) {
            return false;
        }

        // Build the ftp array to pass to VFS.
        $_args = array('hostspec' => $this->_params[$realm]['host'],
                       'port' => $this->_params[$realm]['port'],
                       'pasv' => $this->_params[$realm]['pasv'],
                       'username' => $user,
                       'password' => $password);

        // Create the VFS ftp driver.
        require_once 'VFS.php';
        $_vfs = &VFS::singleton('ftp', $_args);
        $_vfs->setParams($_args);

        // Try to login with the username/password, no realm. This
        // isn't really needed, but allows for a better error message.
        $status = $_vfs->checkCredentials();
        if (is_a($status, 'PEAR_Error')) {
            $this->_error = $status->getMessage();
            $this->_error .= '  ' .  _("Check your username and password");
            return false;
        }

        // Create the forwarding information.
        $address = '&' . $target;
        if ($keeplocal == 'on') {
            $address .= "\n./Maildir/";
        }

        // Set the forward.
        $status = $_vfs->writeData('', '.qmail', $address);
        if (is_a($status, 'PEAR_Error')) {
            $this->_error = $status->getMessage();
            return false;
        }

        // Try to change the permissions, but ignore any errors.
        $_vfs->changePermissions('', '.qmail', '0600');

        return true;
    }

    /**
     * Stops forwarding of mail for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return boolean  True on success, false otherwise.
     */
    function disableForwarding($user, $realm = 'default', $password)
    {
        if (!$this->checkConfiguration($realm)) {
            return false;
        }

        $_args = array('hostspec' => $this->_params[$realm]['host'],
                       'port' => $this->_params[$realm]['port'],
                       'pasv' => $this->_params[$realm]['pasv'],
                       'username' => $user,
                       'password' => $password);

        require_once 'VFS.php';
        $_vfs = &VFS::singleton('ftp', $_args);
        $_vfs->setParams($_args);

        // Try to login with the username/password, no realm.
        $status = $_vfs->checkCredentials();
        if (is_a($status, 'PEAR_Error')) {
            $this->_error = $status->getMessage();
            $this->_error .= '  ' .  _("Check your username and password");
            return false;
        }

        $status = $_vfs->deleteFile('', '.qmail');
        if (is_a($status, 'PEAR_Error')) {
            $this->_error = $status->getMessage();
            $this->_error .= '  ' .  _("Maybe you didn't have a forward enabled?");
            return false;
        }

        return true;
    }

    /**
     * Retrieves current target of mail redirection for a user.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return mixed  The current forwarding mail address, or false.
     */
    function currentTarget($user, $realm = 'default', $password)
    {
        // Make sure the configuration file is correct.
        if (!$this->checkConfiguration($realm)) {
            return false;
        }

        // Build the ftp array to pass to VFS.
        $_args = array('hostspec' => $this->_params[$realm]['host'],
                       'port' => $this->_params[$realm]['port'],
                       'pasv' => $this->_params[$realm]['pasv'],
                       'username' => $user,
                       'password' => $password);

        // Create the VFS ftp driver.
        require_once 'VFS.php';
        $_vfs = &VFS::singleton('ftp', $_args);
        $_vfs->setParams($_args);

        // Try to login with the username/password, no realm.
        $status = $_vfs->checkCredentials();
        if (is_a($status, 'PEAR_Error')) {
            $this->_error = $status->getMessage();
            $this->_error .= '  ' .  _("Check your username and password");
            return false;
        }

        $status = $_vfs->read('', '.qmail');
        if (is_a($status, 'PEAR_Error')) {
            $this->_error = $status->getMessage();
            return false;
        }
        return $status;
    }

    /**
     * Retrieves current state of mail redirection for a user.
     *
     * @todo This function is implemented poorly, and should be rewritten.
     *
     * @param string $user      The username of the user.
     * @param string $realm     The realm of the user.
     * @param string $password  The password of the user.
     *
     * @return mixed  'Y' if forwarding is enabled, or false otherwise.
     */
    function isEnabledForwarding($user, $realm, $password)
    {
        $yn = $this->currentTarget($user, $realm, $password);
        if ($yn) {
            return 'Y';
        } else {
            return false;
        }
    }

}
