/*
 * ----------------------------------------------------
 *
 * TCP socket based CAN message forwarding
 * for CAN-chip emulators 
 * (C) 2004  Lightmaze Solutions AG
 *   Author: Jochen Karrer
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 * ----------------------------------------------------
 */
#include <socket_can.h>
#include <stdint.h>
#include <unistd.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <netinet/in.h>
#include "fio.h"
#include "configfile.h"
#include "sgstring.h"


static void
close_connection(Connection *con)
{
	Connection *cursor,*prev;
	CanController *contr=con->canController;
        close(con->sockfd);
	if(con->rfh_is_active) {
		FIO_RemoveFileHandler(&con->rfh);
		con->rfh_is_active=0;
	}
        if(con->wfh_is_active) {
                FIO_RemoveFileHandler(&con->wfh);
                con->wfh_is_active=0;
        }
	for(prev=NULL,cursor=contr->con_list;cursor;prev=cursor,cursor=cursor->next)
        {
                if(cursor==con)     {
                        if(prev) {
                                prev->next=cursor->next;
                        } else {
                                contr->con_list=cursor->next;
                        }
			break;
                }
        }
        free(con);
	fprintf(stderr,"CAN-Bus socket connection closed\n");
        return;
}

/*
 * ------------------------------------------------------
 * Can Send: Interface function for the Chip Emulator,
 *	sends a CAN-Message to all sockets connected 
 *	a CAN Controller
 * ------------------------------------------------------
 */ 

void
CanSend(CanController *contr,CAN_MSG *msg) {
	/* To lazy for buffering and Writefilehandlers, may be added later */
	Connection *con,*next;
	int result;
	msg->id=htonl(msg->id);	
	for(con=contr->con_list;con; con=next) {
		next=con->next;
		result = write(con->sockfd,msg,sizeof(*msg));
		if(result<0) {
			if(errno==EAGAIN) {
				fprintf(stderr,"Tcp Buffer overrun, skiping CAN-Message\n");	
			} else {
				close_connection(con);
			}
		} else if (result==0) {
			close_connection(con);
		}
	}
	return;
}


static int
read_from_sock(void *cd,int mask)
{
	Connection *con = cd;
	CanController *contr=con->canController;
	int result,count=0;
	char *cbuf=(char*)&con->imsg;
	result=read(con->sockfd,cbuf+con->ibuf_wp,sizeof(CAN_MSG)-con->ibuf_wp);
	if(result>0) {
		count=result;
	} else if(result == 0) {
		close_connection(con);
		return 0;
	} else {
		if(errno==EAGAIN) {
			return 0;
		} else {
			close_connection(con);
			return 0;
		}
	}
	con->ibuf_wp+=count;
	if(con->ibuf_wp>=sizeof(CAN_MSG)) {
		con->imsg.id = ntohl(con->imsg.id);
		con->ibuf_wp=0;	
		contr->cops->receive(contr->clientData,&con->imsg);
	}
	return 0;
	
}

static void
tcp_connect(int sockfd,char *host,unsigned short port,void *cd)
{
	Connection *con = sg_new(Connection);	
	CanController *contr=cd;
	con->canController=contr;
	con->sockfd=sockfd;
	fcntl(sockfd,F_SETFL,O_NONBLOCK);
	if(contr->con_list) {
		con->next=contr->con_list;
	} else {
		con->next=NULL;
	} 
	contr->con_list=con;
	FIO_AddFileHandler(&con->rfh,sockfd,FIO_READABLE,read_from_sock,con);
	con->rfh_is_active=1;
	fprintf(stderr,"New socket connection to emulated CAN-Controller\n");
		
}

/*
 * ----------------------------------------------------------
 * Create a new Cancontroller with a Listening TCP-Socket
 * ----------------------------------------------------------
 */
CanController * 
CanSocketInterface_New(CanChipOperations *cops,const char *name,void *clientData) 
{
	CanController *contr = sg_new(CanController);	
	int port;
	if(Config_ReadInt32(&port,name,"port")<0) {
		free(contr);
		return 0;
	}
	contr->cops=cops;	
	contr->clientData=clientData;
	contr->listen_fd=FIO_InitTcpServer(&contr->tserv,tcp_connect,contr,"127.0.0.1",port);
	if(contr->listen_fd<0) {
		fprintf(stderr,"Can not open TCP Listening Port %d for CAN-Emulator: ",port);
		perror("");
		free(contr);
		return NULL;
	}
	return contr;
}
void 
CanStopRx(CanController *contr) {
	Connection *cursor;
	for(cursor=contr->con_list;cursor;cursor=cursor->next)
        {
		if(cursor->rfh_is_active) {
			cursor->rfh_is_active=0;
			FIO_RemoveFileHandler(&cursor->rfh);
		}
        }
}
void 
CanStartRx(CanController *contr) {
	Connection *cursor;
	for(cursor=contr->con_list;cursor;cursor=cursor->next)
        {
		if(!cursor->rfh_is_active) {
			cursor->rfh_is_active=1;
			FIO_AddFileHandler(&cursor->rfh,cursor->sockfd,FIO_READABLE,read_from_sock,cursor);
		}
        }
}
