/*
 * --------------------------------------------------------
 * Emulation of Samsung NAND Flash 
 *
 * (C) 2007 Jochen Karrer 
 *   Author: Jochen Karrer
 *
 * State: Nothing is implemented
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * --------------------------------------------------------
 */

#include <stdlib.h>
#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <unistd.h>
#include "sgstring.h"
#include "nand.h"
#include "configfile.h"

typedef struct FlashDescription {
	char *type;
	int addrbytes;
	int addrbits;
	uint64_t size;
	int planes;
	int blocks;
	int pages_per_block;
} FlashDescription;

static FlashDescription flash_types[] = {
	{ 
		.type = "K9F1208",
		.addrbytes = 4,
		.addrbits = 26,
		.size = 528 * 32 * 4096,
		.planes = 4,
		.pages_per_block = 32,
		.blocks = 4096
	},
};
/*
 * CMD sequences
 *	Read 1 		0x00/0x01	  (A/B Area 0-255,256-511)
 *	Read 2  	0x50		  (C Area 512-527)
 *	Reset		0xff
 *	Page PGM True	0x80	0x10
 * 	Page PGM Dummy	0x80	0x11
 *	Copyback PGM True  0x00 0x8a 0x10 
 *	Copyback PGM Dummy 0x03 0x8a 0x11
 *	Block Erase	0x60 	0xd0
 *	Multi Plane Block Erase 60--60 	0xd0
 *	Read Status		0x70
 *	Read Multi Plane Status	(0x71)
 * 
 */

#define STATE_IDLE		(0)
#define STATE_CMD		(1)
#define STATE_COLUMN		(2)
#define STATE_ADDR0		(3)
#define STATE_ADDR1		(4)
#define STATE_ADDR_COMPLETE	(5)
#define STATE_PGM_PAGE		(7)
#define STATE_COPYBACK_PGM	(8)
#define STATE_BLOCK_ERASE	(9)
#define STATE_MP_BLOCK_ERASE	(10)
/*
 * This is read1/read2 cmd
 */
static void
cmd_read(NandFlashDevice *nf,uint8_t *data,int ctrl) 
{
	switch(nf->state) {
		case STATE_CMD:
			nf->addr = *data & 0x7f;
			nf->state = STATE_COLUMN;
			break;

		case STATE_COLUMN:
			nf->addr |= *data << 9;
			nf->state = STATE_ADDR0;
			break;

		case STATE_ADDR0:
			nf->addr |= *data << 17;
			nf->state = STATE_ADDR1;
			break;

		case STATE_ADDR1:
			nf->addr |= *data << 25;
			nf->state = STATE_ADDR_COMPLETE;
			nf->transfer_count = 0;
			/* calculate plane from address */
			// read to internal buffer of block
			break;

		case STATE_ADDR_COMPLETE:
			//do_write_cmd(nf,*data,ctrl);
			break;
			
		default:
			fprintf(stderr,"Nand Flash: Illegal state %d\n",nf->state);
			break;
	}
}
void
NandFlash_Access(NandFlashDevice *nf,uint8_t *data,int ctrl) 
{
	
	if(ctrl & NFCTRL_nCE) {
		/* chip not enabled */
		return; 
	}
	if(!(ctrl & NFCTRL_nWE)) {
		if(ctrl & NFCTRL_CLE) {
			nf->state = STATE_CMD;
			nf->cmd = *data;
		} else if(ctrl & NFCTRL_ALE) {
			nf->addr = ((nf->addr << 8) | *data ) &  nf->addrmask;
		} else if(nf->state != STATE_IDLE) {

			switch(nf->cmd) {
				case 0x00:
				case 0x01:
				case 0x50:
					cmd_read(nf,data,ctrl);
					break;

					//nf->page_pointer = 512;
					break;

				case 0x10:
					if(nf->state == STATE_PGM_PAGE) {
						// do_program
						nf->state = STATE_IDLE;
					} else if (nf->state == STATE_COPYBACK_PGM) {
						// do copyback
						nf->state = STATE_IDLE;
					}
					break;

				case 0x11:
					if(nf->state == STATE_PGM_PAGE) {
						nf->state = STATE_IDLE;
					} else if (nf->state == STATE_COPYBACK_PGM) {
						nf->state = STATE_IDLE;
					}
					break;

				case 0x70:	/* Read status */
				case 0x71:	/* Read multp. status */
					break;

				case 0x80:	/* Page PGM */
					nf->state = STATE_PGM_PAGE;
					break;
			}
			return;
		}
	} else {
		// do_read_cmd(nf,*data,ctrl);
	} 
}

/*
 * -----------------------------------------------------------------
 * A nandflash controller can be registered here
 * It will receive the data sent by the Nand Flash and is
 * allowed to send Nandflash commands to the device 
 * -----------------------------------------------------------------
 */
void
NandFlash_RegisterController(NandFlashDevice *nfd,NandFlashController *nfc) 
{
	if(nfd->nfc) {
		fprintf(stderr,"NandFlash: Bug, only one NandflashController allowed\n");
		exit(1);
	}
	nfd->nfc = nfc;
}

NandFlashDevice *
NandFlash_New(const char *name) 
{
	int i,nr_types;
	char *type,*imagedir,*filename;
	FlashDescription *fldescr;
	NandFlashDevice *nfd;
	type = Config_ReadVar(name,"type");	
	if(!type) {
		fprintf(stderr,"No type for Nand Flash \"%s\". Creating nothing\n",name);
		return NULL;
	}
	nr_types = sizeof(flash_types) / sizeof(FlashDescription);
	for(i=0;i < nr_types;i++) {
		fldescr = &flash_types[i];
		if(!strcmp(type,fldescr->type)) {
			break;
		}
	}
	if(i == nr_types) {
		fprintf(stderr,"Flash type \"%s\" not found\n",type);
		exit(1);
	}
	imagedir = Config_ReadVar("global","imagedir");
	if(!imagedir) {
		fprintf(stderr,"No directory for NAND Flash diskimage given\n");
		return NULL;
	}
	filename = alloca(strlen(imagedir) + strlen(name) + 50);
	sprintf(filename,"%s/%s.img",imagedir,name);
	nfd = sg_new(NandFlashDevice);
	nfd->nfplanes = fldescr->planes;
	nfd->addrmask = 1<<(fldescr->addrbits-1);
	nfd->nfplane = sg_calloc(nfd->nfplanes * sizeof(NFPlane));
	nfd->disk_image = DiskImage_Open(filename,fldescr->size,DI_RDWR | DI_CREAT_FF | DI_SPARSE); 
	nfd->type = sg_strdup(type);
	nfd->busySig = SigNode_New("%s.busy",name);
	return nfd;
}
