/*
 * ----------------------------------------------------------------------
 * 32 Bit CRC calculations
 * (C) 2007 Jochen Karrer
 *   Author: Jochen Karrer
 *
 * state: working, no interface defined
 *
 *  This program is free software; you can distribute it and/or modify it
 *  under the terms of the GNU General Public License (Version 2) as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope it will be useful, but WITHOUT
 *  ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 *  FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  59 Temple Place - Suite 330, Boston MA 02111-1307, USA.
 *
 * ----------------------------------------------------
 */

#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "compiler_extensions.h"


static uint32_t 
bitreverse32(uint32_t x)
{
        x = (x >> 16) | (x << 16);
        x = (x >> 8 & 0x00ff00ff) | (x << 8 & 0xff00ff00);
        x = (x >> 4 & 0x0f0f0f0f) | (x << 4 & 0xf0f0f0f0);
        x = (x >> 2 & 0x33333333) | (x << 2 & 0xcccccccc);
        x = (x >> 1 & 0x55555555) | (x << 1 & 0xaaaaaaaa);
        return x;
}

static inline uint8_t 
bitreverse8(uint8_t x) {
        x = (x >> 4 & 0x0f) | (x << 4 & 0xf0);
        x = (x >> 2 & 0x33) | (x << 2 & 0xcc);
        x = (x >> 1 & 0x55) | (x << 1 & 0xaa);
        return x;
}

/*
 * --------------------------------------------------
 * Bitwise version for init of crctabs
 * Takes only one Byte of data because it is
 * normally only used during initialization
 * --------------------------------------------------
 */
static uint32_t 
CRC32_bitwise(uint32_t crc,uint8_t val,uint32_t poly)
{
	int i;
	for(i=7;i>=0;i--) {
		int carry = !!(crc & 0x80000000);
		int inbit = !!(val & (1<<i));
		crc = crc<<1;
		if(inbit ^ carry) {
			crc = crc ^ poly;
		}
	}
	return crc;
}

/*
 * ------------------------------------------------------
 * The bitreverse version
 * ------------------------------------------------------
 */
static uint32_t 
CRC32rev_bitwise(uint32_t crc,uint8_t val,uint32_t poly)
{
	int i;
	for(i=0;i<8;i++) {
		int carry = !!(crc & 0x80000000);
		int inbit = !!(val & (1<<i));
		crc = crc<<1;
		if(inbit ^ carry) {
			crc = crc ^ poly;
		}
	}
	return crc;
}

static uint32_t tab_04C11DB7[256];
static uint32_t tab_04C11DB7_rev[256];
static uint32_t tab_1EDC6F41[256];
static uint32_t tab_1EDC6F41_rev[256];
static uint32_t tab_000000AF[256];
static int tabs_initialized = 0;

/*
 * ------------------------------------------------------------------
 * Initialize one CRC tab for a given polynom. When reflected
 * input is used all calculations are done bit reversed.
 * ------------------------------------------------------------------
 */
static void
CRC32Tab_Init(uint32_t *crctab,uint32_t poly,int reverse) 
{
	int i;
	for(i=0;i<256;i++) {
		if(reverse) {
			crctab[i] = bitreverse32(CRC32rev_bitwise(0,i,poly));
		} else {
			crctab[i] = CRC32_bitwise(0,i,poly);
		}
	}
}

/*
 * ------------------------------------------------------------------
 * Initialize all crc tabs 
 * ------------------------------------------------------------------
 */
static void
CRC32Tabs_Init(void) 
{
	CRC32Tab_Init(tab_04C11DB7,0x04C11DB7,0);
	CRC32Tab_Init(tab_04C11DB7_rev,0x04C11DB7,1);
	CRC32Tab_Init(tab_1EDC6F41,0x1EDC6F41,0);
	CRC32Tab_Init(tab_1EDC6F41_rev,0x1EDC6F41,1);
	CRC32Tab_Init(tab_000000AF,0x000000AF,0);
	tabs_initialized = 1;
}

static uint32_t CRC32(uint32_t crc,uint32_t *crctab,void *vdata,int len);
static uint32_t CRC32rev(uint32_t crc,uint32_t *crctab,void *vdata,int len);

/*
 * -------------------------------------------------------------
 * Self test for correctness. Values are taken from
 * "A PAINLESS GUIDE TO CRC ERROR DETECTION ALGORITHMS" 
 * from Ross N. Williams
 * -------------------------------------------------------------
 */
static void
CRC32Tabs_Test(void) {
	uint32_t crc;
	char *bla = "123456789";
	/* CRC-32/ADCCP Pkzip */
	crc = CRC32rev(0xffffffff,tab_04C11DB7_rev,bla,strlen(bla)) ^ 0xffffffff;

	if(crc != 0xCBF43926) {
		fprintf(stderr,"CRC Test1 failed: %08x\n",crc);
		exit(1);
	}

	/* CRC-32/Posix */
	crc = CRC32(0x0,tab_04C11DB7,bla,strlen(bla)) ^ 0xffffffff;
   	if(crc != 0x765E7680) {
		fprintf(stderr,"CRC Test2 failed: %08x\n",crc);
		exit(1);
	}

	crc = CRC32rev(0xffffffff,tab_1EDC6F41_rev,bla,strlen(bla)) ^ 0xffffffff;
   	if(crc != 0xE3069283) {
		fprintf(stderr,"CRC Test3 failed: %08x\n",crc);
		exit(1);
	}

	/* CRC-32/XFER */
	crc = CRC32(0x0,tab_000000AF,bla,strlen(bla));
   	if(crc != 0xBD0BE338) {
		fprintf(stderr,"CRC Test4 failed: %08x\n",crc);
		exit(1);
	}

	

}

static uint32_t
CRC32rev(uint32_t crc,uint32_t *crctab,void *vdata,int len)
{
	uint8_t *data = (uint8_t *)vdata;
        uint8_t index;
        int i;
	if(unlikely(tabs_initialized == 0)) {
		CRC32Tabs_Init();	
		CRC32Tabs_Test();	
	}
        for(i=0;i<len;i++) {
                index = ( crc  ) ^ data[i];
                crc = ( crc >> 8 ) ^ crctab[index];
        }
	return crc;
}

static uint32_t
CRC32(uint32_t crc,uint32_t *crctab,void *vdata,int len)
{
	uint8_t *data = (uint8_t *)vdata;
        uint8_t index;
        int i;
	if(unlikely(tabs_initialized == 0)) {
		CRC32Tabs_Init();	
		CRC32Tabs_Test();	
	}
        for(i=0;i<len;i++) {
                index = ( crc >> 24 ) ^ data[i];
                crc = ( crc << 8 ) ^ crctab[index];
        }
	return crc;
}

/*
 * ------------------------------------------------
 * Main Programm for standalone test
 * ------------------------------------------------
 */
#ifdef TEST
int 
main() {
	CRC32Tabs_Init();
	CRC32Tabs_Test();
	exit(0);
}
#endif
