/*
 * PumpViewer.h -- Pump Viewer Definitions
 * Copyright (C) 2003 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef _KINO_PUMP_VIEWER_
#define _KINO_PUMP_VIEWER_

#include <pthread.h>

#include <frame.h>
#include <Threader.h>
#include <DVPumpProvider.h>

/** Interface for obtaining the next frame - this is given to the audio
	player on its initialisation. When the input source is finished, this
	method will return NULL.
*/

class FrameSupplier
{
	public:
		// Wait for and return a frame from the pump
		virtual Frame *GetFrame( ) = 0;
		// Get the last frame position associated to the player
		virtual int GetLastFramePosition( ) = 0;
		// Toggle the pause state
		virtual bool TogglePause( ) = 0;
		// Determine if we are paused
		virtual bool IsPaused( ) = 0;
		// Obtain the current playback seed
		virtual double GetSpeed( ) = 0;
};

/** The player interface - note that no implementation is provided or
	assumued by the PumpViewer. 

	The theory is this:

	* When the first frame becomes available, the ShowInit and AudioInit methods
	  are called. These methods set up the video window and audio playback
	  mechanism respectively.
	* The audio playback mechanism is responsible for getting all the frames
	  from the supplier in real time - each time it gets a frame, an internal
	  event (in the PumpViewer) is triggered and if its being waited on, the
	  ShowFrame method is called (note: this means that ShowFrame does not have
	  to complete in real time - frames will simply be dropped if the machine
	  is not capable of rendering them all).
	* The HandleEvents is provided to allow the displayer to pick up events 
	  (such as keyboard or mouse events) associated to the playback mechanism.
	  If it returns false, the pump viewer will close.
	* ShowClose and AudioClose are called to clean up the associated resources
	  when the pump viewer is no longer required.
*/

class FramePlayer
{
	public:
		// Initialise the video previewer
		virtual bool ShowInit( Frame &frame ) = 0;
		// Show a frame
		virtual bool ShowFrame( Frame &frame ) = 0;
		// Close the video previewer
		virtual void ShowClose( ) = 0;
		// Initialise the audio handler
		virtual bool AudioInit( Frame &frame, FrameSupplier *supplier ) = 0;
		// Close the audio handler
		virtual void AudioClose( ) = 0;
		// Handle keyboard/mouse events
		virtual bool HandleEvents( ) = 0;
		// Passthrough implementation
		virtual void PassThrough( Frame &frame ) = 0;		
};

/** The pump viewer class takes the DVPump to be viewed and the FramePlayer
	in the constructor and provides the necessary code to drive the player.
*/

class PumpViewer :
	public Threader,
	protected FrameSupplier
{
	public:
		// Constructor for the pump viewer
		PumpViewer( DVPumpProvider &_pump, FramePlayer &_player ) :
			player( _player ),
			pump( _pump ),
			available( NULL ),
			last_played( NULL )
		{
			pthread_mutex_init( &condition_mutex, NULL );
			pthread_cond_init( &condition, NULL );
		}
		
		// Destructor for the pump viewer
		virtual ~PumpViewer( )
		{
			pthread_mutex_destroy( &condition_mutex );
			pthread_cond_destroy( &condition );
		}
	
		// Drives the FramePlayer until input is exhausted or user quits
		void Thread( )
		{
			if ( pump.GetOutputAvailable( ) > 0 )
			{
				Frame &frame = pump.GetOutputFrame( );
			
				bool running = player.ShowInit( frame );
				
				if ( running )
				{
					running = player.AudioInit( frame, this );
				
					if ( running )
					{
						Frame *ptr = NULL;
						while ( running && ( ptr = GetViewFrame( ) ) != NULL )
							running = player.ShowFrame( *ptr ) && player.HandleEvents( );
						
						player.AudioClose( );
					}
					
					player.ShowClose( );
				}
				
				pump.ClearPump( );
			}
		}
		
		// Implementation of the FrameSupplier GetFrame method
		Frame *GetFrame( )
		{
			Frame *frame = NULL;
			int count = pump.GetOutputAvailable( );
			
			if ( count > 0 )
			{
				frame = &( pump.GetOutputFrame( ) );
				last_played = frame;
				if ( !pump.IsInputCritical( ) )
				{
					pthread_mutex_lock( &condition_mutex );
					available = frame;
					pthread_cond_broadcast( &condition );
					pthread_mutex_unlock( &condition_mutex );
				}
				pump.QueueInputFrame( );
				player.PassThrough( *frame );
			}
			
			if ( frame == NULL )
			{
				pthread_mutex_lock( &condition_mutex );
				available = NULL;
				pthread_cond_broadcast( &condition );
				pthread_mutex_unlock( &condition_mutex );
			}
			
			return frame;
		}

		// Get the position in the playlist of the last frame played
		int GetLastFramePosition( )
		{
			if ( last_played != NULL )
				return last_played->playlist_position;
			else
				return -1;
		}

		bool TogglePause( ) 
		{
			return pump.TogglePause( );
		}
		
		bool IsPaused( )
		{
			return pump.IsPaused( );
		}
		
		void SetBlockable( bool _blockable )
		{
			pump.SetBlockable( _blockable );
		}
		
		double GetSpeed( )
		{
			return pump.GetSpeed( );
		}

	private:
		FramePlayer &player;
		DVPumpProvider &pump;
		Frame *available;
		Frame *last_played;
		pthread_cond_t condition;
		pthread_mutex_t condition_mutex;
		
		// Gets the currently available display frame or waits til one is available
		Frame *GetViewFrame( )
		{
			pthread_mutex_lock( &condition_mutex );
			Frame *ptr = available;			
			if ( ptr == NULL && ( pump.PumpIsNotCleared( ) || pump.GetOutputAvailable( ) ) )
			{
				pthread_cond_wait( &condition, &condition_mutex );
				ptr = available;
			}
			available = NULL;
			pthread_mutex_unlock( &condition_mutex );
			return ptr;
		}
};

#endif
