/***************************************************************************
    smb4kpreviewer  -  This class queries a remote share for a preview
                             -------------------
    begin                : Mo Mai 28 2007
    copyright            : (C) 2007 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KPREVIEWER_H
#define SMB4KPREVIEWER_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <qobject.h>
#include <qstring.h>
#include <qptrqueue.h>

// KDE includes
#include <kprocess.h>

// forward declarations
class Smb4KPreviewItem;

/**
 * This class is part of the core of Smb4K. It queries a remote SMB share for
 * a preview and returns the result.
 *
 * @author Alexander Reinholdt <dustpuppy@users.berlios.de>
 */

class Smb4KPreviewer : public QObject
{
  Q_OBJECT

  public:
    /**
     * The constructor
     *
     * @param parent            The parent object
     *
     * @param name              The name of this object
     */
    Smb4KPreviewer( QObject *parent = 0, const char *name = 0 );

    /**
     * The destructor
     */
    ~Smb4KPreviewer();

    /**
     * Get a preview of the contents of @p item.
     *
     * In the case that @p item represents a 'homes' share, the user will be
     * prompted for the user name with which he wants to log in and the share
     * name of @p item will be set to the result.
     *
     * @param item              The item for which a preview should be
     *                          requested.
     *
     * @returns TRUE if einter the share name is not 'homes' or if it could
     * successfully be set to a user name. Otherwise it returns FALSE.
     */
    bool preview( Smb4KPreviewItem *item );

    /**
     * Using this function, you can find out whether the previewer is running
     * at the moment.
     *
     * @returns TRUE if the previewer is running or FALSE otherwise.
     */
    bool isRunning() { return m_working; }

    /**
     * Abort any action the previewer is performing at the moment and clear
     * the queue.
     */
    void abort();

  signals:
    /**
     * Emits the state the previewer is in
     *
     * @param state             The state
     */
    void state( int state );

    /**
     * Emits the preview after the process exited successfully. Get the contents
     * of the remote share by looping through the Smb4KPreviewItem::contents() list.
     *
     * @param item              The item for which the preview was received.
     */
    void result( Smb4KPreviewItem *item );

    /**
     * This signal is emitted if an error occurred.
     */
    void failed();

  protected:
    /**
     * Reimplemented from QObject
     */
    void timerEvent( QTimerEvent *e );

  protected slots:
    /**
     * This slot receives output from stdout.
     *
     * @param proc          The process
     *
     * @param buf           The buffer
     *
     * @param len           The length of the buffer
     */
    void slotReceivedStdout( KProcess *proc, char *buf, int len );

    /**
     * This slot receives output from stderr.
     *
     * @param proc          The process
     *
     * @param buf           The buffer
     *
     * @param len           The length of the buffer
     */
    void slotReceivedStderr( KProcess *proc, char *buf, int len );

    /**
     * Is called, when the KProcess exited.
     *
     * @param proc          The process that exited
     */
    void slotProcessExited( KProcess *proc );

  private:
    /**
     * The KProcess object
     */
    KProcess *m_proc;

    /**
     * The output buffer
     */
    QString m_buffer;

    /**
     * This is the pointer to the Smb4KPreviewItem that's
     * processed.
     */
    Smb4KPreviewItem *m_item;

    /**
     * Indicates whether the previewer is running or not.
     */
    bool m_working;

    /**
     * This pointer queue holds the pointers to the Smb4KPreviewItem
     * objects that are to be processed.
     */
    QPtrQueue<Smb4KPreviewItem> m_queue;

    /**
     * The timer id
     */
    int m_timer_id;
};

#endif
