/* This file is part of SLgtk, S-Lang bindings for GTK+ widget set % {{{
 *
 * Copyright (C) 2003-2005 Massachusetts Institute of Technology 
 * Copyright (C) 2002 Michael S. Noble <mnoble@space.mit.edu>
 *
 * This software was partially developed by the MIT Center for Space
 * Research under contract SV1-61010 from the Smithsonian Institution.
 * 
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * the supporting documentation, and that the name of the Massachusetts
 * Institute of Technology not be used in advertising or publicity
 * pertaining to distribution of the software without specific, written
 * prior permission.  The Massachusetts Institute of Technology makes
 * no representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 * 
 * THE MASSACHUSETTS INSTITUTE OF TECHNOLOGY DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE MASSACHUSETTS
 * INSTITUTE OF TECHNOLOGY BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 }}} */

#include "gtkextra/gtkextra.h"
#include "module.h"
#include "gtkextra_glue.c"

static void sl_gtk_plot_canvas_put_polygon(void) /*{{{ */
{
   GdkColor fg, bg;
   void *fgp = &fg, *bgp = &bg;
   int fillrule, style;
   unsigned int i;
   float width;
   double *x, *y;
   GtkPlotCanvas      *canvas;
   SLang_Array_Type   *slx        = NULL;
   SLang_Array_Type   *sly        = NULL;
   GtkPlotPoint       *points     = NULL;
   GtkPlotCanvasChild *poly       = NULL;
   Slirp_Opaque	      *canvas_o   = NULL;

   if (usage_err(8,"reg = _gtk_plot_canvas_put_polygon(canvas, x, y, "
	"linestyle, linewidth, fgcolor, bgcolor, fillrule)")	  	||
	-1 == SLang_pop_integer(&fillrule)			  	||
	-1 == pop_nullable(0, (void**)&bgp,(void**)GdkColor_Layout)	||
	-1 == pop_nullable(0, (void**)&fgp,(void**)GdkColor_Layout)	||
	-1 == SLang_pop_float(&width)				  	||
	-1 == SLang_pop_integer(&style)				  	||
	-1 == SLang_pop_array_of_type(&sly,SLANG_DOUBLE_TYPE)	  	||
	-1 == SLang_pop_array_of_type(&slx,SLANG_DOUBLE_TYPE)	  	||
	-1 == SLang_pop_opaque(GtkOpaque_Type,(void*)&canvas, &canvas_o)||
	slx->num_elements != sly->num_elements				||
	slx->num_elements < 2)
   {
	SLang_verror(SL_INTRINSIC_ERROR,
			"error popping or validating polygon, check input");
	goto cleanup;
   }


   if ((points = g_new(GtkPlotPoint,slx->num_elements)) == NULL) {
	SLang_verror(SL_INTRINSIC_ERROR,"out of memory");
	goto cleanup;
   }

   x = (double*)slx->data; y = (double*)sly->data; 
   for (i=0; i < slx->num_elements; i++) {
        points[i].x = x[i];
        points[i].y = y[i];
   }

   if ( (poly = gtk_plot_canvas_put_polygon(canvas, points, i, style, width,
	       					fgp, bgp, fillrule)) == NULL)
	SLang_verror(SL_INTRINSIC_ERROR,"could not create polygon");

   cleanup:

   SLang_free_array(sly);
   SLang_free_array(slx);
   SLang_free_opaque(canvas_o);
   if (fgp) SLang_free_cstruct((VOID_STAR)fgp,GdkColor_Layout);
   if (bgp) SLang_free_cstruct((VOID_STAR)bgp,GdkColor_Layout);
   g_free(points);
   (void) SLang_push_opaque (GtkOpaque_Type, poly, 0);
} /*}}}*/

static void sl_gtk_plot_axis_get_title (void)  /*{{{*/
{
   int which;
   GtkPlot* plot;
   char* result = NULL;
   Slirp_Opaque* plot_o = NULL;

   if (SLang_Num_Function_Args == 2 &&
	SLang_pop_int((int*)&which) == 0 &&
	SLang_pop_opaque(GtkOpaque_Type, (void**)&plot, &plot_o) == 0) {

	if (plot != NULL) {
	   GtkPlotAxis *axis = gtk_plot_get_axis(plot, which);
	   if (axis != NULL)
		result = axis->title.text;
	}
   }
   else
	SLang_verror(SLEU, "_gtk_plot_axis_get_title(GtkPlot,int)");

   if (result == NULL) result = "";
   SLang_free_opaque(plot_o);
   (void) SLang_push_string ((char*) result);
} /*}}}*/

static void polygon_filter(void) /*{{{*/
{
   register double xinters;
   register double x, y;		/* current point being tested */
   register double v1x, v1y;		/* previous vertex */
   register double v2x, v2y;		/* current  vertex */
   double *pointx, *pointy;		/* points to test */
   double *vertx, *verty;		/* polygon vertices to test against */
   int exclude = 0;
   SLang_Array_Type *slpointx = NULL;
   SLang_Array_Type *slpointy = NULL;
   SLang_Array_Type *slvertx  = NULL;
   SLang_Array_Type *slverty  = NULL;
   SLang_Array_Type *slresult = NULL;
   unsigned char *result;
   int vertex, point, nvertices, npoints, num_intersections;

   if (usage_err(4,"result = _polygon_filter(x, y, poly_x, poly_y [,exclude])")	
	|| (SLang_Num_Function_Args == 5 && -1 == SLang_pop_int(&exclude))
	|| -1 == SLang_pop_array_of_type(&slverty,SLANG_DOUBLE_TYPE)
	|| -1 == SLang_pop_array_of_type(&slvertx,SLANG_DOUBLE_TYPE)
	|| -1 == SLang_pop_array_of_type(&slpointy,SLANG_DOUBLE_TYPE)
	|| -1 == SLang_pop_array_of_type(&slpointx,SLANG_DOUBLE_TYPE)
	|| slpointx->num_elements != slpointy->num_elements
	|| slvertx->num_elements != slverty->num_elements
	|| slvertx->num_elements < 2)
   {
	SLang_verror(SL_INTRINSIC_ERROR,
		"error popping or validating points/polygon, check input");
	goto cleanup;
   }

   nvertices = slvertx->num_elements;
   npoints = slpointx->num_elements;
   pointx = slpointx->data; vertx = slvertx->data;
   pointy = slpointy->data; verty = slverty->data;
   result  = g_new(unsigned char, npoints);

   if (npoints > 1) {
	slresult = SLang_create_array(SLANG_UCHAR_TYPE, 0, result, &npoints, 1);
	if (slresult == NULL) {
	   SLang_verror(SL_INTRINSIC_ERROR,"could not create result array");
	   goto cleanup;
	}
   }

   /* Polygon interior test using the classic even/odd ray intersection.
    * Implementation adapted from Paul Bourke
    *		astronomy.swin.edu.au/~pbourke/geometry/insidepoly 
    * and vectorized for S-Lang */
   for (point = 0; point < npoints; point++)  {

	x = pointx[point];
	y = pointy[point];
	v1x = vertx[0];
	v1y = verty[0];
	num_intersections = 0;

	for (vertex = 1; vertex <= nvertices; vertex++) {

	   if (v1x == x && v1y == y) {
		num_intersections = 1;		/* include vertices */
		break;
	   }

	   v2x = vertx[vertex % nvertices];
	   v2y = verty[vertex % nvertices];
	   if (y > MIN(v1y, v2y)) {
		if (y <= MAX(v1y, v2y)) {
		   if (x <= MAX(v1x,v2x)) {
			if (v1y != v2y) {
			   xinters = (y - v1y)*(v2x - v1x)/(v2y - v1y) + v1x;
			   if (v1x == v2x || x <= xinters)
				num_intersections++;
			}
		   }
		}
	   }
	   v1x = v2x;
	   v1y = v2y;
	}

	if (exclude)
	   result[point] = ! (num_intersections % 2);
	else
	   result[point] = (num_intersections % 2);
   }

   if (npoints > 1)
	(void) SLang_push_array(slresult,1);
   else {
	(void) SLang_push_uchar(result[0]);
	g_free(result);
   }

   cleanup:

   SLang_free_array(slpointy);
   SLang_free_array(slpointx);
   SLang_free_array(slverty);
   SLang_free_array(slvertx);
} /*}}}*/

static SLang_Intrin_Fun_Type Manually_Coded_Funcs[] = /*{{{*/
{
   MAKE_INTRINSIC_0("_gtk_plot_canvas_put_polygon",
				sl_gtk_plot_canvas_put_polygon,V),
   MAKE_INTRINSIC_0("_gtk_plot_axis_get_title", sl_gtk_plot_axis_get_title, V),
   MAKE_INTRINSIC_0("_polygon_filter", polygon_filter, V),

   SLANG_END_INTRIN_FUN_TABLE
}; /*}}}*/

/* Module Initialization {{{ */
SLANG_MODULE(gtkextra);
int init_gtkextra_module_ns(char *ns_name)
{
   SLang_NameSpace_Type *ns;

   if (slang_abi_mismatch()) return -1;

   ns = SLns_create_namespace (ns_name);
   if (ns == NULL  || (slns = SLmalloc(strlen(ns_name)+1)) == NULL)
	return -1;

   slirp_debug_pause("gtkextra");
   if (allocate_reserved_opaque_types() == -1)
	return -1;

   patch_ftable(gtkextra_Funcs, O, GtkWidget_Type);
   patch_ftable(Manually_Coded_Funcs, O, GtkWidget_Type);

   if (-1 == SLns_add_intrin_fun_table(ns, gtkextra_Funcs," __GTKEXTRA__"))
	return -1;

   if (-1 == SLns_add_intrin_fun_table (ns, Manually_Coded_Funcs, NULL))
	return -1;

   if (-1 == SLns_add_iconstant_table (ns, gtkextra_IConsts, NULL))
	return -1;

   return 0;
}
/* }}} */
