/////////////////////////////////////////////////////////////////////////////
// File:        coreruntime.h
// Author:      Cesar Mauri Loba (cesar at crea-si dot com)
// Copyright:   (C) 2010 Cesar Mauri Loba - CREA Software Systems
//
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
/////////////////////////////////////////////////////////////////////////////
#ifndef SPCORE_CORERUNTIME_H
#define SPCORE_CORERUNTIME_H

#include "spcore/libexports.h"
#include "spcore/baseobj.h"
#include "config.h"
/*

One of the goals of SITPLUS is to become GUI toolkit agnostic.
Currently it uses wxWidgets and most modules provide GUI's based on this 
toolkit, but the core itself has not an strong dependency (component's 
method GetGUI expects wxWindows and wxPanel types though this is easyly 
changeable by just defining a neutral window type, the core itself 
provides methods such as InitGUISupport, RunMessageLoop and 
CleanupGUISupport which are based on wx).

TODO: provide an interface for non-wx apps and/or other toolkits
and set this define as a build parameter
*/
#ifndef ENABLE_WXWIDGETS
#error "Currently wxWidgets is necessary"
#endif

namespace spcore {

// Forward class declarations
template<class T> class IIterator;
class ITypeFactory;
class CTypeAny;
class IComponentFactory;
class IComponent;
class IModule;
class IInputPin;
class IOutputPin;
class IConfiguration;
class IPaths;
class ILogTarget;

// Core runtime services interface. Call:
//	getSpCoreRuntime
//to obtain a poiinter to it
class ICoreRuntime {
public:
	virtual ~ICoreRuntime() {}

    // Resolves the runtime ID of a named type. Returns the ID or TYPE_INVALID
    // if the specified type doesn't exists.
    virtual int ResolveTypeID(const char * name) = 0;

    virtual SmartPtr<IIterator<ITypeFactory*> > QueryTypes() = 0;

    // Creates a new type instance for a given type name. Returns a pointer
    // to the new instance or NULL if the type doesn't exists or another error
    // occurred.
    virtual SmartPtr<CTypeAny> CreateTypeInstance(const char * typeName) = 0;

    virtual SmartPtr<CTypeAny> CreateTypeInstance(int id) = 0;

    virtual SmartPtr<IIterator<IComponentFactory*> > QueryComponents() = 0;

    virtual SmartPtr<IComponent> CreateComponent(const char * typeName, const char * name, int argc, const char * argv[]) = 0;

	/**
		Create an instance of an output pin

		\param type : string containing the name of the type of the pin
		\param name : string with the name the pin will be given
		\param locked : tells whether to construct a looked (synchronized) pin or not

		\return an smart ptr to the new pin which can be NULL if error
	*/
	virtual SmartPtr<IOutputPin> CreateOutputPin(const char* type, const char* name, bool locked) = 0;

	///**
    //
    //Register a module inside spcore
    //
    //\return 0 on success
    //\return -1 : module already registered
    //\return -2 : wrong core version
    //\return -3 : type name already registered
    //\return -4 : component already registered
    //*/
    virtual int RegisterModule(SmartPtr<IModule> module) = 0;

	///**
	//Tries to load and register a module.
	//\param path to the shared library to load
	//\return 0 on success, <0 otherwise
	// -1 : module already registered
	// -2 : wrong core version
	// -3 : type name already registered
	// -4 : component already registered
	// -5 : defective element provided in library (e.g. anonymous factory)
	// -6 : library not found
	// -7 : wrong library format (symbol not found)
	// -8 : no module
	//*/
    virtual int LoadModule(const char * name, const char * dir = NULL) = 0;

	/**
		Log severity levels
		- LOG_DEBUG:
		This is the most verbose logging level. Only for development and testing.

		- LOG_INFO:
		The Information level is typically used to output information that is useful
		to the running and management of the system.

		- LOG_WARNING:
		Warning is often used for handled 'exceptions' or other important log events.

		- LOG_ERROR:
		Error is used to log all unhandled exceptions which prevent the application
		to run properly.

		- LOG_FATAL:
		Fatal is reserved for special exceptions/conditions where no recovery is possible.
	*/
	enum LogSeverityLevel { LOG_FATAL= 0, LOG_ERROR, LOG_WARNING, LOG_INFO, LOG_DEBUG };

	/**
		Log message
		\param severity: severity level of the message
		\param message: string with the message
		\param module: module name (optional but recommended)
	*/
	virtual void LogMessage (LogSeverityLevel severity, const char* message, const char* module= NULL)= 0;

	/*!
		\brief Register a new log target
		\lt Pointer to the log target to register
	*/
	virtual void RegisterLogTarget (ILogTarget* lt)= 0;

	/*!
		\brief Unregister log target
		\lt Pointer to the log target to unregister
	*/
	virtual void UnregisterLogTarget (ILogTarget* lt)= 0;

	/*!
		\brief Construct and return an IConfiguration object
		\return a smart pointer to a new IConfiguration object
	*/
	virtual SmartPtr<IConfiguration> GetConfiguration() const= 0;

	/*!
		\brief Get an object to report paths
		\return a reference to the IPath object
	*/
	virtual IPaths& GetPaths() const= 0;

	/**
		Returns whether the caller is the main thread
	*/
	virtual bool IsMainThread () const= 0;

#ifdef ENABLE_WXWIDGETS

	/**
		Initializes wxWidgets

		Initializes the wx subsystem. This method must be called before creating
		any window or calling RunMessageLoop

		\params argc, argv
		\return 0 on successful initialization or -1 when error
	*/
	virtual int InitGUISupport (int argc, char** argv)= 0;

	/**
		Runs the main message loop.

		Runs the wx subsystem main loop. This function does not return until
		the last frame has been destroyed as the regular wx message	loop does.

		\return exit code
	*/
	virtual int RunMessageLoop ()= 0;

	/**
		Finalizes wxWidgets

		Finalized the wx subsystem. This method must be called after RunMessageLoop
		has exited.
	*/
	virtual void CleanupGUISupport ()= 0;

	/**
		Forwards a message via the application main thread to the destination pin.
		This method waits until the event has been processed.

		\return		0:	message successfully sent
					-1:	type mismatch or message could not be send
	*/
	virtual int SendMessageMainThreadSync (SmartPtr<const CTypeAny> msg, IInputPin & dst)= 0;

	/**
		Forwards a message via the application main thread to a callback
	*/
	typedef void ToMainThreadCallback (IComponent*, const CTypeAny*);
	virtual void SendMessageMainThreadAsync (const CTypeAny&, IComponent&, ToMainThreadCallback*)= 0;

#endif
};

class ILogTarget
{
public:
	virtual void LogMessage (ICoreRuntime::LogSeverityLevel severity, const char* message)= 0;
};


EXPORT_FUNCTION ICoreRuntime* getSpCoreRuntime();
EXPORT_FUNCTION void freeSpCoreRuntime();


} // namespace spcore
#endif
