
/****************************************************************************
 **
 ** Copyright (C) 2011 Christian B. Huebschle & George M. Sheldrick
 ** All rights reserved.
 ** Contact: chuebsch@moliso.de
 **
 ** This file is part of the ShelXle
 **
 ** This file may be used under the terms of the GNU Lesser
 ** General Public License version 2.1 as published by the Free Software
 ** Foundation and appearing in the file COPYING included in the
 ** packaging of this file.  Please review the following information to
 ** ensure the GNU Lesser General Public License version 2.1 requirements
 ** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
 **
 **
 ****************************************************************************/
#include "chgl.h"
#include <QtGui>
#include <QtOpenGL>

//
//#define HIDE_REASON_SELECT          1
//#define HIDE_REASON_THIS_FRAGMENT   2
//#define HIDE_REASON_OTHER_FRAGMENT  4
//#define HIDE_REASON_HYDROGEN        8
//#define HIDE_REASON_QPEAK          16 
//


inline void __RotateCS( double c, double s, double& X, double& Y ) {
  double T = X;
  X = c*X - s*Y;
  Y = s*T + c*Y;
}

void glTranslateL( const double dx, const double dy, const double dz ) {
  double mat[4][4];

  glGetDoublev( GL_MODELVIEW_MATRIX, (double*)mat );
  mat[3][0] += dx;  mat[3][1] += dy;  mat[3][2] += dz;
  glLoadMatrixd((double*)mat);
}

void glRotateL( const double dang, const double x, const double y, const double z ) {
  double mat[4][4];
#ifndef M_PI
#define	M_PI		3.14159265358979323846	/* pi */
#endif

  double s = z;
  s = sin(dang*M_PI/180);
  const double c = cos(dang*M_PI/180);
  glGetDoublev( GL_MODELVIEW_MATRIX, (double*)mat );
  //  glGetDoublev( GL_PROJECTION_MATRIX, (double*)mat );


  if( x!=0.0 ){
    __RotateCS( c, s, mat[0][1], mat[0][2] );
    __RotateCS( c, s, mat[1][1], mat[1][2] );
    __RotateCS( c, s, mat[2][1], mat[2][2] );
  }else if( y!=0.0 ){
    __RotateCS( c, s, mat[0][2], mat[0][0] );
    __RotateCS( c, s, mat[1][2], mat[1][0] );
    __RotateCS( c, s, mat[2][2], mat[2][0] );
  }else{
    __RotateCS( c, s, mat[0][0], mat[0][1] );
    __RotateCS( c, s, mat[1][0], mat[1][1] );
    __RotateCS( c, s, mat[2][0], mat[2][1] );
  }


  glLoadMatrixd((double*)mat); 

}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
chGL::chGL(QWidget *parent) : QGLWidget(parent) {                                                        ///////////////
  //printf("chGL::chGL\n");
  pause=true;
  murx=0;
  envirange=3.2;
  enviNoQ= new QAction("Exclude Q-peaks from environment listing", this);
  enviButt= new QToolButton(this);
  enviButt->setText("clear ENVI");
  connect(enviButt,SIGNAL(clicked()),this,SLOT(clearEnvi()));
  wireButt= new QAction("Wiremodel while rotate",this);
  wireButt->setCheckable(true);
  wireButt->setChecked(true);
  enviButt->setVisible(false);
  enviNoQ->setCheckable(true);
  enviNoQ->setChecked(false);
  mol=new Molecule();
  warLabel=noWaitLabel=false;
  viewAngle=29.0;
  imFokus=afok=-1;
  pickradius=200.0;
  //printf("__chGL::chGL\n");
  xray_eye=0;
  moving = new QTimer(this);
  moving->setSingleShot(true);
  connect(moving,SIGNAL(timeout()),this,SLOT(updateGL()));
  p=pp=ppp=0;
  fogrange=60;
  xray_vision=false;
  bggradient=true;
  wirbas=bas=0;
  depthcueing=false;
  altemitte=V3(0,0,0);
  hiddenThings=false;
  hideReason=0;
  qcutoff=-9;
  setMinimumSize(200,200);
  myFont=QFont("Arial",16);
  centerSelection = new QAction(this);//for a dirty fix
  rotze=-1;                                                                                              ///////////////
  stereo_mode=0;                                                                                         ///////////////a
  atomsClickable=true;                                                                                   ///////////////
  mouseOverInteraction=true;                                                                             ///////////////
  warfaul=drawUc=false;
  // 0  1  2  3
  // 4  5  6  7
  // 8  9 10 11
  //12 13 14 15
  for (int i=0; i<36; i++)foubas[i]=0;
  MM[0]=-1;
  MM[1]=MM[2]=MM[3]=MM[4]=MM[7]=MM[8]=MM[11]=MM[12]=MM[13]=0;
  MM[5]=-0.242536;
  MM[6]=0.970142;
  MM[9]=0.970142;
  MM[10]=0.242536;
  MM[15]=1.0;
  //MM[1]=MM[2]=MM[3]=MM[4]=MM[6]=MM[7]=MM[8]=MM[9]=MM[11]=MM[12]=MM[13]=0.0;
  MM[14]=-200.0;
  drawDF=drawFO= drawAx=drawAt=drawADP=drawBo=drawLa=drawHb=true;                                                ///////////////
#if (QT_VERSION >= 0x040600) && defined (Q_WS_MAC)
  grabGesture(Qt::PinchGesture);
  grabGesture(Qt::SwipeGesture);
#endif
  //                                                                                                     ///////////////
}                                                                                                        ///////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#if (QT_VERSION >= 0x040600) && defined (Q_WS_MAC)
	bool chGL::event(QEvent *event){
	  if (event->type() == QEvent::Gesture)
	    return gestureEvent(static_cast<QGestureEvent*>(event));
	  return QWidget::event(event);
	}

	bool chGL::gestureEvent(QGestureEvent *event){
	  if (QGesture *swipe = event->gesture(Qt::SwipeGesture))
	    swipeTriggered(static_cast<QSwipeGesture *>(swipe));
	  else
	    if (QGesture *pinch = event->gesture(Qt::PinchGesture))
	      pinchTriggered(static_cast<QPinchGesture *>(pinch));
	  return true;
	}

void chGL::pinchTriggered(QPinchGesture *gesture){
  QPinchGesture::ChangeFlags changeFlags = gesture->changeFlags();
  if (changeFlags & QPinchGesture::RotationAngleChanged) {
    qreal value = gesture->property("rotationAngle").toReal();
    rotZ(value/-36.0);
  }
  if (changeFlags & QPinchGesture::ScaleFactorChanged) {
    qreal value = gesture->property("scaleFactor").toReal();
    gZoom(value-1.0);
  }
  updateGL();
}

void chGL::swipeTriggered(QSwipeGesture *gesture) {
  if (gesture->horizontalDirection() == QSwipeGesture::Left) glRotateL(-5.0,0.0f,1.0f,0.0f);
  if (gesture->horizontalDirection() == QSwipeGesture::Right) glRotateL(5.0,0.0f,1.0f,0.0f);
  if (gesture->verticalDirection() == QSwipeGesture::Up) glRotateL(-5.0,1.0f,0.0f,0.0f);
  if (gesture->verticalDirection() == QSwipeGesture::Down) glRotateL(5.0,1.0f,0.0f,0.0f);
  updateGL();
}
#endif



static inline void transform_point(GLdouble out[4], const GLdouble m[16], const GLdouble in[4]) {
#define M(row,col)  m[col*4+row]
  out[0] =
	  M(0, 0) * in[0] + M(0, 1) * in[1] + M(0, 2) * in[2] + M(0, 3) * in[3];
  out[1] =
	  M(1, 0) * in[0] + M(1, 1) * in[1] + M(1, 2) * in[2] + M(1, 3) * in[3];
  out[2] =
	  M(2, 0) * in[0] + M(2, 1) * in[1] + M(2, 2) * in[2] + M(2, 3) * in[3];
  out[3] =
	  M(3, 0) * in[0] + M(3, 1) * in[1] + M(3, 2) * in[2] + M(3, 3) * in[3];
#undef M
}

static inline bool  posTo2D(V3 obj,
		const GLdouble model[16], const GLdouble proj[16],
		const GLint viewport[4],
		GLdouble * winx, GLdouble * winy) {
  GLdouble in[4], out[4];

  in[0] = obj.x;
  in[1] = obj.y;
  in[2] = obj.z;
  in[3] = 1.0;
  transform_point(out, model, in);
  transform_point(in, proj, out);

  if (in[3] == 0.0) return false;

  in[0] /= in[3];
  in[1] /= in[3];
  in[2] /= in[3];

  *winx = viewport[0] + (1 + in[0]) * viewport[2] / 2;
  *winy = viewport[1] + (1 - in[1]) * viewport[3] / 2;
  return true;

}

void chGL::changeEnviRange(){
  bool ok;
  double r=QInputDialog::getDouble ( this,"Envi-Range", "change environment range", envirange,0.7,8.0,2,&ok);
  if (ok) envirange=r;
}

void chGL::mousePressEvent(QMouseEvent *event) {
  glDisable(GL_SCISSOR_TEST);
  glDisable(GL_CLIP_PLANE1);
  glDisable(GL_CLIP_PLANE2);
  xray_eye=0;
  lastPos = event->pos();
  p=qMin((int)p,mol->showatoms.size()-1);
  pp=qMin((int)pp,mol->showatoms.size()-1);
  ppp=qMin((int)ppp,mol->showatoms.size()-1);

  p=qMax((int)p,0);
  pp=qMax((int)pp,0);
  ppp=qMax((int)ppp,0);
  double nahda=pickradius,da=0;
  int nahdai=-1;
  for (int j=0; j<mol->showatoms.size();j++){
    if (mol->showatoms.at(j).hidden) continue;
    da=(((mol->showatoms.at(j).screenX-event->x())*( mol->showatoms.at(j).screenX-event->x()))+
		    ((mol->showatoms.at(j).screenY-event->y())*( mol->showatoms.at(j).screenY-event->y())));
    nahdai=(da<nahda)?j:nahdai;
    nahda=qMin(nahda,da);
  }

  //  fprintf(stderr,"pressed %d %d %d  %d\n",p,pp,ppp,mol->showatoms.size());
  if (event->buttons() & Qt::MidButton){
    /*    GLint hits;
	  glSelectBuffer(MAXSELECT, selectBuf);
	  (void)glRenderMode(GL_SELECT);
	  glInitNames();
	  glPushName(~0);
	  glPushMatrix();
	  if ((stereo_mode<2)||(stereo_mode>3))   glViewport(0, 0, ww, wh);
	  else glViewport(0, 0, ww/2, wh);
	  glGetIntegerv(GL_VIEWPORT, vp);
	  glMatrixMode(GL_PROJECTION);
	  glLoadIdentity();
	  if ((stereo_mode<2)||(stereo_mode>3))  {
	  gluPickMatrix(event->x(), wh-event->y(), 8, 8, vp);
	  gluPerspective( viewAngle, (double)ww/wh, 50.0, 8000.0 );
	  }
	  else {
	  gluPickMatrix(event->x() % (ww/2), wh-event->y(), 8, 8, vp);
	  gluPerspective( viewAngle, (double)(ww / 2.0)/wh, 50.0, 8000.0 );
	  }
	  glMatrixMode(GL_MODELVIEW);
    //glGetDoublev(GL_MODELVIEW_MATRIX,MM);
    glPushAttrib(GL_ENABLE_BIT);
    glDisable(GL_STENCIL_TEST);
    glDisable(GL_ALPHA_TEST);
    glDisable(GL_LIGHTING);
    glDisable(GL_FOG);
    glDisable(GL_NORMALIZE);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_COLOR_MATERIAL);
    glDisable(GL_LINE_SMOOTH);
    glDisable(GL_DITHER);
    glDisable(GL_BLEND);
    glShadeModel(GL_SMOOTH);

    draw();
    glPopAttrib();
    glPopMatrix();
    /hits = glRenderMode(GL_RENDER); 
    updateGL();*/
    if (nahdai < 0) {
    }else 
      if (nahdai<mol->showatoms.size()) {
	GLuint index=nahdai;
	if (index==((GLuint)-1))return;
	rotze=((int)index<mol->showatoms.size())?index:-1;
	if (rotze>-1){ rCenter->show();

	  glGetDoublev(GL_MODELVIEW_MATRIX,MM);
	  MM[12]=MM[13]=0;
	  glLoadMatrixd(MM);
	}
	updateGL();

      }
  }
  //*/
  //  if((reSe | moveLab| invEditAble | atomsClickable| xdSetupMode) && 
  if (event->buttons() & Qt::LeftButton){
    /*GLint hits;
      glSelectBuffer(MAXSELECT, selectBuf);
      (void)glRenderMode(GL_SELECT);
      glInitNames();
      glPushName(~0);
      glPushMatrix();
      if ((stereo_mode<2)||(stereo_mode>3))   glViewport(0, 0, ww, wh);
      else glViewport(0, 0, ww/2, wh);
      glGetIntegerv(GL_VIEWPORT, vp);
      glMatrixMode(GL_PROJECTION);
      glLoadIdentity();
      gluPickMatrix(event->x(), wh - event->y(), 8, 8, vp);
      if ((stereo_mode<2)||(stereo_mode>3))  gluPerspective( viewAngle, (double)ww/wh, 50.0, 8000.0 );
      else gluPerspective( viewAngle, (double)(ww / 2.0)/wh, 50.0, 8000.0 );
      glMatrixMode(GL_MODELVIEW);
      glPushAttrib(GL_ENABLE_BIT);
      glDisable(GL_STENCIL_TEST);
      glDisable(GL_ALPHA_TEST);
      glDisable(GL_LIGHTING);
      glDisable(GL_FOG);
      glDisable(GL_NORMALIZE);
      glDisable(GL_DEPTH_TEST);
      glDisable(GL_COLOR_MATERIAL);
      glDisable(GL_LINE_SMOOTH);
      glDisable(GL_DITHER);
      glDisable(GL_BLEND);
      glShadeModel(GL_SMOOTH);

      draw();
      glPopAttrib();
      glPopMatrix();
      hits = glRenderMode(GL_RENDER); 

      updateGL();*/
    if (nahdai < 0) {
    }else 
      if (nahdai<mol->showatoms.size()) {
	GLuint index=nahdai;
	if (index==((GLuint)-1))return;
	double w=0,dw=0;
	if ((pp!=p)&&(pp!=index)&&(p!=index)) {
	  w=mol->winkel(mol->showatoms.at(p).pos-mol->showatoms.at(pp).pos,
			  mol->showatoms.at(p).pos-mol->showatoms.at(index).pos);
	  if ((ppp!=p)&&(ppp!=pp)&&(ppp!=index))
	    dw=mol->dieder(mol->showatoms.at(pp).pos-mol->showatoms.at(ppp).pos,
			    mol->showatoms.at(pp).pos-mol->showatoms.at(p).pos,
			    mol->showatoms.at(p).pos-mol->showatoms.at(index).pos);
	}
	if (atomsClickable){
	  emit jumpit(index);
	  if (!inRenameMode){
	    if (event->modifiers()==Qt::NoModifier) {
	      int isschon=-1;
	      for (int i=0; i< mol->selectedatoms.size();i++){
		isschon=((GLuint)mol->selectedatoms.at(i).style==index)?index:isschon;
	      }
	      mol->selectedatoms.clear();
	      if (isschon==-1){
		mol->selectedatoms.append(mol->showatoms[index]);
		mol->selectedatoms.last().style=index;
	      }
	    }
	    if (event->modifiers()==Qt::ShiftModifier) {
	      int min=mol->showatoms.size(), max=-1;
	      for (int i=0; i< mol->selectedatoms.size();i++){
		min=qMin(mol->selectedatoms.at(i).style,min);
		max=qMax(mol->selectedatoms.at(i).style,max);
	      }
	      min=qMin((int)index,min);
	      max=qMax((int)index,max);
	      mol->selectedatoms.clear();
	      for (int i=min; i<=max; i++){
		mol->selectedatoms.append(mol->showatoms[i]);
		mol->selectedatoms.last().style=i;
	      }
	    }
	    if (event->modifiers()==Qt::ControlModifier) {
	      if (mol->selectedatoms.contains(mol->showatoms[index])){
		mol->selectedatoms.removeOne(mol->showatoms[index]);
	      } else {
		mol->selectedatoms.append(mol->showatoms[index]);
		mol->selectedatoms.last().style=index;
	      }
	    }
	    //	  std::cout<<mol->selectedatoms.size()<<mol->selectedatoms.last().Label.toStdString()<<std::endl;
	    updateBondActions();
	    updateGL();
	  }
	  //
	  //
	  V3 hin;
	  double dmsda=0;
	  hin = Normalize(mol->showatoms.at(index).pos-mol->showatoms.at(p).pos);
	  if ((mol->showatoms.at(index).an>-1) && (mol->showatoms.at(p).an)) 
	    dmsda = fabs(((hin*mol->showatoms.at(index).uc)*hin)- ((hin*mol->showatoms.at(p).uc)*hin)); 
	  //
	  bool syok;
	  QString symml;
	  int syid=mol->showatoms.at(index).Label.section(QString::fromUtf8("»"),1).toInt(&syok);
	  if (mol->usedSymmetry.size()&&syok) symml=mol->symmcode2human(mol->usedSymmetry.at(syid-1),syid);
	  syid=mol->showatoms.at(p).Label.section(QString::fromUtf8("»"),1).toInt(&syok);
	  if ((mol->usedSymmetry.size())&&(syok)&&(!symml.contains(mol->symmcode2human(mol->usedSymmetry.at(syid-1),syid)))) 
	    symml+=mol->symmcode2human(mol->usedSymmetry.at(syid-1),syid);

	  syid=mol->showatoms.at(pp).Label.section(QString::fromUtf8("»"),1).toInt(&syok);
	  if ((mol->usedSymmetry.size())&&(syok)&&(!symml.contains(mol->symmcode2human(mol->usedSymmetry.at(syid-1),syid)))) 
	    symml+=mol->symmcode2human(mol->usedSymmetry.at(syid-1),syid);

	  syid=mol->showatoms.at(ppp).Label.section(QString::fromUtf8("»"),1).toInt(&syok);
	  if ((mol->usedSymmetry.size())&&(syok)&&(!symml.contains(mol->symmcode2human(mol->usedSymmetry.at(syid-1),syid)))) 
	    symml+=mol->symmcode2human(mol->usedSymmetry.at(syid-1),syid);

	  if ((index!=p)&&(index!=pp)&&(mol->showatoms.at(index).Label[0]=='Q')){
	    emit bigmessage(QString("<hr><table><tr><td><b>%1</b></td><td align=\"right\"></td></tr><tr><td><b>%1--%2</b>"
				    "</td><td align=\"right\">%5 &Aring;</td></tr><tr><td><b>%1--%2--%3</b></td><td "
				    "align=\"right\">%6&deg;</td></tr><tr><td><b>%1--%2--%3--%4 </b></td><td "
				    "align=\"right\">%7&deg;</td></tr><tr><td><b>%1</b> Peak Height: </td><td "
				    "align=\"right\">%8 e&Aring;<sup>-3</sup></td><td>[%9&hellip;%10]"
				    "</td></tr></table>%11")
			    .arg((mol->showatoms.at(index).Label))                     //1
			    .arg((mol->showatoms.at(p).Label))				//3
			    .arg((mol->showatoms.at(pp).Label))			                //5			                                                //8
			    .arg((mol->showatoms.at(ppp).Label))			                        //9
			    .arg(sqrt(Distance(mol->showatoms.at(index).pos,mol->showatoms.at(p).pos)),8,'f',3) //4
			    .arg(w,8,'f',2)
			    .arg(dw,8,'f',2)
			    .arg(mol->showatoms.at(index).peakHeight,8,'f',2)
			    .arg(mol->pmin,3,'f',2)
			    .arg(mol->pmax,3,'f',2)
			    .arg(symml)

			   );
	  }
	  else if ((index!=p)&&(index!=pp)){
	    emit bigmessage(QString("<br><hr><table><tr><td><b>%1</b> part: %8 residue: %9 %10 (fragment: %12)</td><td align=\"right\">"
				    "</td></tr><tr><td><b>%1--%2</b></td><td align=\"right\">%5 &Aring;</td><td>&nbsp;&nbsp;&nbsp;DMSDA: %13 &Aring;<sup>2</sup></td></tr>"
				    "<tr><td><b>%1--%2--%3</b></td><td align=\"right\">%6&deg;</td></tr><tr><td>"
				    "<b>%1--%2--%3--%4 </b></td><td align=\"right\">%7&deg;</td></tr></table>%11")
			    .arg((mol->showatoms.at(index).Label))			       //1
			    .arg((mol->showatoms.at(p).Label))			               //2
			    .arg((mol->showatoms.at(pp).Label))			       //3
			    .arg((mol->showatoms.at(ppp).Label))			       //4
			    .arg(sqrt(Distance(mol->showatoms.at(index).pos,mol->showatoms.at(p).pos)),8,'f',3) //5
			    .arg(w,8,'f',2)			                            //6
			    .arg(dw,8,'f',2)                                               //7
			    .arg(mol->showatoms.at(index).part)
			    .arg(mol->showatoms.at(index).resiNr)
			    .arg(mol->showatoms.at(index).ResiClass)
			    .arg(symml)
			    .arg(mol->showatoms.at(index).molindex)
			    .arg(dmsda,5,'e',2)
			   );
	  }
	  //				   ,QMessageBox::Ok);
	  ppp=pp;
	  pp=p;
	  p=index;
	}

	//888
      }
  }
}

void chGL::mouseMoveEvent(QMouseEvent *event) {
  double nahda=pickradius,da=0;
  int nahdai=-1;
  bool changed=false;
  for (int j=0; j<mol->showatoms.size();j++){
    if (mol->showatoms.at(j).hidden) continue;
    da=(((mol->showatoms.at(j).screenX-event->x())*( mol->showatoms.at(j).screenX-event->x()))+ 
		    ((mol->showatoms.at(j).screenY-event->y())*( mol->showatoms.at(j).screenY-event->y())));
    nahdai=(da<nahda)?j:nahdai;
    nahda=qMin(nahda,da);
  }

  if ((mouseOverInteraction)&&(!event->buttons())) {
    if (imFokus!=nahdai) changed=true;
    imFokus=nahdai;
    if (imFokus==-1) {
      emit qpfoci(-1.0);
      emit message("");
    }
    else {
      emit message(mol->showatoms.at(imFokus).Label);
      if ((mol->showatoms.at(imFokus).an<0)) emit qpfoci(mol->showatoms.at(imFokus).peakHeight);
      else qpfoci(-1.0);
    }

    if (changed) updateGL();
    return;
  }
  GLfloat dx = GLfloat( event->x() - lastPos.x()) / ww;
  GLfloat dy = GLfloat( event->y() - lastPos.y()) / wh;
  /*
     if ((mouseOverInteraction)&&(!event->buttons())) {
     GLint hits;   glSelectBuffer(MAXSELECT, selectBuf);
     (void)glRenderMode(GL_SELECT);
     glInitNames();
     glPushName(~0);
     glPushMatrix();
     if ((stereo_mode<2)||(stereo_mode>3))  glViewport(0, 0, ww, wh);
     else glViewport(0, 0, ww/2, wh);
     glGetIntegerv(GL_VIEWPORT, vp);
     glMatrixMode(GL_PROJECTION);
     glLoadIdentity();
     if ((stereo_mode<2)||(stereo_mode>3))  {
     gluPickMatrix(event->x(), wh-event->y(), 8, 8, vp);
     gluPerspective( viewAngle, (double)ww/wh, 50.0, 8000.0 );
     }
     else {
     gluPickMatrix(event->x() % (ww/2), wh-event->y(), 8, 8, vp);
     gluPerspective( viewAngle, (double)(ww / 2.0)/wh, 50.0, 8000.0 );
     }
     glMatrixMode(GL_MODELVIEW);
     glPushAttrib(GL_ENABLE_BIT);
     glDisable(GL_STENCIL_TEST);
     glDisable(GL_ALPHA_TEST);
     glDisable(GL_LIGHTING);
     glDisable(GL_FOG);
     glDisable(GL_NORMALIZE);
     glDisable(GL_DEPTH_TEST);
     glDisable(GL_COLOR_MATERIAL);
     glDisable(GL_LINE_SMOOTH);
     glDisable(GL_DITHER);
     glDisable(GL_BLEND);
     glShadeModel(GL_SMOOTH);
  //     glLoadMatrixd(MM);
  bool ala=drawLa;
  drawLa=false;
  draw();
  drawLa=ala;
  glPopAttrib();
  glPopMatrix();
  hits = glRenderMode(GL_RENDER);
  //hits = glRenderMode(GL_SELECT);
  if ((hits <= 0)||(mol->showatoms.size()<(int)selectBuf[(hits-1)*4+3])) {imFokus=-1; emit qpfoci(-1.0); emit message("");}
  else {
  imFokus=selectBuf[(hits-1)*4+3];
  //  printf("Test,%d %d\n",imFokus,hits);
  if (imFokus>=0) emit message(mol->showatoms.at(selectBuf[(hits-1)*4+3]).Label);
  if ((imFokus>=0)&&(mol->showatoms.at(selectBuf[(hits-1)*4+3]).an<0)) emit qpfoci(mol->showatoms.at(selectBuf[(hits-1)*4+3]).peakHeight);
  else emit qpfoci(-1.0);
  }
  updateGL();
  //    highFocus();
  }
  // */
  if (event->buttons() & Qt::MidButton){

    moving->start(80);
    glTranslateL(dx*viewAngle*3,-dy*viewAngle*3,0);
    updateGL();
  }
  if ((event->buttons() & Qt::LeftButton)) {
    {
      moving->start(80);
      double slow=1.0;
      if (event->modifiers()==Qt::ShiftModifier) slow=0.25;
      if (event->modifiers()==Qt::ControlModifier) slow=0.06;
      glRotateL(dy*360.0*slow,1.0f,0.0f,0.0f);
      glRotateL(dx*360.0*slow,0.0f,1.0f,0.0f);
    }
    updateGL();
  }
  else if((event->buttons() & Qt::RightButton)){
    if (invertMouseZoom->checkState()!=Qt::Checked){

      moving->start(80);
      glScaled(1.0-dy,1.0-dy,1.0-dy);
      //     mlsc/=1.0-dy;
    }else {

      moving->start(80);
      glScaled(1.0+dy,1.0+dy,1.0+dy);
      //     mlsc/=1.0+dy;
    }
    updateGL();
  }
  lastPos = event->pos();
}

void chGL::wheelEvent(QWheelEvent *event){
  /*
modifiers:
Qt::NoModifier	        0x00000000	No modifier key is pressed.
Qt::ShiftModifier	    0x02000000	A Shift key on the keyboard is pressed.
Qt::ControlModifier	    0x04000000	A Ctrl key on the keyboard is pressed.
Qt::AltModifier	        0x08000000	An Alt key on the keyboard is pressed.
Qt::MetaModifier	    0x10000000	A Meta key on the keyboard is pressed.
Qt::KeypadModifier	    0x20000000	A keypad button is pressed.
Qt::GroupSwitchModifier	0x40000000	X11 only. A Mode_switch key on the keyboard is pressed.
*/ 
  int numDegrees = event->delta() / 8;
  int numSteps = numDegrees / 15;
  if (event->modifiers()==Qt::NoModifier){
    int d = myFont.pointSize();
    d=(d+numSteps>4)?d+numSteps:d;
    d=qMax(d,7);
    myFont.setPointSize(d);
    updateGL();
  }
  if (event->modifiers()==Qt::ControlModifier){
    emit diffscroll(numSteps,1); 
  } 
  if (event->modifiers()==Qt::ShiftModifier){
    emit diffscroll(numSteps,0); 
  }
  if (event->modifiers()==(Qt::AltModifier|Qt::ShiftModifier)){
    double va=viewAngle+0.1*numSteps;
    setViewAngle(va);
  } 
  if (event->modifiers()==Qt::AltModifier){
    int afr=fogrange;
    fogrange+=numSteps*5;
    fogrange=qMax(fogrange,2);
    fogrange=qMin(fogrange,199);
    //    printf("fogrange = %d\n", fogrange);
    if (afr!=fogrange){
      glFogi(GL_FOG_MODE,GL_LINEAR);
      glFogi(GL_FOG_START,205-fogrange);
      glFogi(GL_FOG_END,205+fogrange);
      GLfloat fgc[4]={backGroundColor.redF(),backGroundColor.greenF(),backGroundColor.blueF(),1.0};
      glFogfv(GL_FOG_COLOR,fgc);
      updateGL();}
  }
  event->accept();
}

void chGL::contextMenuEvent(QContextMenuEvent *event){
  double nahda=pickradius,da=0;
  int nahdai=-1;
  for (int j=0; j<mol->showatoms.size();j++){
    if (mol->showatoms.at(j).hidden) continue;
    da=(((mol->showatoms.at(j).screenX-event->x())*( mol->showatoms.at(j).screenX-event->x()))+
		    ((mol->showatoms.at(j).screenY-event->y())*( mol->showatoms.at(j).screenY-event->y())));
    nahdai=(da<nahda)?j:nahdai;
    nahda=qMin(nahda,da);
  }
  if ((nahdai < 0)||(mol->showatoms.size()<nahdai)) {
    return;
  }
  else {
    GLuint idx=nahdai;
    ImeanThisAtom=idx;
    if (idx>((GLuint) mol->showatoms.size())) return;
    QVariantList row;
    row.clear();
    row.append((int) p);
    row.append((int) idx);
    QVariant qvl(row);
    QList<QAction *> sfas = sfacMenu->actions();
    for (int k=0;k<sfas.size();k++){
        sfas[k]->setEnabled(sfas[k]->text()!=mol->pse(mol->showatoms.at(idx).an));
    }
    QMenu *menu = new QMenu(mol->showatoms.at(idx).Label);
    menu->addMenu(sfacMenu);
    QAction *a;//=menu->addAction(QString("Edit Style of %1").arg(mol->showatoms.at(idx).Label),parent(),SLOT(styleDialog2()));
    //a->setData(idx);
    if ((mol->selectedatoms.isEmpty())&&(mol->showatoms.at(idx).symmGroup==0)) {
      a=menu->addAction(QIcon(":icons/killselected.png"),QString("Delete %1 ").arg(mol->showatoms.at(idx).Label),
		      parent(),SLOT(deleteSelectedAtoms()));
      a->setData(idx);
    }
    a=menu->addAction(QString("Set rotation center to %1").arg(mol->showatoms.at(idx).Label),this,SLOT(setRotationCenter()));
    a->setData(idx);
    a=menu->addAction(QString("Expand contacts around %1").arg(mol->showatoms.at(idx).Label),this,SLOT(expand()));
    a->setData(idx);
    a=menu->addAction(QString("Hide this fragment"),this,SLOT(hideThisFragment()));
    a->setData(idx);
    a=menu->addAction(QString("Hide other fragments"),this,SLOT(hideOtherFragments()));
    a->setData(idx);
    if (!mol->selectedatoms.isEmpty()) a=menu->addAction("Hide selected Atoms",this,SLOT(hideSelected()));
    a=menu->addAction(QString("Select this fragment"),this,SLOT(selectThisFragment()));
    a->setData(idx);
    QMenu *enviMenu = new QMenu("ENVI-Settings");
    enviMenu->addAction("Change envi range", this, SLOT(changeEnviRange()));
    enviMenu->addAction(enviNoQ);
    menu->addMenu(enviMenu);
    if ((int) idx < mol->asymm.size()){
      a=menu->addAction(QString("List ENVIronment of %1").arg(mol->showatoms.at(idx).Label),this,SLOT(envi()));
      a->setData(idx);
    }
    int ssgr=0;
    int adpat=0;
    for (int seli=0; seli<mol->selectedatoms.size(); seli++){
      if ((mol->showatoms.at(seli).symmGroup==0)&&(mol->selectedatoms.at(seli).uf.m22!=666.0))adpat++;
    }
    for (int seli=0; seli<mol->selectedatoms.size(); seli++){
      if (mol->selectedatoms.at(seli).symmGroup>0) ssgr+=mol->selectedatoms.at(seli).symmGroup;
      else {
	ssgr=0;
	break;
      }
    }
    if (ssgr) adpat=0;
    if (ssgr){
      a=menu->addAction(QString("Move selected atoms to selected sites"),
		      parent(),SLOT(moveSymmMateSel()));
    }else{
      if (!mol->selectedatoms.isEmpty()){
	a=menu->addAction(QIcon(":icons/killselected.png"),QString("Delete selected atoms"),
			parent(),SLOT(deleteSelectedAtoms()));
	a->setData(-1);
      }
      if (mol->showatoms.at(idx).symmGroup>0){
	a=menu->addAction(QString("Move %1 here").arg(mol->showatoms.at(idx).Label.section("_",0,0)),
			parent(),SLOT(moveSymmMate()));
	a->setData(idx);
	if (mol->showatoms.at(idx).part<0){
	  a=menu->addAction(QString("Insert a copy of %1 here").arg(mol->showatoms.at(idx).Label.section("_",0,0)),
			  parent(),SLOT(copySymmMate()));
	  a->setData(idx);
	}

      }
    }
    if ((nahdai<mol->asymm.size())&&(mol->knoepfe.at(idx).neighbors.size()==1)&&((mol->showatoms.at(idx).an==6)||(mol->showatoms.at(idx).an==5))) {

      //	printf("%s hat %d Nachbarn\n",mol->showatoms.at(idx).Label.toStdString().c_str(),mol->knoepfe.at(idx).neighbors.size());
      a=menu->addAction(QString("Add dissordered %1 Hydrogen atoms").arg(((mol->showatoms.at(idx).an==5))?"methyl":"amino"),parent(),SLOT(addDissorderedMethyl()));
      a->setData(idx);
      a=menu->addAction(QString("Add ordered %1 Hydrogen atoms").arg(((mol->showatoms.at(idx).an==5))?"methyl":"amino"),parent(),SLOT(addMethyl()));
      a->setData(idx);
    }
    if ((idx < ((GLuint) mol->asymm.size())) && (mol->asymm.at(idx).an>-1)) {
      a=menu->addAction(QString("Use %1 as new label for rename mode").arg(mol->showatoms.at(idx).Label.section("_",0,0)),parent(),SLOT(renameThisAtom()));
      a->setData(idx);
    }
    if (mol->showatoms.at(p).molindex!=mol->showatoms.at(idx).molindex){
      a=menu->addAction(QString("Inherit Labels from molecule around %1 to atoms around %2")
		      .arg(mol->showatoms.at(p).Label)
		      .arg(mol->showatoms.at(idx).Label)
		      ,parent(),SLOT(inheritLabels()));
      a->setData(qvl);

    }

    if (adpat) {
      a=menu->addAction(QIcon(":/icons/sina.png"),QString("make selected atoms isotropic (ISOT)"),parent(),SLOT(sina()));
      a->setData(-1);
    }
    else if ((mol->selectedatoms.isEmpty())&&(mol->showatoms.at(idx).symmGroup==0)&&(mol->showatoms.at(idx).uf.m22!=-666)){
      a=menu->addAction(QIcon(":/icons/sina.png"),QString("make %1 isotropic (ISOT)")
		      .arg(mol->showatoms.at(idx).Label),parent(),SLOT(sina()));
      a->setData(idx);
    }
    menu->exec(event->globalPos());
    delete enviMenu;
    delete menu;
  }
}

void chGL::Listen(){
  murx=1;
}

void chGL::disSelection(){
  //rotCenter();
  mol->selectedatoms.clear();
  updateBondActions();
  updateGL();
}

void chGL::selectPair(const QString &s){
  mol->selectedatoms.clear();
  if (s.isEmpty()) {
    updateGL();
    return;
  }
  QStringList index=s.split(' ',QString::SkipEmptyParts);
  if (index.size()<2) {
    updateGL();
    return;
  }
  for (int i=0; i<index.size();i++){
    mol->selectedatoms.append(mol->showatoms[index.at(i).toInt()]);
    mol->selectedatoms.last().style=index.at(i).toInt();
  }
  //updateBondActions();
  updateGL();
}

void chGL::selectResiByNr(int nr){
  mol->selectedatoms.clear();
  for (int i=0; i<mol->showatoms.size();i++){
    if (mol->showatoms.at(i).resiNr==nr) {
      mol->selectedatoms.append(mol->showatoms[i]);
      mol->selectedatoms.last().style=i;
    }
  }
  draw();
  if (hiddenThings) hideNonSelected();
  updateBondActions();
  updateGL();
}

void chGL::invertSelection(){
  CEnvironment tempa=mol->selectedatoms;
//  printf("invert %d %d\n",tempa.size(),mol->selectedatoms.size());
  mol->selectedatoms.clear();

  for (int i=0; i<mol->showatoms.size();i++){
    if (!tempa.contains(mol->showatoms.at(i))) {
      mol->selectedatoms.append(mol->showatoms[i]);
      mol->selectedatoms.last().style=i;
    }
  }
  draw();
  rehide();
  updateBondActions();
  updateGL();
}

void chGL::updateBondActions(){
  clearSelection->setVisible(!mol->selectedatoms.isEmpty());
  invSelection->setVisible(!mol->selectedatoms.isEmpty());
  centroid->setVisible(mol->selectedatoms.size()>1);
  centerSelection->setVisible((!mol->selectedatoms.isEmpty())||(centerSelection->isChecked()));
  delSelAt->setVisible(!mol->selectedatoms.isEmpty());
  delSelAt->setEnabled(!mol->selectedatoms.isEmpty());
  hideNotSelection->setVisible(!mol->selectedatoms.isEmpty());
  unhide->setVisible(hiddenThings);
  if (mol->selectedatoms.size()!=2){
    addBond->setVisible(false);
    killBond->setVisible(false);
  }
  else{
    int da=0;
    for (int i=0; i<mol->showbonds.size();i++){
      if (((mol->selectedatoms.at(0).style==mol->showbonds.at(i).a1)||
			      (mol->selectedatoms.at(0).style==mol->showbonds.at(i).a2))&&
		      ((mol->selectedatoms.at(1).style==mol->showbonds.at(i).a1)||
		       (mol->selectedatoms.at(1).style==mol->showbonds.at(i).a2))) da=i;
    }
    if (da){
      addBond->setVisible(false);
      killBond->setVisible(true);
      killBond->setData(da);
    }
    else{
      addBond->setVisible(true);
      killBond->setVisible(false);
    }
  }
}

void chGL::connectSelection(){
  if (mol->selectedatoms.size()!=2) return;
  MyBond b;
  b.a1=mol->selectedatoms.at(0).style;
  b.a2=mol->selectedatoms.at(1).style;
  b.ato1=&mol->showatoms[mol->selectedatoms.at(0).style];
  b.ato2=&mol->showatoms[mol->selectedatoms.at(1).style];
  b.length=sqrt(Distance(b.ato1->pos,b.ato2->pos));
  mol->showbonds.append(b);
  glDeleteLists(bas+1,1);
  mol->tubes=0;
  mol->adp=1;
  glNewList(bas+1, GL_COMPILE );{                          //BONDS
    glPushMatrix();{
      glScaled( L, L, L );
      mol->bonds(mol->showbonds);
    }glPopMatrix();
  }glEndList();

  glDeleteLists(bas+8,1);
  glNewList(bas+8, GL_COMPILE );{                          //BONDS //noadp
    glPushMatrix();{
      glScaled( L, L, L );
      mol->adp=0;
      mol->bonds(mol->showbonds);
    }glPopMatrix();
  }glEndList();

  glDeleteLists(bas+9,1);
  glNewList(bas+9, GL_COMPILE );{                          //BONDS tube
    glPushMatrix();{
      glScaled( L, L, L );
      mol->tubes=1;
      mol->adp=0;
      mol->bonds(mol->showbonds);
    }glPopMatrix();
    mol->tubes=0;
  }glEndList();

  mol->selectedatoms.clear();
  updateBondActions();
  updateGL();
}

void chGL::disConnectSelection(int index){
  mol->showbonds.removeAt(index);
  glDeleteLists(bas+1,1);
  mol->tubes=0;
  mol->adp=1;
  glNewList(bas+1, GL_COMPILE );{                          //BONDS
    glPushMatrix();{
      glScaled( L, L, L );
      mol->bonds(mol->showbonds);
    }glPopMatrix();
  }glEndList();

  glDeleteLists(bas+8,1);
  glNewList(bas+8, GL_COMPILE );{                          //BONDS //noadp
    glPushMatrix();{
      glScaled( L, L, L );
      mol->adp=0;
      mol->bonds(mol->showbonds);
    }glPopMatrix();
  }glEndList();

  glDeleteLists(bas+9,1);
  glNewList(bas+9, GL_COMPILE );{                          //BONDS tube
    glPushMatrix();{
      glScaled( L, L, L );
      mol->tubes=1;
      mol->adp=0;
      mol->bonds(mol->showbonds);
    }glPopMatrix();
    mol->tubes=0;
  }glEndList();

    
  mol->selectedatoms.clear();
  updateBondActions();
  updateGL();
}

void chGL::setAtom(bool b){
  drawAt=b;
  updateGL();
}

void chGL::setBond(bool b){
  drawBo=b;
  updateGL();
}

void chGL::setLabel(bool b){
  drawLa=b;
  updateGL();
}

void chGL::setHBond(bool b){
  drawHb=b;
  updateGL();
}

void chGL::setBGGradient(bool b){
  bggradient=b;
  updateGL();
}

void chGL::setDepthCueing(bool b){
  depthcueing=b;
  updateGL();
}

void chGL::setXRayView(bool b){
  xray_vision=b;
  updateGL();
}

void chGL::setADP(bool b){
  drawADP=b;
  updateGL();
}

void chGL::setUnit(bool b){
  drawUc=b;
  updateGL();
}

void chGL::changeBColor(){
  backGroundColor= QColorDialog::getColor(backGroundColor, this);
  updateGL(); 
}

void chGL::changeTColor(){
  labelColor= QColorDialog::getColor(labelColor, this);
  updateGL(); 
}

void chGL::setMolecule(Molecule *m){
//  printf("setMolecule\n");
  if (mol!=m) {
    if (mol) delete mol;
    mol=m;  
  }
  double dim=mol->dimension();
  L=100.0/dim;
}

void chGL::initializeGL(){
  glEnable(GL_LINE_SMOOTH);   
  glHint(GL_LINE_SMOOTH_HINT,GL_NICEST);
  glEnable(GL_POLYGON_SMOOTH);  
  const GLfloat  position[] = {100.0f, 100.0f,100.0f,0.0f};
  const GLfloat  diffuse[]  = { 1.0, 1.0, 1.0, 1.0 };
  const GLfloat  specular[] = { 1.0, 1.0, 1.0, 1.0 };
  const GLfloat  ambient[]  = { 0.5, 0.5, 0.5, 1.0 };

  glLightModeli(  GL_LIGHT_MODEL_LOCAL_VIEWER, 1 );

  glLightfv( GL_LIGHT0, GL_POSITION, position );
  glLightfv( GL_LIGHT0, GL_AMBIENT,  ambient );
  glLightfv( GL_LIGHT0, GL_DIFFUSE,  diffuse );
  glLightfv( GL_LIGHT0, GL_SPECULAR, specular );

  glLightfv( GL_LIGHT1, GL_POSITION, position );
  glLightfv( GL_LIGHT1, GL_DIFFUSE,  diffuse );  
  glLightfv( GL_LIGHT1, GL_AMBIENT,  ambient );
  glLightfv( GL_LIGHT2, GL_DIFFUSE,  diffuse );  

  glDisable(GL_FOG);
  glFogi(GL_FOG_MODE,GL_LINEAR);
  glFogi(GL_FOG_START,200-fogrange);
  glFogi(GL_FOG_END,200+fogrange);
  GLfloat fgc[4]={backGroundColor.redF(),backGroundColor.greenF(),backGroundColor.blueF(),1.0};
  glFogfv(GL_FOG_COLOR,fgc);

  glEnable( GL_LIGHTING );
  glEnable( GL_LIGHT0 );
  glEnable( GL_BLEND);
  glLightModeli(  GL_LIGHT_MODEL_TWO_SIDE, 1);
  glAlphaFunc ( GL_GREATER, 0.01f ) ;
  glBlendFunc ( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA ) ;
  const GLfloat  OBJ_SPE[]   = { 0.8, 0.8, 0.8, 1.0 };  
  const GLfloat  OBJ_SHIN    = 64.0;               
  glMaterialfv( GL_FRONT_AND_BACK, GL_SPECULAR,             OBJ_SPE  );
  glEnable     ( GL_COLOR_MATERIAL ) ;
  glColorMaterial ( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE ) ;    
  glMaterialf(  GL_FRONT_AND_BACK, GL_SHININESS,           OBJ_SHIN );
  glShadeModel( GL_SMOOTH );
  //glShadeModel(GL_FLAT);
  glEnable(GL_NORMALIZE);
  qglClearColor(backGroundColor);

  glEnable(GL_DEPTH_TEST );
  glDepthFunc(GL_LEQUAL);
  gluLookAt(0.0, 200, 50 ,   0.0, 0.0, 0.0,   0.0, -100, 400 );
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
  setMouseTracking(true);
  //Listen inititialisieren ist hier!
  initLists();
  if (foubas[0]|foubas[1]|foubas[2]){
    emit inimibas();
  }
}

void chGL::initLists(){
  if (mol->showatoms.isEmpty()) return;
  setupTexture();
  if (bas) for (int i=0;i<10; i++){
    int lsts=glIsList(bas+i);
    if (lsts) {
      glDeleteLists(bas+i,1);
      //	printf("deleting List %d %d\n",bas+i,lsts);
    }
  }
  bas=glGenLists(10);
  mol->tubes=0;
  mol->dratom=0;
  glLoadMatrixd(MM);
  glNewList(bas, GL_COMPILE );{                          //ATOME
    glPushMatrix();{
      glScaled( L, L, L );
      mol->intern=1;
      mol->adp=1;
      mol->atoms(mol->showatoms,mol->proba);
    }glPopMatrix();
  }glEndList();

  glNewList(bas+1, GL_COMPILE );{                          //BONDS
    glPushMatrix();{
      glScaled( L, L, L );
      mol->bonds(mol->showbonds);
    }glPopMatrix();
  }glEndList();

  glNewList(bas+2, GL_COMPILE );{                          //ATOMS transparent ellipsoids
    glPushMatrix();{
      glScaled( L, L, L );
      mol->intern=0;
      mol->adp=1;
      mol->atoms(mol->showatoms,mol->proba);
    }glPopMatrix();
  }glEndList();


  glNewList(bas+3, GL_COMPILE );{                          //ATOMS balls
    glPushMatrix();{
      glScaled( L, L, L );
      mol->intern=1;
      mol->adp=0;
      mol->atoms(mol->showatoms);
    }glPopMatrix();
  }glEndList();

  glNewList(bas+4, GL_COMPILE );{                          //Unit Cell
    glPushMatrix();{
      glScaled( L, L, L );
      mol->unitCell();
    }glPopMatrix();
  }glEndList();


  glNewList(bas+5, GL_COMPILE );{                          //ATOMS tubes
    glPushMatrix();{
      glScaled( L, L, L );
      mol->tubes=1;
      mol->intern=1;
      mol->adp=0;
      mol->atoms(mol->showatoms);
    }glPopMatrix();
  }glEndList();

  mol->tubes=0;
  glNewList(bas+6, GL_COMPILE );{                          //QPeak-Line-BONDS
    glPushMatrix();{
      glScaled( L, L, L );
      mol->lbond();
    }glPopMatrix();
  }glEndList();

  glNewList(bas+7, GL_COMPILE );{                          //H-BONDS
    glPushMatrix();{
      glScaled( L, L, L );
      emit bigmessage(mol->h_bonds(mol->showbonds,mol->showatoms));
    }glPopMatrix();
  }glEndList();

  glNewList(bas+8, GL_COMPILE );{                          //BONDS //noadp
    glPushMatrix();{
      glScaled( L, L, L );
      mol->adp=0;
      mol->bonds(mol->showbonds);
    }glPopMatrix();
  }glEndList();

  glNewList(bas+9, GL_COMPILE );{                          //BONDS tube
    glPushMatrix();{
      glScaled( L, L, L );
      mol->tubes=1;
      mol->adp=0;
      mol->bonds(mol->showbonds);
    }glPopMatrix();
    mol->tubes=0;
  }glEndList();

  if (wirbas) for (int i=0;i<2; i++){
    int lsts=glIsList(wirbas+i);
    if (lsts) {
      glDeleteLists(wirbas+i,1);
      //	printf("deleting List %d %d\n",bas+i,lsts);
    }
  }
  wirbas=glGenLists(2);
  glNewList(wirbas,GL_COMPILE );{
    glPushMatrix();{
      glScaled( L, L, L );
      mol->dbond(mol->showbonds);
    }glPopMatrix();
  }glEndList();

  glNewList(wirbas+1, GL_COMPILE );{                          //ATOME
    glPushMatrix();{
      glScaled( L, L, L );
      mol->dratom=5;
      mol->intern=0;
      mol->adp=1;
      mol->atoms(mol->showatoms,mol->proba);
    }glPopMatrix();
  }glEndList();
  mol->dratom=0;
  if (!noWaitLabel) moving->start(80);
}

void chGL::setRotationCenter(){
  QAction *action = qobject_cast<QAction *>(sender());
  int index=0;
  if (action)
    index=action->data().toInt();
  else return;
  if (index==(int)((GLuint)-1))return;
  rotze=((int)index<mol->showatoms.size())?index:-1;
  if (rotze>-1){ 
    rCenter->show();
    glGetDoublev(GL_MODELVIEW_MATRIX,MM);
    MM[12]=MM[13]=0;
    glLoadMatrixd(MM);
  }
  updateGL();
}

void chGL::setRotationCenter(int rz){
  rotze=((int)rz<mol->showatoms.size())?rz:-1;
  if (rotze>-1){
    rCenter->show();
    glGetDoublev(GL_MODELVIEW_MATRIX,MM);
    MM[12]=MM[13]=0;
    glLoadMatrixd(MM);
  }
  updateGL();
}

void chGL::expand(){
  QAction *action = qobject_cast<QAction *>(sender());
  int index=0;
  if (action)
    index=action->data().toInt();
  else return;
  if (index==(int)((GLuint)-1))return;
  mol->expandAt(index);
  emit bigmessage(mol->HumanSymmetry);
  murx=1;
  updateGL();
}

void chGL::incFontSize(){
  myFont.setPointSize(myFont.pointSize()+1);
  updateGL();
}

void chGL::decFontSize(){
  myFont.setPointSize(qMax(myFont.pointSize()-1,7));
  updateGL();

}
#include <iostream>
void chGL::setReNaMo(bool b){
  inRenameMode=b;
}

void chGL::rotCenter(){
  rotze=-1;
  updateGL();
  rCenter->hide();
}

void chGL::resizeGL(int width, int height) {
  if (pause) {ww=width; wh=height; return;}
  glViewport(0, 0, ww=width, wh=height);
  glGetIntegerv(GL_VIEWPORT, vp);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();
  gluPerspective( viewAngle, (double)width/height, 5.0, 8000.0 );
}

void chGL::zoom(double speed){
  moving->start(80);
  glScaled(1.0+speed*0.1,1.0+speed*0.1,1.0+speed*0.1);  
  updateGL();  
}

void chGL::rotY(double speed){
  moving->start(80);
  glRotateL(-20.0*speed,0.0f,1.0f,0.0f);
  updateGL();  
}

void chGL::gZoom(double speed){
  moving->start(80);
  glScaled(1.0+speed*0.02,1.0+speed*0.02,1.0+speed*0.02);
  updateGL();
}

void chGL::rotZ(double speed){
  moving->start(80);
  glRotateL(speed,0.0f,0.0f,1.0f);
  updateGL();
}

void chGL::moveY(double speed){
  moving->start(80);
  glTranslateL(0.0,speed,0.0);
  updateGL();  
}

void chGL::moveX(double speed){
  moving->start(80);
  glTranslateL(speed,0.0,0.0);
  //  QMessageBox::information(this,"Px move","xmoved",QMessageBox::Ok);
  updateGL();  
}

void chGL::rotX(double speed){
  moving->start(80);
  glRotateL(-20.0*speed,1.0f,0.0f,0.0f);
  updateGL();  
}


void chGL::zalman(){
  stereo_mode=1;
  minus= 1;
  updateGL();
}

void chGL::parallel(){
  stereo_mode=2;
  minus=1;
  updateGL();
}

void chGL::crosseye(){
  stereo_mode=3;
  minus=-1;
  updateGL();
}

void chGL::anaglyphRedCyan(){
  stereo_mode=4;
  minus=1;
  updateGL();
}

void chGL::nostereo(){
  stereo_mode=0;
  updateGL();
}

void chGL::clearEnvi(){
enviPositions.clear();
labs.clear();
enviButt->setVisible(false);
rotCenter();
}

void chGL::envi(){
  QAction *action = qobject_cast<QAction *>(sender());
  int index=0;
  if (action)
    index=action->data().toInt();
  else return;
  if (index<0) return;
  if (index>=mol->asymm.size())return;
  mol->enviSDM(envirange);
  enviPositions.clear();
  enviKat.clear();
  labs.clear();
  QList<bool> covs;

  V3 ppc,ppf,p0;
 // int ssy=0;
  QString systr;
  QStringList bs;
  QString info;
  enviP0=mol->asymm.at(index).pos;
  info.append(QString("<hr><b>Environment of %1 </b><table border=\"0\" cellspacing=\"0\" cellpadding=\"5\">").arg(mol->asymm[index].Label));
  for (int i = 0; i < mol->envi_sdm.size(); i++){
    if ((mol->envi_sdm.at(i).a2==index)&&mol->envi_sdm.at(i).d<envirange) {
      if ((mol->asymm[mol->envi_sdm.at(i).a1].an==-1)&&(enviNoQ->isChecked())) continue;
      ppf=mol->cell.symmops.at(mol->envi_sdm.at(i).sn) *
              mol->asymm[mol->envi_sdm.at(i).a1].frac +
              mol->cell.trans.at(mol->envi_sdm.at(i).sn) -//-
	      mol->envi_sdm.at(i).floorD;
      mol->frac2kart(ppf,ppc);
      mol->frac2kart(mol->asymm[index].frac,p0);
/*
if (fabs(mol->envi_sdm.at(i).d-sqrt(Distance(p0,ppc)))>0.5){
	ppf=mol->cell.symmops.at(mol->envi_sdm.at(i).sn) *
                mol->asymm[mol->envi_sdm.at(i).a2].frac -
                mol->cell.trans.at(mol->envi_sdm.at(i).sn) -
		mol->envi_sdm.at(i).floorD;
	mol->frac2kart(ppf,ppc);

      }// */
      enviPositions.append(ppc);
      covs.append(mol->envi_sdm.at(i).covalent);
      enviKat.append((mol->envi_sdm.at(i).covalent)?1:0);
      if((abs(mol->asymm[mol->envi_sdm.at(i).a1].an-7)<2)
              &&(abs(mol->asymm[mol->envi_sdm.at(i).a2].an-7)<2)
              &&(fabs(mol->envi_sdm.at(i).d-2.725)<0.275)){enviKat.last()=2;}

      if((mol->asymm[mol->envi_sdm.at(i).a1].an==0)&&(abs(mol->asymm[mol->envi_sdm.at(i).a2].an-7)<2)&&(fabs(mol->envi_sdm.at(i).d-1.875)<0.275)){enviKat.last()=2;}
      if((mol->asymm[mol->envi_sdm.at(i).a2].an==0)&&(abs(mol->asymm[mol->envi_sdm.at(i).a1].an-7)<2)&&(fabs(mol->envi_sdm.at(i).d-1.875)<0.275)){enviKat.last()=2;}
      bool symm=((mol->envi_sdm.at(i).sn)||(!(mol->envi_sdm.at(i).floorD==V3(0,0,0))));
      if (symm) {
	QString sss=QString("%1_%2%3%4:%5,").arg(mol->envi_sdm.at(i).sn+1).arg(5-(int)mol->envi_sdm.at(i).floorD.x).arg(5-(int)mol->envi_sdm.at(i).floorD.y).arg(5-(int)mol->envi_sdm.at(i).floorD.z).arg(mol->asymm[mol->envi_sdm.at(i).a1].molindex);
	if (!bs.contains(sss)){
	  bs.append(sss);
	}

	systr=QString("%1%2").arg(QString::fromUtf8("»")).arg(bs.indexOf(sss)+1);
        labs.append(mol->asymm[ mol->envi_sdm.at(i).a1].Label+systr);
      }
      else labs.append(mol->asymm[ mol->envi_sdm.at(i).a1].Label);
      info.append(QString("<tr><th style=\"background:%6\" >%1%4</th><td style=\"background:%5\" >%7<font color=%3> %2&nbsp;&Aring;</font></b></td>")
                      .arg(mol->asymm[ mol->envi_sdm.at(i).a1].Label)
		      .arg(mol->envi_sdm.at(i).d,8,'f',3)
		      .arg((mol->envi_sdm.at(i).covalent)?"green":"black")
		      .arg(symm?systr:"").arg((labs.size()%2)?"#eeeeee":"white")
		      .arg((labs.size()%2)?"#d4d4e4":"#efefff")
		      .arg((mol->envi_sdm.at(i).covalent)?"<b>":""));
      for (int j=0; j<enviPositions.size()-1; j++){
	double w=
                mol->winkel(p0-enviPositions.at(j),
                                p0-enviPositions.last());
//		printf("%s-%s-%s %g   %g %g %g\n",labs.at(j).toStdString().c_str(),mol->asymm[index].Label.toStdString().c_str(),labs.last().toStdString().c_str(),w,enviPositions.at(j).x ,enviPositions.at(j).y,enviPositions.at(j).z);
	info.append(QString("<td style=\"background:%3\" align=right>%4<font color=%2> %1&deg;</font></b></td>")
			.arg(w,8,'f',2)
			.arg(((mol->envi_sdm.at(i).covalent)&&(covs.at(j)))?"green":"black")
			.arg(((j+1)%2)?(labs.size()%2)?"#d4d4d4":"#e4e4e4":(labs.size()%2)?"#eeeeee":"#ffffff")
                        .arg(((mol->envi_sdm.at(i).covalent)&&(covs.at(j)))?"<b>":""));

      }
      info.append("</tr>\n");
    }
  }
  info.append("<tr><td style=\"background:#efefff\"></td><td style=\"background:#efefff\"></td>");
  for (int j=0; j<enviPositions.size()-1; j++){
    info.append(QString("<th style=\"background:%2\">%1</th>").arg(labs.at(j)).arg(((j+1)%2)?"#d4d4e4":"#efefff"));
  }
  QString symml="";
  for (int j=0; j<bs.size(); j++){
    symml+=mol->symmcode2human(bs.at(j),j+1);
  }
  info.append(QString("</tr>\n</table>%1<hr>\n").arg(symml));
  emit bigmessage(info);

  enviButt->setVisible(true);
  setRotationCenter(index);
}

void chGL::paintGL(){
  //if (pause)  return;
  warLabel = drawLa;
  GLfloat fgc[4]={backGroundColor.redF(),backGroundColor.greenF(),backGroundColor.blueF(),1.0};
  glFogfv(GL_FOG_COLOR,fgc);
  qglClearColor(backGroundColor);
  // printf("%d %d bas %d %g \n",ww,wh,bas,viewAngle);
  glViewport(0, 0, ww, wh);
  glDisable(GL_SCISSOR_TEST);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  //  printf("clear %d %d\n",ww,wh);
  glViewport(0, 0, ww, wh);
  glDisable(GL_STENCIL_TEST);
  // printf("%d ___%d %d %d\n",stereo_mode,ww,wh,mibas);
  glGetIntegerv(GL_VIEWPORT, vp);
  if (!stereo_mode){
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective( viewAngle, (double)ww/wh, 5.0, 8000.0 );
    glMatrixMode(GL_MODELVIEW);
    if (xray_vision) {  
      glPushMatrix();
      glScissor(vp[0],vp[1],vp[2]/2,vp[3]);
      glEnable(GL_SCISSOR_TEST);
      qglClearColor(backGroundColor);
      //glClearColor(0.0,0.0,1.0,0.7);
      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
      xray_eye=1;
      if (!pause) draw();
      glPopMatrix();
      //      }else
      //	if (xray_eye==2)  {
      glPushMatrix();
      glScissor(vp[0],vp[1],vp[2],vp[3]);
      glScissor(vp[2]/2,vp[1],vp[2],vp[3]);
      qglClearColor(backGroundColor);
      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
      glEnable(GL_SCISSOR_TEST);
      xray_eye=2;
      if (!pause) draw();
      glPopMatrix();
      xray_eye=0;
  }else {
    glPushMatrix();
    glDisable(GL_SCISSOR_TEST);
    xray_eye=0;
    if (!pause) draw();
    glPopMatrix();
  }
  }else 
    if (stereo_mode==1){// stereo zalman
      GLint viewport[4];
      glGetIntegerv(GL_VIEWPORT,viewport);

      glPushAttrib(GL_ENABLE_BIT);
      glMatrixMode(GL_PROJECTION);
      glPushMatrix();
      glLoadIdentity();
      glOrtho(0,viewport[2],0,viewport[3],-10.0,10.0);
      glMatrixMode(GL_MODELVIEW);
      glPushMatrix();
      glLoadIdentity();
      glTranslatef(0.33F,0.33F,0.0F); 
      glDisable(GL_STENCIL_TEST);
      glDisable(GL_ALPHA_TEST);
      glDisable(GL_LIGHTING);
      glDisable(GL_FOG);
      glDisable(GL_NORMALIZE);
      glDisable(GL_DEPTH_TEST);
      glDisable(GL_COLOR_MATERIAL);
      glDisable(GL_LINE_SMOOTH);
      glDisable(GL_DITHER);
      glDisable(GL_BLEND);
      glShadeModel(GL_SMOOTH);
      glDisable(0x809D); // GL_MULTISAMPLE_ARB * / 

      glClearStencil(0);
      glColorMask(false,false,false,false);
      glDepthMask(false);
      glClear(GL_STENCIL_BUFFER_BIT);

      glEnable(GL_STENCIL_TEST);
      glStencilFunc(GL_ALWAYS, 1, 1);
      glStencilOp(GL_KEEP, GL_KEEP, GL_REPLACE);

      glLineWidth(1.0);
      glBegin(GL_LINES);
      int h = viewport[3], w=viewport[2];
      int y;
      for(y=0;y<h;y+=2) {
	glVertex2i(0,y);
	glVertex2i(w,y);
      }
      glEnd();

      glColorMask(true,true,true,true);
      glDepthMask(true);

      glMatrixMode(GL_MODELVIEW);
      glPopMatrix();
      glMatrixMode(GL_PROJECTION);
      glPopMatrix();
      //
      glPopAttrib();

      glViewport(0, 0, ww, wh);        
      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
      glViewport(0, 0, ww, wh);        
      glGetIntegerv(GL_VIEWPORT, vp);
      glMatrixMode(GL_PROJECTION);
      glLoadIdentity();
      gluPerspective( viewAngle, (double)ww/wh, 5.0, 8000.0 );
      glMatrixMode(GL_MODELVIEW);
      if (xray_vision){
	glPushMatrix();
	drawLa=false;
	glRotateL(minus*-1.5,0,1,0);
	glEnable(GL_STENCIL_TEST);
	glStencilFunc(GL_EQUAL, 1, 1);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
	glEnable(GL_STENCIL_TEST);// */
	xray_eye=1;
	glScissor(vp[0],vp[1],vp[2]/2,vp[3]);
	glEnable(GL_SCISSOR_TEST);
	//      qglClearColor(backGroundColor);
	//glClearColor(0.0,0.0,1.0,0.7);
	//      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	if (!pause) draw();
	glPopMatrix();
	glPushMatrix();
	drawLa=false;
	glRotateL(minus*-1.5,0,1,0);
	glEnable(GL_STENCIL_TEST);
	glStencilFunc(GL_EQUAL, 1, 1);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
	glEnable(GL_STENCIL_TEST);// */
	xray_eye=2;
	glScissor(vp[0],vp[1],vp[2],vp[3]);
	glScissor(vp[2]/2,vp[1],vp[2],vp[3]);
	glEnable(GL_SCISSOR_TEST);
	if (!pause) draw();
	xray_eye=0;
	glPopMatrix();
      }else{
	glPushMatrix();
	drawLa=false;
	glRotateL(minus*-1.5,0,1,0);
	glEnable(GL_STENCIL_TEST);
	glStencilFunc(GL_EQUAL, 1, 1);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
	glEnable(GL_STENCIL_TEST);// */
	xray_eye=0;
	glDisable(GL_SCISSOR_TEST);
	if (!pause) draw();
	glPopMatrix();
      }
      drawLa= warLabel;
      if (xray_vision) {
	glPushMatrix();
	glRotateL(minus*1.5,0,1,0);
	glEnable(GL_STENCIL_TEST);
	glStencilFunc(GL_EQUAL, 0, 1);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
	glEnable(GL_STENCIL_TEST);// */
	xray_eye=1;
	glScissor(vp[0],vp[1],vp[2]/2,vp[3]);
	glEnable(GL_SCISSOR_TEST);
	//      qglClearColor(backGroundColor);
	//glClearColor(0.0,0.0,1.0,0.7);
	//      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	if (!pause) draw();
	glPopMatrix();
	glPushMatrix();
	glRotateL(minus*1.5,0,1,0);
	glEnable(GL_STENCIL_TEST);
	glStencilFunc(GL_EQUAL, 0, 1);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
	glEnable(GL_STENCIL_TEST);// */
	xray_eye=2;
	glScissor(vp[0],vp[1],vp[2],vp[3]);
	glScissor(vp[2]/2,vp[1],vp[2],vp[3]);
	//      qglClearColor(backGroundColor);
	//      glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glEnable(GL_SCISSOR_TEST);
	if (!pause) draw();
	glPopMatrix();
      }else
      {
	glPushMatrix();
	glRotateL(minus*1.5,0,1,0);
	glEnable(GL_STENCIL_TEST);
	glStencilFunc(GL_EQUAL, 0, 1);
	glStencilOp(GL_KEEP, GL_KEEP, GL_KEEP);
	glEnable(GL_STENCIL_TEST);// */
	xray_eye=0;
	glDisable(GL_SCISSOR_TEST);
	if (!pause) draw();
	glPopMatrix();
      }
    }else  
      if ((stereo_mode>1)&&(stereo_mode<4)) { //stereo_side by side
	if (xray_vision){
	  glPushMatrix();
	  glViewport(0, 0, ww/2, wh);        
	  glMatrixMode(GL_PROJECTION);
	  glLoadIdentity();
	  gluPerspective( viewAngle, (double)(ww/2.0)/wh, 5.0, 8000.0 );
	  glMatrixMode(GL_MODELVIEW);
	  glPushMatrix();
	  glRotateL(1.5*minus,0,1,0);
	  xray_eye=1;
	  //	  glGetIntegerv(GL_VIEWPORT, vp);
	  glScissor(vp[0],vp[1],vp[2]/2,vp[3]);
	  glEnable(GL_SCISSOR_TEST);
	  if (!pause) draw();
	  glPopMatrix();
	  glPopMatrix();
	  glPushMatrix();
	  glViewport(0, 0, ww/2, wh);        
	  glMatrixMode(GL_PROJECTION);
	  glLoadIdentity();
	  gluPerspective( viewAngle, (double)(ww/2.0)/wh, 5.0, 8000.0 );
	  glMatrixMode(GL_MODELVIEW);
	  glPushMatrix();
	  glRotateL(1.5*minus,0,1,0);
	  xray_eye=2;
	  //	  glGetIntegerv(GL_VIEWPORT, vp);
	  glScissor(vp[0],vp[1],vp[2],vp[3]);
	  glScissor(vp[2]/2,vp[1],vp[2],vp[3]);
	  glEnable(GL_SCISSOR_TEST);
	  if (!pause) draw();
	  glPopMatrix();
	  glPopMatrix();
	}else{
	  glPushMatrix();
	  glViewport(0, 0, ww/2, wh);        
	  glMatrixMode(GL_PROJECTION);
	  glLoadIdentity();
	  gluPerspective( viewAngle, (double)(ww/2.0)/wh, 5.0, 8000.0 );
	  glMatrixMode(GL_MODELVIEW);
	  glPushMatrix();
	  glRotateL(1.5*minus,0,1,0);
	  xray_eye=0;
	  glDisable(GL_SCISSOR_TEST);
	  if (!pause) draw();
	  glPopMatrix();
	  glPopMatrix();
	}


	if (xray_vision){
	  glPushMatrix();
	  glViewport( ww / 2 , 0, ww / 2,wh );        
	  glMatrixMode(GL_PROJECTION);
	  glLoadIdentity();
	  gluPerspective( viewAngle, (double)(ww/2.0)/wh, 5.0, 8000.0 );
	  glMatrixMode(GL_MODELVIEW);
	  glPushMatrix();
	  glRotateL(-1.5*minus,0,1,0);
	  xray_eye=1;
	  //	  glGetIntegerv(GL_VIEWPORT, vp);
	  glScissor(vp[0],vp[1],vp[2],vp[3]);
	  glScissor((ww / 2)+vp[0],vp[1],vp[2]/2,vp[3]);
	  glEnable(GL_SCISSOR_TEST);
	  if (!pause) draw();
	  glPopMatrix();
	  glPopMatrix();
	  glPushMatrix();
	  glViewport( ww / 2 , 0,ww / 2,wh );        
	  glMatrixMode(GL_PROJECTION);
	  glLoadIdentity();
	  gluPerspective( viewAngle, (double)(ww/2.0)/wh, 5.0, 8000.0 );
	  glMatrixMode(GL_MODELVIEW);
	  glPushMatrix();
	  glRotateL(-1.5*minus,0,1,0);
	  xray_eye=2;
	  //	  glGetIntegerv(GL_VIEWPORT, vp);
	  glScissor(vp[0],vp[1],vp[2],vp[3]);
	  glScissor((ww / 2)+vp[2]/2,vp[1],vp[2]/2,vp[3]);
	  glEnable(GL_SCISSOR_TEST);
	  if (!pause) draw();
	  glPopMatrix();
	  glPopMatrix();
	}else{
	  glPushMatrix();
	  glViewport( ww / 2 , 0,ww / 2,wh );        
	  glMatrixMode(GL_PROJECTION);
	  glLoadIdentity();
	  gluPerspective( viewAngle, (double)(ww/2.0)/wh, 5.0, 8000.0 );
	  glMatrixMode(GL_MODELVIEW);
	  glPushMatrix();
	  glRotateL(-1.5*minus,0,1,0);
	  xray_eye=0;
	  glDisable(GL_SCISSOR_TEST);
	  if (!pause) draw();
	  glPopMatrix();
	  glPopMatrix();
	}
      }
      else  if (stereo_mode==4){//anaglyph red cyan
	if (xray_vision){
	  glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	  glColorMask(GL_TRUE, GL_FALSE, GL_FALSE, GL_TRUE);
	  glPushMatrix();
	  glViewport(0, 0, ww, wh);        
	  glMatrixMode(GL_PROJECTION);
	  glLoadIdentity();
	  gluPerspective( viewAngle, (double)(ww)/wh, 5.0, 8000.0 );
	  glMatrixMode(GL_MODELVIEW);
	  glPushMatrix();
	  glRotateL(1.5*minus,0,1,0);
	  xray_eye=1;
	  //	  glGetIntegerv(GL_VIEWPORT, vp);
	  glScissor(vp[0],vp[1],vp[2],vp[3]);
	  glScissor(vp[0],vp[1],vp[2]/2,vp[3]);
	  glEnable(GL_SCISSOR_TEST);
	  //bbb
	  if (!pause) draw();
	  //bbb
	  glPopMatrix();
	  glPopMatrix();
	  glPushMatrix();
	  glViewport(0, 0, ww, wh);        
	  glMatrixMode(GL_PROJECTION);
	  glLoadIdentity();
	  gluPerspective( viewAngle, (double)(ww)/wh, 5.0, 8000.0 );
	  glMatrixMode(GL_MODELVIEW);
	  glPushMatrix();
	  glRotateL(1.5*minus,0,1,0);
	  xray_eye=2;
	  //	  glGetIntegerv(GL_VIEWPORT, vp);
	  glScissor(vp[0],vp[1],vp[2],vp[3]);
	  glScissor(vp[2]/2,vp[1],vp[2],vp[3]);
	  glEnable(GL_SCISSOR_TEST);
	  //aaa
	  if (!pause) draw();
	  //aaa
	  glPopMatrix();
	  glPopMatrix();

	  glDisable(GL_SCISSOR_TEST);
	  glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	  glClear(GL_DEPTH_BUFFER_BIT);
	  glColorMask(GL_FALSE, GL_TRUE, GL_TRUE, GL_TRUE);

	  glPushMatrix();
	  glViewport( 0  , 0, ww ,wh );        
	  glMatrixMode(GL_PROJECTION);
	  glLoadIdentity();
	  gluPerspective( viewAngle, (double)(ww)/wh, 5.0, 8000.0 );
	  glMatrixMode(GL_MODELVIEW);
	  glPushMatrix();
	  glRotateL(-1.5*minus,0,1,0);
	  xray_eye=1;
	  //	  glGetIntegerv(GL_VIEWPORT, vp);
	  glScissor(vp[0],vp[1],vp[2],vp[3]);
	  glScissor(vp[0],vp[1],vp[2]/2,vp[3]);
	  glEnable(GL_SCISSOR_TEST);
	  //dddd
	  if (!pause) draw();
	  //dddd
	  glPopMatrix();
	  glPopMatrix();
	  glPushMatrix();
	  glViewport( 0 , 0,ww ,wh );        
	  glMatrixMode(GL_PROJECTION);
	  glLoadIdentity();
	  gluPerspective( viewAngle, (double)(ww)/wh, 5.0, 8000.0 );
	  glMatrixMode(GL_MODELVIEW);
	  glPushMatrix();
	  glRotateL(-1.5*minus,0,1,0);
	  xray_eye=2;
	  //	  glGetIntegerv(GL_VIEWPORT, vp);
	  glScissor(vp[0],vp[1],vp[2],vp[3]);
	  glScissor(vp[2]/2,vp[1],vp[2],vp[3]);
	  glEnable(GL_SCISSOR_TEST);
	  //ccc
	  if (!pause) draw();
	  //ccc
	  glPopMatrix();
	  glPopMatrix();
	  glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	}
	else{
	  glGetIntegerv(GL_VIEWPORT, vp);
	  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	  glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	  glColorMask(GL_TRUE, GL_FALSE, GL_FALSE, GL_TRUE);

	  // set camera for blue eye, red will be filtered.

	  // draw scene
	  glPushMatrix();
	  glViewport(0, 0, ww, wh);        
	  glMatrixMode(GL_PROJECTION);
	  glLoadIdentity();
	  gluPerspective( viewAngle, (double)ww/wh, 5.0, 8000.0 );
	  glMatrixMode(GL_MODELVIEW);
	  glPushMatrix();
	  glRotateL(1.5*minus,0,1,0);
	  xray_eye=0;
	  glDisable(GL_SCISSOR_TEST);
	  if (!pause) draw();
	  glPopMatrix();
	  glPopMatrix();

	  glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	  glClear(GL_DEPTH_BUFFER_BIT);
	  glColorMask(GL_FALSE, GL_TRUE, GL_TRUE, GL_TRUE);

	  // set camera for red eye, blue will be filtered.

	  // draw scene
	  glPushMatrix();
	  glViewport(0, 0, ww, wh);        
	  glMatrixMode(GL_PROJECTION);
	  glLoadIdentity();
	  gluPerspective( viewAngle, (double)ww/wh, 5.0, 8000.0 );
	  glMatrixMode(GL_MODELVIEW);
	  glPushMatrix();
	  glRotateL(-1.5*minus,0,1,0);
	  xray_eye=0;
	  glDisable(GL_SCISSOR_TEST);
	  if (!pause) draw();
	  glPopMatrix();
	  glPopMatrix();
	  glColorMask(GL_TRUE, GL_TRUE, GL_TRUE, GL_TRUE);
	}	 

      }

}

void chGL::setViewAngle(double ang){
  if ((ang>0.001)&&(ang<160.0)){
    //  printf("View angle = %g degrees. %10.8f %f\n",ang,ang/viewAngle,viewAngle);
    glScaled(ang/viewAngle,ang/viewAngle,ang/viewAngle);
    viewAngle=ang;
    homeXY();
  }
}

void chGL::rehide(){
  //  printf("haa %d %d %g\n",hideReason,hiddenThings,qcutoff);
  if (!hiddenThings) return;
  if ((hideReason&HIDE_REASON_SELECT)&&(mol->selectedatoms.size())) {hideNonSelected(); return;}
  else if (hideReason&HIDE_REASON_SELECT) hideReason-=HIDE_REASON_SELECT;
  int gute=frid;
  for (int i=0;i<mol->showatoms.size();i++){
    if (hideReason&HIDE_REASON_QPEAK) if (mol->showatoms[i].an<0) mol->showatoms[i].hidden=(mol->showatoms[i].peakHeight<qcutoff)?1:0;
    if (hideReason&HIDE_REASON_HYDROGEN) mol->showatoms[i].hidden=
      (mol->showatoms.at(i).an==0)?1:mol->showatoms.at(i).hidden;
    if (hideReason&(HIDE_REASON_THIS_FRAGMENT|HIDE_REASON_OTHER_FRAGMENT)){
      if (mol->showatoms.at(i).an>=0){
	if (hideReason&HIDE_REASON_THIS_FRAGMENT) mol->showatoms[i].hidden=
	  (mol->showatoms.at(i).molindex+9999*mol->showatoms.at(i).symmGroup!=gute)?mol->showatoms.at(i).hidden:1;
	if (hideReason&HIDE_REASON_OTHER_FRAGMENT)mol->showatoms[i].hidden=
	  (mol->showatoms.at(i).molindex+9999*mol->showatoms.at(i).symmGroup!=gute)?1:mol->showatoms.at(i).hidden;
      }
    }
  }

}

void chGL::hideNonSelected(){
  for (int i=0;i<mol->showatoms.size();i++) mol->showatoms[i].hidden=1;
  for(int i=0;i<mol->selectedatoms.size();i++){
    if (mol->selectedatoms.at(i).style < mol->showatoms.size())
      mol->showatoms[mol->selectedatoms.at(i).style].hidden=0;
    //   std::cout<<mol->showatoms.at(mol->selectedatoms.at(i).style).Label.toStdString()<<std::endl;
  }
  hiddenThings=true;
  hideReason|=HIDE_REASON_SELECT;
  mol->selectedatoms.clear();
  murx=1;
  updateBondActions();
  updateGL();
}

void chGL::hideSelected(){
  for(int i=0;i<mol->selectedatoms.size();i++){
    if (mol->selectedatoms.at(i).style < mol->showatoms.size())
      mol->showatoms[mol->selectedatoms.at(i).style].hidden=1;
    //   std::cout<<mol->showatoms.at(mol->selectedatoms.at(i).style).Label.toStdString()<<std::endl;
  }
  hiddenThings=true;
  mol->selectedatoms.clear();
  murx=1;
  updateBondActions();
  updateGL();
}

void chGL::hideThisFragment(){
  QAction *action = qobject_cast<QAction *>(sender());
  int index=0;
  if (action)
    index=action->data().toInt();
  else return;
  if (index>mol->showatoms.size()) return;
  int gute=mol->showatoms.at(index).molindex+9999*mol->showatoms.at(index).symmGroup;
  //int gute=mol->showatoms.at(index).molindex;
  for (int i=0;i<mol->showatoms.size();i++) 
    if (mol->showatoms.at(i).an>=0)
      mol->showatoms[i].hidden=
	      (mol->showatoms.at(i).molindex+9999*mol->showatoms.at(i).symmGroup!=gute)?mol->showatoms.at(i).hidden:1;
  //(mol->showatoms.at(i).molindex!=gute)?mol->showatoms.at(i).hidden:1;
  //
  frid=gute;
  hiddenThings=true;
  hideReason|=HIDE_REASON_THIS_FRAGMENT;
  mol->selectedatoms.clear();
  murx=1;
  updateBondActions();
  updateGL();
}

void chGL::selectThisFragment(){
  QAction *action = qobject_cast<QAction *>(sender());
  int index=0;
  if (action)
    index=action->data().toInt();
  else return;
  if (index>mol->showatoms.size()) return;
  mol->selectedatoms.clear();
  int gute=mol->showatoms.at(index).molindex+9999*mol->showatoms.at(index).symmGroup;
  //int gute=mol->showatoms.at(index).molindex;
  for (int i=0;i<mol->showatoms.size();i++) 
    if (mol->showatoms.at(i).an>=0)
      if(mol->showatoms.at(i).molindex+9999*mol->showatoms.at(i).symmGroup==gute){
	mol->selectedatoms.append(mol->showatoms[i]);
	mol->selectedatoms.last().style=i;
      }
  //(mol->showatoms.at(i).molindex!=gute)?mol->showatoms.at(i).hidden:1;
  //
  updateBondActions();
  updateGL();
}

void chGL::hideHydrogens(){
  for (int i=0;i<mol->showatoms.size();i++) mol->showatoms[i].hidden=
    (mol->showatoms.at(i).an==0)?1:mol->showatoms.at(i).hidden;
  hiddenThings=true;
  hideReason|=HIDE_REASON_HYDROGEN;
  hideh->setVisible(false);
  mol->selectedatoms.clear();
  murx=1;
  updateBondActions();
  updateGL();
}

void chGL::hideOtherFragments(){
  QAction *action = qobject_cast<QAction *>(sender());
  int index=0;
  if (action)
    index=action->data().toInt();
  else return;
  if (index==(int)((GLuint)-1))return;
  if (index>mol->showatoms.size()) return;
  int gute=mol->showatoms.at(index).molindex+9999*mol->showatoms.at(index).symmGroup;
  //int gute=mol->showatoms.at(index).molindex;
  for (int i=0;i<mol->showatoms.size();i++) 
    if (mol->showatoms.at(i).an>=0)
      mol->showatoms[i].hidden=
	      //		(mol->showatoms.at(i).molindex!=gute)?1:mol->showatoms.at(i).hidden;
	      (mol->showatoms.at(i).molindex+9999*mol->showatoms.at(i).symmGroup!=gute)?1:mol->showatoms.at(i).hidden;
  hiddenThings=true;
  frid=gute;
  hideReason|=HIDE_REASON_OTHER_FRAGMENT;
  mol->selectedatoms.clear();
  murx=1;
  updateBondActions();
  updateGL();
}

void chGL::highliteQPeak(double co){
  imFokus=-1;
  //printf("co %f\n",co)
  for (int i=0;i<mol->showatoms.size();i++){
    if (mol->showatoms[i].an==-66) continue;
    if (mol->showatoms[i].an>=0) continue;
    if (mol->showatoms[i].hidden)continue;
    if (mol->showatoms[i].peakHeight<=co) {imFokus=i; break;}
  }
  updateGL();
}

void chGL::hideQPeaksBelow(double cutoff){
  int vor=0,nach=0;
  for (int i=0;i<mol->showatoms.size();i++){
    vor += mol->showatoms[i].hidden;
    if (mol->showatoms[i].an < 0) mol->showatoms[i].hidden = (mol->showatoms[i].peakHeight<cutoff)?1:0;
    if (mol->showatoms[i].an == -66) mol->showatoms[i].hidden = 0;
    nach += mol->showatoms[i].hidden;

  }
  if (vor==nach) return;
  hiddenThings=true;
  hideReason|=HIDE_REASON_QPEAK;
  qcutoff=cutoff;
  mol->selectedatoms.clear();
  murx=1;
  updateBondActions();
  updateGL();
}

void chGL::showHidden(){
  int h=0;
  for (int i=0;i<mol->showatoms.size();i++){
    mol->showatoms[i].hidden=0;
    if (mol->showatoms[i].an==0)h++;
  }
  if (h)
    hideh->setVisible(true);
  hiddenThings=false;
  hideReason=0;
  murx=1;
  updateBondActions();
  updateGL();
}

void chGL::hidePartMinusOne(bool off){
  mol->nopm1=off;
  murx=1;
  updateBondActions();
  updateGL();
}

void chGL::setupTexture(){
  deleteTexture(mol->adpwall);
  deleteTexture(mol->adpwall_plaid);
  deleteTexture(mol->hbtex);

  mol->adpwall_plaid=bindTexture(QImage(QString(":/icons/adpwall.png")),GL_TEXTURE_2D);
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );

  mol->adpwall=bindTexture(QImage(QString(":/icons/adpwall3.png")),GL_TEXTURE_2D);
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );

  mol->hbtex=bindTexture(QImage(QString(":/icons/hbb.png")),GL_TEXTURE_2D);
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
}
void chGL::homeXY(){
  glGetDoublev(GL_MODELVIEW_MATRIX,MM);
  MM[12]=MM[13]=0;
  glLoadMatrixd(MM);
  updateGL();
}
void chGL::draw(){
  if (mol->LOD>3) {
    const GLfloat  OBJ_SPE[]   = { 0.8, 0.8, 0.8, 1.0 };  
    const GLfloat  OBJ_SHIN    = 127.0;               
    glMaterialfv( GL_FRONT_AND_BACK, GL_SPECULAR,             OBJ_SPE  );
    glEnable     ( GL_COLOR_MATERIAL ) ;
    glColorMaterial ( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE ) ;    
    glMaterialf(  GL_FRONT_AND_BACK, GL_SHININESS,           OBJ_SHIN );
  }else{
    const GLfloat  OBJ_SPE[]   = { 0.0, 0.0, 0.0, 1.0 };  
    const GLfloat  OBJ_SHIN    = 1.0;               
    glMaterialfv( GL_FRONT_AND_BACK, GL_SPECULAR,             OBJ_SPE  );
    glEnable     ( GL_COLOR_MATERIAL ) ;
    glColorMaterial ( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE ) ;    
    glMaterialf(  GL_FRONT_AND_BACK, GL_SHININESS,           OBJ_SHIN );
  }
  if (murx) {
    //    printf("www %d %d %d\n",mol->showatoms.size(),bas, mibas);
    rehide();
    initLists();
    murx=0;
  }
  V3 ori=V3(0,0,0);///rotation origin
  V3 sumse=V3(0,0,0);
  int zz=0;
  objCnt=0;
  if (mol->showatoms.size()){
    for (int i=0;i<mol->showatoms.size();i++){
      if (!mol->showatoms[i].hidden) objCnt++;
      if ((!mol->showatoms.at(i).hidden)) {//(mol->showatoms.at(i).an>-1)&&
        sumse+=mol->showatoms[i].pos;
	zz++;
      }
    }

    if (zz) sumse*=1.0/zz;
    else {
      for (int i=0;i<mol->showatoms.size();i++){
	if (!mol->showatoms.at(i).hidden) {
	  sumse+=mol->showatoms[i].pos;
	  zz++;
	}
      }
      if (zz) sumse*=1.0/zz;
    }
  }
  //  printf(" zz %d %d \n",zz,mol->showatoms.size());
  glGetDoublev( GL_MODELVIEW_MATRIX, (double*)MM );
  double gmat[16];
  glGetDoublev( GL_MODELVIEW_MATRIX, (double*)gmat );
  ori.x=gmat[0] * sumse.x + gmat[4] * sumse.y + gmat[8] * sumse.z;
  ori.y=gmat[1] * sumse.x + gmat[5] * sumse.y + gmat[9] * sumse.z;
  ori.z=gmat[2] * sumse.x + gmat[6] * sumse.y + gmat[10] * sumse.z;
  if (mol->showatoms.size()<rotze){
    rotze=-1;
    rCenter->hide();
  }
  if (rotze>-1) {
    double gmat[16];
    sumse=mol->showatoms.at(rotze).pos;
    glGetDoublev( GL_MODELVIEW_MATRIX, (double*)gmat );
    ori.x=gmat[0] * mol->showatoms.at(rotze).pos.x + gmat[4] * mol->showatoms.at(rotze).pos.y + gmat[8] *  mol->showatoms.at(rotze).pos.z;
    ori.y=gmat[1] * mol->showatoms.at(rotze).pos.x + gmat[5] * mol->showatoms.at(rotze).pos.y + gmat[9] *  mol->showatoms.at(rotze).pos.z;
    ori.z=gmat[2] * mol->showatoms.at(rotze).pos.x + gmat[6] * mol->showatoms.at(rotze).pos.y + gmat[10] * mol->showatoms.at(rotze).pos.z;
  }
  if (centerSelection->isChecked()){
    if (mol->selectedatoms.isEmpty()) {
      sumse=altemitte;//centerSelection->setChecked(false);
      glGetDoublev( GL_MODELVIEW_MATRIX, (double*)gmat );
      ori.x=gmat[0] * sumse.x + gmat[4] * sumse.y + gmat[8] * sumse.z;
      ori.y=gmat[1] * sumse.x + gmat[5] * sumse.y + gmat[9] * sumse.z;
      ori.z=gmat[2] * sumse.x + gmat[6] * sumse.y + gmat[10] * sumse.z;
    }
    else {
      sumse=V3(0,0,0);
      for (int i=0;i<mol->selectedatoms.size();i++)
	sumse+=mol->selectedatoms[i].pos;
      sumse*=1.0/mol->selectedatoms.size();
      glGetDoublev( GL_MODELVIEW_MATRIX, (double*)gmat );
      ori.x=gmat[0] * sumse.x + gmat[4] * sumse.y + gmat[8] * sumse.z;
      ori.y=gmat[1] * sumse.x + gmat[5] * sumse.y + gmat[9] * sumse.z;
      ori.z=gmat[2] * sumse.x + gmat[6] * sumse.y + gmat[10] * sumse.z;


    }
  }
  glPushMatrix();
  double mat[16];
  glEnable(GL_BLEND);
  glEnable(GL_COLOR_MATERIAL);
  glGetDoublev( GL_MODELVIEW_MATRIX, (double*)mat );
  glLoadIdentity();
  glDisable(GL_CLIP_PLANE1);
  glDisable(GL_CLIP_PLANE2);
  double 
	  clip_plane1[]={1.0,0.0,-1.0,-190},
	  clip_plane2[]={-1.0,0.0,-1.0,-190};
  glClipPlane(GL_CLIP_PLANE1,clip_plane1);
  glClipPlane(GL_CLIP_PLANE2,clip_plane2);
  glDisable( GL_LIGHTING ); 
  glDisable( GL_DEPTH_TEST ); 
  if ((bggradient)&&(!depthcueing)) { 
    glPushMatrix();
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective( 29, (double)ww/wh, 5.0, 8000.0 );
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glBegin(GL_QUADS);
    double xx = ((double) ww / wh) * 2.0,
	   yy = 1.77777777778;
    glColor4f(1.0,1.0,1.0,0.5);
    //glTexCoord2d(-1,-1);
    glVertex3f(-xx,-yy,-6.0);
    glColor4f(1.0,1.0,1.0,0.5);
    //glTexCoord2d(0,-1);
    glVertex3f( xx,-yy,-6.0);
    //glTexCoord2d(0,0);
    glColor4f(0.3,0.3,0.3,0.7);
    glVertex3f( xx, yy,-6.0);
    glColor4f(0.3,0.3,0.3,0.7);
    //glTexCoord2d(-1,0);
    glVertex3f(-xx, yy,-6.0);
    glEnd();
    glPopMatrix();
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective( viewAngle, (double)ww/wh, 5.0, 8000.0 );
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();
  }  
  if (depthcueing) glEnable(GL_FOG);
  else glDisable(GL_FOG);
  glEnable( GL_LIGHTING ); 
  glEnable( GL_DEPTH_TEST ); 
  if (xray_eye==1)  {
    glEnable(GL_CLIP_PLANE2);
  }else 
    if (xray_eye==2)  {
      glEnable(GL_CLIP_PLANE1);
    }
  glLoadMatrixd(mat);
  glPopMatrix();
  glPushMatrix();
  if (!(altemitte==sumse)){
    //printf("neuemitte %f %f %f\n",sumse.x,sumse.y,sumse.z);
    altemitte=sumse;
    emit neuemitte(altemitte);
  }
  glTranslateL(-L*ori.x,-L*ori.y,-L*ori.z);
  if ((wireButt->isChecked())&&(moving->isActive())) {
   if (glIsList(wirbas)) glCallList(wirbas);//wirebonds
   if ((drawADP)&&(glIsList(wirbas+1))) glCallList(wirbas+1);//adps
   if (glIsList(bas+6)) glCallList(bas+6);//qpeak-bonds
  }
  else {
    if (bas) {
      //     if (drawAt&&drawADP) {glEnable(GL_BLEND); glCallList(bas+2);}//transparent ellipsoids
      if (drawBo&&glIsList(bas+1)) {
	if (mol->bondColorStyle){
	  qglColor(mol->bondColor);
	  glDisable(GL_COLOR_MATERIAL);
	}
	if (drawADP) glCallList(bas+1);
	else if (tubes) glCallList(bas+9); 
	else glCallList(bas+8);//bonds
	glEnable(GL_COLOR_MATERIAL);
      }
      if (drawHb)  {
	// glCallList(bas+7);//h-bonds
	glPushMatrix();{
	  glScaled( L, L, L );
	  mol->tubes=(tubes&&!drawADP)?1:0;
	  mol->adp=(drawADP)?1:0;
	  mol->h_bonds2(mol->showbonds,mol->showatoms);
	}glPopMatrix();

      }
      if (qPeakBonds->isChecked())glCallList(bas+6);//bonds
      if (drawAt&&drawADP&&glIsList(bas)) {
	glCallList(bas);//ellipsoids
      }
      if (drawAt&&drawADP) glCallList(bas+2);//transparent ellipsoids
      if (drawAt&&!drawADP) {
	if (tubes) {glDisable(GL_BLEND);glCallList(bas+5);} //tubes
	else {glDisable(GL_BLEND);glCallList(bas+3);}//balls
      }
    }//bas
  }//wireButt

  if (foubas[0]|foubas[1]|foubas[2]) {
    double max;
    int Pers=0;
    double mm[16];
    glGetDoublev( GL_MODELVIEW_MATRIX, (double*)mm );
    max= (fabs(mm[2])>fabs(mm[6]))?mm[2]:mm[6];
    max=(fabs(max)<fabs(mm[10]))?mm[10]:max;
    if ((max==mm[2])&&(max>0.0)) Pers=0; else
      if ((max==mm[2])&&(max<0.0)) Pers=1; else
	if ((max==mm[6])&&(max>0.0)) Pers=2; else
	  if ((max==mm[6])&&(max<0.0)) Pers=3; else
	    if ((max==mm[10])&&(max>0.0)) Pers=4; else
	      if ((max==mm[10])&&(max<0.0)) Pers=5;
    //	printf ("Pers = %d\n",Pers);
    if (!niceTrans->isChecked())Pers=0;
    glDisable(GL_CULL_FACE);
    if (fillMap->isChecked()) glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
    else glPolygonMode(GL_FRONT_AND_BACK,GL_LINE);
    if (lighting->isChecked())  glEnable(GL_LIGHTING); 
    else glDisable(GL_LIGHTING);
    glEnable(GL_BLEND);
    if (foact->isChecked()) glCallList(foubas[2+3*Pers]);
    if (fofcact->isChecked()) {
      glCallList(foubas[0+3*Pers]);
      glCallList(foubas[1+3*Pers]);
    }
    glEnable(GL_LIGHTING);
    glDisable(GL_BLEND);
    glEnable(GL_CULL_FACE);
    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
  }
  if (!mol->selectedatoms.isEmpty()){
    glPushMatrix();{
      glScaled( L, L, L );
      mol->tubes=0;
      mol->intern=1;
      mol->adp=0;
      mol->dratom=1;
      mol->atoms(mol->selectedatoms);
      qglColor(labelColor);
      mol->dratom=0;
      glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
    }glPopMatrix();
  }
  if (highlightParts->isChecked()){//PART highlighter
    int alod=mol->LOD;
    mol->LOD=3;
    glPushMatrix();{
      glScaled( L, L, L );
      mol->tubes=(tubes&&!drawADP)?1:0;
      mol->intern=0;
      mol->adp=(drawADP)?1:0;
      mol->dratom=2;
      mol->atoms(mol->showatoms);
      qglColor(labelColor);
    }glPopMatrix();
    glPushMatrix();{
      glScaled( L, L, L );
      mol->bonds(mol->showbonds);
    }glPopMatrix();
    mol->dratom=0;
    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
    mol->LOD=alod;
  }
  if (mol->duplicateAtoms.size()){//Duplicate highlighter
    int alod=mol->LOD;
    mol->LOD=3;
    glPushMatrix();{
      glScaled( L, L, L );
      mol->tubes=(tubes&&!drawADP)?1:0;
      mol->intern=0;
      mol->adp=(drawADP)?1:0;
      mol->dratom=2;
      mol->atoms(mol->duplicateAtoms);

      qglColor(QColor("springgreen"));
      glDisable(GL_DEPTH_TEST);
      if (stereo_mode!=1)
	for (int ii=0; ii<mol->duplicateAtoms.size();ii++)renderText(
			mol->duplicateAtoms.at(ii).pos.x,
			mol->duplicateAtoms.at(ii).pos.y,
			mol->duplicateAtoms.at(ii).pos.z,
			"="+mol->duplicateAtoms.at(ii).Label+"=",
			myFont);

      qglColor(labelColor);
    }glPopMatrix();/*
		      glPushMatrix();{
		      glScaled( L, L, L );
		      mol->bonds(mol->showbonds);
		      }glPopMatrix();*/
    mol->dratom=0;
    glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
    mol->LOD=alod;
  }
  if ((imFokus>-1)&&(!drawLa)&&(imFokus<mol->showatoms.size()))
    if (!warLabel){
      CEnvironment fokat;
      fokat.append(mol->showatoms.at(imFokus));
      glPushMatrix();{
	glScaled( L, L, L );
	mol->tubes=0;
	mol->intern=1;
	mol->adp=0;
	mol->dratom=3;
	mol->atoms(fokat);
	qglColor(labelColor);
	mol->dratom=0;
	glDisable(GL_DEPTH_TEST);
	if (stereo_mode!=1)renderText( 
			fokat.at(0).pos.x,
			fokat.at(0).pos.y,
			fokat.at(0).pos.z, 
			fokat.at(0).Label,
			myFont);
	glPolygonMode(GL_FRONT_AND_BACK,GL_FILL);
	glEnable(GL_DEPTH_TEST);
      }glPopMatrix();

    } 
  if (enviPositions.size()){
  glPushMatrix();
  glDisable( GL_DEPTH_TEST );
  glEnable(GL_LINE_STIPPLE);
  glEnable(GL_BLEND);
  glLineWidth(2);
  glDisable(GL_LIGHTING);
  glLineStipple(3,21845);

  glScaled( L, L, L );
  glBegin(GL_LINES);
  for (int k=0;k<enviPositions.size();k++){
      if ((enviNoQ->isChecked())&&(labs.at(k).startsWith('Q',Qt::CaseInsensitive))) continue;
      switch(enviKat.at(k)){
      case 1:glColor3d(0.0,0.8,0.0); break;
      case 2:glColor3d(0.7,0.4,0.0);break;
      default:
          glColor3d(0.0,0.0,0.0);
      }
      //if ((Distance(enviP0,enviPositions.at(k)))<envirange*envirange)
      {
  glVertex3d(enviP0.x,enviP0.y,enviP0.z);
  glVertex3d(enviPositions.at(k).x,enviPositions.at(k).y,enviPositions.at(k).z);
      }
  }
  glEnd();
  for (int k=0;k<enviPositions.size();k++){
    if ((enviNoQ->isChecked())&&(labs.at(k).startsWith('Q',Qt::CaseInsensitive))) continue;
    switch(enviKat.at(k)){
	    case 1:glColor3d(0.0,0.5,0.0); break;
	    case 2:glColor3d(0.7,0.4,0.0);break;
	    default:
		   glColor3d(0.0,0.0,0.0);
    }
    renderText((enviPositions.at(k).x+enviP0.x)/2.0,
		    (enviPositions.at(k).y+enviP0.y)/2.0,
		    (enviPositions.at(k).z+enviP0.z)/2.0,
		    QString::number(sqrt(Distance(enviP0,enviPositions.at(k))),'f',3)+"Ang",nonAtomFont);
    renderText(enviPositions.at(k).x,
		    enviPositions.at(k).y,
		    enviPositions.at(k).z,
		    labs.at(k),nonAtomFont);
  }
  glDisable(GL_LINE_STIPPLE);
  glEnable(GL_LIGHTING);
  glEnable( GL_DEPTH_TEST );
  glPopMatrix();
  }

  if (bas&&drawUc) glCallList(bas+4); //unit cell
  if ((drawLa)&&(!moving->isActive())&&drawUc) {

    glClear( GL_DEPTH_BUFFER_BIT);
    V3 uz0f,uz1f,uz2f,uz3f;
    V3 uz0k,uz1k,uz2k,uz3k;
    uz0f.x=0.0;  uz0f.y=0.0;  uz0f.z=0.0;
    uz1f.x=1.0;  uz1f.y=0.0;  uz1f.z=0.0;
    uz2f.x=0.0;  uz2f.y=1.0;  uz2f.z=0.0;
    uz3f.x=0.0;  uz3f.y=0.0;  uz3f.z=1.0;
    mol->frac2kart(uz0f,uz0k);
    mol->frac2kart(uz1f,uz1k);
    mol->frac2kart(uz2f,uz2k);
    mol->frac2kart(uz3f,uz3k);
    glPushMatrix();{
      glScaled(L,L,L);{
	qglColor(labelColor);

	renderText(uz0k.x,uz0k.y,uz0k.z,"0",myFont);
	glColor4f(1.0f,0.0f,0.0f,1.0);
	renderText(uz1k.x,uz1k.y,uz1k.z,"a",myFont);

	glColor4f(0.0f,1.0f,0.0f,1.0);
	renderText(uz2k.x,uz2k.y,uz2k.z,"b",myFont);

	glColor4f(0.0f,0.0f,1.0f,1.0);
	renderText(uz3k.x,uz3k.y,uz3k.z,"c",myFont);
      }glPopMatrix();
    }
  }
  if ((drawLa)&&(!moving->isActive())) {
    glDisable(GL_CLIP_PLANE1);
    glDisable(GL_CLIP_PLANE2);
    nonAtomFont = myFont;
    nonAtomFont.setPointSize(qMax(((myFont.pointSize()*2)/3),7));
    glClear( GL_DEPTH_BUFFER_BIT);
    glPushMatrix();{
      glScaled(L,L,L);{
	for (int j=0;j<mol->showatoms.size();j++){
	  if (mol->showatoms.at(j).hidden) continue;
	  if ((mol->nopm1)&&(mol->showatoms.at(j).symmGroup)&&(mol->showatoms.at(j).part==-1)) continue;
	  if (imFokus==j) qglColor(Qt::yellow); else {
	    if (mol->showatoms.at(j).symmGroup) {
	      if ((backGroundColor.saturation()>128)&&(fabs(backGroundColor.hue()-240)<30)) qglColor(Qt::lightGray);
	      else {
		if (backGroundColor.value()<128) qglColor(QColor(Qt::blue).lighter(150));
		else qglColor(Qt::darkBlue);
	      }
	    }
	    else qglColor(labelColor);
	  }
	  GLdouble model[16];
	  glGetDoublev( GL_MODELVIEW_MATRIX, (double*)model );
	  double pla1=0,pla2=0;
	  //{1.0,0.0,-1.0,-190},
	  GLdouble in[4], out[4];

	  in[0] = mol->showatoms.at(j).pos.x;
	  in[1] = mol->showatoms.at(j).pos.y;
	  in[2] = mol->showatoms.at(j).pos.z;
	  in[3] = 1.0;
	  transform_point(out, model, in);
	  pla1=(out[0]*clip_plane1[0]+out[1]*clip_plane1[1]+out[2]*clip_plane1[2]+out[3]*clip_plane1[3]);
	  pla2=(out[0]*clip_plane2[0]+out[1]*clip_plane2[1]+out[2]*clip_plane2[2]+out[3]*clip_plane2[3]);
          if ((xray_eye)&&(pla1<0)&&(pla2<0))continue;
	  {
	    if (imFokus==j) {
	      renderText( 
			      mol->showatoms.at(j).pos.x,
			      mol->showatoms.at(j).pos.y,
			      mol->showatoms.at(j).pos.z, 
			      mol->showatoms.at(j).Label,
			      myFont);
	      continue;

	    }else {
	      if ((mol->showatoms.at(j).an<0)) renderText(//(mol->selectedatoms.isEmpty())&&
			      mol->showatoms.at(j).pos.x,
			      mol->showatoms.at(j).pos.y,
			      mol->showatoms.at(j).pos.z, 
			      mol->showatoms.at(j).Label,
			      nonAtomFont);
	      else
		if ((!(mol->AtomStyle[mol->showatoms.at(j).an]&ATOM_STYLE_NOLABEL)))//(mol->selectedatoms.isEmpty())&&
		  renderText(
				  mol->showatoms.at(j).pos.x,
				  mol->showatoms.at(j).pos.y,
				  mol->showatoms.at(j).pos.z,
				  mol->showatoms.at(j).Label,
				  (mol->selectedatoms.isEmpty())?myFont:nonAtomFont);
	    }
	  }
	}
      }
      for (int j=0;j<mol->selectedatoms.size();j++){
	qglColor(Qt::yellow);
	renderText(
			mol->selectedatoms.at(j).pos.x,
			mol->selectedatoms.at(j).pos.y,
			mol->selectedatoms.at(j).pos.z,
			mol->selectedatoms.at(j).Label,
			myFont);
      }
    }glPopMatrix();
  }  
  {
    glPushMatrix();
    glScaled(L,L,L);{

      GLdouble model[16];
      GLdouble proj[16];
      GLint viewport[4];
      glGetIntegerv(GL_VIEWPORT, viewport);
      glGetDoublev( GL_PROJECTION_MATRIX , (double*)proj );
      glGetDoublev( GL_MODELVIEW_MATRIX, (double*)model );
      for (int j=0;j<mol->showatoms.size();j++){
	{mol->showatoms[j].screenX=-200; mol->showatoms[j].screenY=-200;}
	if (mol->showatoms.at(j).hidden) continue;
	double pla1=0,pla2=0;
	//{1.0,0.0,-1.0,-190},
	GLdouble in[4], out[4];

	in[0] = mol->showatoms.at(j).pos.x;
	in[1] = mol->showatoms.at(j).pos.y;
	in[2] = mol->showatoms.at(j).pos.z;
	in[3] = 1.0;
	transform_point(out, model, in);
	pla1=(out[0]*clip_plane1[0]+out[1]*clip_plane1[1]+out[2]*clip_plane1[2]+out[3]*clip_plane1[3]);
	pla2=(out[0]*clip_plane2[0]+out[1]*clip_plane2[1]+out[2]*clip_plane2[2]+out[3]*clip_plane2[3]);
	if ((xray_eye)&&(pla1<0)&&(pla2<0))continue;

	if ((mol->nopm1)&&(mol->showatoms.at(j).symmGroup)&&(mol->showatoms.at(j).part==-1)) continue;
	if (!posTo2D(mol->showatoms.at(j).pos,model,proj,viewport, &mol->showatoms[j].screenX, &mol->showatoms[j].screenY))
	{mol->showatoms[j].screenX=-200; mol->showatoms[j].screenY=-200;}
      }
    }
    glPopMatrix();
  }

  glPopMatrix();  
  afok=imFokus;
}

