 /*==========================================================================
                SeqAn - The Library for Sequence Analysis
                          http://www.seqan.de 
 ============================================================================
  Copyright (C) 2007

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 3 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  Lesser General Public License for more details.

 ============================================================================
  $Id: score_pam_matrix.h 954 2007-07-27 11:48:23Z doering@PCPOOL.MI.FU-BERLIN.DE $
 ==========================================================================*/

#ifndef SEQAN_HEADER_SCORE_PAM_MATRIX_H
#define SEQAN_HEADER_SCORE_PAM_MATRIX_H



namespace SEQAN_NAMESPACE_MAIN
{

struct Pam_Data_Dayhoff_MDM78{};
struct Pam_Data_Jones_PRI29{};
struct Pam_Data_Jones_PET91_SWISS15{};
struct Pam_Data_Jones_PET91_SWISS22{};
struct Pam_Data_Jones_All_Membrane{};
struct Pam_Data_Jones_Single_Membrane{};
struct Pam_Data_Jones_Multi_Membrane{};

/*template <typename TSourceValue, typename TSequenceValue, typename TScoure>
// TSourceValue: type of source data matrices (e.g. double)
void _retrievePamData(_TempMembersPam<TValue> & _member, Score <TValue, Pam<TSequenceValue, TSource> > & _score){
}

*/

template <typename TValue, typename TSequenceValue>
struct _TempMembersPam {
	enum{dim = ValueSize<AminoAcid>::VALUE};
	TValue min_score;	// range of PAM matrix
	double single_prob[dim];	// normalized frequencies of amino scids
	double trans_prob[dim][dim];// mutation probability matrix for distance of 1 PAM
	double log_prob[dim][dim];  // logarithm of odds matrix
};



/**
.Internal._TempMembers
..summary:Container containing temporary information necessary for Pam matrix calculation.
..cat:Classes
..signature:_TempMembers<TValue, TSequenceValue>
..param.TValue:Type of the score values.
...default:$int$
..param.TSequenceValue:Type of alphabet underlying the matrix.
...default:$AminoAcid$
.Internal._extendAlphabetPam.param._member.type:Class._TempMembers
.Internal._extrapolatePam.param._member.type:Class._TempMembers
.Internal._finishPam.param._member.type:Class._TempMembers
.Internal._computeLogOddsPam.param._member.type:Class._TempMembers
.Internal._computeEntropyPam.param._member.type:Class._TempMembers
*/



template <typename TSequenceValue = AminoAcid, typename TSource = Pam_Data_Dayhoff_MDM78>
struct Pam;




template <typename TValue, typename TSequenceValue, typename TSource>
void _starting_data_pam(_TempMembersPam<TValue, TSequenceValue> & _member, Score <TValue, Pam<TSequenceValue, TSource> > & _score){
}

template <typename TValue>
void _starting_data_pam(_TempMembersPam<TValue, AminoAcid> & _member, Score <TValue, Pam<AminoAcid, Pam_Data_Dayhoff_MDM78> > &){
	const int dim = ValueSize<AminoAcid>::VALUE;

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Amino acid exchange data, computation as described in ATLAS OF PROTEIN SEQUENCE AND STRUCTURE, Supplement 2
// Ref. A Model of Evolutionary Change in Proteins (Margaret Dayhoff, R.M. Schwartz, B.C. Orcutt, 1978)
// Accepted point mutations: 1572
/////////////////////////////////////////////////////////////////////////////////////////////////////////////

 double TRANS_DH[dim][dim]= // dim = 24, 1 PAM probability matrix (Dayhoff 1978)
{
{0.986693	, 0.000109553	, 0.000398041	, 0.00056237	, 0.000120508	, 0.000339613	, 0.000971367	, 0.00211437	, 7.66868e-005	, 0.000241016	, 0.000346917	, 0.00020815	, 0.000105901	, 7.30351e-005	, 0.00125986	, 0.00281915	, 0.00215454	, 0.			, 7.30351e-005	, 0.00133289	,0., 0., 0., 0.},
{0.000233351, 0.99135		, 0.000132232	, 0.			, 7.77837e-005	, 0.000933404	, 0				,7.77837e-005	, 0.000801172	, 0.000233351	, 0.000132232	, 0.00371028	, 0.000132232	, 5.44486e-005	, 0.000521151	, 0.00106564	, 0.000155567	, 0.000210016	, 2.33351e-005	, 0.000155567	,0., 0., 0., 0.},
{0.000857731, 0.000133775	, 0.982169		, 0.00418636	, 0.			, 0.000393455	, 0.000739695	, 0.00122758	, 0.00177842	, 0.000283287	, 0.000291157	, 0.00253385	, 0.			, 5.50837e-005	, 0.000212466	, 0.00339945	, 0.00132988	, 2.36073e-005	, 0.000283287	, 0.000102298	,0., 0., 0., 0.},
{0.00104535	, 0.0			, 0.0036112		, 0.985895		, 0.			, 0.000515886	, 0.00564081	, 0.00109965	, 0.000291883	, 8.82437e-005	, 0.			, 0.000576978	, 0.			, 0				, 6.78798e-005	, 0.000665222	, 0.000386915	, 0.			, 0.			, 0.000115396	,0., 0., 0., 0.},
{0.000313665, 9.50499e-005	, 0.			, 0.			, 0.997339		,		0.		, 0.			, 9.50499e-005	, 9.50499e-005	, 0.000161585	, 0.			, 0.			, 0.			, 0.			, 9.50499e-005	, 0.00111208	, 9.50499e-005	, 0.			, 0.00028515	, 0.000313665	,0., 0., 0., 0.},
{0.000773469, 0.000998024	, 0.000415844	, 0.000632082	, 0.			, 0.987624		, 0.00350972	, 0.000249506	, 0.002021		, 6.6535e-005	, 0.000623765	, 0.00122258	, 0.000166337	, 0.			, 0.000773469	, 0.000390893	, 0.000307724	, 0.			, 0.			, 0.000224555	,0., 0., 0., 0.},
{0.00170869	, 0.0			, 0.000603821	, 0.00533804	, 0.			, 0.00271077	, 0.986427		, 0.000719447	, 0.000147744	, 0.000224827	, 9.63545e-005	, 0.000668058	, 4.49654e-005	, 0.			, 0.000256945	, 0.000552432	, 0.000199133	, 0.			, 6.42363e-005	, 0.000237674	,0., 0., 0., 0.},
{0.00207892	, 3.59054e-005	, 0.000560125	, 0.000581668	, 3.59054e-005	, 0.000107716	, 0.000402141	, 0.99348		, 3.59054e-005	, 0.			, 6.10392e-005	, 0.000215433	, 2.51338e-005	, 6.10392e-005	, 0.000175937	, 0.00161574	, 0.000179527	, 0.			, 0.			, 0.000348283	,0., 0., 0., 0.},
{0.000198745, 0.000974795	, 0.00213887	, 0.000406953	, 9.46402e-005	, 0.00229976	, 0.000217673	, 9.46402e-005	, 0.991217		, 2.83921e-005	, 0.000378561	, 0.000217673	, 0.			, 0.00018928	, 0.000473201	, 0.000246065	, 0.000132496	, 2.83921e-005	, 0.000378561	, 0.000283921	,0., 0., 0., 0.},
{0.000569298, 0.000258772	, 0.000310526	, 0.000112135	, 0.000146637	, 6.90059e-005	, 0.000301901	, 0.			, 2.58772e-005	, 0.987225		, 0.00218231	, 0.000370907	, 0.000491667	, 0.000776316	, 6.03801e-005	, 0.000172515	, 0.00111272	, 0.			, 0.000112135	, 0.00570161	,0., 0., 0., 0.},
{0.000354108, 6.33666e-005	, 0.000137916	, 0.			, 0.			, 0.000279559	, 5.59117e-005	, 6.33666e-005	, 0.000149098	, 0.000943045	, 0.994677		, 0.000145371	, 0.000771582	, 0.000622484	, 0.00016028	, 0.000119278	, 0.000193827	, 4.84568e-005	, 8.57313e-005	, 0.00112942	,0., 0., 0., 0.},
{0.000225336, 0.00188571	, 0.00127295	, 0.000336028	, 0.			, 0.000581131	, 0.00041114	, 0.000237196	, 9.09252e-005	, 0.000169991	, 0.000154178	, 0.992548		, 0.000355794	, 0.			, 0.000169991	, 0.00066415	, 0.000790654	, 0.			, 3.95327e-005	, 7.72056e-005	,0., 0., 0., 0.},
{0.000625429, 0.000366631	, 0.			, 0.			, 0.			, 0.00043133	, 0.000150966	, 0.000150966	, 0.			, 0.00122929	, 0.00446427	, 0.00194099	, 0.987491		, 0.000366631	, 8.6266e-005	, 0.00043133	, 0.000603862	, 0.			, 0.			, 0.00166062	,0., 0., 0., 0.},
{0.000159996, 5.59986e-005	, 5.59986e-005	, 0.			, 0.			, 0.			, 0.			, 0.000135997	, 0.000159996	, 0.000719982	, 0.00133597	, 0.			, 0.000135997	, 0.994544		, 5.59986e-005	, 0.000319992	, 7.9998e-005	, 7.9998e-005	, 0.00207995	, 7.9998e-005	,0., 0., 0., 0.},
{0.00216589	, 0.000420622	, 0.000169504	, 6.27794e-005	, 6.27794e-005	, 0.000583849	, 0.000251118	, 0.000307619	, 0.000313897	, 4.39456e-005	, 0.000269951	, 0.000269951	, 2.51118e-005	, 4.39456e-005	, 0.992548		, 0.00168877	, 0.00045829	, 0.			, 0.			, 0.000313897	,0., 0., 0., 0.},
{0.00353024	, 0.00062648	, 0.00197547	, 0.000448139	, 0.000535023	, 0.000214924	, 0.000393265	, 0.00205778	, 0.000118894	, 9.1457e-005	, 0.000146331	, 0.000768239	, 9.1457e-005	, 0.000182914	, 0.0012301		, 0.984032		, 0.0031827		, 7.77384e-005	, 0.000100603	, 0.000196633	,0., 0., 0., 0.},
{0.00320656	, 0.000108697	, 0.000918491	, 0.000309787	, 5.43486e-005	, 0.00020109	, 0.000168481	, 0.000271743	, 7.6088e-005	, 0.000701096	, 0.000282613	, 0.00108697	, 0.000152176	, 5.43486e-005	, 0.000396744	, 0.00378266	, 0.987092		, 0.			, 0.000125002	, 0.00101088	,0., 0., 0., 0.},
{0.			, 0.000818633	, 9.09592e-005	, 0.			, 0.			, 0.			, 0.			, 0.			, 9.09592e-005	, 0.			, 0.000394156	, 0.			, 0.			, 0.000303197	, 0.			, 0.000515435	, 0.			, 0.997605		, 0.000181918	, 0				,0., 0., 0., 0.},
{0.000212704, 3.19057e-005	, 0.000382868	, 0.			, 0.000319057	, 0.			, 0.000106352	, 0.			, 0.000425409	, 0.000138258	, 0.00024461	, 0.000106352	, 0.			, 0.00276516	, 0.			, 0.000233975	, 0.00024461	, 6.38113e-005	, 0.994544		, 0.000180799	,0., 0., 0., 0.},
{0.00179442	, 9.83243e-005	, 6.39108e-005	, 8.35756e-005	, 0.000162235	, 0.000132738	, 0.0001819		, 0.000476873	, 0.000147486	, 0.00324962	, 0.00148961	, 8.35756e-005	, 0.000378548	, 4.91621e-005	, 0.000245811	, 0.000211397	, 0.000914416	, 0.			, 8.35756e-005	, 0.990153		,0., 0., 0., 0.},
{0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000},
{0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000},
{0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000},
{0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000, 0.0000}
};




double OCCURENCE_DH[dim]=  {0.08713, 0.0409, 0.04043, 0.04687, 0.03347, 0.03826, 0.04953, 0.08861, 0.03362, 0.03689, 0.08536, 0.08048, 0.01475, 0.03977, 0.05068, 0.06958, 0.05854, 0.01049, 0.02992, 0.06472, 0., 0., 0., 0.};

	arrayCopyForward(OCCURENCE_DH,OCCURENCE_DH+dim,_member.single_prob);
	arrayCopyForward(* TRANS_DH,* TRANS_DH+dim*dim,* _member.trans_prob);
}


template <typename TValue>
void _starting_data_pam(_TempMembersPam<TValue, AminoAcid> & _member, Score <TValue, Pam<AminoAcid, Pam_Data_Jones_PRI29> > & _score){
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Amino acid exchange data generated from PIR Release 29.0
// Ref. Jones D.T., Taylor W.R. and Thornton J.M. (1992) CABIOS 8:275-282
// Usable sequences: 28266
// Final alignments: 7855
// Accepted point mutations: 90079
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
 	const int dim = ValueSize<AminoAcid>::VALUE;

	const double TRANS_JONES_PIR29[dim][dim] = // dim = 24, Normalized Frequencies of Amino Acids (Jones 1992, PIR Release 29.0)
{{0.98762,  0.00063,  0.00039,  0.00064,  0.00046,  0.00043,  0.00075,  0.00138,  0.00024,  0.00028,  0.00024,  0.00024,  0.00038,  0.00012,  0.00151,  0.00267,  0.00334,  0.00010,  0.00012,  0.00224, 0., 0., 0., 0.},
 {0.00043,  0.98931,  0.00023,  0.00010,  0.00056,  0.00159,  0.00021,  0.00075,  0.00177,  0.00012,  0.00021,  0.00305,  0.00020,  0.00006,  0.00045,  0.00057,  0.00035,  0.00072,  0.00012,  0.00012, 0., 0., 0., 0.},
 {0.00022,  0.00019,  0.98693,  0.00230,  0.00013,  0.00036,  0.00023,  0.00028,  0.00170,  0.00023,  0.00006,  0.00109,  0.00025,  0.00006,  0.00007,  0.00207,  0.00091,  0.00005,  0.00035,  0.00007, 0., 0., 0., 0.},
 {0.00043,  0.00010,  0.00285,  0.98947,  0.00011,  0.00028,  0.00365,  0.00073,  0.00062,  0.00009,  0.00004,  0.00014,  0.00010,  0.00002,  0.00009,  0.00031,  0.00024,  0.00006,  0.00029,  0.00017, 0., 0., 0., 0.},
 {0.00012,  0.00022,  0.00006,  0.00004,  0.99429,  0.00003,  0.00001,  0.00011,  0.00014,  0.00003,  0.00006,  0.00002,  0.00008,  0.00013,  0.00003,  0.00042,  0.00009,  0.00028,  0.00037,  0.00014, 0., 0., 0., 0.},
 {0.00023,  0.00127,  0.00035,  0.00022,  0.00006,  0.98943,  0.00146,  0.00012,  0.00240,  0.00004,  0.00030,  0.00111,  0.00017,  0.00003,  0.00061,  0.00022,  0.00021,  0.00011,  0.00008,  0.00008, 0., 0., 0., 0.},
 {0.00061,  0.00025,  0.00034,  0.00435,  0.00004,  0.00221,  0.99062,  0.00076,  0.00020,  0.00006,  0.00007,  0.00110,  0.00010,  0.00004,  0.00012,  0.00019,  0.00019,  0.00009,  0.00005,  0.00028, 0., 0., 0., 0.},
 {0.00133,  0.00107,  0.00050,  0.00103,  0.00041,  0.00022,  0.00090,  0.99319,  0.00016,  0.00007,  0.00008,  0.00019,  0.00015,  0.00005,  0.00021,  0.00146,  0.00027,  0.00048,  0.00009,  0.00040, 0., 0., 0., 0.},
 {0.00007,  0.00077,  0.00092,  0.00027,  0.00016,  0.00132,  0.00007,  0.00005,  0.98804,  0.00006,  0.00013,  0.00009,  0.00013,  0.00009,  0.00030,  0.00017,  0.00014,  0.00003,  0.00123,  0.00003, 0., 0., 0., 0.},
 {0.00019,  0.00012,  0.00029,  0.00009,  0.00008,  0.00005,  0.00005,  0.00005,  0.00013,  0.98748,  0.00115,  0.00010,  0.00249,  0.00050,  0.00007,  0.00023,  0.00127,  0.00006,  0.00021,  0.00457, 0., 0., 0., 0.},
 {0.00028,  0.00037,  0.00012,  0.00007,  0.00025,  0.00067,  0.00010,  0.00009,  0.00055,  0.00203,  0.99320,  0.00014,  0.00332,  0.00230,  0.00100,  0.00057,  0.00025,  0.00059,  0.00026,  0.00162, 0., 0., 0., 0.},
 {0.00018,  0.00348,  0.00153,  0.00016,  0.00005,  0.00159,  0.00104,  0.00015,  0.00025,  0.00012,  0.00009,  0.99145,  0.00038,  0.00003,  0.00012,  0.00026,  0.00054,  0.00007,  0.00006,  0.00008, 0., 0., 0., 0.},
 {0.00012,  0.00009,  0.00014,  0.00004,  0.00009,  0.00009,  0.00004,  0.00005,  0.00014,  0.00112,  0.00084,  0.00015,  0.98819,  0.00010,  0.00003,  0.00007,  0.00051,  0.00009,  0.00007,  0.00075, 0., 0., 0., 0.},
 {0.00006,  0.00005,  0.00006,  0.00002,  0.00025,  0.00003,  0.00003,  0.00003,  0.00016,  0.00039,  0.00102,  0.00002,  0.00018,  0.99328,  0.00009,  0.00041,  0.00006,  0.00021,  0.00210,  0.00027, 0., 0., 0., 0.},
 {0.00101,  0.00045,  0.00009,  0.00008,  0.00009,  0.00076,  0.00010,  0.00015,  0.00067,  0.00007,  0.00056,  0.00010,  0.00006,  0.00011,  0.99248,  0.00134,  0.00063,  0.00007,  0.00007,  0.00012, 0., 0., 0., 0.},
 {0.00248,  0.00079,  0.00350,  0.00042,  0.00147,  0.00039,  0.00021,  0.00140,  0.00052,  0.00032,  0.00044,  0.00032,  0.00022,  0.00072,  0.00186,  0.98591,  0.00309,  0.00030,  0.00050,  0.00028, 0., 0., 0., 0.},
 {0.00260,  0.00040,  0.00129,  0.00028,  0.00025,  0.00031,  0.00018,  0.00022,  0.00038,  0.00147,  0.00016,  0.00054,  0.00131,  0.00008,  0.00073,  0.00259,  0.98711,  0.00015,  0.00020,  0.00059, 0., 0., 0., 0.},
 {0.00002,  0.00020,  0.00002,  0.00001,  0.00019,  0.00004,  0.00002,  0.00009,  0.00002,  0.00002,  0.00009,  0.00002,  0.00005,  0.00007,  0.00002,  0.00006,  0.00004,  0.99604,  0.00010,  0.00005, 0., 0., 0., 0.},
 {0.00005,  0.00008,  0.00027,  0.00018,  0.00060,  0.00006,  0.00003,  0.00004,  0.00181,  0.00014,  0.00009,  0.00003,  0.00010,  0.00173,  0.00004,  0.00023,  0.00011,  0.00024,  0.99361,  0.00007, 0., 0., 0., 0.},
 {0.00194,  0.00015,  0.00012,  0.00022,  0.00047,  0.00013,  0.00030,  0.00036,  0.00009,  0.00586,  0.00117,  0.00009,  0.00214,  0.00044,  0.00015,  0.00026,  0.00066,  0.00024,  0.00015,  0.98807, 0., 0., 0., 0.},
 {0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000, 0., 0., 0., 0.}, 
 {0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000, 0., 0., 0., 0.}, 
 {0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000, 0., 0., 0., 0.}, 
 {0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000,  0.00000, 0., 0., 0., 0.}};

	const double OCC_JONES_PIR29[dim]= // dim = 24, Normalized Frequencies of Amino Acids (Jones 1992, PIR Release 29.0)
{0.076097, 0.0514338, 0.041855, 0.0519453, 0.0203737, 0.0409894, 0.0619809, 0.0734405, 0.0224972, 0.0515443, 0.0913336, 0.0587453, 0.0232275, 0.0402897, 0.0508347, 0.070812, 0.0592891, 0.0140877, 0.033163, 0.0660604, 0., 0., 0., 0.};

	arrayCopyForward(OCC_JONES_PIR29,OCC_JONES_PIR29+dim,_member.single_prob);
	arrayCopyForward(* TRANS_JONES_PIR29,* TRANS_JONES_PIR29+dim*dim,* _member.trans_prob);
}


template <typename TValue>
void _starting_data_pam(_TempMembersPam<TValue, AminoAcid> & _member, Score <TValue, Pam<AminoAcid, Pam_Data_Jones_PET91_SWISS15> > & _score){
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Amino acid exchange data generated from SWISSPROT Release 15.0
// Ref. Jones D.T., Taylor W.R. and Thornton J.M. (1992) CABIOS 8:275-282
// Usable sequences: 16130
// Final alignments: 3677
// Accepted point mutations: 59190
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
   	const int dim = ValueSize<AminoAcid>::VALUE;
	const double TRANS_JONES_SWISS15[dim][dim] = // dim = 24, Normalized Frequencies of Amino Acids (Jones 1992, Swissprot 15.0)
{{0.98759, 0.00041, 0.00043, 0.00063, 0.00044, 0.00043, 0.00082, 0.00135, 0.00017, 0.00028, 0.00024, 0.00028, 0.00036, 0.00011, 0.00150, 0.00297, 0.00351, 0.00007, 0.00011, 0.00226},
 {0.00027, 0.98962, 0.00023, 0.00008, 0.00052, 0.00154, 0.00016, 0.00070, 0.00164, 0.00012, 0.00019, 0.00334, 0.00022, 0.00003, 0.00036, 0.00051, 0.00033, 0.00065, 0.00012, 0.00009},
 {0.00024, 0.00019, 0.98707, 0.00235, 0.00013, 0.00033, 0.00025, 0.00033, 0.00171, 0.00021, 0.00006, 0.00108, 0.00014, 0.00003, 0.00005, 0.00214, 0.00100, 0.00001, 0.00030, 0.00007},
 {0.00042, 0.00008, 0.00284, 0.98932, 0.00005, 0.00027, 0.00398, 0.00066, 0.00053, 0.00006, 0.00003, 0.00014, 0.00010, 0.00002, 0.00007, 0.00030, 0.00022, 0.00003, 0.00023, 0.00016},
 {0.00012, 0.00021, 0.00006, 0.00002, 0.99450, 0.00002, 0.00001, 0.00011, 0.00015, 0.00003, 0.00003, 0.00001, 0.00008, 0.00014, 0.00003, 0.00044, 0.00009, 0.00023, 0.00043, 0.00013},
 {0.00023, 0.00125, 0.00031, 0.00021, 0.00004, 0.98955, 0.00140, 0.00010, 0.00233, 0.00003, 0.00029, 0.00122, 0.00019, 0.00002, 0.00066, 0.00022, 0.00021, 0.00007, 0.00010, 0.00007},
 {0.00066, 0.00020, 0.00036, 0.00478, 0.00003, 0.00211, 0.99042, 0.00070, 0.00015, 0.00007, 0.00006, 0.00107, 0.00011, 0.00003, 0.00012, 0.00019, 0.00020, 0.00007, 0.00004, 0.00029},
 {0.00129, 0.00102, 0.00058, 0.00095, 0.00041, 0.00017, 0.00083, 0.99369, 0.00015, 0.00004, 0.00005, 0.00020, 0.00010, 0.00004, 0.00016, 0.00139, 0.00024, 0.00041, 0.00004, 0.00035},
 {0.00005, 0.00074, 0.00092, 0.00024, 0.00017, 0.00130, 0.00006, 0.00005, 0.98867, 0.00004, 0.00012, 0.00012, 0.00008, 0.00011, 0.00026, 0.00017, 0.00011, 0.00003, 0.00134, 0.00003},
 {0.00019, 0.00013, 0.00026, 0.00006, 0.00008, 0.00004, 0.00006, 0.00003, 0.00010, 0.98722, 0.00122, 0.00011, 0.00253, 0.00041, 0.00005, 0.00021, 0.00134, 0.00007, 0.00016, 0.00504},
 {0.00028, 0.00034, 0.00012, 0.00006, 0.00015, 0.00064, 0.00009, 0.00006, 0.00049, 0.00212, 0.99328, 0.00013, 0.00350, 0.00230, 0.00097, 0.00054, 0.00025, 0.00049, 0.00022, 0.00161},
 {0.00022, 0.00390, 0.00150, 0.00017, 0.00003, 0.00176, 0.00103, 0.00016, 0.00031, 0.00012, 0.00009, 0.99101, 0.00037, 0.00001, 0.00013, 0.00028, 0.00057, 0.00005, 0.00005, 0.00007},
 {0.00011, 0.00010, 0.00008, 0.00004, 0.00010, 0.00011, 0.00004, 0.00003, 0.00008, 0.00113, 0.00090, 0.00015, 0.98845, 0.00010, 0.00004, 0.00007, 0.00049, 0.00005, 0.00004, 0.00071},
 {0.00006, 0.00003, 0.00003, 0.00001, 0.00028, 0.00002, 0.00002, 0.00002, 0.00018, 0.00031, 0.00101, 0.00001, 0.00018, 0.99357, 0.00006, 0.00038, 0.00006, 0.00022, 0.00222, 0.00024},
 {0.00099, 0.00036, 0.00006, 0.00006, 0.00006, 0.00081, 0.00010, 0.00011, 0.00058, 0.00005, 0.00053, 0.00011, 0.00008, 0.00008, 0.99278, 0.00140, 0.00059, 0.00004, 0.00006, 0.00011},
 {0.00264, 0.00069, 0.00344, 0.00040, 0.00147, 0.00037, 0.00021, 0.00129, 0.00051, 0.00028, 0.00040, 0.00032, 0.00019, 0.00065, 0.00190, 0.98548, 0.00325, 0.00021, 0.00043, 0.00028},
 {0.00267, 0.00038, 0.00137, 0.00025, 0.00028, 0.00031, 0.00019, 0.00019, 0.00028, 0.00149, 0.00016, 0.00057, 0.00123, 0.00008, 0.00069, 0.00278, 0.98670, 0.00005, 0.00012, 0.00067},
 {0.00001, 0.00018, 0.00000, 0.00001, 0.00016, 0.00002, 0.00002, 0.00008, 0.00002, 0.00002, 0.00008, 0.00001, 0.00003, 0.00008, 0.00001, 0.00004, 0.00001, 0.99684, 0.00011, 0.00003},
 {0.00004, 0.00008, 0.00023, 0.00015, 0.00068, 0.00008, 0.00002, 0.00002, 0.00189, 0.00010, 0.00008, 0.00003, 0.00006, 0.00179, 0.00004, 0.00020, 0.00006, 0.00024, 0.99377, 0.00005},
 {0.00193, 0.00011, 0.00011, 0.00021, 0.00041, 0.00012, 0.00031, 0.00032, 0.00008, 0.00630, 0.00117, 0.00008, 0.00201, 0.00040, 0.00014, 0.00027, 0.00076, 0.00016, 0.00011, 0.98772}};


	const double OCC_JONES_SWISS15[dim]= // dim = 24, Normalized Frequencies of Amino Acids (Jones 1992, Swissprot 15.0)
{0.0772425, 0.0508841, 0.0426636, 0.0514551, 0.0202286, 0.041147, 0.061926, 0.0738581, 0.0230227, 0.0528239, 0.0913305, 0.0594405, 0.0235088, 0.0400974, 0.0505928, 0.0685255, 0.0587051, 0.0141565, 0.0323276, 0.0660635};


	
	arrayCopyForward(OCC_JONES_SWISS15,OCC_JONES_SWISS15+dim,_member.single_prob);
	arrayCopyForward(* TRANS_JONES_SWISS15,* TRANS_JONES_SWISS15+dim*dim,* _member.trans_prob);
}
template <typename TValue>
void _starting_data_pam(_TempMembersPam<TValue, AminoAcid> & _member, Score <TValue, Pam<AminoAcid, Pam_Data_Jones_PET91_SWISS22> > & _score){
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Amino acid exchange data generated from SWISSPROT Release 22.0
// Ref. Jones D.T., Taylor W.R. and Thornton J.M. (1992) CABIOS 8:275-282
//
// Usable sequences: 23824
// Final alignments: 5437
// Accepted point mutations: 92883
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
	const int dim = ValueSize<AminoAcid>::VALUE;
	const double TRANS_JONES_SWISS22[dim][dim] = // dim = 24, transition probability matrix of amino acids (Jones 1992, Swissprot 22.0)
{{0.98754, 0.00044, 0.00042, 0.00062, 0.00043, 0.00044, 0.00080, 0.00136, 0.00021, 0.00029, 0.00023, 0.00027, 0.00042, 0.00011, 0.00148, 0.00287, 0.00360, 0.00007, 0.00008, 0.00226}, 
 {0.00030, 0.98974, 0.00023, 0.00008, 0.00058, 0.00159, 0.00015, 0.00070, 0.00168, 0.00011, 0.00019, 0.00331, 0.00023, 0.00003, 0.00038, 0.00052, 0.00033, 0.00065, 0.00010, 0.00009}, 
 {0.00023, 0.00019, 0.98720, 0.00223, 0.00015, 0.00037, 0.00025, 0.00035, 0.00165, 0.00020, 0.00005, 0.00111, 0.00013, 0.00004, 0.00007, 0.00212, 0.00098, 0.00003, 0.00030, 0.00007}, 
 {0.00042, 0.00008, 0.00269, 0.98954, 0.00005, 0.00025, 0.00392, 0.00067, 0.00057, 0.00006, 0.00004, 0.00014, 0.00008, 0.00002, 0.00008, 0.00031, 0.00020, 0.00002, 0.00024, 0.00016}, 
 {0.00011, 0.00022, 0.00007, 0.00002, 0.99432, 0.00002, 0.00001, 0.00012, 0.00014, 0.00003, 0.00005, 0.00001, 0.00006, 0.00015, 0.00003, 0.00044, 0.00008, 0.00023, 0.00041, 0.00012}, 
 {0.00023, 0.00125, 0.00035, 0.00020, 0.00004, 0.98955, 0.00130, 0.00011, 0.00241, 0.00004, 0.00029, 0.00118, 0.00018, 0.00002, 0.00067, 0.00022, 0.00021, 0.00008, 0.00010, 0.00008}, 
 {0.00065, 0.00018, 0.00036, 0.00470, 0.00003, 0.00198, 0.99055, 0.00074, 0.00016, 0.00007, 0.00006, 0.00111, 0.00011, 0.00003, 0.00011, 0.00018, 0.00020, 0.00006, 0.00004, 0.00027}, 
 {0.00130, 0.00099, 0.00059, 0.00095, 0.00043, 0.00019, 0.00087, 0.99350, 0.00017, 0.00004, 0.00005, 0.00020, 0.00011, 0.00004, 0.00018, 0.00146, 0.00024, 0.00040, 0.00006, 0.00034}, 
 {0.00006, 0.00075, 0.00089, 0.00025, 0.00016, 0.00136, 0.00006, 0.00005, 0.98864, 0.00004, 0.00013, 0.00011, 0.00007, 0.00009, 0.00026, 0.00017, 0.00011, 0.00002, 0.00130, 0.00003}, 
 {0.00020, 0.00012, 0.00025, 0.00006, 0.00009, 0.00005, 0.00006, 0.00003, 0.00009, 0.98729, 0.00122, 0.00011, 0.00255, 0.00047, 0.00006, 0.00021, 0.00131, 0.00005, 0.00017, 0.00511}, 
 {0.00028, 0.00035, 0.00011, 0.00006, 0.00021, 0.00066, 0.00009, 0.00006, 0.00051, 0.00209, 0.99330, 0.00013, 0.00354, 0.00227, 0.00093, 0.00054, 0.00024, 0.00048, 0.00022, 0.00165}, 
 {0.00021, 0.00376, 0.00153, 0.00015, 0.00004, 0.00170, 0.00105, 0.00016, 0.00027, 0.00012, 0.00008, 0.99100, 0.00038, 0.00002, 0.00012, 0.00027, 0.00060, 0.00006, 0.00005, 0.00008}, 
 {0.00013, 0.00010, 0.00007, 0.00004, 0.00007, 0.00010, 0.00004, 0.00003, 0.00008, 0.00113, 0.00092, 0.00015, 0.98818, 0.00010, 0.00004, 0.00007, 0.00053, 0.00006, 0.00004, 0.00076}, 
 {0.00006, 0.00002, 0.00004, 0.00002, 0.00031, 0.00002, 0.00002, 0.00002, 0.00016, 0.00035, 0.00099, 0.00002, 0.00017, 0.99360, 0.00007, 0.00037, 0.00005, 0.00021, 0.00214, 0.00025}, 
 {0.00098, 0.00037, 0.00008, 0.00008, 0.00007, 0.00083, 0.00009, 0.00013, 0.00058, 0.00005, 0.00052, 0.00011, 0.00008, 0.00009, 0.99270, 0.00144, 0.00060, 0.00003, 0.00005, 0.00012}, 
 {0.00257, 0.00069, 0.00342, 0.00041, 0.00152, 0.00037, 0.00021, 0.00137, 0.00050, 0.00027, 0.00040, 0.00032, 0.00020, 0.00063, 0.00194, 0.98556, 0.00324, 0.00024, 0.00043, 0.00026}, 
 {0.00275, 0.00037, 0.00135, 0.00023, 0.00025, 0.00030, 0.00019, 0.00020, 0.00027, 0.00142, 0.00015, 0.00060, 0.00131, 0.00007, 0.00069, 0.00276, 0.98665, 0.00007, 0.00012, 0.00066}, 
 {0.00001, 0.00018, 0.00001, 0.00001, 0.00016, 0.00003, 0.00001, 0.00008, 0.00001, 0.00001, 0.00007, 0.00001, 0.00003, 0.00008, 0.00001, 0.00005, 0.00002, 0.99686, 0.00010, 0.00004}, 
 {0.00003, 0.00006, 0.00022, 0.00015, 0.00067, 0.00008, 0.00002, 0.00003, 0.00182, 0.00010, 0.00008, 0.00003, 0.00006, 0.00171, 0.00003, 0.00020, 0.00007, 0.00023, 0.99392, 0.00005}, 
 {0.00194, 0.00012, 0.00011, 0.00020, 0.00041, 0.00013, 0.00029, 0.00031, 0.00008, 0.00627, 0.00118, 0.00009, 0.00212, 0.00041, 0.00015, 0.00025, 0.00074, 0.00017, 0.00011, 0.98761}};

	const double OCC_JONES_SWISS22[dim]= // dim = 24, normalized frequencies of amino acids (Jones 1992, Swissprot 22.0)
{0.0767477, 0.0516907, 0.0426448, 0.0515445, 0.0198027, 0.0407523, 0.0618296, 0.0731516, 0.0229438, 0.0537609, 0.0919042, 0.0586762, 0.0238262, 0.0401265, 0.0509007, 0.0687652, 0.0585647, 0.0142613, 0.0321015, 0.0660051};


	arrayCopyForward(OCC_JONES_SWISS22,OCC_JONES_SWISS22+dim,_member.single_prob);
	arrayCopyForward(* TRANS_JONES_SWISS22,* TRANS_JONES_SWISS22+dim*dim,* _member.trans_prob);
}

template <typename TValue>
void _starting_data_pam(_TempMembersPam<TValue, AminoAcid> & _member, Score <TValue, Pam<AminoAcid, Pam_Data_Jones_All_Membrane> > & _score){

/////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Amino acid exchange data generated from SWISSPROT Release 23.0
// (from all transmembranal segments)
// Ref. Jones D.T., Taylor W.R. and Thornton J.M. (1994) FEBS Letters
// 339:269-275
// Usable sequences: 2137 (plus additional sequences from homology searches)
// Final alignments: 3155
// Accepted point mutations: 4845
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
	const int dim = ValueSize<AminoAcid>::VALUE;
	const double TRANS_JONES_ALL_MEMBRANE[dim][dim] = // dim = 24, transition probability matrix of transmembranal segments amino acids (Jones 1994, Swissprot 23.0)
{{0.98950, 0.00138, 0.00011, 0.00081, 0.00061, 0.00029, 0.00064, 0.00218, 0.00037, 0.00038, 0.00027, 0.00046, 0.00031, 0.00028, 0.00135, 0.00360, 0.00399, 0.00000, 0.00003, 0.00252},
{0.00021, 0.98590, 0.00000, 0.00012, 0.00009, 0.00154, 0.00032, 0.00030, 0.00129, 0.00003, 0.00005, 0.00487, 0.00059, 0.00000, 0.00004, 0.00009, 0.00010, 0.00129, 0.00000, 0.00000},
{0.00002, 0.00000, 0.99368, 0.00162, 0.00005, 0.00051, 0.00000, 0.00000, 0.00049, 0.00003, 0.00003, 0.00101, 0.00009, 0.00001, 0.00008, 0.00058, 0.00038, 0.00005, 0.00003, 0.00002},
{0.00007, 0.00007, 0.00078, 0.99200, 0.00000, 0.00000, 0.00128, 0.00020, 0.00025, 0.00001, 0.00000, 0.00018, 0.00003, 0.00000, 0.00004, 0.00000, 0.00012, 0.00000, 0.00003, 0.00003},
{0.00013, 0.00013, 0.00006, 0.00000, 0.98964, 0.00000, 0.00000, 0.00018, 0.00012, 0.00003, 0.00007, 0.00000, 0.00003, 0.00045, 0.00000, 0.00087, 0.00026, 0.00037, 0.00073, 0.00041},
{0.00004, 0.00138, 0.00039, 0.00000, 0.00000, 0.99158, 0.00171, 0.00001, 0.00160, 0.00001, 0.00010, 0.00055, 0.00009, 0.00000, 0.00020, 0.00013, 0.00004, 0.00000, 0.00000, 0.00000},
{0.00006, 0.00020, 0.00000, 0.00139, 0.00000, 0.00117, 0.99241, 0.00029, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00007, 0.00004, 0.00000, 0.00000, 0.00006},
{0.00157, 0.00145, 0.00000, 0.00174, 0.00061, 0.00007, 0.00225, 0.99468, 0.00006, 0.00009, 0.00000, 0.00000, 0.00009, 0.00005, 0.00028, 0.00116, 0.00024, 0.00023, 0.00000, 0.00046},
{0.00006, 0.00138, 0.00045, 0.00046, 0.00009, 0.00190, 0.00000, 0.00001, 0.99329, 0.00003, 0.00001, 0.00000, 0.00003, 0.00000, 0.00000, 0.00000, 0.00008, 0.00000, 0.00092, 0.00002},
{0.00043, 0.00026, 0.00022, 0.00012, 0.00019, 0.00007, 0.00000, 0.00014, 0.00018, 0.98579, 0.00172, 0.00000, 0.00499, 0.00088, 0.00016, 0.00040, 0.00296, 0.00005, 0.00013, 0.00763},
{0.00042, 0.00053, 0.00028, 0.00000, 0.00052, 0.00117, 0.00000, 0.00000, 0.00012, 0.00237, 0.99274, 0.00009, 0.00475, 0.00333, 0.00147, 0.00078, 0.00051, 0.00092, 0.00019, 0.00220},
{0.00005, 0.00349, 0.00062, 0.00023, 0.00000, 0.00044, 0.00000, 0.00000, 0.00000, 0.00000, 0.00001, 0.99164, 0.00012, 0.00000, 0.00000, 0.00002, 0.00004, 0.00000, 0.00016, 0.00001},
{0.00010, 0.00125, 0.00017, 0.00012, 0.00005, 0.00022, 0.00000, 0.00004, 0.00006, 0.00140, 0.00097, 0.00037, 0.98465, 0.00011, 0.00000, 0.00002, 0.00063, 0.00005, 0.00016, 0.00077},
{0.00021, 0.00000, 0.00006, 0.00000, 0.00160, 0.00000, 0.00000, 0.00005, 0.00000, 0.00057, 0.00158, 0.00000, 0.00025, 0.99311, 0.00000, 0.00058, 0.00018, 0.00009, 0.00172, 0.00032},
{0.00033, 0.00007, 0.00011, 0.00012, 0.00000, 0.00037, 0.00000, 0.00010, 0.00000, 0.00003, 0.00023, 0.00000, 0.00000, 0.00000, 0.99555, 0.00016, 0.00020, 0.00000, 0.00003, 0.00001},
{0.00194, 0.00033, 0.00179, 0.00000, 0.00226, 0.00051, 0.00043, 0.00087, 0.00000, 0.00019, 0.00027, 0.00009, 0.00003, 0.00042, 0.00036, 0.98844, 0.00265, 0.00005, 0.00070, 0.00011},
{0.00198, 0.00033, 0.00106, 0.00070, 0.00061, 0.00015, 0.00021, 0.00016, 0.00025, 0.00130, 0.00016, 0.00018, 0.00099, 0.00012, 0.00040, 0.00244, 0.98657, 0.00005, 0.00010, 0.00041},
{0.00000, 0.00184, 0.00006, 0.00000, 0.00038, 0.00000, 0.00000, 0.00007, 0.00000, 0.00001, 0.00013, 0.00000, 0.00003, 0.00003, 0.00000, 0.00002, 0.00002, 0.99593, 0.00006, 0.00016},
{0.00001, 0.00000, 0.00006, 0.00012, 0.00108, 0.00000, 0.00000, 0.00000, 0.00178, 0.00003, 0.00004, 0.00046, 0.00016, 0.00072, 0.00004, 0.00040, 0.00006, 0.00009, 0.99493, 0.00002},
{0.00287, 0.00000, 0.00011, 0.00046, 0.00221, 0.00000, 0.00075, 0.00072, 0.00012, 0.00767, 0.00161, 0.00009, 0.00276, 0.00049, 0.00004, 0.00024, 0.00095, 0.00083, 0.00006, 0.98485}};

	const double OCC_JONES_ALL_MEMBRANE[dim]= // dim = 24, normalized frequencies of transmembranal segments amino acids (Jones 1994, Swissprot 23.0)
{0.105109, 0.015666, 0.0184526, 0.00889738, 0.0219083, 0.0140969, 0.00965252, 0.075845, 0.0167686, 0.118751, 0.163453, 0.0112335, 0.0332713, 0.077722, 0.0259932, 0.0567615, 0.0522683, 0.022326, 0.0323664, 0.119458};


	arrayCopyForward(OCC_JONES_ALL_MEMBRANE,OCC_JONES_ALL_MEMBRANE+dim,_member.single_prob);
	arrayCopyForward(* TRANS_JONES_ALL_MEMBRANE,* TRANS_JONES_ALL_MEMBRANE+dim*dim,* _member.trans_prob);
}





template <typename TValue>
void _starting_data_pam(_TempMembersPam<TValue, AminoAcid> & _member, Score <TValue, Pam<AminoAcid, Pam_Data_Jones_Single_Membrane> > & _score){
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Amino acid exchange data generated from SWISSPROT Release 23.0
// (from single-spanning transmembranal segments)
// Ref. Jones D.T., Taylor W.R. and Thornton J.M. (1994) FEBS Letters
// 339:269-275
// Usable sequences: 1154 (plus additional sequences from homology searches)
// Final alignments: 1765
// Accepted point mutations: 1237
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
	const int dim = ValueSize<AminoAcid>::VALUE;
	const double TRANS_JONES_SINGLE_MEMBRANE[dim][dim] = // dim = 24, transition probability matrix of single-spanning transmembranal segments amino acids (Jones 1994, Swissprot 23.0)
{{0.99008, 0.00056, 0.00000, 0.00071, 0.00021, 0.00121, 0.00257, 0.00096, 0.00000, 0.00076, 0.00027, 0.00101, 0.00014, 0.00036, 0.00140, 0.00144, 0.00576, 0.00000, 0.00000, 0.00279},
{0.00011, 0.98194, 0.00000, 0.00000, 0.00042, 0.00181, 0.00000, 0.00055, 0.00358, 0.00009, 0.00009, 0.00674, 0.00242, 0.00000, 0.00000, 0.00008, 0.00008, 0.00505, 0.00000, 0.00000},
{0.00000, 0.00000, 0.98851, 0.00000, 0.00021, 0.00000, 0.00000, 0.00000, 0.00107, 0.00009, 0.00011, 0.00034, 0.00000, 0.00000, 0.00000, 0.00085, 0.00049, 0.00000, 0.00000, 0.00005},
{0.00004, 0.00000, 0.00000, 0.99435, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00041, 0.00000, 0.00000, 0.00005},
{0.00004, 0.00037, 0.00037, 0.00000, 0.99288, 0.00000, 0.00000, 0.00014, 0.00072, 0.00003, 0.00000, 0.00000, 0.00000, 0.00087, 0.00000, 0.00017, 0.00000, 0.00048, 0.00103, 0.00005},
{0.00007, 0.00056, 0.00000, 0.00000, 0.00000, 0.98793, 0.00000, 0.00000, 0.00107, 0.00003, 0.00011, 0.00101, 0.00000, 0.00000, 0.00070, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000},
{0.00011, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.98375, 0.00055, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00008, 0.00000, 0.00000, 0.00000, 0.00008},
{0.00075, 0.00223, 0.00000, 0.00000, 0.00063, 0.00000, 0.01027, 0.99508, 0.00000, 0.00024, 0.00000, 0.00000, 0.00014, 0.00000, 0.00070, 0.00076, 0.00049, 0.00000, 0.00000, 0.00085},
{0.00000, 0.00186, 0.00111, 0.00000, 0.00042, 0.00181, 0.00000, 0.00000, 0.99213, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00069, 0.00000},
{0.00089, 0.00056, 0.00111, 0.00000, 0.00021, 0.00060, 0.00000, 0.00036, 0.00000, 0.98741, 0.00144, 0.00000, 0.00712, 0.00182, 0.00000, 0.00051, 0.00470, 0.00000, 0.00000, 0.00439},
{0.00043, 0.00074, 0.00185, 0.00000, 0.00000, 0.00302, 0.00000, 0.00000, 0.00000, 0.00192, 0.99422, 0.00034, 0.00157, 0.00372, 0.00444, 0.00102, 0.00041, 0.00096, 0.00000, 0.00158},
{0.00011, 0.00372, 0.00037, 0.00000, 0.00000, 0.00181, 0.00000, 0.00000, 0.00000, 0.00000, 0.00002, 0.98719, 0.00043, 0.00000, 0.00000, 0.00000, 0.00016, 0.00000, 0.00086, 0.00000},
{0.00004, 0.00316, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00005, 0.00000, 0.00152, 0.00025, 0.00101, 0.98079, 0.00000, 0.00000, 0.00000, 0.00016, 0.00000, 0.00086, 0.00116},
{0.00018, 0.00000, 0.00000, 0.00000, 0.00251, 0.00000, 0.00000, 0.00000, 0.00000, 0.00076, 0.00117, 0.00000, 0.00000, 0.99111, 0.00000, 0.00068, 0.00008, 0.00000, 0.00155, 0.00028},
{0.00021, 0.00000, 0.00000, 0.00000, 0.00000, 0.00181, 0.00000, 0.00014, 0.00000, 0.00000, 0.00043, 0.00000, 0.00000, 0.00000, 0.99158, 0.00034, 0.00000, 0.00000, 0.00000, 0.00003},
{0.00060, 0.00019, 0.00371, 0.00000, 0.00042, 0.00000, 0.00086, 0.00041, 0.00000, 0.00018, 0.00027, 0.00000, 0.00000, 0.00058, 0.00094, 0.99018, 0.00186, 0.00000, 0.00292, 0.00015},
{0.00252, 0.00019, 0.00222, 0.00353, 0.00000, 0.00000, 0.00000, 0.00027, 0.00000, 0.00177, 0.00011, 0.00067, 0.00028, 0.00007, 0.00000, 0.00195, 0.98403, 0.00000, 0.00000, 0.00044},
{0.00000, 0.00391, 0.00000, 0.00000, 0.00042, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00009, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.99207, 0.00000, 0.00015},
{0.00000, 0.00000, 0.00000, 0.00000, 0.00126, 0.00000, 0.00000, 0.00000, 0.00143, 0.00000, 0.00000, 0.00169, 0.00071, 0.00066, 0.00000, 0.00144, 0.00000, 0.00000, 0.99209, 0.00000},
{0.00384, 0.00000, 0.00074, 0.00141, 0.00042, 0.00000, 0.00257, 0.00150, 0.00000, 0.00518, 0.00139, 0.00000, 0.00640, 0.00080, 0.00023, 0.00051, 0.00138, 0.00144, 0.00000, 0.98794}};

	const double OCC_JONES_SINGLE_MEMBRANE[dim]= // dim = 24, normalized frequencies of single-spanning transmembranal segments amino acids (Jones 1994, Swissprot 23.0)
{0.113719, 0.0217113, 0.010909, 0.00572793, 0.0192977, 0.00669743, 0.00472479, 0.0887765, 0.0113026, 0.1326, 0.176902, 0.0119861, 0.0284011, 0.0554819, 0.0172838, 0.047751, 0.0498503, 0.016816, 0.0235148, 0.156547};



	arrayCopyForward(OCC_JONES_SINGLE_MEMBRANE,OCC_JONES_SINGLE_MEMBRANE+dim,_member.single_prob);
	arrayCopyForward(* TRANS_JONES_SINGLE_MEMBRANE,* TRANS_JONES_SINGLE_MEMBRANE+dim*dim,* _member.trans_prob);
}



template <typename TValue>
void _starting_data_pam(_TempMembersPam<TValue, AminoAcid> & _member, Score <TValue, Pam<AminoAcid, Pam_Data_Jones_Multi_Membrane> > & _score){	
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Amino acid exchange data generated from SWISSPROT Release 23.0
// (from multi-spanning transmembranal segments)
// Ref. Jones D.T., Taylor W.R. and Thornton J.M. (1994) FEBS Letters
// 339:269-275
// Usable sequences: 983 (plus additional sequences from homology searches)
// Final alignments: 1405
// Accepted point mutations: 3612
/////////////////////////////////////////////////////////////////////////////////////////////////////////////
 	const int dim = ValueSize<AminoAcid>::VALUE;
	const double TRANS_JONES_MULTI_MEMBRANE[dim][dim] = // dim = 24, transition probability matrix of multi-spanning transmembranal segments amino acids (Jones 1994, Swissprot 23.0)
{{0.98927, 0.00164, 0.00013, 0.00083, 0.00073, 0.00016, 0.00024, 0.00268, 0.00044, 0.00026, 0.00029, 0.00025, 0.00032, 0.00024, 0.00133, 0.00424, 0.00346, 0.00000, 0.00004, 0.00243},
 {0.00022, 0.98854, 0.00000, 0.00014, 0.00000, 0.00140, 0.00037, 0.00019, 0.00089, 0.00001, 0.00003, 0.00398, 0.00000, 0.00000, 0.00005, 0.00009, 0.00010, 0.00040, 0.00000, 0.00000},
 {0.00003, 0.00000, 0.99459, 0.00207, 0.00000, 0.00058, 0.00000, 0.00000, 0.00037, 0.00001, 0.00000, 0.00124, 0.00012, 0.00002, 0.00010, 0.00049, 0.00034, 0.00006, 0.00004, 0.00000},
 {0.00008, 0.00010, 0.00099, 0.99143, 0.00000, 0.00000, 0.00147, 0.00029, 0.00030, 0.00001, 0.00000, 0.00025, 0.00004, 0.00000, 0.00005, 0.00000, 0.00003, 0.00000, 0.00004, 0.00003},
 {0.00016, 0.00000, 0.00000, 0.00000, 0.98857, 0.00000, 0.00000, 0.00019, 0.00000, 0.00004, 0.00010, 0.00000, 0.00004, 0.00036, 0.00000, 0.00111, 0.00039, 0.00034, 0.00063, 0.00058},
 {0.00003, 0.00174, 0.00046, 0.00000, 0.00000, 0.99200, 0.00184, 0.00002, 0.00170, 0.00000, 0.00010, 0.00050, 0.00012, 0.00000, 0.00019, 0.00016, 0.00005, 0.00000, 0.00000, 0.00000},
 {0.00003, 0.00031, 0.00000, 0.00166, 0.00000, 0.00124, 0.99388, 0.00017, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.00007, 0.00005, 0.00000, 0.00000, 0.00005},
 {0.00186, 0.00102, 0.00000, 0.00207, 0.00060, 0.00008, 0.00110, 0.99450, 0.00015, 0.00002, 0.00000, 0.00000, 0.00008, 0.00006, 0.00019, 0.00127, 0.00016, 0.00029, 0.00000, 0.00026},
 {0.00008, 0.00123, 0.00033, 0.00055, 0.00000, 0.00190, 0.00000, 0.00004, 0.99344, 0.00004, 0.00002, 0.00000, 0.00004, 0.00000, 0.00000, 0.00000, 0.00010, 0.00000, 0.00098, 0.00003},
 {0.00028, 0.00010, 0.00007, 0.00014, 0.00018, 0.00000, 0.00000, 0.00004, 0.00022, 0.98517, 0.00184, 0.00000, 0.00445, 0.00065, 0.00019, 0.00037, 0.00242, 0.00006, 0.00016, 0.00916},
 {0.00045, 0.00041, 0.00000, 0.00000, 0.00066, 0.00099, 0.00000, 0.00000, 0.00015, 0.00256, 0.99220, 0.00000, 0.00559, 0.00320, 0.00086, 0.00072, 0.00055, 0.00086, 0.00024, 0.00246},
 {0.00003, 0.00327, 0.00066, 0.00028, 0.00000, 0.00033, 0.00000, 0.00000, 0.00000, 0.00000, 0.00000, 0.99329, 0.00008, 0.00000, 0.00000, 0.00002, 0.00000, 0.00000, 0.00000, 0.00001},
 {0.00011, 0.00000, 0.00020, 0.00014, 0.00006, 0.00025, 0.00000, 0.00004, 0.00007, 0.00138, 0.00124, 0.00025, 0.98578, 0.00011, 0.00000, 0.00005, 0.00078, 0.00006, 0.00000, 0.00058},
 {0.00020, 0.00000, 0.00007, 0.00000, 0.00133, 0.00000, 0.00000, 0.00008, 0.00000, 0.00049, 0.00173, 0.00000, 0.00028, 0.99358, 0.00000, 0.00058, 0.00023, 0.00011, 0.00176, 0.00038},
 {0.00038, 0.00010, 0.00013, 0.00014, 0.00000, 0.00033, 0.00000, 0.00008, 0.00000, 0.00005, 0.00016, 0.00000, 0.00000, 0.00000, 0.99624, 0.00012, 0.00029, 0.00000, 0.00004, 0.00000},
 {0.00247, 0.00041, 0.00139, 0.00000, 0.00290, 0.00058, 0.00037, 0.00107, 0.00000, 0.00019, 0.00027, 0.00012, 0.00008, 0.00040, 0.00024, 0.98779, 0.00294, 0.00006, 0.00020, 0.00009},
 {0.00179, 0.00041, 0.00086, 0.00014, 0.00091, 0.00016, 0.00024, 0.00012, 0.00030, 0.00113, 0.00018, 0.00000, 0.00118, 0.00015, 0.00052, 0.00262, 0.98721, 0.00006, 0.00012, 0.00039},
 {0.00000, 0.00072, 0.00007, 0.00000, 0.00036, 0.00000, 0.00000, 0.00010, 0.00000, 0.00001, 0.00013, 0.00000, 0.00004, 0.00003, 0.00000, 0.00002, 0.00003, 0.99692, 0.00008, 0.00016},
 {0.00001, 0.00000, 0.00007, 0.00014, 0.00097, 0.00000, 0.00000, 0.00000, 0.00184, 0.00005, 0.00005, 0.00000, 0.00000, 0.00073, 0.00005, 0.00012, 0.00008, 0.00011, 0.99561, 0.00003},
 {0.00252, 0.00000, 0.00000, 0.00028, 0.00272, 0.00000, 0.00049, 0.00039, 0.00015, 0.00858, 0.00165, 0.00012, 0.00177, 0.00047, 0.00000, 0.00016, 0.00078, 0.00069, 0.00008, 0.98336}};

	const double OCC_JONES_MULTI_MEMBRANE[dim]= // dim = 24, normalized frequencies of multi-spanning transmembranal segments amino acids (Jones 1994, Swissprot 23.0)
{0.102589, 0.0135273, 0.0209838, 0.010018, 0.0228986, 0.0167893, 0.0113139, 0.0712285, 0.0187691, 0.113696, 0.158274, 0.0111395, 0.0351428, 0.085559, 0.0291013, 0.0597448, 0.0531459, 0.0242471, 0.0353314, 0.1065};


	arrayCopyForward(OCC_JONES_MULTI_MEMBRANE,OCC_JONES_MULTI_MEMBRANE+dim,_member.single_prob);
	arrayCopyForward(* TRANS_JONES_MULTI_MEMBRANE,* TRANS_JONES_MULTI_MEMBRANE+dim*dim,* _member.trans_prob);
}


}// namespace SEQAN_NAMESPACE_MAIN


#endif //#ifndef SEQAN_HEADER_...

