/*  Screem:  screem-xml-model.c
 *
 *  a GtkTreeModel representation of an xmlDoc
 * 
 *  Columns are fixed based off what screem requires:
 *
 *  name
 *  text
 *  start position in text stream
 *  end position in text stream
 *  valid flag
 *  icon
 *  xmlnodeptr
 * 
 *  Copyright (C) 2004  David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 */

#include <libxml/tree.h>

#include <gdk-pixbuf/gdk-pixbuf.h>

#include <gobject/gvaluecollector.h>

#include <string.h>

#include "screem-xml-model.h"

/* iter format:
 * iter->stamp = stamp
 * iter->user_data = xmlNodePtr
 */

#define VALID_ITER ( priv->stamp == iter->stamp )

typedef struct {
	guint start;
	guint end;
	gboolean valid;
	GdkPixbuf *icon;
} ScreemXmlNode;

struct ScreemXmlModelPrivate {
	xmlDocPtr doc;

	guint stamp;

	gint columns;
	GType column_types[ SCREEM_XML_MODEL_NUM_COLS ];
};

static void screem_xml_model_init( ScreemXmlModel *model );
static void screem_xml_model_class_init( ScreemXmlModelClass *klass );
static void screem_xml_model_model_init( GtkTreeModelIface *iface );
static void screem_xml_model_finalize( GObject *object );


/* TreeModelIface impl */
static guint screem_xml_model_get_flags( GtkTreeModel *model );
static gint screem_xml_model_get_n_columns( GtkTreeModel *model );
static GType screem_xml_model_get_column_type( GtkTreeModel *model,
		gint index );
static gboolean screem_xml_model_get_iter( GtkTreeModel *model,
		GtkTreeIter *iter, GtkTreePath *path );
static GtkTreePath *screem_xml_model_get_path( GtkTreeModel *model,
		GtkTreeIter *iter );
static void screem_xml_model_get_value( GtkTreeModel *model,
		GtkTreeIter *iter, gint column, GValue *value );
static gboolean screem_xml_model_iter_next( GtkTreeModel *model,
		GtkTreeIter *iter );
static gboolean screem_xml_model_iter_children( GtkTreeModel *model,
		GtkTreeIter *iter, GtkTreeIter *parent );
static gboolean screem_xml_model_iter_has_child( GtkTreeModel *model,
		GtkTreeIter *iter );
static gint screem_xml_model_iter_n_children( GtkTreeModel *model,
		GtkTreeIter *iter );
static gboolean screem_xml_model_iter_nth_child( GtkTreeModel *model,
		GtkTreeIter *iter, GtkTreeIter *parent, gint n );
static gboolean screem_xml_model_iter_parent( GtkTreeModel *model,
		GtkTreeIter *iter, GtkTreeIter *child );
static void screem_xml_model_ref_node( GtkTreeModel *model,
		GtkTreeIter *iter );
static void screem_xml_model_unref_node( GtkTreeModel *model,
		GtkTreeIter *iter );


static GObjectClass *parent_class = NULL;

GType screem_xml_model_get_type( void )
{
	static GType type = 0;

	if( ! type ) {

		static const GTypeInfo info = {
			sizeof( ScreemXmlModelClass ),
			NULL,
			NULL,
			(GClassInitFunc)screem_xml_model_class_init,
			NULL,
			NULL,
			sizeof( ScreemXmlModel ),
			0,
			(GInstanceInitFunc)screem_xml_model_init

		};

		static const GInterfaceInfo iinfo = {
			(GInterfaceInitFunc)screem_xml_model_model_init,
			NULL,
			NULL
		};

		type = g_type_register_static( G_TYPE_OBJECT,
					"ScreemXmlModel",
					&info, 0 );
		g_type_add_interface_static( type,
					GTK_TYPE_TREE_MODEL,
					&iinfo );
		
	}

	return type;
}

static void screem_xml_model_init( ScreemXmlModel *model )
{
	ScreemXmlModelPrivate *priv;
	
	priv = model->priv = g_new0( ScreemXmlModelPrivate, 1 );
	model->priv->stamp = g_random_int();

	priv->columns = SCREEM_XML_MODEL_NUM_COLS;

	priv->column_types[ SCREEM_XML_MODEL_NAME ] = G_TYPE_STRING;
	priv->column_types[ SCREEM_XML_MODEL_TEXT ] = G_TYPE_STRING;
	priv->column_types[ SCREEM_XML_MODEL_START ] = G_TYPE_UINT;
	priv->column_types[ SCREEM_XML_MODEL_END ] = G_TYPE_UINT;
	priv->column_types[ SCREEM_XML_MODEL_VALID ] = G_TYPE_BOOLEAN;
	priv->column_types[ SCREEM_XML_MODEL_ICON ] = GDK_TYPE_PIXBUF;
	priv->column_types[ SCREEM_XML_MODEL_NODE ] = G_TYPE_POINTER;
}

static void screem_xml_model_class_init( ScreemXmlModelClass *klass )
{
	GObjectClass *obj_class;

	obj_class = G_OBJECT_CLASS( klass );
	parent_class = g_type_class_peek_parent( klass );

	obj_class->finalize = screem_xml_model_finalize;
}

static void screem_xml_model_model_init( GtkTreeModelIface *iface )
{
	iface->get_flags = screem_xml_model_get_flags;
	iface->get_n_columns = screem_xml_model_get_n_columns;
	iface->get_column_type = screem_xml_model_get_column_type;
	iface->get_iter = screem_xml_model_get_iter;
	iface->get_path = screem_xml_model_get_path;
	iface->get_value = screem_xml_model_get_value;
	iface->iter_next = screem_xml_model_iter_next;
	iface->iter_children = screem_xml_model_iter_children;
	iface->iter_has_child = screem_xml_model_iter_has_child;
	iface->iter_n_children = screem_xml_model_iter_n_children;
	iface->iter_nth_child = screem_xml_model_iter_nth_child;
	iface->iter_parent = screem_xml_model_iter_parent;
	iface->ref_node = screem_xml_model_ref_node;
	iface->unref_node = screem_xml_model_unref_node;
}


static void screem_xml_model_finalize( GObject *obj )
{
	ScreemXmlModel *model;
	ScreemXmlModelPrivate *priv;
	
	model = SCREEM_XML_MODEL( obj );
	priv = SCREEM_XML_MODEL( model )->priv;
	
	g_free( priv );
	
	parent_class->finalize( obj );
}

/* Private */
static xmlNodePtr screem_xml_model_node( void )
{
	xmlNodePtr node;
	ScreemXmlNode *snode;
	
	node = xmlNewNode( NULL, (const xmlChar*)"" );
	node->_private = g_new0( ScreemXmlNode, 1 );
	snode = (ScreemXmlNode*)node->_private;

	snode->start = 0;
	snode->end = 0;
	snode->valid = FALSE;
	snode->icon = NULL;
	
	return node;
}

static gboolean screem_xml_model_real_set_value( ScreemXmlModel *model,
		GtkTreeIter *iter, gint column, GValue *value )
{
	ScreemXmlModelPrivate *priv;
	const gchar *tmp;
	xmlNodePtr node;
	ScreemXmlNode *snode;
	gboolean ret;
	
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), FALSE );
	g_return_val_if_fail( iter != NULL, FALSE );
	g_return_val_if_fail( value != NULL, FALSE );
	
	priv = SCREEM_XML_MODEL( model )->priv;
	
	g_return_val_if_fail( VALID_ITER, FALSE );
	g_return_val_if_fail( column < priv->columns, FALSE );

	node = iter->user_data;
		
	if( ! node->_private ) {
		node->_private = g_new0( ScreemXmlNode, 1 );
		snode = (ScreemXmlNode*)node->_private;
		snode->start = 0;
		snode->end = 0;
		snode->valid = FALSE;
		snode->icon = NULL;
	} else {
		snode = node->_private;
	}
	
	ret = FALSE;
	switch( column ) {
		case SCREEM_XML_MODEL_NAME:
			tmp = g_value_get_string( value );
			xmlNodeSetName( node, (const xmlChar*)tmp );
			ret = TRUE;
			break;
		case SCREEM_XML_MODEL_TEXT:
			tmp = g_value_get_string( value );
			xmlNodeSetContent( node, (const xmlChar*)tmp );
			ret = TRUE;
			break;
		case SCREEM_XML_MODEL_START:
			snode->start = g_value_get_uint( value );
			ret = TRUE;
			break;
		case SCREEM_XML_MODEL_END:
			snode->end = g_value_get_uint( value );
			ret = TRUE;
			break;
		case SCREEM_XML_MODEL_VALID:
			snode->valid = g_value_get_boolean( value );
			ret = TRUE;
			break;
		case SCREEM_XML_MODEL_ICON:
			if( snode->icon ) {
				g_object_unref( snode->icon );
			}
			snode->icon = g_value_get_object( value );
			if( snode->icon ) {
				g_object_ref( snode->icon );
			}
			ret = TRUE;
			break;
		case SCREEM_XML_MODEL_NODE:
			g_warning( "SCREEM_XML_MODEL_NODE is read only\n" );
			break;
		default:
			g_warning( "Invalid column\n" );
			break;
	}

	return ret;
}


/* TreeModelIface implementation */
static guint screem_xml_model_get_flags( GtkTreeModel *model )
{
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), 0 );

	return 0;
}

static gint screem_xml_model_get_n_columns( GtkTreeModel *model )
{
	ScreemXmlModelPrivate *priv;
	
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), 0 );
	
	priv = SCREEM_XML_MODEL( model )->priv;
	
	return priv->columns;
}

static GType screem_xml_model_get_column_type( GtkTreeModel *model, 
		gint index )
{
	ScreemXmlModelPrivate *priv;
	
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), 
			G_TYPE_INVALID );
	
	priv = SCREEM_XML_MODEL( model )->priv;

	g_return_val_if_fail( index < priv->columns, G_TYPE_INVALID );
	
	return priv->column_types[ index ];
}

static gboolean screem_xml_model_get_iter( GtkTreeModel *model, 
		GtkTreeIter *iter, GtkTreePath *path )
{
	ScreemXmlModelPrivate *priv;
	gint depth;
	gint *indicies;
	xmlNodePtr node;
	gboolean ret;
	gint i;
	
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), FALSE );
	g_return_val_if_fail( iter != NULL, FALSE );
	
	priv = SCREEM_XML_MODEL( model )->priv;
	
	depth = gtk_tree_path_get_depth( path );
	indicies = gtk_tree_path_get_indices( path );
	
	g_return_val_if_fail( indicies != NULL, FALSE );
	
	ret = FALSE;

	if( depth == 1 && indicies[ 0 ] != 0 ) {
		iter->stamp = 0;
		return FALSE;
	}
	
	node = xmlDocGetRootElement( priv->doc );
	depth --;
	indicies ++;

	while( depth > 0 ) {
		node = node->children;
		for( i = 0; node && ( i < *indicies ); ++ i ) {
			node = node->next;
		}

		depth --;
		indicies ++;
	}	
	
	if( node ) {
		ret = TRUE;
		iter->stamp = priv->stamp;
		iter->user_data = node;
	}
	
	return ret;
}

static GtkTreePath *screem_xml_model_get_path( GtkTreeModel *model, 
		GtkTreeIter *iter )
{
	ScreemXmlModelPrivate *priv;
	GtkTreePath *path;
	xmlNodePtr node;
	gint i;
	
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), NULL );
	g_return_val_if_fail( iter != NULL, NULL );
	
	priv = SCREEM_XML_MODEL( model )->priv;

	g_return_val_if_fail( VALID_ITER, NULL );
	
	path = gtk_tree_path_new();
	
	node = iter->user_data;
	
	while( node ) {
		for( i = 0;  node->prev; ++ i ) {
			node = node->prev;
		}
		node = node->parent;
		if( node ) {
			gtk_tree_path_prepend_index( path, i );
		}
	}
	
	return path;
}

static void screem_xml_model_get_value( GtkTreeModel *model,
		GtkTreeIter *iter, gint column, GValue *value )
{
	ScreemXmlModelPrivate *priv;
	xmlNodePtr node;
	ScreemXmlNode *snode;
	
	g_return_if_fail( SCREEM_IS_XML_MODEL( model ) );
	g_return_if_fail( iter != NULL );
	g_return_if_fail( value != NULL );
	
	priv = SCREEM_XML_MODEL( model )->priv;

	g_return_if_fail( VALID_ITER );
	g_return_if_fail( column < priv->columns );

	node = (xmlNodePtr)iter->user_data;
	
	g_return_if_fail( node != NULL );
	
	snode = (ScreemXmlNode*)node->_private;
		
	g_value_init( value, priv->column_types[ column ] );
	
	switch( column ) {
		case SCREEM_XML_MODEL_NAME:
			g_value_set_string( value, 
					(const gchar*)node->name );
			break;
		case SCREEM_XML_MODEL_TEXT:
			/* node->xmlContent */
			g_value_set_string( value, 
					(const gchar*)node->content );
			break;
		case SCREEM_XML_MODEL_START:
			if( snode ) {
				g_value_set_uint( value, snode->start );
			}
			break;
		case SCREEM_XML_MODEL_END:
			if( snode ) {
				g_value_set_uint( value, snode->end );
			}
			break;
		case SCREEM_XML_MODEL_VALID:
			if( snode ) {
				g_value_set_uint( value, snode->valid );
			}
			break;
		case SCREEM_XML_MODEL_ICON:
			if( snode ) {
				g_value_set_object( value, snode->icon );
			}
			break;
		case SCREEM_XML_MODEL_NODE:
			g_value_set_pointer( value, node );
			break;
		default:
			g_warning( "Invalid column\n" );
			break;
	}
}

static gboolean screem_xml_model_iter_next( GtkTreeModel *model,
		GtkTreeIter *iter )
{
	ScreemXmlModelPrivate *priv;
	gboolean ret;
	xmlNodePtr node;
	
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), FALSE );
	g_return_val_if_fail( iter != NULL, FALSE );
	
	priv = SCREEM_XML_MODEL( model )->priv;

	g_return_val_if_fail( VALID_ITER, FALSE );

	node = iter->user_data;
	ret = FALSE;
	if( node->next ) {
		iter->user_data = node->next;
		ret = TRUE;
	}

	return ret;
}

static gboolean screem_xml_model_iter_children( GtkTreeModel *model,
		GtkTreeIter *iter, GtkTreeIter *parent )
{
	ScreemXmlModelPrivate *priv;
	gboolean ret;
	xmlNodePtr node;
	
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), FALSE );
	g_return_val_if_fail( iter != NULL, FALSE );

	ret = FALSE;
	
	if( ! parent ) {
		ret = gtk_tree_model_get_iter_first( model, iter );
	} else {
		priv = SCREEM_XML_MODEL( model )->priv;
		g_return_val_if_fail( priv->stamp == parent->stamp, 
				FALSE );
	
		node = parent->user_data;
		ret = FALSE;
		if( node->children ) {
			iter->stamp = parent->stamp;
			iter->user_data = node->children;
			ret = TRUE;
		}
	}
	
	return ret;
}

static gboolean screem_xml_model_iter_has_child( GtkTreeModel *model, 
		GtkTreeIter *iter )
{
	ScreemXmlModelPrivate *priv;
	xmlNodePtr node;

	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), FALSE );

	priv = SCREEM_XML_MODEL( model )->priv;
	g_return_val_if_fail( VALID_ITER, FALSE );

	node = iter->user_data;
	
	return ( node->children  != NULL );
}

static gint screem_xml_model_iter_n_children( GtkTreeModel *model,
		GtkTreeIter *iter )
{
	ScreemXmlModelPrivate *priv;
	xmlNodePtr node;
	gint ret;
	
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), 0 );

	priv = SCREEM_XML_MODEL( model )->priv;
	g_return_val_if_fail( VALID_ITER, FALSE );
	
	ret = 0;
	
	if( ! iter ) {
		node = xmlDocGetRootElement( priv->doc );
	} else {
		node = iter->user_data;
	}
	if( node ) {
		node = node->children;
	}
	for( ret = 0; node; ++ ret ) {
		node = node->next;
	}
	
	return ret;
}

static gboolean screem_xml_model_iter_nth_child( GtkTreeModel *model,
		GtkTreeIter *iter, GtkTreeIter *parent, gint n )
{
	ScreemXmlModelPrivate *priv;
	gboolean ret;
	xmlNodePtr node;
	
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), FALSE );
	g_return_val_if_fail( iter != NULL, FALSE );
		
	priv = SCREEM_XML_MODEL( model )->priv;

	ret = FALSE;
	if( ! parent ) {
		/* no parent, only the root node is available here,
		 * anything else doesn't exist */
		if( n == 0 ) {
			iter->stamp = priv->stamp;
			iter->user_data = xmlDocGetRootElement( priv->doc );
			ret = TRUE;
		}
	} else {
		g_return_val_if_fail( priv->stamp == parent->stamp, FALSE );
		node = parent->user_data;
		for( node = node->children; node && n > 0; n -- ) {
			node = node->next;
		}
		if( node ) {
			iter->stamp = parent->stamp;
			iter->user_data = node;
			ret = TRUE;
		}
	}
	
	return ret;
}

static gboolean screem_xml_model_iter_parent( GtkTreeModel *model,
		GtkTreeIter *iter, GtkTreeIter *child )
{
	ScreemXmlModelPrivate *priv;
	gboolean ret;
	xmlNodePtr node;
	
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ),
				FALSE );
	g_return_val_if_fail( iter != NULL, FALSE );
	g_return_val_if_fail( child != NULL, FALSE );

	priv = SCREEM_XML_MODEL( model )->priv;
	g_return_val_if_fail( priv->stamp == child->stamp, FALSE );

	node = child->user_data;
	node = node->parent;

	ret = FALSE;
	if( node ) {
		iter->stamp = child->stamp;
		iter->user_data = node;
		ret = TRUE;
	}
	
	return ret;
}

static void screem_xml_model_ref_node( GtkTreeModel *model, 
		GtkTreeIter  *iter )
{
}

static void screem_xml_model_unref_node( GtkTreeModel *model,
		GtkTreeIter  *iter )
{
}

/* Public */
GtkTreeModel *screem_xml_model_new( void )
{
	GtkTreeModel *model;
	
	model = g_object_new( SCREEM_TYPE_XML_MODEL, NULL );

	return model;
}

GtkTreeModel *screem_xml_model_new_with_doc( xmlDocPtr doc )
{
	GtkTreeModel *model;
	ScreemXmlModelPrivate *priv;
	
	model = g_object_new( SCREEM_TYPE_XML_MODEL, NULL );

	priv = SCREEM_XML_MODEL( model )->priv;

	priv->doc = doc;
	
	return model;
}

xmlDocPtr screem_xml_model_get_doc( ScreemXmlModel *model )
{
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), NULL );

	return model->priv->doc;
}

gboolean screem_xml_model_prepend( ScreemXmlModel *model, 
		GtkTreeIter *it, GtkTreeIter *parent )
{
	ScreemXmlModelPrivate *priv;
	xmlDocPtr doc;
	xmlNodePtr node;
	xmlNodePtr n;
	gboolean ret;
	
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), FALSE );

	priv = model->priv;

	doc = priv->doc;
	
	ret = FALSE;
	
	if( ! parent ) {
		/* setting root node, only do this if there isn't
		 * one */
		node = xmlDocGetRootElement( doc );
		if( ! node ) {
			n = screem_xml_model_node();
			xmlDocSetRootElement( doc, n );
			ret = TRUE;
		} 
	} else {
		node = parent->user_data;
		if( ! node->children ) {
			n = screem_xml_model_node();
			node = xmlAddChild( node, n );
		} else {
			node = node->children;
			n = screem_xml_model_node();
			xmlAddPrevSibling( node, n );
			node = n;
		}

		ret = TRUE;
	}

	/* build iter */
	if( ret ) {
		it->stamp = ++ priv->stamp;
		it->user_data = node;
	}
	
	return ret;

}

gboolean screem_xml_model_append( ScreemXmlModel *model, 
		GtkTreeIter *it, GtkTreeIter *parent )
{
	ScreemXmlModelPrivate *priv;
	xmlDocPtr doc;
	xmlNodePtr node;
	xmlNodePtr n;
	gboolean ret;
	
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), FALSE );

	priv = model->priv;

	doc = priv->doc;
	
	ret = FALSE;
	
	if( ! parent ) {
		/* setting root node, only do this if there isn't
		 * one */
		node = xmlDocGetRootElement( doc );
		if( ! node ) {
			node = screem_xml_model_node();
			xmlDocSetRootElement( doc, node );
			ret = TRUE;
		} 
	} else {
		node = parent->user_data;
		n = screem_xml_model_node();
		node = xmlAddChild( node, n );
		
		ret = TRUE;
	}

	/* build iter */
	if( ret ) {
		it->stamp = ++ priv->stamp;
		it->user_data = node;
	}
	
	return ret;
}

xmlNodePtr screem_xml_model_get_xml_node( ScreemXmlModel *model,
		GtkTreeIter *it )
{
	ScreemXmlModelPrivate *priv;
	xmlNodePtr node;
	
	g_return_val_if_fail( SCREEM_IS_XML_MODEL( model ), NULL );
	g_return_val_if_fail( it != NULL, NULL );

	priv = model->priv;
	
	g_return_val_if_fail( priv->stamp == it->stamp, NULL );
	
	node = it->user_data;

	return node;
}

void screem_xml_model_set_value( ScreemXmlModel *model,
		GtkTreeIter *iter, gint column, GValue *value )
{
	ScreemXmlModelPrivate *priv;
	gboolean emit;
	
	g_return_if_fail( SCREEM_IS_XML_MODEL( model ) );
	g_return_if_fail( iter != NULL );
	
	priv = SCREEM_XML_MODEL( model )->priv;
	
	g_return_if_fail( VALID_ITER );

	emit = screem_xml_model_real_set_value( model, iter,
			column, value );
	
	if( emit ) {
		GtkTreePath *path;

		path = gtk_tree_model_get_path( GTK_TREE_MODEL( model ),
				iter );
		gtk_tree_model_row_changed( GTK_TREE_MODEL( model ),
				path, iter );
		gtk_tree_path_free( path );
	}

}

void screem_xml_model_set_valist( ScreemXmlModel *model,
		GtkTreeIter *iter, va_list va_args )
{
	ScreemXmlModelPrivate *priv;
	gint col;
	gboolean emit;

	g_return_if_fail( SCREEM_IS_XML_MODEL( model ) );
	g_return_if_fail( iter != NULL );
	
	priv = SCREEM_XML_MODEL( model )->priv;
	
	g_return_if_fail( VALID_ITER );

	col = va_arg( va_args, gint );

	emit = FALSE;
	
	while( col != -1 ) {
		GValue value = { 0 };
		gchar *error = NULL;
		
		if( col > priv->columns ) {
			g_warning( "%s: Invalid column number %d added to iter (remember to end your list of columns with a -1", G_STRLOC, col );
			break;
		}
		
		g_value_init( &value, priv->column_types[ col ] );
		G_VALUE_COLLECT( &value, va_args, 0, &error );
		
		if( error ) {
			g_warning( "%s: %s", G_STRLOC, error );
			g_free( error );
			
			/* value leaked, unknown state if an
			 * error occured */
			break;
		}
	
		emit |= screem_xml_model_real_set_value( model, iter,
			col, &value );
	
		g_value_unset( &value );
		col = va_arg( va_args, gint);
	}
	
	if( emit ) {
		GtkTreePath *path;

		path = gtk_tree_model_get_path( GTK_TREE_MODEL( model ),
				iter );
		gtk_tree_model_row_changed( GTK_TREE_MODEL( model ),
				path, iter );
		gtk_tree_path_free( path );
	}
}

void screem_xml_model_set( ScreemXmlModel *model, GtkTreeIter *iter,
		... )
{
	ScreemXmlModelPrivate *priv;
	va_list va_args;
	
	g_return_if_fail( SCREEM_IS_XML_MODEL( model ) );
	g_return_if_fail( iter != NULL );
	
	priv = SCREEM_XML_MODEL( model )->priv;

	g_return_if_fail( VALID_ITER );
	
	va_start( va_args, iter );
	screem_xml_model_set_valist( model, iter, va_args );
	va_end( va_args );

}

