/*  Screem:  screem-plugin_manager.c
 *
 *  Copyright (C) 2004 David A Knight
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 *
 *  For contact information with the author of this source code please see
 *  the AUTHORS file.  If there is no AUTHORS file present then check the
 *  about box under the help menu for a contact address
 */
#include <config.h>

#include <gtk/gtk.h>

#include <string.h>

#include <libgnomevfs/gnome-vfs.h>

#include "screem-plugin-manager.h"

#include "screem-window.h"

#include "fileops.h"

struct ScreemPluginManagerPrivate {
	GSList *plugins;
};

enum {
	ARG_0
};

static void screem_plugin_manager_class_init( ScreemPluginManagerClass *klass );
static void screem_plugin_manager_init( ScreemPluginManager *plugin_manager );
static void screem_plugin_manager_finalize( GObject *object );
static void screem_plugin_manager_set_prop( GObject *object, guint property_id,
		const GValue *value, GParamSpec *pspec );
static void screem_plugin_manager_get_prop( GObject *object, guint property_id,
		GValue *value, GParamSpec *pspec );


/* G Object stuff */
#define PARENT_TYPE G_TYPE_OBJECT

static gpointer parent_class;

static void screem_plugin_manager_class_init( ScreemPluginManagerClass *klass )
{
	GObjectClass *object_class;

	object_class = G_OBJECT_CLASS( klass );

	object_class->finalize = screem_plugin_manager_finalize;
	parent_class = g_type_class_peek_parent( klass );

	object_class->get_property = screem_plugin_manager_get_prop;
	object_class->set_property = screem_plugin_manager_set_prop;
}

static void screem_plugin_manager_init( ScreemPluginManager *plugin_manager )
{
	plugin_manager->priv = g_new0( ScreemPluginManagerPrivate, 1 );

}

static void screem_plugin_manager_finalize( GObject *object )
{
	ScreemPluginManager *plugin_manager;
	ScreemPluginManagerPrivate *priv;
	
	plugin_manager = SCREEM_PLUGIN_MANAGER( object );
	priv = plugin_manager->priv;

	g_free( priv );
	
	G_OBJECT_CLASS( parent_class )->finalize( object );
}

static void screem_plugin_manager_set_prop( GObject *object, guint property_id,
				    const GValue *value, GParamSpec *pspec )
{
	ScreemPluginManager *plugin_manager;
	ScreemPluginManagerPrivate *priv;
	
	plugin_manager = SCREEM_PLUGIN_MANAGER( object );
	priv = plugin_manager->priv;
	
	switch( property_id ) {
		default:
			break;
	}
}

static void screem_plugin_manager_get_prop( GObject *object, guint property_id,
				    GValue *value, GParamSpec *pspec)
{
	ScreemPluginManager *plugin_manager;
	ScreemPluginManagerPrivate *priv;
	
	plugin_manager = SCREEM_PLUGIN_MANAGER( object );
	priv = plugin_manager->priv;
	
	switch( property_id ) {
		default:
			break;
	}
}

GType screem_plugin_manager_get_type()
{
	static GType type = 0;
	
	if( ! type ) {
		static const GTypeInfo info = {
			sizeof( ScreemPluginManagerClass ),
			NULL, /* base init */
			NULL, /* base finalise */
			(GClassInitFunc)screem_plugin_manager_class_init,
			NULL, /* class finalise */
			NULL, /* class data */
			sizeof( ScreemPluginManager ),
			0, /* n_preallocs */
			(GInstanceInitFunc)screem_plugin_manager_init
		};

		type = g_type_register_static( PARENT_TYPE,
					       "ScreemPluginManager",
					       &info, 0 );
	}

	return type;
}

/* Private API */
static void 
screem_plugin_manager_register_plugin( ScreemPluginManager *manager,
		ScreemPluginDetails *details )
{
	ScreemPluginManagerPrivate *priv;

	g_return_if_fail( SCREEM_IS_PLUGIN_MANAGER( manager ) );
	g_return_if_fail( details != NULL );

	priv = manager->priv;

	priv->plugins = g_slist_prepend( priv->plugins, details );
}

static void
screem_plugin_manager_unregister_plugin( ScreemPluginManager *manager,
		ScreemPluginDetails *details )
{
	ScreemPluginManagerPrivate *priv;

	g_return_if_fail( SCREEM_IS_PLUGIN_MANAGER( manager ) );
	g_return_if_fail( details != NULL );

	priv = manager->priv;

	priv->plugins = g_slist_remove( priv->plugins, details );
}

static gint plugin_filter( const GnomeVFSFileInfo *info )
{
	gint ret;
	const gchar *ext;

	ext = strrchr( info->name, '.' );
	ret = -1;
	if( ext ) {
		ret = strcmp( ".so", ext );
	}

	return ret;
}

typedef void (*GetDetails)( ScreemPluginDetails **details);

static void load_plugins( ScreemPluginManager *manager,
		const gchar *directory )
{
	GSList *files;
	GSList *tmp;

	GModule *mod;
	GetDetails get_details;
	GetDetails *get;

	ScreemPluginDetails *details;
	
	g_return_if_fail( directory != NULL );

	files = screem_vfs_scandir( directory, plugin_filter, 
					(GCompareFunc)strcmp,
					FALSE );
	get = &get_details;
	
	for( tmp = files; tmp; tmp = tmp->next ) {
		details = NULL;
		if( ( mod = g_module_open( tmp->data, 
					G_MODULE_BIND_LAZY ) ) ) {
			if( g_module_symbol( mod, 
					"get_details",
					 (gpointer*)get ) ) {
				get_details( &details );
			}

			if( ( ! details ) ||
			    ( details->api_version !=
			      SCREEM_PLUGIN_REQUIRED_VERSION ) ) {
				g_module_close( mod );
			} 		
		} else {
			g_print( "PLUGIN ERROR: %s\n",
					g_module_error() );
		}
		g_free( tmp->data );

		if( details ) {
			details->mod = mod;
			screem_plugin_manager_register_plugin( manager,
					details );			
		}
	}
	g_slist_free( files );
}

/* Public API */

ScreemPluginManager *screem_plugin_manager_new( void )
{
	ScreemPluginManager *plugin_manager;

	plugin_manager = SCREEM_PLUGIN_MANAGER( g_object_new( SCREEM_TYPE_PLUGIN_MANAGER, 
				NULL ) );

	return plugin_manager;
}


void screem_plugin_manager_scan_plugins( ScreemPluginManager *manager )
{
	const gchar *search_paths[ 7 ] = {
		PLUGIN_PATH,
                "/usr/lib/",
                "/usr/X11R6/lib",
		NULL,
                NULL
	};
	gint path = 0;
	gchar *directory = NULL;
	
	g_return_if_fail( SCREEM_IS_PLUGIN_MANAGER( manager ) );
	
	/* add these into search_paths here rather than in the declaration
	   as the values would be set at compile rather than runtime
	   otherwise */
	search_paths[ 3 ] = g_getenv( "SCREEM_LIB_PATH" );

	/* global paths to look for the plugins are searched in order,
	   if plugins are found in one of them no more are checked */
	do {
		g_free( directory );
                directory = g_strdup_printf( "%sscreem%cplugins%c",
                                             search_paths[ path ],
                                             G_DIR_SEPARATOR,
                                             G_DIR_SEPARATOR );
                path ++;
        } while( ( search_paths[ path ] ) &&
                 ( ! g_file_test( directory, G_FILE_TEST_IS_DIR ) ) );

	/* now load the global plugins */
	load_plugins( manager, directory );

	g_free( directory );

	/* now load from the users $HOME/.screem directory */
	directory = screem_get_dot_dir();

	if( g_file_test( directory, G_FILE_TEST_IS_DIR ) ) {
		load_plugins( manager, directory );
	}
	
	g_free( directory );
}

ScreemPluginDetails *screem_plugin_manager_get_plugin_by_name( ScreemPluginManager *manager,
		const gchar *name )
{
	ScreemPluginManagerPrivate *priv;
	ScreemPluginDetails *ret;
	ScreemPluginDetails *details;
	GSList *tmp;
	
	g_return_val_if_fail( SCREEM_IS_PLUGIN_MANAGER( manager ),
			NULL );
	g_return_val_if_fail( name !=  NULL, NULL );
	
	priv = manager->priv;
	ret = NULL;
	for( tmp = priv->plugins; tmp; tmp = tmp->next ) {
		details = (ScreemPluginDetails*)tmp->data;
		if( ! strcmp( details->name, name ) ){
			ret = details;
			break;
		}
	}

	return ret;
}

void screem_plugin_manager_add_window( ScreemPluginManager *manager,
		ScreemWindow *window )
{
	ScreemPluginManagerPrivate *priv;
	ScreemPluginDetails *details;
	GSList *tmp;
	ScreemPlugin *plugin;
	
	g_return_if_fail( SCREEM_IS_PLUGIN_MANAGER( manager ) );
	g_return_if_fail( SCREEM_IS_WINDOW( window ) );

	priv = manager->priv;
	for( tmp = priv->plugins; tmp; tmp = tmp->next ) {
		details = (ScreemPluginDetails*)tmp->data;

		plugin = details->create();
		if( plugin ) {

			g_object_set( G_OBJECT( plugin ),
					"window", window,
					NULL );
			if( ! screem_plugin_setup( plugin ) ) {
				g_object_unref( plugin );
				plugin = NULL;
			} else {
				/* plugin setup, FIXME, leaked */
			}
		}
	}
}

void screem_plugin_manager_remove_window( ScreemPluginManager *manager,
		ScreemWindow *window )
{
	g_return_if_fail( SCREEM_IS_PLUGIN_MANAGER( manager ) );
	g_return_if_fail( SCREEM_IS_WINDOW( window ) );

	/* FIXME: cleanup ScreemPlugin objects created against
	 * window */
}

