/*
 *  Copyright (C) 2003 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include "config.h"

#include "screem-dropdown-action.h"
#include "screem-window.h"

#include <gtk/gtkimage.h>
#include <gtk/gtkmenuitem.h>
#include <gtk/gtkimagemenuitem.h>
#include <gtk/gtkmenushell.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkmenutoolbutton.h>
#include <gtk/gtktoolbar.h>

#define SCREEM_DROPDOWN_ACTION_GET_PRIVATE(object)(G_TYPE_INSTANCE_GET_PRIVATE ((object), SCREEM_TYPE_DROPDOWN_ACTION, ScreemDropdownActionPrivate))

struct _ScreemDropdownActionPrivate
{
	ScreemWindow *window;
	gchar *arrow_tooltip;
	gchar *menu_path;
	gboolean menu_set;
};

enum
{
	PROP_0,
	PROP_ARROW_TOOLTIP,
	PROP_MENU_PATH,
	PROP_WINDOW
};

static void
connect_proxy (GtkAction *action, GtkWidget *proxy);

/* GObject stuff */
G_DEFINE_TYPE( ScreemDropdownAction, screem_dropdown_action, GTK_TYPE_ACTION ) 

static void
screem_dropdown_action_init (ScreemDropdownAction *action)
{
	action->priv = SCREEM_DROPDOWN_ACTION_GET_PRIVATE (action);
}

static void
screem_dropdown_action_finalize (GObject *object)
{
	ScreemDropdownAction *action = SCREEM_DROPDOWN_ACTION (object);

	g_free (action->priv->arrow_tooltip);
	g_free (action->priv->menu_path);

	G_OBJECT_CLASS( screem_dropdown_action_parent_class )->finalize (object);
}

static void
screem_dropdown_action_set_property (GObject *object,
				     guint prop_id,
				     const GValue *value,
				     GParamSpec *pspec)
{
	ScreemDropdownAction *nav = SCREEM_DROPDOWN_ACTION (object);

	switch (prop_id)
	{
		case PROP_ARROW_TOOLTIP:
			nav->priv->arrow_tooltip = g_value_dup_string (value);
			g_object_notify (object, "tooltip");
			break;
		case PROP_MENU_PATH:
			nav->priv->menu_path = g_value_dup_string (value);
			break;
		case PROP_WINDOW:
			nav->priv->window = SCREEM_WINDOW (g_value_get_object (value));
			break;
	}
}

static void
screem_dropdown_action_get_property (GObject *object,
				     guint prop_id,
				     GValue *value,
				     GParamSpec *pspec)
{
	ScreemDropdownAction *nav = SCREEM_DROPDOWN_ACTION (object);

	switch (prop_id)
	{
		case PROP_ARROW_TOOLTIP:
			g_value_set_string (value, nav->priv->arrow_tooltip);
			break;
		case PROP_MENU_PATH:
			g_value_set_string (value, nav->priv->menu_path);
			break;
		case PROP_WINDOW:
			g_value_set_object (value, nav->priv->window);
			break;
	}
}

static void
screem_dropdown_action_class_init (ScreemDropdownActionClass *class)
{
	GObjectClass *object_class = G_OBJECT_CLASS (class);
	GtkActionClass *action_class = GTK_ACTION_CLASS (class);

	object_class->finalize = screem_dropdown_action_finalize;
	object_class->set_property = screem_dropdown_action_set_property;
	object_class->get_property = screem_dropdown_action_get_property;

	action_class->toolbar_item_type = GTK_TYPE_MENU_TOOL_BUTTON;
	action_class->connect_proxy = connect_proxy;

	g_object_class_install_property (object_class,
					 PROP_ARROW_TOOLTIP,
					 g_param_spec_string ("arrow-tooltip",
							      "Arrow Tooltip",
							      "Arrow Tooltip",
							      NULL,
							      G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
					 PROP_MENU_PATH,
					 g_param_spec_string ("menu-path",
							      "Path to dropdown menu",
							      "Menu UI path",
							      NULL,
							      G_PARAM_READWRITE));

	g_object_class_install_property (object_class,
					 PROP_WINDOW,
					 g_param_spec_object ("window",
							      "Window",
							      "The dropdown window",
							      G_TYPE_OBJECT,
							      G_PARAM_READWRITE));

	g_type_class_add_private (object_class, sizeof (ScreemDropdownActionPrivate));
}

/* static stuff */
static void
menu_activated_cb( GtkMenuToolButton *button, 
		ScreemDropdownAction *action )
{

}

static gboolean
set_tooltip_cb (GtkMenuToolButton *proxy,
		GtkTooltips *tooltips,
		const char *tip,
		const char *tip_private,
		ScreemDropdownAction *action)
{
	gtk_menu_tool_button_set_arrow_tooltip (proxy, tooltips,
						action->priv->arrow_tooltip,
						NULL);

	/* don't stop emission */
	return FALSE;
}

static void
connect_proxy (GtkAction *action, GtkWidget *proxy)
{
	if (GTK_IS_MENU_TOOL_BUTTON (proxy))
	{
		ScreemDropdownAction *ddaction;
		GtkWidget *menu;
		GtkUIManager *merge;

		ddaction = SCREEM_DROPDOWN_ACTION( action );
		merge = GTK_UI_MANAGER( ddaction->priv->window->merge );
	
		if( ddaction->priv->menu_path ) {
	       		menu = gtk_ui_manager_get_widget( merge,
					ddaction->priv->menu_path );
	
			gtk_menu_tool_button_set_menu (GTK_MENU_TOOL_BUTTON (proxy), menu);
		}
		g_signal_connect (proxy, "show-menu",
				  G_CALLBACK (menu_activated_cb), action);

		g_signal_connect (proxy, "set-tooltip",
				  G_CALLBACK (set_tooltip_cb), action);
	}

	GTK_ACTION_CLASS (screem_dropdown_action_parent_class)->connect_proxy (action, proxy);
}


/* public stuff */
ScreemDropdownAction *screem_dropdown_action_new( const gchar *name,
		const gchar *label, const gchar *tooltip,
		const gchar *stock_id,
		const gchar *arrow_tooltip,
		const gchar *menu_path,
		ScreemWindow *window )
{
	ScreemDropdownAction *action;

	action = g_object_new( SCREEM_TYPE_DROPDOWN_ACTION, 
			"name", name,
			"label", label,
			"tooltip", tooltip,
			"stock-id", stock_id,
			"arrow-tooltip", arrow_tooltip,
			"menu-path", menu_path,
			"window", window,
			NULL );

	return action;
}

