/*  Screem:  screem-apps-model.c
 *
 *  Copyright (C) 2004 David A Knight
 *
 *  Based off panel-run-dialog.c from gnome-panel
 *  Copyright (C) 2003 Frank Worsley <fworsley@shaw.ca>
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA
 */

#include "config.h"

#include <glib/gi18n.h>

#include <gtk/gtk.h>
#include <gdk/gdkpixbuf.h>

#if GNOMEMENU_VERSION >= 2011001

#define GMENU_I_KNOW_THIS_IS_UNSTABLE
#include <gmenu-tree.h>

#else

#define MENU_I_KNOW_THIS_IS_UNSTABLE
#include <menu-tree.h>

typedef MenuTree GMenuTree;
typedef MenuTreeEntry GMenuTreeEntry;
typedef MenuTreeDirectory GMenuTreeDirectory;

#define gmenu_tree_entry_get_name menu_tree_entry_get_name
#define gmenu_tree_directory_get_contents menu_tree_directory_get_entries
#define gmenu_tree_lookup(x,y) menu_tree_lookup(x)
#define gmenu_tree_get_directory_from_path menu_tree_get_directory_from_path
	
#define gmenu_tree_unref menu_tree_unref
#define gmenu_tree_entry_get_name menu_tree_entry_get_name
#define gmenu_tree_entry_get_comment menu_tree_entry_get_comment
#define gmenu_tree_entry_get_icon menu_tree_entry_get_icon
#define gmenu_tree_entry_get_desktop_file_path menu_tree_entry_get_desktop_file_path

#endif


#include <string.h>

#include "screem-apps-model.h"

#include "support.h"

static GdkPixbuf *make_menu_icon( GtkIconTheme *gtk_theme,
		const char *icon, const char *fallback,
		gint size, gboolean *long_operation );
static gboolean add_icon_idle( ScreemAppsModel *model );
static int compare_applications( GMenuTreeEntry *a, GMenuTreeEntry *b );
static GSList
*get_all_applications_from_dir( GMenuTreeDirectory *directory,
		GSList *list );
static GSList *get_all_applications( void );
static gboolean build_applications_model( GtkListStore *store );

struct ScreemAppsModelPrivate {
	guint load_id;

	GSList *icons;
	guint icons_id;	
	
	GtkIconTheme *theme;
};

enum {
	LAST_SIGNAL
};

enum {
	PROP_0
};

/*static guint screem_apps_model_signals[ LAST_SIGNAL ] = { 0 };*/
	
G_DEFINE_TYPE( ScreemAppsModel, screem_apps_model, GTK_TYPE_LIST_STORE )

static void screem_apps_model_finalize( GObject *object );
static void screem_apps_model_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec );
static void screem_apps_model_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec );

static void screem_apps_model_class_init( ScreemAppsModelClass *klass )
{
	GObjectClass *obj_class;
	
	obj_class = G_OBJECT_CLASS( klass );
	obj_class->finalize = screem_apps_model_finalize;
	obj_class->get_property = screem_apps_model_get_prop;
	obj_class->set_property = screem_apps_model_set_prop;

}

static void screem_apps_model_init( ScreemAppsModel *apps_model )
{
	ScreemAppsModelPrivate *priv;
	GtkListStore *store;
	GType columns[] = {
		GDK_TYPE_PIXBUF, 
		G_TYPE_STRING, 
		G_TYPE_STRING, 
		G_TYPE_STRING, 
		G_TYPE_STRING, 
		G_TYPE_STRING,
		G_TYPE_STRING
	};
	store = GTK_LIST_STORE( apps_model );

	apps_model->priv = priv = g_new0( ScreemAppsModelPrivate, 1 );
	
	gtk_list_store_set_column_types( store, 
			AVAILABLE_APP_NUM_COLUMNS, columns );

	priv->load_id = g_idle_add( (GSourceFunc)build_applications_model, 
			apps_model );

	priv->theme = gtk_icon_theme_get_default();
	g_object_ref( priv->theme );
}

static void screem_apps_model_finalize( GObject *object )
{
	ScreemAppsModel *apps_model;
	ScreemAppsModelPrivate *priv;
	
	g_return_if_fail( object != NULL );
	g_return_if_fail( SCREEM_IS_APPS_MODEL( object ) );

	apps_model = SCREEM_APPS_MODEL( object );

	priv = apps_model->priv;

	g_object_unref( priv->theme );
	if( priv->load_id ) {
		g_source_remove( priv->load_id );
	}
	if( priv->icons_id ) {
		g_source_remove( priv->icons_id );
	}
	
	g_free( priv );
	
	G_OBJECT_CLASS( screem_apps_model_parent_class )->finalize( object );
}

static void screem_apps_model_set_prop( GObject *object, guint prop_id,
				const GValue *value, GParamSpec *spec )
{
	ScreemAppsModel *apps_model;
	ScreemAppsModelPrivate *priv;
	
	apps_model = SCREEM_APPS_MODEL( object );
	priv = apps_model->priv;

	switch( prop_id ) {
		default:
			break;
	}
}

static void screem_apps_model_get_prop( GObject *object, guint prop_id,
				GValue *value, GParamSpec *spec )
{
	ScreemAppsModel *apps_model;
	ScreemAppsModelPrivate *priv;
	
	apps_model = SCREEM_APPS_MODEL( object );
	priv = apps_model->priv;

	switch( prop_id ) {
		default:
			break;
	}
}

/* static stuff */
static GdkPixbuf *make_menu_icon( GtkIconTheme *gtk_theme,
		const char *icon, const char *fallback,
		gint size, gboolean *long_operation )
{
	GdkPixbuf *pb;
	gchar *file;

	g_return_val_if_fail( size > 0, NULL );

	file = NULL;
	if( icon != NULL ) {
		file = screem_lookup_icon( gtk_theme, icon, size, 0 );
	}
	
	if( file == NULL && fallback != NULL ) {
		file = screem_lookup_icon( gtk_theme, fallback, 
				size, 0 );
	}

	if( file == NULL ) {
		return NULL;
	}

	if( long_operation != NULL ) {
		*long_operation = TRUE;
	}

	pb = NULL;

	if (pb == NULL) {
		pb = gdk_pixbuf_new_from_file_at_size( file, 
				size, size, NULL );
	}

	if (pb == NULL) {
		g_free( file );
		return NULL;
	}

	if( ( gdk_pixbuf_get_width( pb ) != size ||
	      gdk_pixbuf_get_height( pb ) != size ) ) {
		
		GdkPixbuf *pb2;
		pb2 = gdk_pixbuf_scale_simple( pb, size, size,
				GDK_INTERP_BILINEAR );
		g_object_unref( G_OBJECT( pb ) );
		pb = pb2;
	}

	g_free( file );

	return pb;
}

static gboolean add_icon_idle( ScreemAppsModel *model )
{
	ScreemAppsModelPrivate *priv;
	GtkTreeIter  iter;
	GtkTreePath *path;
	GdkPixbuf   *pixbuf;
	char        *file;
	int          icon_height;
	gboolean     long_operation = FALSE;
	
	priv = model->priv;
	
	do {
		if( ! priv->icons ) {
			priv->icons_id = 0;
			return FALSE;
		}

		path = priv->icons->data;
		priv->icons->data = NULL;
		priv->icons = g_slist_delete_link( priv->icons,
				priv->icons );

		if( ! gtk_tree_model_get_iter( GTK_TREE_MODEL( model ),
					&iter, path ) ) {
			gtk_tree_path_free( path );
			continue;
		}
		
		gtk_tree_path_free( path );

		gtk_tree_model_get( GTK_TREE_MODEL( model ), &iter,
				AVAILABLE_APP_COLUMN_ICON_FILE, &file,
				-1);

		icon_height = 16;
		gtk_icon_size_lookup( GTK_ICON_SIZE_MENU, NULL, 
				&icon_height );
		
		pixbuf =make_menu_icon( priv->theme,
				file, NULL, icon_height, 
				&long_operation );

		if( pixbuf ) {
			gtk_list_store_set( GTK_LIST_STORE( model ),
					&iter, 
					AVAILABLE_APP_COLUMN_ICON, 
					pixbuf, -1 );
			g_object_unref( pixbuf );
		}
		g_free( file );
		
	/* don't go back into the main loop if this wasn't very hard
	 * to do */
	} while( ! long_operation );

	return TRUE;
}

static int compare_applications( GMenuTreeEntry *a, GMenuTreeEntry *b )
{
	return g_utf8_collate( gmenu_tree_entry_get_name( a ),
			gmenu_tree_entry_get_name( b ) );
}
	
static GSList *get_all_applications_from_dir( GMenuTreeDirectory *directory,
		GSList *list )
{
#if GNOMEMENU_VERSION >= 2011001

	list = g_slist_concat( list,
			gmenu_tree_directory_get_contents( directory ) );
#else
	GSList *subdirs;
	GSList *l;

	list = g_slist_concat( list,
			menu_tree_directory_get_entries( directory ) );

	subdirs = menu_tree_directory_get_subdirs( directory );
	for( l = subdirs; l; l = l->next ) {
		MenuTreeDirectory *subdir = l->data;

		list = get_all_applications_from_dir( subdir, list );

		menu_tree_directory_unref( subdir );
	}
	g_slist_free( subdirs );
	
#endif
	return list;
}

static GSList *get_all_applications( void )
{
	GMenuTree          *tree;
	GMenuTreeDirectory *root;
	GSList            *retval;

	/* used to contain /Network as well, this seems to
	 * cause a lockup with gnome 2.12 though so it has
	 * been removed */
	static const gchar *directories[] = {
		"/Internet"
	};
	static const int num = G_N_ELEMENTS( directories );
	gint i;
	
	tree = gmenu_tree_lookup( "applications.menu", 
			GMENU_TREE_FLAGS_NONE );

	retval = NULL;
	for( i = 0; i < num; ++ i ) {
		root = gmenu_tree_get_directory_from_path( tree,
				directories[ i ] );
		if( root ) {
			retval = get_all_applications_from_dir( root, 
					retval );
#if GNOMEMENU_VERSION >= 2011001
			gmenu_tree_item_unref( root );
#else
			menu_tree_directory_unref( root );
#endif
		}
	}
	gmenu_tree_unref( tree );

	retval = g_slist_sort( retval,
			(GCompareFunc)compare_applications );

	return retval;
}

static gboolean build_applications_model( GtkListStore *store )
{
	ScreemAppsModel *model;
	ScreemAppsModelPrivate *priv;
	GSList *all_applications;
	GSList *l;
	const gchar *name;
	const gchar *comment;
	const gchar *icon;
	const gchar *file_path;
	const gchar *prev_name;
	GKeyFile *ditem;
	GtkTreeIter iter;
	GtkTreePath *path;

	gchar *tmp;
	gint i;
	
	model = SCREEM_APPS_MODEL( store );
	priv = model->priv;
	
	/* create list store */
	all_applications = get_all_applications();
	
	/* Strip duplicates, non applications */
	prev_name = NULL;
	i = 0;
	for( l = all_applications; l; l = l->next ) {
		GMenuTreeEntry *entry = l->data;

		name = gmenu_tree_entry_get_name( entry );
		comment = gmenu_tree_entry_get_comment( entry );
		icon = gmenu_tree_entry_get_icon( entry );
		file_path = gmenu_tree_entry_get_desktop_file_path( entry );
	
		if( prev_name && ! strcmp( name, prev_name ) ) {	
#if GNOMEMENU_VERSION >= 2011001
			gmenu_tree_item_unref( entry );
#else
			menu_tree_entry_unref( entry );
#endif
			continue;
		}
		ditem = g_key_file_new();
		tmp = NULL;
		if( g_key_file_load_from_file( ditem, file_path,
					G_KEY_FILE_NONE, NULL ) ) {
			tmp = g_key_file_get_string( ditem,
					"Desktop Entry", "Type", NULL );
			if( tmp && ! strcmp( "Application", tmp ) ) {
				g_free( tmp );
				tmp = g_key_file_get_string( ditem,
						"Desktop Entry",
						"Exec", NULL );
			} else {
				g_free( tmp );
				tmp = NULL;
			}
			g_key_file_free( ditem );
       	        }
		if( tmp ) {
			prev_name = name;
		
			gtk_list_store_insert_with_values( store, 
					&iter, i ++,
					AVAILABLE_APP_COLUMN_ICON, NULL,
					AVAILABLE_APP_COLUMN_ICON_FILE,
					icon,
					AVAILABLE_APP_COLUMN_NAME, name,
					AVAILABLE_APP_COLUMN_COMMENT, 
					comment,
					AVAILABLE_APP_COLUMN_PATH, 
					file_path,
					AVAILABLE_APP_COLUMN_EXEC, tmp,
					-1 );
			path = gtk_tree_model_get_path( GTK_TREE_MODEL( store ),
					&iter );
			if( path ) {
				priv->icons = g_slist_prepend( priv->icons,
						path );
			}
		}
#if GNOMEMENU_VERSION >= 2011001
		gmenu_tree_item_unref( entry );
#else
		menu_tree_entry_unref( entry );
#endif
	}
	g_slist_free( all_applications );

	priv->load_id = 0;

	if( ! priv->icons_id ) {
		priv->icons_id = g_idle_add_full( G_PRIORITY_LOW, 
				(GSourceFunc) add_icon_idle,
				store, NULL);
	}
	
	return FALSE;
}


/* public stuff */

ScreemAppsModel *screem_apps_model_new( void )
{
	ScreemAppsModel *apps_model;

	apps_model = g_object_new( SCREEM_TYPE_APPS_MODEL, NULL );

	return apps_model;
}

void screem_apps_model_setup_cell_layout( GtkWidget *widget )
{
	GtkCellRenderer *prenderer;
	GtkCellRenderer *renderer;
	GtkTreeViewColumn *col;
	
	if( GTK_IS_CELL_LAYOUT( widget ) ) {
		gtk_cell_layout_clear( GTK_CELL_LAYOUT( widget ) );
	
		prenderer = gtk_cell_renderer_pixbuf_new();
		gtk_cell_layout_pack_start( GTK_CELL_LAYOUT( widget ), prenderer,
				FALSE );
		gtk_cell_layout_set_attributes( GTK_CELL_LAYOUT( widget ),
				prenderer, "pixbuf", AVAILABLE_APP_COLUMN_ICON,
				NULL );

		renderer = gtk_cell_renderer_text_new ();
		gtk_cell_layout_pack_start( GTK_CELL_LAYOUT( widget ), renderer,
				TRUE );
		gtk_cell_layout_set_attributes( GTK_CELL_LAYOUT( widget ),
				renderer, "text", 
				AVAILABLE_APP_COLUMN_NAME, NULL );
	} else if( GTK_IS_TREE_VIEW( widget ) ) {
		prenderer = gtk_cell_renderer_pixbuf_new();
		renderer = gtk_cell_renderer_text_new ();
		
		col = gtk_tree_view_column_new();
		gtk_tree_view_column_pack_start( col, prenderer, FALSE );
		gtk_tree_view_column_pack_start( col, renderer, TRUE );
		
		gtk_tree_view_column_set_attributes( col, prenderer,
				"pixbuf", AVAILABLE_APP_COLUMN_ICON, NULL );
		gtk_tree_view_column_set_attributes( col, renderer,
				"text", AVAILABLE_APP_COLUMN_NAME, NULL ); 

		gtk_tree_view_append_column( GTK_TREE_VIEW( widget ), col );
	} else {
		g_warning( "Incorrect widget passed to screem_apps_model_setup_cell_layout\n" );
	}
}

GtkTreeModel *screem_apps_model_get_default( void )
{
	static gpointer model = NULL;

	if( ! model ) {
		model = screem_apps_model_new();
		g_object_add_weak_pointer( G_OBJECT( model ), &model );
	} else {
		g_object_ref( model );
	}

	return GTK_TREE_MODEL( model );
}

