/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.jogl.sprite;

import java.nio.FloatBuffer;

import javax.media.opengl.GL;

import org.scilab.forge.scirenderer.buffers.ElementsBuffer;
import org.scilab.forge.scirenderer.implementation.jogl.JoGLDrawingTools;
import org.scilab.forge.scirenderer.sprite.SpriteAnchorPosition;
import org.scilab.forge.scirenderer.tranformations.Transformation;
import org.scilab.forge.scirenderer.tranformations.TransformationManager;
import org.scilab.forge.scirenderer.tranformations.Vector3d;

/**
 * @author Pierre Lando
 */
public class QuadsTextureRenderer extends AbstractTextureRenderer {

    /**
     * Default constructor.
     *
     * @param sprite the sprite this renderer draw.
     */
    QuadsTextureRenderer(JoGLSprite sprite) {
        super(sprite, false);
    }

    @Override
    public void draw(JoGLDrawingTools drawingTools, SpriteAnchorPosition anchor, ElementsBuffer positions) {
        GL gl = drawingTools.getGl();
        update(gl);

        gl.glEnable(GL.GL_TEXTURE_2D);
        bindTexture(gl);

        gl.glEnable(GL.GL_BLEND);
        gl.glBlendFunc(GL.GL_SRC_ALPHA, GL.GL_ONE_MINUS_SRC_ALPHA);
        gl.glEnable(GL.GL_ALPHA_TEST);
        gl.glAlphaFunc(GL.GL_GREATER, 0);

        /**
         * Draw the sprite.
         */
        TransformationManager transformationManager = drawingTools.getTransformationManager();
        Transformation canvasProjection = transformationManager.getCanvasProjection();

        boolean needProjection = drawingTools.getTransformationManager().isUsingSceneCoordinate();
        drawingTools.getTransformationManager().useWindowCoordinate();

        FloatBuffer data = positions.getData();
        data.rewind();
        while (data.remaining() >= positions.getElementsSize()) {
            Vector3d position = new Vector3d(data.get(), data.get(), data.get());
            if (positions.getElementsSize() == 4) {
                data.get();
            }

            Vector3d projected;
            if (needProjection) {
                projected = canvasProjection.project(position);
            } else {
                projected = position;
            }

            int x = (int) (projected.getX() + getAnchorDeltaX(anchor));
            int y = (int) (projected.getY() + getAnchorDeltaY(anchor));
            double z = projected.getZ();

            gl.glBegin(GL.GL_QUADS);
                gl.glTexCoord2f(0, 1);  gl.glVertex3d(x - getImage().getWidth() / 2f, y - getImage().getHeight() / 2f, z);
                gl.glTexCoord2f(0, 0);  gl.glVertex3d(x - getImage().getWidth() / 2f, y + getImage().getHeight() / 2f, z);
                gl.glTexCoord2f(1, 0);  gl.glVertex3d(x + getImage().getWidth() / 2f, y + getImage().getHeight() / 2f, z);
                gl.glTexCoord2f(1, 1);  gl.glVertex3d(x + getImage().getWidth() / 2f, y - getImage().getHeight() / 2f, z);
            gl.glEnd();
        }

        if (needProjection) {
            drawingTools.getTransformationManager().useSceneCoordinate();
        }

        gl.glDisable(GL.GL_TEXTURE_2D);
        gl.glDisable(GL.GL_BLEND);
        gl.glDisable(GL.GL_ALPHA_TEST);
    }
}
