/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.jogl.sprite;

import org.scilab.forge.scirenderer.buffers.ElementsBuffer;
import org.scilab.forge.scirenderer.implementation.jogl.HardwareFailException;
import org.scilab.forge.scirenderer.implementation.jogl.JoGLDrawingTools;
import org.scilab.forge.scirenderer.sprite.SpriteAnchorPosition;
import org.scilab.forge.scirenderer.tranformations.Transformation;
import org.scilab.forge.scirenderer.tranformations.TransformationFactory;

import javax.media.opengl.GL;

/**
 * An implementation of a {@link JoGLSpriteRenderer} using OpenGl texture and point sprite for rendering.
 *
 * @author Pierre Lando
 */
class PointRenderer extends AbstractTextureRenderer implements JoGLSpriteRenderer {

    /**
     * The current OpenGl point size.
     */
    private final int pointSize;

    /**
     * Default constructor.
     * @param drawingTools the current drawing tools.
     * @param sprite the sprite this object render.
     * @throws HardwareFailException when the sprite size is bigger than the OpenGl point sprite size.
     */
    PointRenderer(JoGLDrawingTools drawingTools, JoGLSprite sprite) throws HardwareFailException {
        super(sprite, true);

        /**
         * Compute the point size.
         */
        pointSize = Math.max(getImage().getWidth(), getImage().getHeight());

        float[] range = drawingTools.getGLCapacity().getAliasedPointSizeRange();

        if (pointSize > range[1]) {
            throw new HardwareFailException("The 'GL_ALIASED_POINT_SIZE_RANGE' is too small to draw this sprite.");
        }
    }

    @Override
    public void draw(JoGLDrawingTools drawingTools, SpriteAnchorPosition anchor, ElementsBuffer positions) {
        GL gl = drawingTools.getGl();
        update(gl);

        setAnchor(drawingTools, anchor);


        gl.glAlphaFunc(GL.GL_GREATER, 0.0f);
        gl.glPointSize(pointSize);

        gl.glEnable(GL.GL_TEXTURE_2D);
        bindTexture(gl);
        gl.glEnable(GL.GL_POINT_SPRITE);
        gl.glEnable(GL.GL_POINT_SMOOTH);
        gl.glEnable(GL.GL_ALPHA_TEST);
        gl.glTexEnvi(GL.GL_POINT_SPRITE, GL.GL_COORD_REPLACE, GL.GL_TRUE);

        drawingTools.getCanvas().getBuffersManager().bindVertexBuffer(gl, positions);
        gl.glDrawArrays(GL.GL_POINTS, 0, positions.getSize());
        gl.glDisableClientState(GL.GL_VERTEX_ARRAY);

        gl.glDisable(GL.GL_TEXTURE_2D);
        gl.glDisable(GL.GL_POINT_SMOOTH);
        gl.glDisable(GL.GL_POINT_SPRITE);
        gl.glDisable(GL.GL_ALPHA_TEST);

        drawingTools.getTransformationManager().getProjectionStack().pop();
    }

    /**
     * Change the projection matrix to correctly place the anchor.
     * @param drawingTools the drawing tools.
     * @param anchor the anchor.
     */
    private void setAnchor(JoGLDrawingTools drawingTools, SpriteAnchorPosition anchor) {
        double x = getAnchorDeltaX(anchor);
        double y = getAnchorDeltaY(anchor);

        double w = drawingTools.getCanvas().getWidth() / 2.0;
        double h = drawingTools.getCanvas().getHeight() / 2.0;
        Transformation t = TransformationFactory.getTranslateTransformation(x / w, y / h, 0);
        drawingTools.getTransformationManager().getProjectionStack().pushLeftMultiply(t);
    }
}
