/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2012 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.jogl.sprite;

import org.scilab.forge.scirenderer.buffers.ElementsBuffer;
import org.scilab.forge.scirenderer.implementation.jogl.JoGLCanvas;
import org.scilab.forge.scirenderer.implementation.jogl.JoGLDrawingTools;
import org.scilab.forge.scirenderer.sprite.Sprite;
import org.scilab.forge.scirenderer.sprite.SpriteAnchorPosition;
import org.scilab.forge.scirenderer.sprite.SpriteManager;
import org.scilab.forge.scirenderer.sprite.TextEntity;
import org.scilab.forge.scirenderer.tranformations.Vector3d;

import java.awt.Dimension;
import java.awt.font.FontRenderContext;
import java.awt.font.TextLayout;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.Stack;

/**
 * @author Pierre Lando
 */
public class JoGLSpriteManager implements SpriteManager {

    /**
     * The current canvas.
     */
    private final JoGLCanvas canvas;

    /**
     * The stack of dead sprite.
     * This sprites will be disposed at the next call to {@link this#glSynchronize}.
     */
    private final Stack<JoGLSprite> deadSprites = new Stack<JoGLSprite>();

    /**
     * Set of current sprite.
     */
    private final Set<JoGLSprite> spriteSet = new HashSet<JoGLSprite>();

    /**
     * Default constructor.
     * @param canvas current canvas.
     */
    public JoGLSpriteManager(JoGLCanvas canvas) {
        this.canvas = canvas;
    }


    @Override
    public Sprite createSprite(int width, int height) {
        JoGLSprite sprite = new JoGLSprite(canvas, width, height);
        spriteSet.add(sprite);
        return sprite;
    }

    @Override
    public Sprite createSprite(Dimension dimension) {
        return createSprite(dimension.width, dimension.height);
    }

    @Override
    public Sprite createRotatableSprite(int width, int height) {
        JoGLRotatableSprite sprite = new JoGLRotatableSprite(canvas, width, height);
        spriteSet.add(sprite);
        return sprite;
    }

    @Override
    public void dispose(Sprite sprite) {
        if ((sprite != null) && (sprite instanceof JoGLSprite)) {
            JoGLSprite localSprite = (JoGLSprite) sprite;
            deadSprites.push(localSprite);
            spriteSet.remove(localSprite);
        }
    }

    @Override
    public void dispose(Collection<Sprite> sprites) {
        if (sprites != null) {
            for (Sprite sprite: sprites) {
                dispose(sprite);
            }
        }
    }

    @Override
    public Dimension getSize(TextEntity textEntity) {
        try {
            FontRenderContext frc = new FontRenderContext(null, textEntity.isTextAntiAliased(), textEntity.isTextUseFractionalMetrics());
            TextLayout textLayout = new TextLayout(textEntity.getText(), textEntity.getFont(), frc);
            Dimension dimension = new Dimension();
            dimension.setSize(textLayout.getBounds().getWidth(), textLayout.getBounds().getHeight());
            return dimension;
        } catch (Exception e) {
            return new Dimension(0, 0);
        }
    }

    /**
     * Perform the sprite drawing.
     * @param drawingTools the drawing tools.
     * @param sprite       sprite to draw.
     * @param anchor       anchor position.
     * @param positions    position where draw the sprite.
     */
    public void draw(JoGLDrawingTools drawingTools, Sprite sprite, SpriteAnchorPosition anchor, ElementsBuffer positions) {
        if ((sprite != null) && (sprite instanceof JoGLSprite)) {
            JoGLSprite localSprite = (JoGLSprite) sprite;
            if (spriteSet.contains(localSprite)) {
                localSprite.draw(drawingTools, anchor, positions);
            }
        }
    }

    /**
     * Perform the sprite drawing.
     * @param drawingTools  the drawing tools.
     * @param sprite        sprite to draw.
     * @param anchor        anchor position.
     * @param positions     position where draw the sprite.
     * @param rotationAngle rotation angle.
     */
    public void draw(JoGLDrawingTools drawingTools, Sprite sprite, SpriteAnchorPosition anchor, ElementsBuffer positions, double rotationAngle) {
        if (sprite != null) {
            if (sprite instanceof JoGLRotatableSprite) {
                JoGLRotatableSprite localSprite = (JoGLRotatableSprite) sprite;
                if (spriteSet.contains(localSprite)) {
                    localSprite.draw(drawingTools, anchor, positions, rotationAngle);
                }
            } else if (sprite instanceof JoGLSprite) {
                JoGLSprite localSprite = (JoGLSprite) sprite;
                if (spriteSet.contains(localSprite)) {
                    localSprite.draw(drawingTools, anchor, positions);
                }
            }
        }
    }

    /**
     * Perform the sprite drawing.
     * @param drawingTools drawing tools.
     * @param sprite       sprite to draw.
     * @param anchor       anchor position.
     * @param position     position where draw the sprite.
     */
    public void draw(JoGLDrawingTools drawingTools, Sprite sprite, SpriteAnchorPosition anchor, Vector3d position) {
        if ((sprite != null) && (sprite instanceof JoGLSprite)) {
            JoGLSprite localSprite = (JoGLSprite) sprite;
            if (spriteSet.contains(localSprite)) {
                localSprite.draw(drawingTools, anchor, position);
            }
        }
    }

    /**
     * Perform the sprite drawing.
     * @param drawingTools  drawing tools.
     * @param sprite        sprite to draw.
     * @param anchor        anchor position.
     * @param position      position where draw the sprite.
     * @param rotationAngle rotation angle.
     */
    public void draw(JoGLDrawingTools drawingTools, Sprite sprite, SpriteAnchorPosition anchor, Vector3d position, double rotationAngle) {
        if (sprite != null) {
            if (sprite instanceof JoGLRotatableSprite) {
                JoGLRotatableSprite localSprite = (JoGLRotatableSprite) sprite;
                if (spriteSet.contains(localSprite)) {
                    localSprite.draw(drawingTools, anchor, position, rotationAngle);
                }
            } else if (sprite instanceof JoGLSprite) {
                JoGLSprite localSprite = (JoGLSprite) sprite;
                if (spriteSet.contains(localSprite)) {
                    localSprite.draw(drawingTools, anchor, position);
                }
            }
        }
    }

    /**
     * Synchronize to OpenGl context.
     * Mostly consist to dispose dead sprites resources.
     * @param drawingTools the current drawing tools.
     */
    public void glSynchronize(JoGLDrawingTools drawingTools) {
        while (!deadSprites.isEmpty()) {
            deadSprites.pop().dispose(drawingTools);
        }
    }

    /**
     * Ask all {@link JoGLSprite} to reload.
     * This is needed when the OpenGl context has been lost.
     * @param drawingTools the current drawing tools.
     */
    public void glReload(JoGLDrawingTools drawingTools) {
        for (JoGLSprite sprite : spriteSet) {
            sprite.reload(drawingTools);
        }
    }
}
