/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.implementation.jogl.buffers;

import org.scilab.forge.scirenderer.buffers.BuffersManager;
import org.scilab.forge.scirenderer.buffers.DataBuffer;
import org.scilab.forge.scirenderer.buffers.ElementsBuffer;
import org.scilab.forge.scirenderer.buffers.IndicesBuffer;

import javax.media.opengl.GL;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.Stack;

/**
 * @author Pierre Lando
 */
public class JoGLBuffersManager implements BuffersManager {

    /**
     * Set off current buffers.
     */
    private final Set<JoGLDataBuffer> buffers = new HashSet<JoGLDataBuffer>();

    /**
     * Set of dead buffers.
     */
    private final Stack<JoGLDataBuffer> deadBuffers = new Stack<JoGLDataBuffer>();

    /**
     * Default constructor.
     */
    public JoGLBuffersManager() {
    }

    @Override
    public ElementsBuffer createElementsBuffer() {
        JoGLElementsBuffer newBuffer = new JoGLElementsBuffer();
        buffers.add(newBuffer);
        return newBuffer;
    }

    @Override
    public IndicesBuffer createIndicesBuffer() {
        JoGLIndicesBuffer newBuffer = new JoGLIndicesBuffer();
        buffers.add(newBuffer);
        return newBuffer;
    }

    @Override
    public void dispose(DataBuffer buffer) {
        JoGLDataBuffer localBuffer = getLocalBuffer(buffer);
        if (localBuffer != null) {
            buffers.remove(localBuffer);
            deadBuffers.push(localBuffer);
        }
    }

    @Override
    public void dispose(Collection<? extends DataBuffer> buffers) {
        for (DataBuffer buffer : buffers) {
            dispose(buffer);
        }
    }

    /**
     * Called when previous OpenGl context is gone.
     * @param gl the new OpenGl context.
     */
    public void glReload(GL gl) {
        for (JoGLDataBuffer buffer : buffers) {
            buffer.reload(gl);
        }
    }

    /**
     * Called before rendering for synchronisation.
     * Clean dead buffers.
     * @param gl the OpenGl context.
     */
    public void glSynchronize(GL gl) {
        while (!deadBuffers.isEmpty()) {
            deadBuffers.pop().dispose(gl);
        }
    }

    public int bindVertexBuffer(GL gl, ElementsBuffer buffer) {
        JoGLElementsBuffer localBuffer = getLocalElementsBuffer(buffer);
        if (localBuffer != null) {
            return localBuffer.bindAsVertexBuffer(gl);
        } else {
            return 0;
        }
    }

    public int bindNormalsBuffer(GL gl, ElementsBuffer buffer) {
        JoGLElementsBuffer localBuffer = getLocalElementsBuffer(buffer);
        if (localBuffer != null) {
            return localBuffer.bindAsNormalsBuffer(gl);
        } else {
            return 0;
        }
    }

    public int bindTextureCoordinatesBuffer(GL gl, ElementsBuffer buffer) {
        JoGLElementsBuffer localBuffer = getLocalElementsBuffer(buffer);
        if (localBuffer != null) {
            return localBuffer.bindAsTextureCoordinatesBuffer(gl);
        } else {
            return 0;
        }
    }

    public int bindColorsBuffer(GL gl, ElementsBuffer buffer) {
        JoGLElementsBuffer localBuffer = getLocalElementsBuffer(buffer);
        if (localBuffer != null) {
            return localBuffer.bindAsColorsBuffer(gl);
        } else {
            return 0;
        }
    }

    public int bindIndicesBuffer(GL gl, IndicesBuffer indices) {
        JoGLDataBuffer localBuffer = getLocalBuffer(indices);
        if (localBuffer != null) {
            gl.glBindBuffer(GL.GL_ELEMENT_ARRAY_BUFFER, localBuffer.getGlName(gl));
            return indices.getSize();
        } else {
            return 0;
        }
    }

    /**
     * This method check buffer to be from here.
     * @param buffer the given buffer.
     * @return the JoGL instance of the buffer.
     */
    private JoGLDataBuffer getLocalBuffer(DataBuffer buffer) {
        if (buffer instanceof JoGLDataBuffer) {
            JoGLDataBuffer localBuffer = (JoGLDataBuffer) buffer;
            if (buffers.contains(localBuffer)) {
                return localBuffer;
            }
        }
        return null;
    }


    /**
     * This method check buffer to be from here.
     * @param buffer the given buffer.
     * @return the JoGL instance of the buffer.
     */
    private JoGLIndicesBuffer getLocalIndicesBuffer(IndicesBuffer buffer) {
        if (buffer instanceof JoGLIndicesBuffer) {
            JoGLIndicesBuffer localBuffer = (JoGLIndicesBuffer) buffer;
            if (buffers.contains(localBuffer)) {
                return localBuffer;
            }
        }
        return null;
    }

    /**
     * This method check buffer to be from here.
     * @param buffer the given buffer.
     * @return the JoGL instance of the buffer.
     */
    private JoGLElementsBuffer getLocalElementsBuffer(ElementsBuffer buffer) {
        if (buffer instanceof JoGLElementsBuffer) {
            JoGLElementsBuffer localBuffer = (JoGLElementsBuffer) buffer;
            if (buffers.contains(localBuffer)) {
                return localBuffer;
            }
        }
        return null;
    }
}
