/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2011 - DIGITEO - Manuel Juliachs
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.examples.rotatableSprite;

import org.scilab.forge.scirenderer.Canvas;
import org.scilab.forge.scirenderer.Drawer;
import org.scilab.forge.scirenderer.DrawingTools;
import org.scilab.forge.scirenderer.buffers.ElementsBuffer;
import org.scilab.forge.scirenderer.shapes.appearance.Appearance;
import org.scilab.forge.scirenderer.shapes.appearance.Color;
import org.scilab.forge.scirenderer.sprite.Sprite;
import org.scilab.forge.scirenderer.sprite.SpriteAnchorPosition;
import org.scilab.forge.scirenderer.sprite.SpriteDrawer;
import org.scilab.forge.scirenderer.sprite.SpriteDrawingTools;
import org.scilab.forge.scirenderer.sprite.SpriteManager;
import org.scilab.forge.scirenderer.sprite.TextEntity;
import org.scilab.forge.scirenderer.tranformations.DegenerateMatrixException;
import org.scilab.forge.scirenderer.tranformations.Transformation;
import org.scilab.forge.scirenderer.tranformations.TransformationFactory;
import org.scilab.forge.scirenderer.tranformations.Vector3d;

import java.awt.Dimension;

/**
 * A rotatable sprite drawer.
 * Draws one rotating sprite for each existing sprite anchor position value.
 *
 * @author Manuel Juliachs
 */
public class RotatableSpriteDrawerExample implements Drawer {

    /** The buffer used to position anchor sprites. */
    private ElementsBuffer position;

    /** The sprites' coordinates. */
    float[] positions;

    /** The sprites' text strings. */
    private static final String[] textStrings = {"Center anchor", "Left anchor", "Lower-left anchor",
                                                  "Down anchor", "Lower-right anchor", "Right anchor",
                                                  "Upper-right anchor", "Up anchor", "Upper-left anchor"};

    /** The anchor positions. */
    private static final SpriteAnchorPosition[] anchorPositions = {SpriteAnchorPosition.CENTER, SpriteAnchorPosition.LEFT, SpriteAnchorPosition.LOWER_LEFT,
                                                                   SpriteAnchorPosition.DOWN, SpriteAnchorPosition.LOWER_RIGHT, SpriteAnchorPosition.RIGHT,
                                                                   SpriteAnchorPosition.UPPER_RIGHT, SpriteAnchorPosition.UP, SpriteAnchorPosition.UPPER_LEFT};

    /** The text sprites. */
    private Sprite[] textSprites;

    /** The sprite used to draw anchors. */
    private Sprite anchorSprite;

    /** The message sprite. */
    private Sprite messageSprite;

    /** The sprite manager. */
    private SpriteManager spriteManager;

    /** The number of sprites to draw, equal to the number of available sprite anchor position values. */
    private static final int NUM_SPRITES = 9;

    /** The sprites' text entities. */
    private TextEntity[] textEntities;

    /** The rotation angle applied to all the sprites. */
    private float rotationAngle = 0.0f;

    /** The step added to the rotation angle at each draw call. */
    private float rotationAngleStep = 2.5f;

    /* Arbitrary displacement used to position the sprites. */
    private float deltaPos = 0.8f;

    /** The margin used for the text sprites. */
    private static final int MARGIN = 4;

    /** The line width of the sprites' box. */
    private int lineWidth = 2;

    /** The line's half width. */
    private int halfLineWidth = lineWidth / 2;

    /** Specifies whether anchor sprites are displayed or not. */
    private boolean anchorsDisplayed;

    /** Specifies the sprites' rotation direction. */
    private boolean clockwiseRotation;

    /** Specifies whether the message is displayed or not. */
    private boolean messageDisplayed;

    /**
     * Constructor.
     * @param canvas the canvas to use.
     */
    public RotatableSpriteDrawerExample(Canvas canvas) {
        spriteManager = canvas.getSpriteManager();

        anchorsDisplayed = true;
        clockwiseRotation = false;
        messageDisplayed = true;


        textEntities = new TextEntity[NUM_SPRITES];

        for (int i = 0; i < NUM_SPRITES; i++) {
            TextEntity textEntity = new TextEntity(textStrings[i]);
            textEntity.setTextColor(new Color(0.0f, 0.0f, 0.0f));
            textEntity.setTextAntiAliased(false);
            textEntities[i] = textEntity;
        }

        textSprites = new Sprite[NUM_SPRITES];

        for (int i = 0; i < NUM_SPRITES; i++) {
            Dimension dimension = spriteManager.getSize(textEntities[i]);

            int boxWidth = (int) dimension.getWidth() + 2*MARGIN + lineWidth;
            int boxHeight = (int) dimension.getHeight() + 2*MARGIN + lineWidth;

            textSprites[i] = canvas.getSpriteManager().createRotatableSprite(boxWidth, boxHeight);
            textSprites[i].setDrawer(getTextSpriteDrawer(i));
        }

        /* The sprite used to visualize anchor points */
        anchorSprite = canvas.getSpriteManager().createSprite(10, 10);
        anchorSprite.setDrawer(getSpriteDrawer());

        position = canvas.getBuffersManager().createElementsBuffer();

        positions = new float[]{0, 0, 0,
                                -deltaPos, 0, 0,
                                -deltaPos, -deltaPos, 0,
                                0, -deltaPos, 0,
                                deltaPos, -deltaPos, 0,
                                deltaPos, 0, 0,
                                deltaPos, deltaPos, 0,
                                0, deltaPos, 0,
                                -deltaPos, deltaPos, 0};

        position.setData(positions, 3);

        /* The message sprite */
        messageSprite = createMessage(canvas);
    }

    @Override
    public void draw(DrawingTools drawingTools) {
        drawingTools.clear(new Color(1.0f, 1.0f, 1.0f));

        Transformation projection = null;
        try {
            projection = TransformationFactory.getPreferredAspectRatioTransformation(drawingTools.getCanvas().getDimension(), 1f);
        } catch (DegenerateMatrixException ignored) {
            // Should not occur.
        }

        drawingTools.getTransformationManager().getProjectionStack().push(projection);

        for (int i = 0; i < NUM_SPRITES; i++) {
            drawingTools.draw(textSprites[i], anchorPositions[i], new Vector3d(positions[3*i], positions[3*i+1], positions[3*i+2]), rotationAngle);
        }

        if (anchorsDisplayed) {
            drawingTools.draw(anchorSprite, SpriteAnchorPosition.CENTER, position);
        }

        if (messageDisplayed) {
            drawingTools.draw(messageSprite, SpriteAnchorPosition.UPPER_LEFT, new Vector3d(-1.0, 1.0, 0.0));
        }


        if (clockwiseRotation) {
            rotationAngle -= rotationAngleStep;
        } else {
            rotationAngle += rotationAngleStep;
        }

        rotationAngle %= 360.0f;
    }

    /**
     * Returns the anchor sprite drawer.
     * @return the anchor sprite drawer.
     */
    public SpriteDrawer getSpriteDrawer() {

        return new SpriteDrawer() {

            @Override
            public void draw(SpriteDrawingTools drawingTools) {
                drawingTools.fillDisc(0, 0, 10, new Color(1, 0, 0));


                Appearance appearance = new Appearance();
                appearance.setLineColor(new Color(0, 0, 0));
                drawingTools.drawCircle(0, 0, 10, appearance);
            }

            @Override
            public OriginPosition getOriginPosition() {
                return SpriteDrawer.OriginPosition.CENTER;
            }
        };
    }

   /**
    * Returns the sprite drawer associated to a particular text sprite.
    * @param textSpriteIndex the sprite index.
    * @return the text sprite drawer.
    */
   public SpriteDrawer getTextSpriteDrawer(final int textSpriteIndex) {

        if (textSpriteIndex > NUM_SPRITES-1 || textSpriteIndex < 0) {
            return null;
        }

        return new SpriteDrawer() {

            @Override
            public void draw(SpriteDrawingTools drawingTools) {
                TextEntity textEntity = textEntities[textSpriteIndex];

                drawingTools.draw(textEntity, 0 + MARGIN + halfLineWidth, 0 + MARGIN + halfLineWidth);

                double width = spriteManager.getSize(textEntity).getWidth();
                double height = spriteManager.getSize(textEntity).getHeight();

                int boxHeight = textSprites[textSpriteIndex].getHeight();
                int boxWidth = textSprites[textSpriteIndex].getWidth();

                Appearance appearance = new Appearance();
                appearance.setLineColor(new Color(0, 0, 0));
                appearance.setLineWidth((float) lineWidth);

                drawingTools.drawPolyline(new int[]{halfLineWidth, halfLineWidth, boxWidth - halfLineWidth, halfLineWidth,
                                                    boxWidth - halfLineWidth, boxHeight - halfLineWidth, halfLineWidth, boxHeight - halfLineWidth,
                                                    halfLineWidth, halfLineWidth},
                                                    appearance);
            }

            @Override
            public OriginPosition getOriginPosition() {
                return SpriteDrawer.OriginPosition.UPPER_LEFT;
            }

        };
    }

    /**
     * Toggles display of the anchors.
     */
    public void toggleAnchorDisplay() {
        anchorsDisplayed = !anchorsDisplayed;
    }

    /**
     * Switches the rotation direction.
     */
    public void switchRotationDirection() {
        clockwiseRotation = !clockwiseRotation;
    }

    /**
     * Toggles display of the help message.
     */
    public void toggleMessageDisplay() {
        messageDisplayed = !messageDisplayed;
    }

    /**
     * Creates a help message.
     * @param canvas the canvas where the message sprite is created.
     * @return a sprite that draws the message.
     */
    private Sprite createMessage(Canvas canvas) {
        final TextEntity text = new TextEntity("Press 'T' to toggle anchor drawing, 'D' to switch the rotation direction, 'M' to toggle this message.");
        Dimension dimension = canvas.getSpriteManager().getSize(text);
        Sprite message = canvas.getSpriteManager().createSprite(dimension.width, dimension.height);
        message.setDrawer(new SpriteDrawer() {

            @Override
            public void draw(SpriteDrawingTools drawingTools) {
                drawingTools.draw(text, 0, 0);
            }

            @Override
            public OriginPosition getOriginPosition() {
                return OriginPosition.UPPER_LEFT;
            }
        });

        return message;
    }

}
