/*
 * Scilab ( http://www.scilab.org/ ) - This file is part of Scilab
 * Copyright (C) 2009-2011 - DIGITEO - Pierre Lando
 *
 * This file must be used under the terms of the CeCILL.
 * This source file is licensed as described in the file COPYING, which
 * you should have received as part of this distribution.  The terms
 * are also available at
 * http://www.cecill.info/licences/Licence_CeCILL_V2-en.txt
 */

package org.scilab.forge.scirenderer.examples.milkDrop;


import org.scilab.forge.scirenderer.Canvas;
import org.scilab.forge.scirenderer.buffers.ElementsBuffer;
import org.scilab.forge.scirenderer.buffers.IndicesBuffer;
import org.scilab.forge.scirenderer.shapes.geometry.DefaultGeometry;
import org.scilab.forge.scirenderer.shapes.geometry.Geometry;

import java.nio.FloatBuffer;
import java.nio.IntBuffer;

/**
 * An utility class to create milk drop geometry.
 * @author Pierre Lando
 */
public final class MilkDropFactory {

    // TODO : use double precision.


    private static final int HALF_SIDE = 10;
    private static final int SIDE = 2 * HALF_SIDE + 1;
    private static final int ELEMENTS_SIZE = 4;

    /**
     * Return a geometry representing a milk drop.
     * @param canvas the canvas where the buffers are created.
     * @return a geometry representing a milk drop.
     */
    public static Geometry createMilkDrop(Canvas canvas) {
        FloatBuffer vertices = FloatBuffer.allocate(SIDE * SIDE * ELEMENTS_SIZE);
        FloatBuffer colors   = FloatBuffer.allocate(SIDE * SIDE * ELEMENTS_SIZE);
        IntBuffer   indices  = IntBuffer.allocate((SIDE-1) * (SIDE-1) * 2 * 3);
        vertices.rewind();
        colors.rewind();
        indices.rewind();

        for (int x = -HALF_SIDE; x <= HALF_SIDE; x++) {
            for (int y = -HALF_SIDE; y <= HALF_SIDE; y++) {
                double d = Math.sqrt((x * x) + (y * y));
                double z = HALF_SIDE * ((d==0) ? 1 : Math.sin(d) / d);

                vertices.put(x);
                vertices.put(y);
                vertices.put((float) z);
                vertices.put(1);

                colors.put((float) z);
                colors.put((float) (1-z));
                colors.put(0);
                colors.put(1);
            }
        }

        for (int x = 0; x < SIDE-1; x++) {
            for (int y = 0; y < SIDE-1; y++) {
                    /**
                     * (x, y)  -- (x+1, y)
                     *    |          |
                     *    |          |
                     * (x, y+1)--(x+1, y+1)
                     *
                     *  (x,y) => (x*SIDE + y);
                     *
                     */

                if (((x >= 0) && (y < 0)) || ((x < 0) && (y >= 0))){
                    indices.put( (x * SIDE) + y);
                    indices.put( ((x + 1) * SIDE) + y);
                    indices.put( ((x + 1) * SIDE) + (y + 1));

                    indices.put( (x * SIDE) + y);
                    indices.put( ((x + 1) * SIDE) + (y + 1));
                    indices.put( (x * SIDE) + (y + 1));
                } else {
                    indices.put( (x * SIDE) + y);
                    indices.put( ((x + 1) * SIDE) + y);
                    indices.put( (x * SIDE) + (y + 1));

                    indices.put( ((x + 1) * SIDE) + y);
                    indices.put( ((x + 1) * SIDE) + (y + 1));
                    indices.put( (x * SIDE) + (y + 1));
                }
            }
        }

        ElementsBuffer vertexBuffer = canvas.getBuffersManager().createElementsBuffer();
        ElementsBuffer colorBuffer = canvas.getBuffersManager().createElementsBuffer();
        IndicesBuffer  indicesBuffer = canvas.getBuffersManager().createIndicesBuffer();

        vertices.rewind();
        colors.rewind();
        indices.rewind();

        vertexBuffer.setData(vertices, ELEMENTS_SIZE);
        colorBuffer.setData(colors, ELEMENTS_SIZE);
        indicesBuffer.setData(indices);

        DefaultGeometry geometry = new DefaultGeometry();
        geometry.setFillDrawingMode(Geometry.FillDrawingMode.TRIANGLES);
        geometry.setVertices(vertexBuffer);
        geometry.setColors(colorBuffer);
        geometry.setIndices(indicesBuffer);

        return geometry;
    }
}



