//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function varargout = CL_dat_cjd2cal(cjd)
// Modified (1950.0) Julian day to calendar date
//
// Calling Sequence
// [year,month,day [hour,minute,second] = CL_dat_cjd2cal(cjd)
// [cal] = CL_dat_jd2cal(jd) // cal=[year;month;day;hour;minute;second]
//
// Description
// <itemizedlist><listitem>
// <p>Converts a 1950.0 Julian day (julian day from 1950.0) into a calendar date.</p>
// <p>The output arguments are either: </p>
// <p> - year, month, day, hour, minute, second: 3 or 6 output arguments are expected. 3 is only accepted if the julian day is a whole number. </p>
// <p> - cal: colum vector containing year, month, day, hour, minute, second. </p>
// <p></p></listitem>
// <listitem>
// <p>See <link linkend="Dates and Time">Dates and Time</link> for more details.</p>
// </listitem>
// </itemizedlist>
//
// Parameters
// cjd: Modified Julian day from 1950.0 (1xN)
// year: Year (1xN)
// month: Month (1xN)
// day: Day (1xN)
// hour: Hours (1xN)
// minute: Minutes (1xN)
// second: Seconds (1xN)
//
// Authors
// CNES - DCT/SB
//
// See also
// CL_dat_cal2cjd
// CL_dat_jd2cal
//
// Examples
// cjd1 = 18262;
// [year,month,day,hour,minute,second] = CL_dat_cjd2cal(cjd1)
// d = CL_dat_cjd2cal(cjd1)
// cjd2  = 21965.50260416667;
// [year,month,day,hour,minute,second] = CL_dat_cjd2cal(cjd2)
// d = CL_dat_cjd2cal(cjd2)
// [year,month,day,hour,minute,second] = CL_dat_cjd2cal([cjd1, cjd2])
// d = CL_dat_cjd2cal([cjd1, cjd2])

// Declarations:


// Code:
[lhs, rhs] = argn();

if (lhs <> 1 & lhs <> 3 & lhs <> 6)
  CL__error("Wrong number of output arguments: 1, 3 or 6 expected.");
end
if (lhs == 3)
  I = find (cjd <> round(cjd));
  if ~isempty(I); CL__error("Wrong number of output arguments: 3 only for integer dates"); end
end 

if (rhs <> 1)
  CL__error("Wrong number of input arguments");
end

// algorithm

jd = floor(cjd) + 2433282.5; // Julian day (0h): same year,month,day

[year,month,day,hour,minute,second] = CL_dat_jd2cal(jd+0.1); // hour,minute,second: ignored! 

aux = (cjd - floor(cjd)).*24;
hour = floor(aux);
aux = (aux-hour).*60;
minute = floor(aux);
second = (aux-minute).*60;

// output

if (lhs == 1)
  varargout(1) = [year;month;day;hour;minute;second];

else
  varargout(1) = year;
  varargout(2) = month;
  varargout(3) = day;
  if (lhs >= 4) 
     varargout(4) = hour;
     varargout(5) = minute;
     varargout(6) = second;
  end
end

endfunction
