// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#ifndef __frscale_h__
#define __frscale_h__

#include <stdlib.h>

class FrScale {
 public:
  enum ColorScaleType {LINEARSCALE, LOGSCALE, SQUAREDSCALE, 
		       SQRTSCALE, IISSCALE, HISTEQUSCALE};
  enum ClipScope {GLOBAL, LOCAL};
  enum ClipMode {MINMAX, ZSCALE, ZMAX, AUTOCUT, USERCLIP};
  enum MinMaxMode {AUTOSCAN, SCAN, SAMPLE, DATAMIN, IRAFMIN};
  enum ScanMode {NODATASEC, DATASEC, UNODATASEC, UDATASEC, RESET};

 private:
  ColorScaleType colorScaleType_; // color scale type
  ClipScope clipScope_;       // color scale clip scope
  ClipMode clipMode_;         // color scale clip mode
  MinMaxMode mmMode_;         // method to use for determining minmax
  double low_;                // low cut level for all data
  double high_;               // high cut level for all data
  double min_;                // min for all data
  double max_;                // max for all data
  double uLow_;               // low cut via user for all data
  double uHigh_;              // high cut via user for all data
  float zContrast_;           // zscale slope transfer function
  int zSampleSize_;           // zscale optimal sample size
  int zSampleLine_;           // zscale number of lines to sample
  int mmIncr_;                // minmax sampling incr
  float autoCutPer_;          // autoCut percentage
  ScanMode scanMode_;         // use keyword DATASEC			  
  double* histogramX_;        // scale histogram
  double* histogramY_;        // scale histogram
  double* histequ_;           // image histogram equalization xfer function
  int histogramXSize_;
  int histogramYSize_;
  int histequSize_;
  int preserve_;

 public:
  FrScale();
  ~FrScale();
  FrScale(const FrScale&);
  FrScale& operator=(const FrScale&);

  ColorScaleType colorScaleType() {return colorScaleType_;}
  ClipScope clipScope() {return clipScope_;}
  ClipMode clipMode() {return clipMode_;}
  MinMaxMode mmMode() {return mmMode_;}
  double low() {return low_;}
  double high() {return high_;}
  double min() {return min_;}
  double max() {return max_;}
  double uLow() {return uLow_;}
  double uHigh() {return uHigh_;}
  float zContrast() {return zContrast_;}
  int zSampleSize() {return zSampleSize_;}
  int zSampleLine() {return zSampleLine_;}
  int mmIncr() {return mmIncr_;}
  float autoCutPer() {return autoCutPer_;}
  ScanMode scanMode() {return scanMode_;}
  double* histogramX() {return histogramX_;}
  double* histogramY() {return histogramY_;}
  double* histequ() {return histequ_;}
  int preserve() {return preserve_;}
  
  void setColorScaleType(ColorScaleType v) {colorScaleType_ = v;}
  void setClipScope(ClipScope v) {clipScope_ = v;}
  void setClipMode(ClipMode v) {clipMode_ = v;}
  void setMMMode(MinMaxMode v) {mmMode_ = v;}
  void setLow(double v) {low_ = v;}
  void setHigh(double v) {high_ = v;}
  void setMin(double v) {min_ = v;}
  void setMax(double v) {max_ = v;}
  void setULow(double v) {uLow_ = v;}
  void setUHigh(double v) {uHigh_ = v;}
  void setZContrast(float v) {zContrast_ = v;}
  void setZSampleSize(int v) {zSampleSize_ = v;}
  void setZSampleLine(int v) {zSampleLine_ = v;}
  void setMMIncr(int v) {mmIncr_ = v;}
  void setAutoCutPer(float v) {autoCutPer_ = v;}
  void setScanMode(ScanMode v) {scanMode_ = v;}
  void setHistogramX(double* v) {histogramX_ =v;}
  void setHistogramY(double* v) {histogramY_ = v;}
  void setHistequ(double* v) {histequ_ = v;}
  void setPreserve(int r) {preserve_ = r;}

  double* initHistogramX(int);
  double* initHistogramY(int);
  double* initHistequ(int);

  void deleteHistogramX();
  void deleteHistogramY();
  void deleteHistequ();
};

#endif
