// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#ifndef __fitsimage_h__
#define __fitsimage_h__

#include "framebase.h"
#include "vector.h"
#include "fitsdata.h"
#include "coord.h"

class FitsFile;

#include "wcs.h"

class FitsImage {
protected:
  FrameBase* parent;            // pointer to parent

  char* objectName;         // object name
  char* rootBaseFileName;   // base root filename
  char* fullBaseFileName;   // base full filename
  char* rootFileName;       // root filename
  char* fullFileName;       // full filename
  char* iisFileName;        // alt file name for iis

  FitsFile* fits_;          // fits file native
  FitsFile* hist_;          // fits file bin
  FitsFile* smooth_;        // fits file image analysis

  FitsData* fitsdata_;      // raw fits image data
  FitsData* smoothdata_;    // raw fits image data

  FitsFile* image_;         // final image
  FitsData* data_;          // final raw data

  int width_;
  int height_;
  int depth_;
  int bitpix_;

  double rotation_;          // local rotation
  Vector zoom_;              // local scaling;
  Orientation orientation_; // local orientation
  Vector origin_;            // local translation in image space

  Vector histCursor;        // start point to bin about
  Vector actualHistCursor;  // actual bin center

  FitsHist::Function binFunction_; // binning operation
  Vector binFactor_;        // binning factor
  int binBufferSize_;       // size of bin buffer
  int binDepth_;            // binning depth

  char buf[32];             // tmp storage for returning strings

  // Mosaic items
  int keyLTMV;
  int keyATMV;
  int keyDTMV;
  int keyDATASEC;

  Vector ccdsum;            // CCDSUM keyword

  BBox detsize;             // DETSIZE keyword
  BBox detsec;              // DETSEC keyword
  BBox datasec;             // DATASEC keyword

  Matrix amplifierToCanvas;
  Matrix amplifierToPhysical;
  Matrix amplifierToImage;
  Matrix amplifierToRef;
  Matrix amplifierToPanner;
  Matrix amplifierToUser;
  Matrix canvasToAmplifier;
  Matrix canvasToPhysical;
  Matrix canvasToData;
  Matrix canvasToDetector;
  Matrix canvasToImage;
  Matrix canvasToRef;
  Matrix physicalToAmplifier;
  Matrix physicalToCanvas;
  Matrix physicalToData;
  Matrix physicalToDetector;
  Matrix physicalToImage;
  Matrix physicalToRef;
  Matrix physicalToUser;
  Matrix physicalToPanner;
  Matrix dataToCanvas;
  Matrix dataToPhysical;
  Matrix dataToImage;
  Matrix dataToMagnifier;
  Matrix dataToRef;
  Matrix dataToPanner;
  Matrix dataToUser;
  Matrix dataToWidget;
  Matrix detectorToCanvas;
  Matrix detectorToPhysical;
  Matrix detectorToImage;
  Matrix detectorToPanner;
  Matrix detectorToRef;
  Matrix detectorToUser;
  Matrix imageToCanvas;
  Matrix imageToPhysical;
  Matrix imageToAmplifier;
  Matrix imageToDetector;
  Matrix imageToData;
  Matrix imageToRef;
  Matrix imageToPanner;
  Matrix imageToUser;
  Matrix imageToWidget;
  Matrix magnifierToData;
  Matrix pannerToAmplifier;
  Matrix pannerToPhysical;
  Matrix pannerToData;
  Matrix pannerToDetector;
  Matrix pannerToImage;
  Matrix refToPhysical;
  Matrix refToData;
  Matrix refToImage;
  Matrix refToAmplifier;
  Matrix refToDetector;
  Matrix refToCanvas;
  Matrix userToAmplifier;
  Matrix userToPhysical;
  Matrix userToData;
  Matrix userToDetector;
  Matrix userToImage;
  Matrix widgetToData;
  Matrix widgetToImage;
  Matrix dataToPS;
  Matrix PSToData;

  // this is for optimization of frame
  double* mWidgetToData;
  double* mPannerToData;
  double* mMagnifierToData;
  double* mPSToData;

  int iparams[5];           // image clip/render bbox
  int dparams[5];           // DATASEC clip/render bbox
  int imparams[5];          // user specified minmax scan bbox
  int irparams[5];          // user specified valid data bbox
  int dmparams[5];          // DATASEC user specified minmax scan bbox
  int drparams[5];          // DATASEC user specified valid data bbox

  int iisMode;              // flag to indicate if iis
  Vector iisz;              // z1 and z2
  int iiszt;                // 0-Uniary, 1-Linear, 2-Log

  FitsImage* nextMosaic_;   // next in list
  FitsImage* nextSlice_;    // next in list

  WorldCoor** wcs;          // wcs list
  Vector* wcstan;           // list of tangent points

  void reset();

  void process(const char*);
  Matrix nextHist(const Vector&);
  Vector getHistCenter();

  char* root(const char*);
  char* strip(const char*);
  void parseSection(char*, Vector*, Vector*);

  int smooth();

public:
  FitsImage(FrameBase*);
  virtual ~FitsImage();

  void load();
  void analysis();

  FitsImage* nextMosaic() {return nextMosaic_;}
  FitsImage* nextSlice() {return nextSlice_;}
  void setNextMosaic(FitsImage* n) {nextMosaic_ = n;}
  void setNextSlice(FitsImage* n) {nextSlice_ = n;}

  FitsFile* fitsFile() {return fits_;}
  FitsFile* imageFile() {return image_;}

  int isValid() {return image_ ? 1 : 0;}
  int isHist() {return hist_ ? 1 : 0;}
  int isImage() {return fits_ ? fits_->isImage() : 0;}
  int isTable() {return fits_ ? fits_->isTable() : 0;}

  void close() {if (fits_) fits_->done();}

  Matrix updateHistCenter();
  Matrix updateHistCursor();
  Matrix updateHist(const Vector&);
  void calcHistSize();
  Vector getHistDim();
  Vector getHistColMinMax(const char*);
  Vector getHistCursor() {return actualHistCursor;}
  const char* getHistFilter();
  const char* getHistX();
  const char* getHistY();
  const char* getHistZ();
  char* getHistList();
  Vector getHistZlim();

  FitsHist::Function binFunction() {return binFunction_;}
  Vector binFactor() {return binFactor_;}
  int binBufferSize() {return binBufferSize_;}
  int binDepth() {return binDepth_;}

  void setBinCursor(const Vector&);
  void setBinFunction(FitsHist::Function f) {binFunction_ = f;}
  void setBinFactor(const Vector&);
  void setBinToFactor(const Vector&);
  void setBinBufferSize(int s) {binBufferSize_ = s;}
  void setBinDepth(int d) {binDepth_ = d < 1 ? 1 : d;}
  void setBinX(const char*);
  void setBinY(const char*);
  void setBinZ(const char*);
  void setBinZlim(const Vector&);
  void setBinFilter(const char*);
  void setBinSliceFilter(const char*);

  const char* getObjectName() {return objectName;}
  const char* getRootBaseFileName() {return rootBaseFileName;}
  const char* getFullBaseFileName() {return fullBaseFileName;}
  const char* getRootFileName() {return rootFileName;}
  const char* getFullFileName() {return fullFileName;}
  const char* iisGetFileName() {return iisFileName;}
  void iisSetFileName(const char*);
  void setFileName(const char*);
  void updateFileName();

  int width() {return width_;}
  int height() {return height_;}
  int depth() {return depth_;}
  int bitpix() {return bitpix_;}
  int ext() {return fits_->ext();}
  BBox getIRAFbb();

  int* getDataParams(FrScale::ScanMode);   // return bbox in IMAGE
  int* getMinMaxParams(FrScale::ScanMode);
  void setDataParams(int,int,int,int);
  void setMinMaxParams(int,int,int,int);

  int hasLTMV() {return keyLTMV;}
  int hasATMV() {return keyATMV;}
  int hasDTMV() {return keyDTMV;}
  int isIIS() {return iisMode;}

  char* pix2wcs(Vector, CoordSystem, SkyFrame, SkyFormat, char*, int);
  Vector pix2wcs(Vector, CoordSystem);
  Vector pix2wcs(Vector, CoordSystem, SkyFrame);
  Vector wcs2pix(Vector, CoordSystem);
  Vector wcs2pix(Vector, CoordSystem, SkyFrame);
  Vector pix2wcsDist(Vector, CoordSystem);
  Vector wcs2pixDist(Vector, CoordSystem);
  double wcsdist(Vector, Vector, CoordSystem);


  void initWCS(FitsHead*);
  void listWCS(ostream&, CoordSystem);
  void resetWCS();
  void replaceWCS(FitsHead*);
  void appendWCS(FitsHead*);
  void initWCS0(const Vector&);
  void resetWCS0();

  int processKeywords();
  int processKeywordsIRAF();
  int processKeywordsWCS();

  double rotation() {return rotation_;}
  Orientation orientation() {return orientation_;}

  WorldCoor* getWCS(CoordSystem sys) 
    {return (wcs && wcs[sys-WCS]) ? wcs[sys-WCS] : NULL;}
  const char* getWCSName(CoordSystem sys) 
    {return (wcs && wcs[sys-WCS]) ? wcs[sys-WCS]->wcsname : NULL;}
  Orientation getWCSOrientation(CoordSystem, SkyFrame);
  double getWCSRotation(CoordSystem, SkyFrame);
  Vector getWCScrval(CoordSystem);
  Vector getWCScrpix(CoordSystem);
  Vector getWCScdelt(CoordSystem);
  Vector getWCStan(CoordSystem);

  int hasWCS(CoordSystem);
  int hasWCSEqu(CoordSystem);
  int hasWCSLinear(CoordSystem);

  void updateMatrices(FrameBase::MosaicType,
		      Matrix&, Matrix&, Matrix&, Matrix&, Matrix&);
  void updateMagnifierMatrices(Matrix&);

  void updatePS(Matrix);

  double* getmWidgetToData() {return mWidgetToData;}
  double* getmPannerToData() {return mPannerToData;}
  double* getmMagnifierToData() {return mMagnifierToData;}
  double* getmPSToData() {return mPSToData;}

  Matrix& getAmplifierToPhysical() {return amplifierToPhysical;}
  Matrix& getAmplifierToImage() {return amplifierToImage;}
  Matrix& getAmplifierToRef() {return amplifierToRef;}
  Matrix& getAmplifierToUser() {return amplifierToUser;}
  Matrix& getCanvasToAmplifier() {return canvasToAmplifier;}
  Matrix& getCanvasToPhysical() {return canvasToPhysical;}
  Matrix& getCanvasToData() {return canvasToData;}
  Matrix& getCanvasToDetector() {return canvasToDetector;}
  Matrix& getCanvasToImage() {return canvasToImage;}
  Matrix& getCanvasToRef() {return canvasToRef;}
  Matrix& getPhysicalToAmplifier() {return physicalToAmplifier;}
  Matrix& getPhysicalToDetector() {return physicalToDetector;}
  Matrix& getPhysicalToImage() {return physicalToImage;}
  Matrix& getPhysicalToRef() {return physicalToRef;}
  Matrix& getPhysicalToUser() {return physicalToUser;}
  Matrix& getDataToCanvas() {return dataToCanvas;}
  Matrix& getDataToPhysical() {return dataToPhysical;}
  Matrix& getDataToImage() {return dataToImage;}
  Matrix& getDataToPanner() {return dataToPanner;}
  Matrix& getDataToRef() {return dataToRef;}
  Matrix& getDataToUser() {return dataToUser;}
  Matrix& getDataToWidget() {return dataToWidget;}
  Matrix& getDetectorToPhysical() {return detectorToPhysical;}
  Matrix& getDetectorToImage() {return detectorToImage;}
  Matrix& getDetectorToRef() {return detectorToRef;}
  Matrix& getDetectorToUser() {return detectorToUser;}
  Matrix& getImageToCanvas() {return imageToCanvas;}
  Matrix& getImageToPhysical() {return imageToPhysical;}
  Matrix& getImageToAmplifier() {return imageToAmplifier;}
  Matrix& getImageToDetector() {return imageToDetector;}
  Matrix& getImageToData() {return imageToData;}
  Matrix& getImageToRef() {return imageToRef;}
  Matrix& getImageToUser() {return imageToUser;}
  Matrix& getImageToWidget() {return imageToWidget;}
  Matrix& getMagnifierToData() {return magnifierToData;}
  Matrix& getRefToAmplifier() {return refToAmplifier;}
  Matrix& getRefToPhysical() {return refToPhysical;}
  Matrix& getRefToData() {return refToData;}
  Matrix& getRefToDetector() {return refToDetector;}
  Matrix& getRefToImage() {return refToImage;}
  Matrix& getRefToCanvas() {return refToCanvas;}
  Matrix& getPannerToAmplifier() {return pannerToAmplifier;}
  Matrix& getPannerToPhysical() {return pannerToPhysical;}
  Matrix& getPannerToData() {return pannerToData;}
  Matrix& getPannerToDetector() {return pannerToDetector;}
  Matrix& getPannerToImage() {return pannerToImage;}
  Matrix& getPhysicalToData() {return physicalToData;}
  Matrix& getUserToAmplifier() {return userToAmplifier;}
  Matrix& getUserToPhysical() {return userToPhysical;}
  Matrix& getUserToData() {return userToData;}
  Matrix& getUserToDetector() {return userToDetector;}
  Matrix& getUserToImage() {return userToImage;}
  Matrix& getWidgetToData() {return widgetToData;}
  Matrix& getWidgetToImage() {return widgetToImage;}

  Vector mapFromRef(const Vector&, CoordSystem, SkyFrame =FK5);
  void mapFromRef(const Vector&, CoordSystem, SkyFrame, SkyFormat, char*,int);
  Vector mapToRef(const Vector&, CoordSystem, SkyFrame =FK5);
  double mapLenFromRef(double, CoordSystem, SkyFormat =DEGREES);
  Vector mapLenFromRef(const Vector&, CoordSystem, SkyFormat =DEGREES);
  double mapLenToRef(double, CoordSystem, SkyFormat =DEGREES);
  Vector mapLenToRef(const Vector&, CoordSystem, SkyFormat =DEGREES);
  double mapDistFromRef(const Vector&, const Vector&, CoordSystem,
		       SkyFormat =DEGREES);
  Vector mapLen(const Vector&, const Matrix&);

// we assume that 'data' is valid

  const char* getValue(const Vector& v);
  float getValueFloat(const Vector& v) {return data_->getValueFloat(v);}
  float getValueFloat(int i) {return data_->getValueFloat(i);}
  double getValueDouble(const Vector& v) {return data_->getValueDouble(v);}
  double getValueDouble(int i) {return data_->getValueDouble(i);}
  void setClip(double l, double h) {data_->setClip(l,h);}

  const char* getMin() {return data_ ? data_->getMin() : NULL;}
  const char* getMax() {return data_ ? data_->getMax() : NULL;}

  double getMinDouble() {return data_ ? data_->getMinDouble() : 0;}
  double getMaxDouble() {return data_ ? data_->getMaxDouble() : 0;}

  const char* getLow() {return data_ ? data_->getLow() : NULL;}
  const char* getHigh() {return data_ ? data_->getHigh() : NULL;}
  double getLowDouble() {return data_ ? data_->getLowDouble() : 0;}
  double getHighDouble() {return data_ ? data_->getHighDouble() : 0;}

  void updateClip(FrScale* frs)
    {if (data_) data_->updateClip(frs,getMinMaxParams(frs->scanMode()));}
  void updateWCSbb(CoordSystem, int);

  int hasDATAMIN() {return data_ ? data_->hasDATAMIN() : 0;}
  int hasDATASEC() {return keyDATASEC;}
  int hasIRAFMIN() {return data_ ? data_->hasIRAFMIN() : 0;}

  void bin(double* b, int l, double mn, double mx, FrScale::ScanMode ds)
    {data_->bin(b,l,mn,mx,getDataParams(ds));}

  char* display(FitsHead*);
  char* displayHeader();
  char* displayPrimary();
  FitsHead* getHead();
  char* getCards();
  int getNCards();
  char* getKeyword(const char*);
  int findKeyword(const char*);

  int FitsImage::saveFitsImageFile(const char* fn, int compress)
    {return image_ ? image_->saveFitsImageFile(fn, compress) : 0;}

  int FitsImage::saveFitsImageChannel(Tcl_Interp* interp, const char* ch, 
				      int compress)
    {return image_ ? image_->saveFitsImageChannel(interp, ch, compress) : 0;}

  int FitsImage::saveFitsImageSocket(int s, int compress)
    {return image_ ? image_->saveFitsImageSocket(s, compress) : 0;}

  int FitsImage::saveFitsTableFile(const char* fn, int compress)
    {return fits_ ? fits_->saveFitsTableFile(fn, compress) : 0;}

  int FitsImage::saveFitsTableChannel(Tcl_Interp* interp, const char* ch, 
				      int compress)
    {return fits_ ? fits_->saveFitsTableChannel(interp, ch, compress): 0;}

  int FitsImage::saveFitsTableSocket(int s, int compress)
    {return fits_ ? fits_->saveFitsTableSocket(s, compress) : 0;}

  int FitsImage::saveArrayFile(const char* fn)
    {return image_ ? image_->saveArrayFile(fn) : 0;}

  int FitsImage::saveArrayChannel(Tcl_Interp* interp, const char* ch)
    {return image_ ? image_->saveArrayChannel(interp, ch) : 0;}

  int FitsImage::saveArraySocket(int s)
    {return image_ ? image_->saveArraySocket(s) : 0;}
};

class FitsImageMaster : public FitsImage {
public:
  FitsImageMaster(FrameBase* p) : FitsImage(p) {}
  virtual ~FitsImageMaster();
};

class FitsImageSlave : public FitsImage {
public:
  FitsImageSlave(FrameBase*, const char*, FitsFile*);
  virtual ~FitsImageSlave() {}
};

// Fits

class FitsImageFitsAlloc : public FitsImageMaster {
public:
  FitsImageFitsAlloc(FrameBase*, const char*, FitsFile::FlushMode);
};

class FitsImageFitsAllocGZ : public FitsImageMaster {
public:
  FitsImageFitsAllocGZ(FrameBase*, const char*, FitsFile::FlushMode);
};

class FitsImageFitsChannel : public FitsImageMaster {
public:
  FitsImageFitsChannel(FrameBase*, Tcl_Interp*, const char*,const char*,
		       FitsFile::FlushMode);
};

class FitsImageFitsMMap : public FitsImageMaster {
public:
  FitsImageFitsMMap(FrameBase*, const char*);
};

class FitsImageFitsSMMap : public FitsImageMaster {
public:
  FitsImageFitsSMMap(FrameBase*, const char*, const char*);
};

class FitsImageFitsMMapIncr : public FitsImageMaster {
public:
  FitsImageFitsMMapIncr(FrameBase*, const char*);
};

class FitsImageFitsShare : public FitsImageMaster {
public:
  FitsImageFitsShare(FrameBase*, FrameBase::ShmType, int, const char*);
};

class FitsImageFitsSShare : public FitsImageMaster {
public:
  FitsImageFitsSShare(FrameBase*, FrameBase::ShmType, int, int, const char*);
};

class FitsImageFitsSocket : public FitsImageMaster {
public:
  FitsImageFitsSocket(FrameBase*, int, const char*, FitsFile::FlushMode);
};

class FitsImageFitsSocketGZ : public FitsImageMaster {
public:
  FitsImageFitsSocketGZ(FrameBase*, int, const char*, FitsFile::FlushMode);
};

class FitsImageFitsVar : public FitsImageMaster {
public:
  FitsImageFitsVar(FrameBase*, Tcl_Interp*, const char*, const char*);
};

// Fits Next

class FitsImageFitsNextAlloc : public FitsImageMaster {
public:
  FitsImageFitsNextAlloc(FrameBase*, const char*, FitsFile*);
};

class FitsImageFitsNextAllocGZ : public FitsImageMaster {
public:
  FitsImageFitsNextAllocGZ(FrameBase*, const char*, FitsFile*);
};

class FitsImageFitsNextChannel : public FitsImageMaster {
public:
  FitsImageFitsNextChannel(FrameBase*, const char*, FitsFile*);
};

class FitsImageFitsNextMMap : public FitsImageMaster {
public:
  FitsImageFitsNextMMap(FrameBase*, const char*, FitsFile*);
};

class FitsImageFitsNextSMMap : public FitsImageMaster {
public:
  FitsImageFitsNextSMMap(FrameBase*, const char*, FitsFile*);
};

class FitsImageFitsNextMMapIncr : public FitsImageMaster {
public:
  FitsImageFitsNextMMapIncr(FrameBase*, const char*, FitsFile*);
};

class FitsImageFitsNextShare : public FitsImageMaster {
public:
  FitsImageFitsNextShare(FrameBase*, const char*, FitsFile*);
};

class FitsImageFitsNextSShare : public FitsImageMaster {
public:
  FitsImageFitsNextSShare(FrameBase*, const char*, FitsFile*);
};

class FitsImageFitsNextSocket : public FitsImageMaster {
public:
  FitsImageFitsNextSocket(FrameBase*, const char*, FitsFile*);
};

class FitsImageFitsNextSocketGZ : public FitsImageMaster {
public:
  FitsImageFitsNextSocketGZ(FrameBase*, const char*, FitsFile*);
};

class FitsImageFitsNextVar : public FitsImageMaster {
public:
  FitsImageFitsNextVar(FrameBase*, const char*, FitsFile*);
};

class FitsImageFitsNextHist : public FitsImageMaster {
public:
  FitsImageFitsNextHist(FrameBase*, FitsImage*, FitsFile*);
};

// Array

class FitsImageArrAlloc : public FitsImageMaster {
public:
  FitsImageArrAlloc(FrameBase*, const char*, FitsFile::FlushMode);
};

class FitsImageArrAllocGZ : public FitsImageMaster {
public:
  FitsImageArrAllocGZ(FrameBase*, const char*, FitsFile::FlushMode);
};

class FitsImageArrChannel : public FitsImageMaster {
public:
  FitsImageArrChannel(FrameBase*, Tcl_Interp*, const char*, const char*, 
		      FitsFile::FlushMode);
};

class FitsImageArrMMap : public FitsImageMaster {
public:
  FitsImageArrMMap(FrameBase*, const char*);
};

class FitsImageArrMMapIncr : public FitsImageMaster {
public:
  FitsImageArrMMapIncr(FrameBase*, const char*);
};

class FitsImageArrShare : public FitsImageMaster {
public:
  FitsImageArrShare(FrameBase*, FrameBase::ShmType, int, const char*);
};

class FitsImageArrSocket : public FitsImageMaster {
public:
  FitsImageArrSocket(FrameBase*, int, const char*, FitsFile::FlushMode);
};

class FitsImageArrSocketGZ : public FitsImageMaster {
public:
  FitsImageArrSocketGZ(FrameBase*, int, const char*, FitsFile::FlushMode);
};

class FitsImageArrVar : public FitsImageMaster {
public:
  FitsImageArrVar(FrameBase*, Tcl_Interp*, const char*, const char*);
};

// Mosaic

class FitsImageMosaicAlloc : public FitsImageMaster {
public:
  FitsImageMosaicAlloc(FrameBase*, const char*, FitsFile::FlushMode);
};

class FitsImageMosaicAllocGZ : public FitsImageMaster {
public:
  FitsImageMosaicAllocGZ(FrameBase*, const char*, FitsFile::FlushMode);
};

class FitsImageMosaicChannel : public FitsImageMaster {
public:
  FitsImageMosaicChannel(FrameBase*, Tcl_Interp*, const char*, const char*,
			 FitsFile::FlushMode);
};

class FitsImageMosaicMMap : public FitsImageMaster {
public:
  FitsImageMosaicMMap(FrameBase*, const char*);
};

class FitsImageMosaicMMapIncr : public FitsImageMaster {
public:
  FitsImageMosaicMMapIncr(FrameBase*, const char*);
};

class FitsImageMosaicShare : public FitsImageMaster {
public:
  FitsImageMosaicShare(FrameBase*, FrameBase::ShmType, int, const char*);
};

class FitsImageMosaicSocket : public FitsImageMaster {
public:
  FitsImageMosaicSocket(FrameBase*, int, const char*, FitsFile::FlushMode);
};

class FitsImageMosaicSocketGZ : public FitsImageMaster {
public:
  FitsImageMosaicSocketGZ(FrameBase*, int, const char*, FitsFile::FlushMode);
};

class FitsImageMosaicVar : public FitsImageMaster {
public:
  FitsImageMosaicVar(FrameBase*, Tcl_Interp*, const char*, const char*);
};

// Mosaic Next

class FitsImageMosaicNextAlloc : public FitsImageMaster {
public:
  FitsImageMosaicNextAlloc(FrameBase*, const char*, FitsFile*, 
			   FitsFile::FlushMode);
};

class FitsImageMosaicNextAllocGZ : public FitsImageMaster {
public:
  FitsImageMosaicNextAllocGZ(FrameBase*, const char*, FitsFile*, 
			     FitsFile::FlushMode);
};

class FitsImageMosaicNextChannel : public FitsImageMaster {
public:
  FitsImageMosaicNextChannel(FrameBase*, const char*, FitsFile*, 
			     FitsFile::FlushMode);
};

class FitsImageMosaicNextMMap : public FitsImageMaster {
public:
  FitsImageMosaicNextMMap(FrameBase*, const char*, FitsFile*);
};

class FitsImageMosaicNextMMapIncr : public FitsImageMaster {
public:
  FitsImageMosaicNextMMapIncr(FrameBase*, const char*, FitsFile*);
};

class FitsImageMosaicNextShare : public FitsImageMaster {
public:
  FitsImageMosaicNextShare(FrameBase*, const char*, FitsFile*);
};

class FitsImageMosaicNextSocket : public FitsImageMaster {
public:
  FitsImageMosaicNextSocket(FrameBase*, const char*, FitsFile*, 
			    FitsFile::FlushMode);
};

class FitsImageMosaicNextSocketGZ : public FitsImageMaster {
public:
  FitsImageMosaicNextSocketGZ(FrameBase*, const char*, FitsFile*, 
			      FitsFile::FlushMode);
};

class FitsImageMosaicNextVar : public FitsImageMaster {
public:
  FitsImageMosaicNextVar(FrameBase*, const char*, FitsFile*);
};

// IIS

class FitsImageIIS : public FitsImageMaster {
public:
  FitsImageIIS(FrameBase*, int, int);
  void iisErase();
  const char* iisGet(int x, int y);
  void iisSet(int x, int y, const char* d, int l);
  void iisWCS(const Matrix&, const Vector&, int);
};

#endif
