// Copyright (C) 1999-2005
// Smithsonian Astrophysical Observatory, Cambridge, MA, USA
// For conditions of distribution and use, see copyright notice in "copyright"

#include "compass.h"
#include "framebase.h"
#include "fitsimage.h"
#include "util.h"

Compass::Compass(const Compass& a) : Marker(a)
{
  p1 = a.p1;
  p2 = a.p2;
  pp1 = a.pp1;
  pp2 = a.pp2;
  radius = a.radius;
  coordSystem = a.coordSystem;
  skyFrame = a.skyFrame;
  df =  Tk_GetFont(parent->interp, parent->tkwin, Tk_NameOfFont(a.df));
  northText = dupstr(a.northText);
  eastText = dupstr(a.eastText);
  northArrow = a.northArrow;
  eastArrow = a.eastArrow;
}

Compass::Compass(FrameBase* p, const Vector& ctr,
		 double r,
		 const char* n, const char* e, int na, int ea,
		 CoordSystem sys, SkyFrame sky, 
		 const char* clr, int w, const char* f, 
		 const char* t, unsigned short prop, const char* c,
		 const List<Tag>& tag, const List<CallBack>& cb) 
  : Marker(p, ctr, 0, clr, w, f, t, prop, c, tag, cb)
{
  df = Tk_GetFont(parent->interp, parent->tkwin, "helvetica 9 normal");
  coordSystem = sys;
  skyFrame = sky;
  radius = r;

  northText = dupstr(n);
  eastText = dupstr(e);
  northArrow = na;
  eastArrow = ea;

  strcpy(type,"compass");
  handle = new Vector[3];
  numHandle = 3;

  updateBBox();
}

Compass::~Compass()
{
  if (df)
    Tk_FreeFont(df);

  if (northText)
    delete [] northText;

  if (eastText)
    delete [] eastText;
}

void Compass::updateBBox()
{
  FitsImage* ptr = parent->findFits(center);

  // calc p1, p2
  Vector orval = ptr->mapFromRef(center, coordSystem, skyFrame);
  double delta = ptr->mapLenFromRef(1, coordSystem);
  Vector npix = ptr->mapToRef(Vector(orval[0],orval[1]+delta), 
			      coordSystem, skyFrame);
  Vector epix = ptr->mapToRef(Vector(orval[0]+delta,orval[1]), 
			      coordSystem, skyFrame);
  Vector north = (npix - center).normalize();
  Vector east =  (epix - center).normalize();

  // calc text points pp1,pp2
  Tk_FontMetrics metrics;
  Tk_GetFontMetrics(font, &metrics);
  //  double rr = (metrics.linespace*.25)/zz[0]+r;

  Matrix r  = Scale(radius)*Scale(parent->zoom()).invert();
  Matrix rr = Scale(metrics.linespace*.25)*Scale(parent->zoom()).invert();
  p1 = center + north*r;
  p2 = center + east*r;
  pp1 = p1 + north*rr;
  pp2 = p2 + east*rr;

  // generate handles
  handle[0] = center * parent->refToCanvas;
  handle[1] = p1 * parent->refToCanvas;
  handle[2] = p2 * parent->refToCanvas;

  // bound marker
  bbox = BBox(center * parent->refToCanvas);
  bbox.bound(p1 * parent->refToCanvas);
  bbox.bound(p2 * parent->refToCanvas);

  // make room for handles
  bbox.expand(3);

  // make room for text
  Vector a = center * parent->refToCanvas;
  Vector b = p1 * parent->refToCanvas;
  Vector c = p2 * parent->refToCanvas;
  Vector bb = pp1 * parent->refToCanvas;
  Vector cc = pp2 * parent->refToCanvas;

  if (northText) {
    float r1 = Tk_TextWidth(font, northText, strlen(northText))/2.;
    float r2 = metrics.linespace/2.;
    double angle = (b-a).angle();

    Vector bbb = bb * Translate(r1*cos(angle),r2*sin(angle));

    bbox.bound(bbb * Translate(-r1,(metrics.ascent-metrics.descent)/2.));
    bbox.bound(bbb * Translate(-r1,-metrics.ascent/2.-3*metrics.descent/2.));
    bbox.bound(bbb * Translate(r1,(metrics.ascent-metrics.descent)/2.));
    bbox.bound(bbb * Translate(r1,-metrics.ascent/2.-3*metrics.descent/2.));
  }

  if (eastText) {
    float r1 = Tk_TextWidth(font, eastText, strlen(eastText))/2.;
    float r2 = metrics.linespace/2.;
    double angle = (c-a).angle();

    Vector ccc = cc * Translate(r1*cos(angle),r2*sin(angle));

    bbox.bound(ccc * Translate(-r1,(metrics.ascent-metrics.descent)/2.));
    bbox.bound(ccc * Translate(-r1,-metrics.ascent/2.-3*metrics.descent/2.));
    bbox.bound(ccc * Translate(r1,(metrics.ascent-metrics.descent)/2.));
    bbox.bound(ccc * Translate(r1,-metrics.ascent/2.-3*metrics.descent/2.));
  }

  // calculate overall bbox
  calcAllBBox();
}

void Compass::edit(const Vector& v, int h)
{
  radius = (v * Translate(-center) * Scale(parent->zoom())).length();

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void Compass::ps(int mode)
{
  Marker::ps(mode);

  Vector a = center * parent->refToCanvas;
  Vector b = p1 * parent->refToCanvas;
  Vector c = p2 * parent->refToCanvas;
  Vector bb = pp1 * parent->refToCanvas;
  Vector cc = pp2 * parent->refToCanvas;

  ostringstream str;
  str << "newpath " 
      << a.TkCanvasPs(parent->canvas) << "moveto"
      << b.TkCanvasPs(parent->canvas) << "lineto"
      << " stroke" << endl;

  str << "newpath " 
      << a.TkCanvasPs(parent->canvas) << "moveto"
      << c.TkCanvasPs(parent->canvas) << "lineto"
      << " stroke" << endl;
  str << ends;
  Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);

  if (northArrow)
    psArrow(b, (a-b).normalize());
  if (eastArrow)
    psArrow(c, (a-c).normalize());

  if (font) {
    Tcl_DString psFont;
    Tcl_DStringInit(&psFont);
    int psSize = Tk_PostscriptFontName(font, &psFont);

    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);

    if (northText) {
      float r1 = Tk_TextWidth(font, northText, strlen(northText))/2.;
      float r2 = metrics.linespace/2.;
      double angle = (b-a).angle();
      Vector bbb = bb * 
	Translate(r1*cos(angle),r2*sin(angle)) * 
	Translate(-r1,(metrics.ascent-metrics.descent)/2.);

      ostringstream str;
      str << '/' << Tcl_DStringValue(&psFont) 
	  << " findfont " << psSize << " scalefont setfont" << endl;
      
      str << bbb.TkCanvasPs(parent->canvas) << " moveto" << endl
	  << '(' << psQuote(northText) << ')' << endl
	  << " show" << endl << ends;
      Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
    }

    if (eastText) {
      float r1 = Tk_TextWidth(font, eastText, strlen(eastText))/2.;
      float r2 = metrics.linespace/2.;
      double angle = (c-a).angle();
      Vector ccc = cc *
	Translate(r1*cos(angle),r2*sin(angle)) * 
	Translate(-r1,(metrics.ascent-metrics.descent)/2.);

      ostringstream str;
      str << '/' << Tcl_DStringValue(&psFont) 
	  << " findfont " << psSize << " scalefont setfont" << endl;
      
      str << ccc.TkCanvasPs(parent->canvas) << " moveto" << endl
	  << '(' << psQuote(eastText) << ')' << endl
	  << " show" << endl << ends;
      Tcl_AppendResult(parent->interp, str.str().c_str(), NULL);
    }
    Tcl_DStringFree(&psFont);
  }
}

int Compass::isIn(const Vector& vv)
{
  /*
    v[0]-- x value of point being tested
    v[1]-- y value of point being tested

    This algorithm is from "An Introduction to Ray Tracing", Academic Press,
    1989, edited by Andrew Glassner, pg 53
    -- a point lies in a polygon if a line is extended from the point to 
    infinite in any direction and the number of intersections with the 
    polygon is odd.
    This is valid for both concave and convex polygons.
    Points on a vertex are considered inside.
    Points on a edge are considered inside.
  */

  Vector v = vv * parent->canvasToRef; // xform to ref coords
  int crossings = 0;   // number of crossings

  Vector bb[4];
  bb[0] = center - v;
  bb[1] = p1 - v;
  bb[2] = p2 - v;
  bb[3] = bb[0];

  Vector v1;
  Vector v2 = bb[0];

  int sign = ((v2[1])>=0) ? 1 : -1; // init sign

  // for all edges

  int done = 0;

  for (int j=1; j<4; j++) {
    // look at next two vertices
    v1 = v2;
    v2 = bb[j];

    int nextSign = (v2[1]>=0) ? 1 : -1; // sign holder for p2

    if (sign != nextSign) {
      if (v1[0]>0 && v2[0]>0)
	crossings++;
      else if (v1[0]>0 || v2[0]>0) {
	if (v1[0]-(v1[1]*(v2[0]-v1[0])/(v2[1]-v1[1])) > 0)
	  crossings++;
      }
      sign = nextSign;
    }
  }

  return fmod(float(crossings),float(2)) ? 1 : 0; // if odd, point is inside
}

void Compass::setRadius(double r)
{
  radius = r;

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void Compass::setArrows(int n, int e)
{
  northArrow = n;
  eastArrow = e;

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void Compass::setLabels(const char* n, const char* e)
{
  northText = dupstr(n);
  eastText = dupstr(e);

  updateBBox();
  doCallBack(CallBack::EDITCB);
}

void Compass::setCoordSystem(CoordSystem sys, SkyFrame sky)
{
  coordSystem = sys;
  skyFrame = sky;

  updateBBox();
}

// Private

void Compass::render(Drawable drawable, const Matrix& mx, RenderMode mode)
{
  switch (mode) {
  case SRC:
    setLineNoDash();
    XSetForeground(display, gc, color);
    XSetClipRectangles(display, gc, 0, 0, parent->rectWidget, 1, Unsorted);
    break;
  case XOR:
    setLineDash();
    XSetForeground(display, gc, parent->getWhiteColor());
    XSetClipRectangles(display, gc, 0, 0, parent->rectWindow, 1, Unsorted);
    break;
  }

  Vector a = center * mx;
  Vector b = p1 * mx;
  Vector c = p2 * mx;
  Vector bb = pp1 * mx;
  Vector cc = pp2 * mx;

  XDRAWLINE(display, drawable, gc, (int)a[0], (int)a[1], (int)b[0], (int)b[1]);
  XDRAWLINE(display, drawable, gc, (int)a[0], (int)a[1], (int)c[0], (int)c[1]);

  if (northArrow)
    renderArrow(drawable, b, (a-b).normalize());

  if (eastArrow)
    renderArrow(drawable, c, (a-c).normalize());

  if (font) {
    Tk_FontMetrics metrics;
    Tk_GetFontMetrics(font, &metrics);

    if (northText) {
      float r1 = Tk_TextWidth(font, northText, strlen(northText))/2.;
      float r2 = metrics.linespace/2.;
      double angle = (b-a).angle();
      Vector bbb = bb * 
	Translate(r1*cos(angle),r2*sin(angle)) * 
	Translate(-r1,(metrics.ascent-metrics.descent)/2.);

      XDrawString(display, drawable, gc, (int)bbb[0], (int)bbb[1], northText, 
		  strlen(northText));
    }

    if (eastText) {
      float r1 = Tk_TextWidth(font, eastText, strlen(eastText))/2.;
      float r2 = metrics.linespace/2.;
      double angle = (c-a).angle();
      Vector ccc = cc *
	Translate(r1*cos(angle),r2*sin(angle)) * 
	Translate(-r1,(metrics.ascent-metrics.descent)/2.);

      XDrawString(display, drawable, gc, (int)ccc[0], (int)ccc[1], eastText, 
		  strlen(eastText));
    }
  }
}

// list

void Compass::list(ostream& str, CoordSystem sys, SkyFrame sky,
		 SkyFormat format, int conj, int strip)
{
  if (!strip) {
    FitsImage* ptr = parent->findFits(center);
    listPre(str, sys, sky, ptr, strip, 1);

    switch (sys) {
    case IMAGE:
    case PHYSICAL:
    case DETECTOR:
    case AMPLIFIER:
      {
	Vector v = ptr->mapFromRef(center,sys);
	str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ',' 
	    << ptr->mapLenFromRef(radius,sys) << ')';
      }
      break;
    default:
      if (ptr->hasWCS(sys)) {
	if (ptr->hasWCSEqu(sys)) {
	  switch (format) {
	  case DEGREES:
	    {
	      Vector v = ptr->mapFromRef(center,sys,sky);
	      str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
		  << ptr->mapLenFromRef(radius,sys,ARCSEC) << '"' << ')';
	    }
	    break;
	  case SEXAGESIMAL:
	    {
	      char buf[64];
	      ptr->mapFromRef(center,sys,sky,format,buf,64);
	      char ra[16];
	      char dec[16];
	      string x(buf);
	      istringstream wcs(x);
	      wcs >> ra >> dec;
	      str << type << '(' << ra << ',' << dec << ',' 
		  << ptr->mapLenFromRef(radius,sys,ARCSEC) 
		  << '"' << ')';
	    }
	    break;
	  }
	}
	else {
	  Vector v = ptr->mapFromRef(center,sys);
	  str << type << '(' << setprecision(8) << v[0] << ',' << v[1] << ','
	      << ptr->mapLenFromRef(radius,sys) << ')';
	}
      }
    }

    listPost(str, conj, strip);
  }
}

void Compass::listPost(ostream& str, int conj, int strip)
{
  if (conj)
    str << " ||";

  FitsImage* ptr = parent->findFits(center);
  str << " compass=";
  listCoordSystem(str, coordSystem, skyFrame, ptr->hasWCSEqu(coordSystem));
  str << " {" << northText << "} {" << eastText << "} " 
      << northArrow << ' ' << eastArrow;
  listProperties(str, 0);
}
